/*
	Author: Marco Costalba (C) 2005-2006

	Copyright: See COPYING file that comes with this distribution
*/
#ifndef LISTVIEW_H
#define LISTVIEW_H

#include <qobject.h>
#include <qlistview.h>
#include <qptrvector.h>
#include "common.h"

class QFont;
class Git;
class StateInfo;
class Domain;
class FileHistory;

class ListViewItem: public QListViewItem {
public:
	ListViewItem(QListView* p, ListViewItem* a, Git* g, SCRef sha,
		     const QPtrVector<QPixmap>& pm, bool e, unsigned long t, FileHistory* f);

	int getLaneType(uint pos) const;
	void setDiffTarget(bool b);
	void paintCell(QPainter* p, const QColorGroup& cg, int c, int w, int a);
	SCRef getRefs();
	void setEven(bool b) { isEvenLine = b; }
	void setHighlighted(bool b) { isHighlighted = b; }
	bool highlighted() { return isHighlighted; }

private:
	void setupData(const Rev& c);
	QPixmap* getGraph(const Rev& c);
	void addTextPixmap(SCRef text, const QColor& color, bool bold = false);
	QPixmap* doAddTextPixmap(SCRef text, const QColor& color, int col, bool bold);
	const QString timeDiff(unsigned long secs) const;
	bool changedFiles(SCRef c);

	Git* git;
	FileHistory* fh;
	const QPtrVector<QPixmap>& pms;
	unsigned long secs;
	bool populated, isEvenLine, isHighlighted, isDiffTarget;
};

class ListView: public QObject {
Q_OBJECT
public:
	ListView(Domain* d, Git* g, QListView* l, FileHistory* f);
	~ListView();
	void clear();
	const QString getSha(int id);
	void updateIdValues();
	void getSelectedItems(QStringList& selectedItems);
	bool update();
	void addNewRevs(const QValueVector<QString>& shaVec);
	void repaintAll(QFont& f);

	bool filterNextContextMenuRequest;

signals:
	void lanesContextMenuRequested(const QStringList&, const QStringList&);
	void droppedRevisions(const QString&, const QStringList&, const QString&);
	void contextMenu(const QString&, int);

public slots:
	void on_newRevsAdded(const FileHistory* fh, const QValueVector<QString>& shaVec);

protected:
	bool eventFilter(QObject* obj, QEvent* ev);

private slots:
	void on_contextMenuRequested(QListViewItem*);
	void on_currentChanged(QListViewItem* item);
	void on_mouseButtonPressed(int, QListViewItem*, const QPoint&, int);
	void on_clicked(QListViewItem*);
	void on_onItem(QListViewItem*);
	void on_contentsMoving(int, int newY);

private:
	void setupListView();
	bool filterRightButtonPressed(QMouseEvent* e);
	bool filterDropEvent(QDropEvent* e);
	bool getLaneParentsChilds(ListViewItem* item, int x, SList p, SList c);
	const QString getRefs(QListViewItem* item);
	void setHighlight(SCRef diffToSha);

	Domain* d;
	Git* git;
	QListView* lv;
	StateInfo* st;
	FileHistory* fh;
	ListViewItem* lastItem; // QListView::lastItem() is slow
	ListViewItem* diffTarget; // cannot use QGuardedPtr, not QObject inherited
	uint processedRevs;
};

#endif

