/* pyparted.c
    Python bindings to libparted.
    Copyright (C) 2000, 2001, 2002 Progeny Linux Systems, Inc.
    AUTHOR: Eric Gillespie, Jr.

    This file is part of python-parted.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License, version 2,  as
    published by the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#define _GNU_SOURCE

#include <stdio.h>

#include <Python.h>
#include <parted/parted.h>
#include <parted/constraint.h>

extern void init_parted(void);

typedef struct {
	PyObject_HEAD;
	char *name;
	void *obj;
} PyPartedObject;

static PyTypeObject PyPartedType;

static PyObject *
PyPartedObject_new(char *name, void *obj)
{
	PyPartedObject *self;

	if (!name)
		return NULL;

	if (!obj)
		return Py_BuildValue("");

	self = PyObject_NEW(PyPartedObject, &PyPartedType);
	if (!self)
		return NULL;
	self->name = strdup(name);
	self->obj = obj;

	return (PyObject *) self;
}

static void
PyPartedObject_dealloc(PyPartedObject *self)
{
	free(self->name);

	PyMem_DEL(self);
}

static int
PyPartedObject_compare(PyPartedObject *self, PyPartedObject *v)
{
	if (self->obj == v->obj)
		return 0;
	else if (self->obj > v->obj)
		return -1;
	else
		return 1;
}

static long
PyPartedObject_hash(PyPartedObject *self)
{
	return (long) self->obj;
}

static PyObject *
PyPartedObject_repr(PyPartedObject *self)
{
	char *msg;
	PyObject *r;

	asprintf(&msg, "<%s object at %p>", self->name,
		 self->obj);
	r = PyString_FromString(msg);
	free(msg);

	return r;
}

static char PyPartedType__doc__[] = "This is the type of parted objects";

static PyTypeObject PyPartedType = {
	PyObject_HEAD_INIT(&PyType_Type)
	0,
	"PartedObject",
	sizeof(PyPartedObject),
	0,
	(destructor) PyPartedObject_dealloc,
	(printfunc) 0,
	(getattrfunc) 0,
	(setattrfunc) 0,
	(cmpfunc) PyPartedObject_compare,
	(reprfunc) PyPartedObject_repr,
	0,
	0,
	0,
	(hashfunc) PyPartedObject_hash,
	(ternaryfunc) 0,
	(reprfunc) 0,
	0L, 0L, 0L, 0L,
	PyPartedType__doc__
};

static PyObject *
init(PyObject *self, PyObject *args)
{
	ped_init();
	return Py_BuildValue("");
}

static PyObject *
device_get_model(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyString_FromString(((PedDevice *)o->obj)->model);
}

static PyObject *
device_get_path(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyString_FromString(((PedDevice *)o->obj)->path);
}

static PyObject *
device_get_type(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyInt_FromLong(((PedDevice *)o->obj)->type);
}

static PyObject *
device_get_sector_size(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyInt_FromLong(((PedDevice *)o->obj)->sector_size);
}

static PyObject *
device_get_heads(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyInt_FromLong(((PedDevice *)o->obj)->heads);
}

static PyObject *
device_get_sectors(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyInt_FromLong(((PedDevice *)o->obj)->sectors);
}

static PyObject *
device_get_cylinders(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyInt_FromLong(((PedDevice *)o->obj)->cylinders);
}

static PyObject *
device_get_geom_known(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyInt_FromLong(((PedDevice *)o->obj)->geom_known);
}

static PyObject *
device_get_host(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyInt_FromLong(((PedDevice *)o->obj)->host);
}

static PyObject *
device_get_did(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyInt_FromLong(((PedDevice *)o->obj)->did);
}

static PyObject *
device_get_length(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyLong_FromLongLong(((PedDevice *)o->obj)->length);
}

static PyObject *
device_get_open_count(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyInt_FromLong(((PedDevice *)o->obj)->open_count);
}

static PyObject *
device_get_dirty(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyInt_FromLong(((PedDevice *)o->obj)->dirty);
}

static PyObject *
device_open(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyInt_FromLong(ped_device_open(o->obj));
}

static PyObject *
device_close(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyInt_FromLong(ped_device_close(o->obj));
}

static PyObject *
device_read(PyObject *self, PyObject *args)
{
	return Py_BuildValue("");
}

static PyObject *
device_write(PyObject *self, PyObject *args)
{
	return Py_BuildValue("");
}

static PyObject *
device_sync(PyObject *self, PyObject *args)
{
	return Py_BuildValue("");
}

static PyObject *
device_probe_all(PyObject *self, PyObject *args)
{
	ped_device_probe_all();

	return Py_BuildValue("");
}

static PyObject *
device_get(PyObject *self, PyObject *args)
{
	char *name;

	if (!PyArg_ParseTuple(args, "z", &name))
		return NULL;

	if (name)
		return PyPartedObject_new("PedDevice", ped_device_get(name));
	else
		return Py_BuildValue("");
}

static PyObject *
device_get_next(PyObject *self, PyObject *args)
{
	PyPartedObject *o;
	PedDevice *dev;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	if (PyObject_IsTrue((PyObject *)o))
		dev = (PedDevice *)o->obj;
	else
		dev = NULL;

	return PyPartedObject_new("PedDevice", ped_device_get_next(dev));
}

static PyObject *
disk_type_get_next(PyObject *self, PyObject *args)
{
	PyPartedObject *o;
	PedDiskType *dev;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	if (PyObject_IsTrue((PyObject *)o))
		dev = (PedDiskType *)o->obj;
	else
		dev = NULL;

	return PyPartedObject_new("PedDiskType", ped_disk_type_get_next(dev));
}

static PyObject *
disk_type_get_name(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyString_FromString(((PedDiskType *)o->obj)->name);
}

static PyObject *
disk_type_get_ops(PyObject *self, PyObject *args)
{
	return Py_BuildValue("");
}

static PyObject *
disk_type_get(PyObject *self, PyObject *args)
{
	char *name;

	if (!PyArg_ParseTuple(args, "z", &name))
		return NULL;

	if (name)
		return PyPartedObject_new("PedDiskType",
					  ped_disk_type_get(name));
	else
		return Py_BuildValue("");
}

static PyObject *
disk_get_dev(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyPartedObject_new("PedDevice", ((PedDisk *)o->obj)->dev);
}

static PyObject *
disk_get_type(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyPartedObject_new("PedDiskType", ((PedDisk *)o->obj)->type);
}

static PyObject *
disk_get_part_list(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

/* 	printf("name is %s %p\n", */
/* 	       ((PedDisk *)o->obj)->type->name, */
/* 	       ((PedDisk *)o->obj)->part_list); */

	return PyPartedObject_new("PedPartition",
				  ((PedDisk *)o->obj)->part_list);
}

static PyObject *
disk_open(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyPartedObject_new("PedDisk",
				  ped_disk_open(((PedDevice *)o->obj)));
}

static PyObject *
disk_create(PyObject *self, PyObject *args)
{
	PyPartedObject *o, *o2;

	if (!PyArg_ParseTuple(args, "OO", &o, &o2))
		return NULL;

	return PyPartedObject_new("PedDisk",
				  ped_disk_create((PedDevice *)o->obj,
						  (PedDiskType *)o2->obj));
}

static PyObject *
disk_close(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyInt_FromLong(ped_disk_close((PedDisk *)o->obj));
}

static PyObject *
disk_read(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyInt_FromLong(ped_disk_read((PedDisk *)o->obj));
}

static PyObject *
disk_write(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyInt_FromLong(ped_disk_write((PedDisk *)o->obj));
}

static PyObject *
disk_add_partition(PyObject *self, PyObject *args)
{
	PyPartedObject *o, *o2;

	if (!PyArg_ParseTuple(args, "OO", &o, &o2))
		return NULL;

	return PyInt_FromLong(ped_disk_add_partition((PedDisk *)o->obj,
						     ((PedPartition *)o2->obj),
                                                     ped_constraint_any((PedDisk *)o->obj)));
}

static PyObject *
disk_delete_partition(PyObject *self, PyObject *args)
{
	PyPartedObject *o, *o2;

	if (!PyArg_ParseTuple(args, "OO", &o, &o2))
		return NULL;

	return PyInt_FromLong(ped_disk_delete_partition((PedDisk *)o->obj,
							((PedPartition *)o2->obj)));
}

static PyObject *
disk_delete_all(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyInt_FromLong(ped_disk_delete_all((PedDisk *)o->obj));
}

static PyObject *
disk_set_partition_geom(PyObject *self, PyObject *args)
{
	PyPartedObject *o, *o2;
	PedSector start, end;

	if (!PyArg_ParseTuple(args, "OOLL", &o, &o2, &start, &end))
		return NULL;

	return PyInt_FromLong(ped_disk_set_partition_geom((PedDisk *)o->obj,
							  (PedPartition *)o2->obj,
                                                          ped_constraint_any((PedDisk *)o->obj),
							  start, end));
}

static PyObject *
disk_maximize_partition(PyObject *self, PyObject *args)
{
	PyPartedObject *o, *o2;

	if (!PyArg_ParseTuple(args, "OO", &o, &o2))
		return NULL;

	return PyInt_FromLong(ped_disk_maximize_partition((PedDisk *)o->obj,
							  (PedPartition *)o2->obj,
                                                          ped_constraint_any((PedDisk *)o->obj)));
}

static PyObject *
disk_get_max_partition_geometry(PyObject *self, PyObject *args)
{
	PyPartedObject *o, *o2;

	// That's odd. Accoring to the API doc, this function
	// exists, but it isn't actually in the library.

	return Py_BuildValue("");

	if (!PyArg_ParseTuple(args, "OO", &o, &o2))
		return NULL;

#if 0
	return PyPartedObject_new("PedGeometry",
				  ped_disk_get_max_partition_geometry((PedDisk *)o->obj,
								      (PedPartition *)o2->obj));
#endif
}

static PyObject *
disk_minimize_extended_partition(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyInt_FromLong(ped_disk_minimize_extended_partition((PedDisk *)o->obj));
}

static PyObject *
disk_next_partition(PyObject *self, PyObject *args)
{
	PyPartedObject *o, *o2;

	o2 = NULL;
	if (!PyArg_ParseTuple(args, "O|O", &o, &o2))
		return NULL;

/* 	printf("o = %s\n", */
/* 	       ((PedDisk *)o->obj)->type->name); */

	if (o2 == NULL)
	  return 
	    PyPartedObject_new("PedPartition",
			       ped_disk_next_partition((PedDisk *)o->obj,
						       NULL));
	else
	  return 
	    PyPartedObject_new("PedPartition",
			       ped_disk_next_partition((PedDisk *)o->obj,
						       (PedPartition *)o2->obj));
}

static PyObject *
disk_get_partition(PyObject *self, PyObject *args)
{
	PyPartedObject *o;
	int num;

	if (!PyArg_ParseTuple(args, "Oi", &o, &num))
		return NULL;

	return PyPartedObject_new("PedPartition",
				  ped_disk_get_partition((PedDisk *)o->obj,
							 num));
}

static PyObject *
disk_get_partition_by_sector(PyObject *self, PyObject *args)
{
	PyPartedObject *o;
	PedSector sect;

	if (!PyArg_ParseTuple(args, "Ol", &o, &sect))
		return NULL;

	return PyPartedObject_new("PedPartition",
				  ped_disk_get_partition_by_sector((PedDisk *)o->obj,
								   sect));
}

static PyObject *
disk_get_extended_partition(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyPartedObject_new("PedPartition",
				  ped_disk_extended_partition((PedDisk *)o->obj));
}

static PyObject *
geometry_get_disk(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyPartedObject_new("PedDisk",
				  ((PedGeometry *)o->obj)->disk);
}

static PyObject *
geometry_get_start(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyLong_FromLongLong(((PedGeometry *)o->obj)->start);
}

static PyObject *
geometry_get_length(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyLong_FromLongLong(((PedGeometry *)o->obj)->length);
}

static PyObject *
geometry_get_end(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyLong_FromLongLong(((PedGeometry *)o->obj)->end);
}

static PyObject *
partition_new(PyObject *self, PyObject *args)
{
	PyPartedObject *o, *fs_type;
	PedPartitionType type;
	long start, end;

	end = 0;
	if (!PyArg_ParseTuple(args, "OiOll", &o, &type, &fs_type,
			      &start, &end))
		return NULL;

/* 	printf("o = %s, type = %d, fs_type = %s, start = %ld, end = %ld\n", */
/* 	       ((PedDisk *)o->obj)->dev->path, type,  */
/* 	       fs_type->name, start, end); */

	return PyPartedObject_new("PedPartition",
				  ped_partition_new((PedDisk *)o->obj, type,
						    (PedFileSystemType *)fs_type->obj,
						    (PedSector)start, 
						    (PedSector)end));
}

static PyObject *
partition_get_prev(PyObject *self, PyObject *args)
{
	return Py_BuildValue("");
}

static PyObject *
partition_get_next(PyObject *self, PyObject *args)
{
	PyPartedObject *o;
	PedPartition *dev;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	dev = (PedPartition *)o->obj;
	return PyPartedObject_new("PedPartition", dev->next);
}

static PyObject *
partition_get_geom(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyPartedObject_new("PedGeometry",
				  &((PedPartition *)o->obj)->geom);
}

static PyObject *
partition_get_num(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyInt_FromLong(((PedPartition *)o->obj)->num);
}

static PyObject *
partition_get_hidden(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

        return PyInt_FromLong(ped_partition_get_flag((PedPartition *)o->obj, PED_PARTITION_HIDDEN));
}

static PyObject *
partition_get_type(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyInt_FromLong(((PedPartition *)o->obj)->type);
}

static PyObject *
partition_get_part_list(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyPartedObject_new("PedPartition",
				  ((PedPartition *)o->obj)->part_list);
}

static PyObject *
partition_get_bootable(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

        return Py_BuildValue("b", ped_partition_get_flag((PedPartition *)o->obj, PED_PARTITION_BOOT));
}

static PyObject *
partition_set_bootable(PyObject *self, PyObject *args)
{
	PyPartedObject *o;
	int i;

	if (!PyArg_ParseTuple(args, "Ob", &o, &i))
		return NULL;

        ped_partition_set_flag((PedPartition *)o->obj, PED_PARTITION_BOOT, 1);

	return Py_BuildValue("");
}

static PyObject *
partition_get_fs_type(PyObject *self, PyObject *args)
{
	PyPartedObject *o;
	PedFileSystemType *fs_type;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	fs_type = (PedFileSystemType *) ((PedPartition *)o->obj)->fs_type;
	return PyPartedObject_new("PedFileSystemType", fs_type);
}

static PyObject *
partition_set_fs_type(PyObject *self, PyObject *args)
{
	PyPartedObject *o, *o2;

	if (!PyArg_ParseTuple(args, "OO", &o, &o2))
		return NULL;

	((PedPartition *)o->obj)->fs_type = (PedFileSystemType *)o2->obj;

	return Py_BuildValue("");
}

static PyObject *
partition_is_active(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyInt_FromLong(ped_partition_is_active((PedPartition *)o->obj));
}

static PyObject *
partition_is_busy(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyInt_FromLong(ped_partition_is_busy((PedPartition *)o->obj));
}

/* HACKHACKHACK */

/*
 * Due to parted brain damage, we have to peek into an internal
 * structure in order to mark a partition as an EFI boot partition.
 * This DosPartitionData structure is copied from
 * include/parted/disk_dos.h and EFI_PMBR_OSTYPE_EFI is copied from
 * include/parted/disk_gtp.h.
 */

#define EFI_PMBR_OSTYPE_EFI 0xEF
typedef struct {
	unsigned char	system;
	int		boot;
	int		hidden;
	int		raid;
	int		lvm;
	int		lba;
} DosPartitionData;

static PyObject *
partition_is_efi(PyObject *self, PyObject *args)
{
    PyPartedObject *o;
    PedPartition *part;
    DosPartitionData *dos_data;

    if (!PyArg_ParseTuple(args, "O", &o)) {
        return NULL;
    }

    part = o->obj;
    dos_data = part->disk_specific;
    PED_ASSERT(dos_data != NULL, return PyInt_FromLong(0));

    return PyInt_FromLong(dos_data->system == EFI_PMBR_OSTYPE_EFI);
}

static PyObject *
partition_mark_efi(PyObject *self, PyObject *args)
{
    PyPartedObject *o;
    PedPartition *part;
    DosPartitionData *dos_data;

    if (!PyArg_ParseTuple(args, "O", &o)) {
        return NULL;
    }

    part = o->obj;
    dos_data = part->disk_specific;
    PED_ASSERT(dos_data != NULL, return PyInt_FromLong(0));

    dos_data->system = EFI_PMBR_OSTYPE_EFI;

    return PyInt_FromLong(1);
}

static PyObject *
partition_set_system(PyObject *self, PyObject *args)
{
	PyPartedObject *o, *o2;

	if (!PyArg_ParseTuple(args, "OO", &o, &o2))
		return NULL;

	return PyInt_FromLong(ped_partition_set_system((PedPartition *)o->obj,
						       (PedFileSystemType *)o2->obj));
}

static PyObject *
partition_destroy(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	ped_partition_destroy((PedPartition *)o->obj);

	return Py_BuildValue("");
}

static PyObject *
file_system_type_get_next(PyObject *self, PyObject *args)
{
	PyPartedObject *o;
	PedFileSystemType *fs_type;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	if (PyObject_IsTrue((PyObject *)o))
		fs_type = (PedFileSystemType *)o->obj;
	else
		fs_type = NULL;

	return PyPartedObject_new("PedFileSystemType",
				  ped_file_system_type_get_next(fs_type));
}

static PyObject *
file_system_type_get_name(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyString_FromString(((PedFileSystemType *)o->obj)->name);
}

static PyObject *
file_system_type_get_ops(PyObject *self, PyObject *args)
{
	return Py_BuildValue("");
}

static PyObject *
file_system_type_get(PyObject *self, PyObject *args)
{
	char *name;

	if (!PyArg_ParseTuple(args, "s", &name))
		return NULL;

	return PyPartedObject_new("PedFileSystemType",
				  ped_file_system_type_get(name));
}

static PyObject *
file_system_create(PyObject *self, PyObject *args)
{
	PyPartedObject *o, *o2;

	if (!PyArg_ParseTuple(args, "OO", &o, &o2))
		return NULL;

	return PyPartedObject_new("PedFileSystem",
				  ped_file_system_create((PedGeometry *)o->obj,
							 (PedFileSystemType *)o2->obj));
}

static PyObject *
file_system_get_type(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyPartedObject_new("PedFileSystemType",
				  ((PedFileSystem *)o->obj)->type);
}

static PyObject *
file_system_get_geom(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyPartedObject_new("PedGeometry",
				  ((PedFileSystem *)o->obj)->geom);
}

static PyObject *
file_system_get_resize_constraint(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyPartedObject_new("PedConstraint",
				  ped_file_system_get_resize_constraint((PedFileSystem *)o->obj));
}

static PyObject *
file_system_close(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyInt_FromLong(ped_file_system_close((PedFileSystem *)o->obj));
}

static PyObject *
file_system_check(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyInt_FromLong(ped_file_system_check((PedFileSystem *)o->obj));
}

static PyObject *
file_system_copy(PyObject *self, PyObject *args)
{
	PyPartedObject *o, *o2;

	if (!PyArg_ParseTuple(args, "OO", &o, &o2))
		return NULL;

	return PyInt_FromLong(ped_file_system_copy((PedFileSystem *)o->obj,
						   (PedGeometry *)o2->obj));
}

static PyObject *
file_system_resize(PyObject *self, PyObject *args)
{
	PyPartedObject *o, *o2;

	if (!PyArg_ParseTuple(args, "OO", &o, &o2))
		return NULL;

	return PyInt_FromLong(ped_file_system_resize((PedFileSystem *)o->obj,
						     (PedGeometry *)o2->obj));
}

static PyObject *
file_system_probe(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyPartedObject_new("PedFileSystemType",
				  ped_file_system_probe((PedGeometry *)o->obj));
}

static PyObject *
file_system_open(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyPartedObject_new("PedFileSystem",
				  ped_file_system_open((PedGeometry *)o->obj));
}

static PyObject *
alignment_new(PyObject *self, PyObject *args)
{
	PedSector offset, grain_size;

	if (!PyArg_ParseTuple(args, "LL", &offset, &grain_size))
		return NULL;

	return PyPartedObject_new("PedAlignment",
				  ped_alignment_new(offset, grain_size));
}

static PyObject *
alignment_get_offset(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyLong_FromLongLong(((PedAlignment *)o->obj)->offset);
}

static PyObject *
alignment_get_grain_size(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyLong_FromLongLong(((PedAlignment *)o->obj)->grain_size);

}

static PyObject *
constraint_new(PyObject *self, PyObject *args)
{
	PyPartedObject *o, *o2, *o3, *o4;
	PedSector min_size;

	if (!PyArg_ParseTuple(args, "OOOOL", &o, &o2, &o3, &o4, &min_size))
		return NULL;

	return PyPartedObject_new("PedConstraint",
				  ped_constraint_new((PedAlignment *)o->obj,
                                                     (PedAlignment *)o2->obj, 
                                                     (PedGeometry *)o3->obj,
                                                     (PedGeometry *)o4->obj,
                                                     min_size));
}

static PyObject *
constraint_get_start_align(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyPartedObject_new("PedAlignment",
                                  ((PedConstraint *)o->obj)->start_align);
}

static PyObject *
constraint_get_end_align(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyPartedObject_new("PedAlignment",
                                  ((PedConstraint *)o->obj)->end_align);
}

static PyObject *
constraint_get_start_range(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyPartedObject_new("PedGeometry",
                                  ((PedConstraint *)o->obj)->start_range);
}

static PyObject *
constraint_get_end_range(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyPartedObject_new("PedGeometry",
                                  ((PedConstraint *)o->obj)->end_range);
}

static PyObject *
constraint_get_min_size(PyObject *self, PyObject *args)
{
	PyPartedObject *o;

	if (!PyArg_ParseTuple(args, "O", &o))
		return NULL;

	return PyLong_FromLongLong(((PedConstraint *)o->obj)->min_size);
}

static PyObject *py_exception_handler = NULL;

static PedExceptionOption
ped_exception_handler(PedException *exception)
{
	PyObject *r;

	if (py_exception_handler) {
		r = PyObject_CallFunction(py_exception_handler, "sii",
					  exception->message, exception->type,
					  exception->options);

		if (!r) {
			PyErr_Print();
			PyErr_Clear();
			return PED_EXCEPTION_UNHANDLED;
		}

		return PyInt_AsLong(r);
	} else
		return PED_EXCEPTION_UNHANDLED;
}

static PyObject *
exception_set_handler(PyObject *self, PyObject *args)
{
	if (!PyArg_ParseTuple(args, "O", &py_exception_handler))
		return NULL;

	return Py_BuildValue("");
}

static PyObject *
done(PyObject *self, PyObject *args)
{
	ped_done();
	return Py_BuildValue("");
}

static PyMethodDef parted_methods[] = {
	{ "init", init, METH_VARARGS, NULL },
	{ "device_get_next", device_get_next, METH_VARARGS, NULL },
        { "device_get_model", device_get_model, METH_VARARGS, NULL },
	{ "device_get_path", device_get_path, METH_VARARGS, NULL },
	{ "device_get_type", device_get_type, METH_VARARGS, NULL },
	{ "device_get_sector_size", device_get_sector_size, METH_VARARGS, NULL },
	{ "device_get_heads", device_get_heads, METH_VARARGS, NULL },
	{ "device_get_sectors", device_get_sectors, METH_VARARGS, NULL },
	{ "device_get_cylinders", device_get_cylinders, METH_VARARGS, NULL },
	{ "device_get_geom_known", device_get_geom_known, METH_VARARGS, NULL },
	{ "device_get_host", device_get_host, METH_VARARGS, NULL },
	{ "device_get_did", device_get_did, METH_VARARGS, NULL },
	{ "device_get_length", device_get_length, METH_VARARGS, NULL },
	{ "device_get_open_count", device_get_open_count, METH_VARARGS, NULL },
	{ "device_get_dirty", device_get_dirty, METH_VARARGS, NULL },
	{ "device_open", device_open, METH_VARARGS, NULL },
	{ "device_close", device_close, METH_VARARGS, NULL },
	{ "device_read", device_read, METH_VARARGS, NULL },
	{ "device_write", device_write, METH_VARARGS, NULL },
	{ "device_sync", device_sync, METH_VARARGS, NULL },
	{ "device_probe_all", device_probe_all, METH_VARARGS, NULL },
	{ "device_get", device_get, METH_VARARGS, NULL },
	{ "device_get_next", device_get_next, METH_VARARGS, NULL },
	{ "device_get_next", device_get_next, METH_VARARGS, NULL },
	{ "disk_type_get_next", disk_type_get_next, METH_VARARGS, NULL },
	{ "disk_type_get_name", disk_type_get_name, METH_VARARGS, NULL },
	{ "disk_type_get_ops", disk_type_get_ops, METH_VARARGS, NULL },
	{ "disk_type_get", disk_type_get, METH_VARARGS, NULL },
	{ "disk_type_get_next", disk_type_get_next, METH_VARARGS, NULL },
	{ "disk_type_get_next", disk_type_get_next, METH_VARARGS, NULL },
	{ "disk_get_dev", disk_get_dev, METH_VARARGS, NULL },
	{ "disk_get_type", disk_get_type, METH_VARARGS, NULL },
	{ "disk_get_part_list", disk_get_part_list, METH_VARARGS, NULL },
	{ "disk_open", disk_open, METH_VARARGS, NULL },
	{ "disk_create", disk_create, METH_VARARGS, NULL },
	{ "disk_close", disk_close, METH_VARARGS, NULL },
	{ "disk_read", disk_read, METH_VARARGS, NULL },
	{ "disk_write", disk_write, METH_VARARGS, NULL },
	{ "disk_add_partition", disk_add_partition, METH_VARARGS, NULL },
	{ "disk_delete_partition", disk_delete_partition, METH_VARARGS, NULL },
	{ "disk_delete_all", disk_delete_all, METH_VARARGS, NULL },
	{ "disk_set_partition_geom", disk_set_partition_geom, METH_VARARGS, NULL },
	{ "disk_maximize_partition", disk_maximize_partition, METH_VARARGS, NULL },
	{ "disk_get_max_partition_geometry", disk_get_max_partition_geometry, METH_VARARGS, NULL },
	{ "disk_minimize_extended_partition", disk_minimize_extended_partition, METH_VARARGS, NULL },
	{ "disk_next_partition", disk_next_partition, METH_VARARGS, NULL },
	{ "disk_get_partition", disk_get_partition, METH_VARARGS, NULL },
	{ "disk_get_partition_by_sector", disk_get_partition_by_sector, METH_VARARGS, NULL },
	{ "disk_get_extended_partition", disk_get_extended_partition, METH_VARARGS, NULL },
	{ "geometry_get_disk", geometry_get_disk, METH_VARARGS, NULL },
	{ "geometry_get_start", geometry_get_start, METH_VARARGS, NULL },
	{ "geometry_get_length", geometry_get_length, METH_VARARGS, NULL },
	{ "geometry_get_end", geometry_get_end, METH_VARARGS, NULL },
	{ "partition_new", partition_new, METH_VARARGS, NULL },
	{ "partition_get_prev", partition_get_prev, METH_VARARGS, NULL },
	{ "partition_get_next", partition_get_next, METH_VARARGS, NULL },
	{ "partition_get_geom", partition_get_geom, METH_VARARGS, NULL },
	{ "partition_get_num", partition_get_num, METH_VARARGS, NULL },
	{ "partition_get_hidden", partition_get_hidden, METH_VARARGS, NULL },
	{ "partition_get_type", partition_get_type, METH_VARARGS, NULL },
	{ "partition_get_part_list", partition_get_part_list, METH_VARARGS, NULL },
	{ "partition_get_bootable", partition_get_bootable, METH_VARARGS, NULL },
	{ "partition_set_bootable", partition_set_bootable, METH_VARARGS, NULL },
	{ "partition_get_fs_type", partition_get_fs_type, METH_VARARGS, NULL },
	{ "partition_set_fs_type", partition_set_fs_type, METH_VARARGS, NULL },
	{ "partition_is_active", partition_is_active, METH_VARARGS, NULL },
	{ "partition_is_busy", partition_is_busy, METH_VARARGS, NULL },
        { "partition_is_efi", partition_is_efi, METH_VARARGS, NULL },
        { "partition_mark_efi", partition_mark_efi, METH_VARARGS, NULL },
	{ "partition_set_system", partition_set_system, METH_VARARGS, NULL },
	{ "partition_destroy", partition_destroy, METH_VARARGS, NULL },
	{ "file_system_type_get_next", file_system_type_get_next, METH_VARARGS, NULL },
	{ "file_system_type_get_name", file_system_type_get_name, METH_VARARGS, NULL },
	{ "file_system_type_get_ops", file_system_type_get_ops, METH_VARARGS, NULL },
	{ "file_system_type_get", file_system_type_get, METH_VARARGS, NULL },
	{ "file_system_create", file_system_create, METH_VARARGS, NULL },
	{ "file_system_get_type", file_system_get_type, METH_VARARGS, NULL },
	{ "file_system_get_geom", file_system_get_geom, METH_VARARGS, NULL },
	{ "file_system_close", file_system_close, METH_VARARGS, NULL },
	{ "file_system_check", file_system_check, METH_VARARGS, NULL },
	{ "file_system_copy", file_system_copy, METH_VARARGS, NULL },
	{ "file_system_resize", file_system_resize, METH_VARARGS, NULL },
	{ "file_system_probe", file_system_probe, METH_VARARGS, NULL },
	{ "file_system_open", file_system_open, METH_VARARGS, NULL },
	{ "file_system_get_resize_constraint", file_system_get_resize_constraint, METH_VARARGS, NULL },
	{ "alignment_new", alignment_new, METH_VARARGS, NULL },
	{ "alignment_get_offset", alignment_get_offset, METH_VARARGS, NULL },
	{ "alignment_get_grain_size", alignment_get_grain_size, METH_VARARGS, NULL },
	{ "constraint_new", constraint_new, METH_VARARGS, NULL },
	{ "constraint_get_start_align", constraint_get_start_align, METH_VARARGS, NULL },
	{ "constraint_get_end_align", constraint_get_end_align, METH_VARARGS, NULL },
	{ "constraint_get_start_range", constraint_get_start_range, METH_VARARGS, NULL },
	{ "constraint_get_end_range", constraint_get_end_range, METH_VARARGS, NULL },
	{ "constraint_get_min_size", constraint_get_min_size, METH_VARARGS, NULL },
	{ "exception_set_handler", exception_set_handler, METH_VARARGS, NULL },
	{ "done", done, METH_VARARGS, NULL },

	{ NULL, 0, 0, NULL }
};

void
init_parted(void)
{
	PyObject *m;

	m = Py_InitModule("_parted", parted_methods);

	ped_exception_set_handler(ped_exception_handler);

	if (PyErr_Occurred())
		Py_FatalError("Can't initialize module _parted");
}
