//
// DynamicAnyHolder.h
//
// $Id: //poco/1.3/Foundation/include/Poco/DynamicAnyHolder.h#2 $
//
// Library: Foundation
// Package: Core
// Module:  DynamicAnyHolder
//
// Definition of the DynamicAnyHolder class.
//
// Copyright (c) 2007, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef Foundation_DynamicAnyHolder_INCLUDED
#define Foundation_DynamicAnyHolder_INCLUDED


#include "Poco/Foundation.h"
#include "Poco/NumberFormatter.h"
#include "Poco/NumberParser.h"
#include "Poco/String.h"
#include "Poco/Exception.h"
#include <typeinfo>
#undef min
#undef max
#include <limits>


namespace Poco {


class Foundation_API DynamicAnyHolder
	/// Interface for a data holder used by the DynamicAny class. 
	/// Provides methods to convert between data types.
	/// Only data types for which a convert method exists are supported, which are
	/// all C++ built-in types with addition of std::string.
{
public:
	DynamicAnyHolder();
		/// Creates the DynamicAnyHolder.

	virtual ~DynamicAnyHolder();
		/// Destroys the DynamicAnyHolder.

	virtual DynamicAnyHolder* clone() const = 0;
		/// Deep-copies the DynamicAnyHolder.
		
	virtual const std::type_info& type() const = 0;
		/// Returns the type information of the stored content.

	virtual void convert(Int8& val) const = 0;
	virtual void convert(Int16& val) const = 0;
	virtual void convert(Int32& val) const = 0;
	virtual void convert(Int64& val) const = 0;
	virtual void convert(UInt8& val) const = 0;
	virtual void convert(UInt16& val) const = 0;
	virtual void convert(UInt32& val) const = 0;
	virtual void convert(UInt64& val) const = 0;

#ifndef POCO_LONG_IS_64_BIT
	void convert(long& val) const;
	void convert(unsigned long& val) const;
#endif

	virtual void convert(bool& val) const = 0;
	virtual void convert(float& val) const = 0;
	virtual void convert(double& val) const = 0;
	virtual void convert(char& val) const = 0;
	virtual void convert(std::string& val) const = 0;

protected:
	template <typename F, typename T>
	void convertToSmaller(const F& from, T& to) const
		/// This function is meant to convert signed integral values from
		/// larger to smaller type. It checks the upper and lower bound and
		/// if from value is within limits of type T (i.e. check calls do not throw), 
		/// it is converted.
	{
		poco_static_assert (std::numeric_limits<F>::is_specialized);
		poco_static_assert (std::numeric_limits<T>::is_specialized);
		poco_static_assert (std::numeric_limits<F>::is_signed);
		poco_static_assert (std::numeric_limits<T>::is_signed);

		checkUpperLimit(from, to); 
		checkLowerLimit(from, to);
		to = static_cast<T>(from);
	}

	template <typename F, typename T>
	void convertToSmallerUnsigned(const F& from, T& to) const
		/// This function is meant for converting unsigned integral data types,
		/// from larger to smaller type. Since lower limit is always 0 for unigned types, 
		/// only the upper limit is checked, thus saving some cycles compared to the signed 
		/// version of the function. If the value to be converted is smaller than
		/// the maximum value for the target type, the conversion is performed.
	{
		poco_static_assert (std::numeric_limits<F>::is_specialized);
		poco_static_assert (std::numeric_limits<T>::is_specialized);
		poco_static_assert (!std::numeric_limits<F>::is_signed);
		poco_static_assert (!std::numeric_limits<T>::is_signed);

		checkUpperLimit(from, to); 
		to = static_cast<T>(from);
	}

	template <typename F, typename T>
	void convertSignedToUnsigned(const F& from, T& to) const
		/// This function is meant for converting signed integral data types to
		/// unsigned data types. Negative values can not be converted and if one is 
		/// encountered, RangeException is thrown. 
		/// If uper limit is within the target data type limits, the converiosn is performed.
	{
		poco_static_assert (std::numeric_limits<F>::is_specialized);
		poco_static_assert (std::numeric_limits<T>::is_specialized);
		poco_static_assert (std::numeric_limits<F>::is_signed);
		poco_static_assert (!std::numeric_limits<T>::is_signed);

		if (from < 0)
			throw RangeException("Value too small.");
		checkUpperLimit(from, to); 
		to = static_cast<T>(from);
	}

	template <typename F, typename T>
	void convertUnsignedToSigned(const F& from, T& to) const
		/// This function is meant for converting unsigned integral data types to
		/// unsigned data types. Negative values can not be converted and if one is 
		/// encountered, RangeException is thrown. 
		/// If uper limit is within the target data type limits, the converiosn is performed.
	{
		poco_static_assert (std::numeric_limits<F>::is_specialized);
		poco_static_assert (std::numeric_limits<T>::is_specialized);
		poco_static_assert (!std::numeric_limits<F>::is_signed);
		poco_static_assert (std::numeric_limits<T>::is_signed);

		checkUpperLimit(from, to); 
		to = static_cast<T>(from);
	}

private:
	template <typename F, typename T>
	void checkUpperLimit(const F& from, T& to) const
	{
		if (from > std::numeric_limits<T>::max()) 
			throw RangeException("Value too large.");
	}

	template <typename F, typename T>
	void checkLowerLimit(const F& from, T& to) const
	{
		if (from < std::numeric_limits<T>::min()) 
			throw RangeException("Value too small.");
	}
};


//
// inlines
//
#ifndef POCO_LONG_IS_64_BIT
inline void DynamicAnyHolder::convert(long& val) const
{
	Int32 tmp;
	convert(tmp);
	val = tmp;
}


inline void DynamicAnyHolder::convert(unsigned long& val) const
{
	UInt32 tmp;
	convert(tmp);
	val = tmp;
}
#endif


template <typename T>
class DynamicAnyHolderImpl: public DynamicAnyHolder
	/// Template based implementation of a DynamicAnyHolder. 
	/// Conversion work happens in the template specializations of this class.
	///
	/// DynamicAny can be used for any type for which a specialization for
	/// DynamicAnyHolderImpl is available.
	///
	/// DynamicAnyHolderImpl throws following exceptions:
	///		NotImplementedException (if the specialization for a type does not exist)
	///		RangeException (if an attempt is made to assign a numeric value outside of the target min/max limits
	///		SyntaxException (if an attempt is made to convert a string containing non-numeric characters to number)
	///
	/// All specializations must additionally implement a public member function:
	///     const T& value() const
	/// returning a const reference to the actual stored value.
{
public:
	DynamicAnyHolderImpl()
	{
	}

	~DynamicAnyHolderImpl()
	{
	}
	
	const std::type_info& type() const
	{
		return typeid(T);
	}

	void convert(Int8&) const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}

	void convert(Int16&) const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}

	void convert(Int32&) const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}

	void convert(Int64&) const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}
	
	void convert(UInt8&) const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}

	void convert(UInt16&) const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}

	void convert(UInt32&) const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}

	void convert(UInt64&) const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}

	void convert(bool& val) const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}

	void convert(float& val) const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}

	void convert(double& val) const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}

	void convert(char& val) const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}

	void convert(std::string&) const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}

	DynamicAnyHolder* clone() const
	{
		throw NotImplementedException("No DynamicAnyHolder specialization for type", typeid(T).name());
	}
};


template <>
class DynamicAnyHolderImpl<Int8>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(Int8 val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}
	
	const std::type_info& type() const
	{
		return typeid(Int8);
	}

	void convert(Int8& val) const
	{
		val = _val;
	}

	void convert(Int16& val) const
	{
		val = _val;
	}
	
	void convert(Int32& val) const
	{
		val = _val;
	}

	void convert(Int64& val) const
	{
		val = _val;
	}

	void convert(UInt8& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(UInt16& val) const
	{
		convertSignedToUnsigned(_val, val);
	}
	
	void convert(UInt32& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(UInt64& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(bool& val) const
	{
		val = (_val != 0);
	}

	void convert(float& val) const
	{
		val = static_cast<float>(_val);
	}

	void convert(double& val) const
	{
		val = static_cast<double>(_val);
	}

	void convert(char& val) const
	{
		val = static_cast<char>(_val);
	}

	void convert(std::string& val) const
	{
		val = NumberFormatter::format(_val);
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}
	
	const Int8& value() const
	{
		return _val;
	}

private:
	Int8 _val;
};


template <>
class DynamicAnyHolderImpl<Int16>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(Int16 val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}
	
	const std::type_info& type() const
	{
		return typeid(Int16);
	}

	void convert(Int8& val) const
	{
		convertToSmaller(_val, val);
	}

	void convert(Int16& val) const
	{
		val = _val;
	}
	
	void convert(Int32& val) const
	{
		val = _val;
	}

	void convert(Int64& val) const
	{
		val = _val;
	}

	void convert(UInt8& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(UInt16& val) const
	{
		convertSignedToUnsigned(_val, val);
	}
	
	void convert(UInt32& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(UInt64& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(bool& val) const
	{
		val = (_val != 0);
	}

	void convert(float& val) const
	{
		val = static_cast<float>(_val);
	}

	void convert(double& val) const
	{
		val = static_cast<double>(_val);
	}

	void convert(char& val) const
	{
		UInt8 tmp;
		convert(tmp);
		val = static_cast<char>(tmp);
	}

	void convert(std::string& val) const
	{
		val = NumberFormatter::format(_val);
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}

	const Int16& value() const
	{
		return _val;
	}

private:
	Int16 _val;
};


template <>
class DynamicAnyHolderImpl<Int32>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(Int32 val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}

	const std::type_info& type() const
	{
		return typeid(Int32);
	}

	void convert(Int8& val) const
	{
		convertToSmaller(_val, val);
	}

	void convert(Int16& val) const
	{
		convertToSmaller(_val, val);
	}
	
	void convert(Int32& val) const
	{
		val = _val;
	}

	void convert(Int64& val) const
	{
		val = _val;
	}

	void convert(UInt8& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(UInt16& val) const
	{
		convertSignedToUnsigned(_val, val);
	}
	
	void convert(UInt32& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(UInt64& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(bool& val) const
	{
		val = (_val != 0);
	}

	void convert(float& val) const
	{
		val = static_cast<float>(_val);
	}

	void convert(double& val) const
	{
		val = static_cast<double>(_val);
	}

	void convert(char& val) const
	{
		UInt8 tmp;
		convert(tmp);
		val = static_cast<char>(tmp);
	}

	void convert(std::string& val) const
	{
		val = NumberFormatter::format(_val);
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}

	const Int32& value() const
	{
		return _val;
	}

private:
	Int32 _val;
};


template <>
class DynamicAnyHolderImpl<Int64>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(Int64 val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}

	const std::type_info& type() const
	{
		return typeid(Int64);
	}

	void convert(Int8& val) const
	{
		convertToSmaller(_val, val);
	}

	void convert(Int16& val) const
	{
		convertToSmaller(_val, val);
	}
	
	void convert(Int32& val) const
	{
		convertToSmaller(_val, val);
	}

	void convert(Int64& val) const
	{
		val = _val;
	}

	void convert(UInt8& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(UInt16& val) const
	{
		convertSignedToUnsigned(_val, val);
	}
	
	void convert(UInt32& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(UInt64& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(bool& val) const
	{
		val = (_val != 0);
	}

	void convert(float& val) const
	{
		val = static_cast<float>(_val);
	}

	void convert(double& val) const
	{
		val = static_cast<double>(_val);
	}

	void convert(char& val) const
	{
		UInt8 tmp;
		convert(tmp);
		val = static_cast<char>(tmp);
	}

	void convert(std::string& val) const
	{
		val = NumberFormatter::format(_val);
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}

	const Int64& value() const
	{
		return _val;
	}

private:
	Int64 _val;
};


template <>
class DynamicAnyHolderImpl<UInt8>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(UInt8 val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}
	
	const std::type_info& type() const
	{
		return typeid(UInt8);
	}

	void convert(Int8& val) const
	{
		convertUnsignedToSigned(_val, val);
	}

	void convert(Int16& val) const
	{
		convertUnsignedToSigned(_val, val);
	}
	
	void convert(Int32& val) const
	{
		val = static_cast<Int32>(_val);
	}

	void convert(Int64& val) const
	{
		val = static_cast<Int64>(_val);
	}

	void convert(UInt8& val) const
	{
		val = _val;
	}

	void convert(UInt16& val) const
	{
		val = _val;
	}
	
	void convert(UInt32& val) const
	{
		val = _val;
	}

	void convert(UInt64& val) const
	{
		val = _val;
	}

	void convert(bool& val) const
	{
		val = (_val != 0);
	}

	void convert(float& val) const
	{
		val = static_cast<float>(_val);
	}

	void convert(double& val) const
	{
		val = static_cast<double>(_val);
	}

	void convert(char& val) const
	{
		UInt8 tmp;
		convert(tmp);
		val = static_cast<char>(tmp);
	}

	void convert(std::string& val) const
	{
		val = NumberFormatter::format(_val);
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}

	const UInt8& value() const
	{
		return _val;
	}

private:
	UInt8 _val;
};


template <>
class DynamicAnyHolderImpl<UInt16>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(UInt16 val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}

	const std::type_info& type() const
	{
		return typeid(UInt16);
	}

	void convert(Int8& val) const
	{
		convertUnsignedToSigned(_val, val);
	}

	void convert(Int16& val) const
	{
		convertUnsignedToSigned(_val, val);
	}
	
	void convert(Int32& val) const
	{
		convertUnsignedToSigned(_val, val);
	}

	void convert(Int64& val) const
	{
		val = static_cast<Int64>(_val);
	}

	void convert(UInt8& val) const
	{
		convertToSmallerUnsigned(_val, val);
	}

	void convert(UInt16& val) const
	{
		val = _val;
	}
	
	void convert(UInt32& val) const
	{
		val = _val;
	}

	void convert(UInt64& val) const
	{
		val = _val;
	}

	void convert(bool& val) const
	{
		val = (_val != 0);
	}

	void convert(float& val) const
	{
		val = static_cast<float>(_val);
	}

	void convert(double& val) const
	{
		val = static_cast<double>(_val);
	}

	void convert(char& val) const
	{
		UInt8 tmp;
		convert(tmp);
		val = static_cast<char>(tmp);
	}

	void convert(std::string& val) const
	{
		val = NumberFormatter::format(_val);
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}

	const UInt16& value() const
	{
		return _val;
	}

private:
	UInt16 _val;
};


template <>
class DynamicAnyHolderImpl<UInt32>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(UInt32 val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}

	const std::type_info& type() const
	{
		return typeid(UInt32);
	}

	void convert(Int8& val) const
	{
		convertUnsignedToSigned(_val, val);
	}

	void convert(Int16& val) const
	{
		convertUnsignedToSigned(_val, val);
	}
	
	void convert(Int32& val) const
	{
		convertUnsignedToSigned(_val, val);
	}

	void convert(Int64& val) const
	{
		convertUnsignedToSigned(_val, val);
	}

	void convert(UInt8& val) const
	{
		convertToSmallerUnsigned(_val, val);
	}

	void convert(UInt16& val) const
	{
		convertToSmallerUnsigned(_val, val);
	}
	
	void convert(UInt32& val) const
	{
		val = _val;
	}

	void convert(UInt64& val) const
	{
		val = _val;
	}

	void convert(bool& val) const
	{
		val = (_val != 0);
	}

	void convert(float& val) const
	{
		val = static_cast<float>(_val);
	}

	void convert(double& val) const
	{
		val = static_cast<double>(_val);
	}

	void convert(char& val) const
	{
		UInt8 tmp;
		convert(tmp);
		val = static_cast<char>(tmp);
	}

	void convert(std::string& val) const
	{
		val = NumberFormatter::format(_val);
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}

	const UInt32& value() const
	{
		return _val;
	}

private:
	UInt32 _val;
};


template <>
class DynamicAnyHolderImpl<UInt64>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(UInt64 val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}

	const std::type_info& type() const
	{
		return typeid(UInt64);
	}

	void convert(Int8& val) const
	{
		convertUnsignedToSigned(_val, val);
	}

	void convert(Int16& val) const
	{
		convertUnsignedToSigned(_val, val);
	}
	
	void convert(Int32& val) const
	{
		convertUnsignedToSigned(_val, val);
	}

	void convert(Int64& val) const
	{
		convertUnsignedToSigned(_val, val);
	}

	void convert(UInt8& val) const
	{
		convertToSmallerUnsigned(_val, val);
	}

	void convert(UInt16& val) const
	{
		convertToSmallerUnsigned(_val, val);
	}
	
	void convert(UInt32& val) const
	{
		convertToSmallerUnsigned(_val, val);
	}

	void convert(UInt64& val) const
	{
		val = _val;
	}

	void convert(bool& val) const
	{
		val = (_val != 0);
	}

	void convert(float& val) const
	{
		val = static_cast<float>(_val);
	}

	void convert(double& val) const
	{
		val = static_cast<double>(_val);
	}

	void convert(char& val) const
	{
		UInt8 tmp;
		convert(tmp);
		val = static_cast<char>(tmp);
	}

	void convert(std::string& val) const
	{
		val = NumberFormatter::format(_val);
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}

	const UInt64& value() const
	{
		return _val;
	}

private:
	UInt64 _val;
};


template <>
class DynamicAnyHolderImpl<bool>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(bool val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}

	const std::type_info& type() const
	{
		return typeid(bool);
	}

	void convert(Int8& val) const
	{
		val = static_cast<Int8>(_val ? 1 : 0);
	}

	void convert(Int16& val) const
	{
		val = static_cast<Int16>(_val ? 1 : 0);
	}
	
	void convert(Int32& val) const
	{
		val = static_cast<Int32>(_val ? 1 : 0);
	}

	void convert(Int64& val) const
	{
		val = static_cast<Int64>(_val ? 1 : 0);
	}

	void convert(UInt8& val) const
	{
		val = static_cast<UInt8>(_val ? 1 : 0);
	}

	void convert(UInt16& val) const
	{
		val = static_cast<UInt16>(_val ? 1 : 0);
	}
	
	void convert(UInt32& val) const
	{
		val = static_cast<UInt32>(_val ? 1 : 0);
	}

	void convert(UInt64& val) const
	{
		val = static_cast<UInt64>(_val ? 1 : 0);
	}

	void convert(bool& val) const
	{
		val = _val;
	}

	void convert(float& val) const
	{
		val = (_val ? 1.0f : 0.0f);
	}

	void convert(double& val) const
	{
		val = (_val ? 1.0 : 0.0);
	}

	void convert(char& val) const
	{
		val = static_cast<char>(_val ? 1 : 0);
	}

	void convert(std::string& val) const
	{
		val = (_val ? "true" : "false");
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}

	const bool& value() const
	{
		return _val;
	}

private:
	bool _val;
};


template <>
class DynamicAnyHolderImpl<float>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(float val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}

	const std::type_info& type() const
	{
		return typeid(float);
	}

	void convert(Int8& val) const
	{
		convertToSmaller(_val, val);
	}

	void convert(Int16& val) const
	{
		convertToSmaller(_val, val);
	}
	
	void convert(Int32& val) const
	{
		convertToSmaller(_val, val);
	}

	void convert(Int64& val) const
	{
		convertToSmaller(_val, val);
	}

	void convert(UInt8& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(UInt16& val) const
	{
		convertSignedToUnsigned(_val, val);
	}
	
	void convert(UInt32& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(UInt64& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(bool& val) const
	{
		val = !(_val <= std::numeric_limits<float>::min() && 
			_val >= -1 * std::numeric_limits<float>::min());
	}

	void convert(float& val) const
	{
		val = _val;
	}

	void convert(double& val) const
	{
		val = _val;
	}

	void convert(char& val) const
	{
		UInt8 tmp;
		convert(tmp);
		val = static_cast<char>(tmp);
	}

	void convert(std::string& val) const
	{
		val = NumberFormatter::format(_val);
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}

	const float& value() const
	{
		return _val;
	}

private:
	float _val;
};


template <>
class DynamicAnyHolderImpl<double>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(double val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}

	const std::type_info& type() const
	{
		return typeid(double);
	}

	void convert(Int8& val) const
	{
		convertToSmaller(_val, val);
	}

	void convert(Int16& val) const
	{
		convertToSmaller(_val, val);
	}
	
	void convert(Int32& val) const
	{
		convertToSmaller(_val, val);
	}

	void convert(Int64& val) const
	{
		convertToSmaller(_val, val);
	}

	void convert(UInt8& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(UInt16& val) const
	{
		convertSignedToUnsigned(_val, val);
	}
	
	void convert(UInt32& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(UInt64& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(bool& val) const
	{
		val = !(_val <= std::numeric_limits<double>::min() && 
			_val >= -1 * std::numeric_limits<double>::min());
	}

	void convert(float& val) const
	{
		double fMin = -1 * std::numeric_limits<float>::max();
		double fMax = std::numeric_limits<float>::max();

		if (_val < fMin) throw RangeException("Value too small.");
		if (_val > fMax) throw RangeException("Value too large.");

		val = static_cast<float>(_val);
	}

	void convert(double& val) const
	{
		val = _val;
	}

	void convert(char& val) const
	{
		UInt8 tmp;
		convert(tmp);
		val = static_cast<char>(tmp);
	}

	void convert(std::string& val) const
	{
		val = NumberFormatter::format(_val);
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}

	const double& value() const
	{
		return _val;
	}

private:
	double _val;
};


template <>
class DynamicAnyHolderImpl<char>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(char val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}

	const std::type_info& type() const
	{
		return typeid(char);
	}

	void convert(Int8& val) const
	{
		val = static_cast<Int8>(_val);
	}

	void convert(Int16& val) const
	{
		val = static_cast<UInt8>(_val);
	}
	
	void convert(Int32& val) const
	{
		val = static_cast<UInt8>(_val);
	}

	void convert(Int64& val) const
	{
		val = static_cast<UInt8>(_val);
	}

	void convert(UInt8& val) const
	{
		val = static_cast<UInt8>(_val);
	}

	void convert(UInt16& val) const
	{
		val = static_cast<UInt8>(_val);
	}
	
	void convert(UInt32& val) const
	{
		val = static_cast<UInt8>(_val);
	}

	void convert(UInt64& val) const
	{
		val = static_cast<UInt8>(_val);
	}

	void convert(bool& val) const
	{
		val = (_val != '\0');
	}

	void convert(float& val) const
	{
		val = static_cast<float>(_val);
	}

	void convert(double& val) const
	{
		val = static_cast<double>(_val);
	}

	void convert(char& val) const
	{
		val = _val;
	}

	void convert(std::string& val) const
	{
		val = std::string(1, _val);
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}

	const char& value() const
	{
		return _val;
	}

private:
	char _val;
};


template <>
class DynamicAnyHolderImpl<std::string>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(const char* pVal): _val(pVal)
	{
	}

	DynamicAnyHolderImpl(const std::string& val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}

	const std::type_info& type() const
	{
		return typeid(std::string);
	}

	void convert(Int8& val) const
	{
		int v = NumberParser::parse(_val);
		convertToSmaller(v, val);
	}

	void convert(Int16& val) const
	{
		int v = NumberParser::parse(_val);
		convertToSmaller(v, val);
	}
	
	void convert(Int32& val) const
	{
		val = NumberParser::parse(_val);
	}

	void convert(Int64& val) const
	{
		val = NumberParser::parse64(_val);
	}

	void convert(UInt8& val) const
	{
		unsigned int v = NumberParser::parseUnsigned(_val);
		convertToSmallerUnsigned(v, val);
	}

	void convert(UInt16& val) const
	{
		unsigned int v = NumberParser::parseUnsigned(_val);
		convertToSmallerUnsigned(v, val);
	}
	
	void convert(UInt32& val) const
	{
		val = NumberParser::parseUnsigned(_val);
	}

	void convert(UInt64& val) const
	{
		val = NumberParser::parseUnsigned64(_val);
	}

	void convert(bool& val) const
	{
		static const std::string VAL_FALSE("false");
		static const std::string VAL_INTFALSE("0");

		if (_val == VAL_INTFALSE || (icompare(_val, VAL_FALSE) == 0))
			val = false;
		else
			val = true;
	}

	void convert(float& val) const
	{
		double v = NumberParser::parseFloat(_val);
		convertToSmaller(v, val);
	}

	void convert(double& val) const
	{
		val = NumberParser::parseFloat(_val);
	}

	void convert(char& val) const
	{
		if (_val.empty())
			val = '\0';
		else
			val = _val[0];
	}

	void convert(std::string& val) const
	{
		val = _val;
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}

	const std::string& value() const
	{
		return _val;
	}

private:
	std::string _val;
};


#ifndef POCO_LONG_IS_64_BIT


template <>
class DynamicAnyHolderImpl<long>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(long val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}

	const std::type_info& type() const
	{
		return typeid(long);
	}

	void convert(Int8& val) const
	{
		convertToSmaller(_val, val);
	}

	void convert(Int16& val) const
	{
		convertToSmaller(_val, val);
	}
	
	void convert(Int32& val) const
	{
		val = static_cast<Int32>(_val);
	}

	void convert(Int64& val) const
	{
		val = static_cast<Int64>(_val);
	}

	void convert(UInt8& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(UInt16& val) const
	{
		convertSignedToUnsigned(_val, val);
	}
	
	void convert(UInt32& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(UInt64& val) const
	{
		convertSignedToUnsigned(_val, val);
	}

	void convert(bool& val) const
	{
		val = (_val != 0);
	}

	void convert(float& val) const
	{
		val = static_cast<float>(_val);
	}

	void convert(double& val) const
	{
		val = static_cast<double>(_val);
	}

	void convert(char& val) const
	{
		UInt8 tmp;
		convert(tmp);
		val = static_cast<char>(tmp);
	}

	void convert(std::string& val) const
	{
		val = NumberFormatter::format(_val);
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}

	const long& value() const
	{
		return _val;
	}

private:
	long _val;
};


template <>
class DynamicAnyHolderImpl<unsigned long>: public DynamicAnyHolder
{
public:
	DynamicAnyHolderImpl(unsigned long val): _val(val)
	{
	}

	~DynamicAnyHolderImpl()
	{
	}

	const std::type_info& type() const
	{
		return typeid(unsigned long);
	}

	void convert(Int8& val) const
	{
		convertUnsignedToSigned(_val, val);
	}

	void convert(Int16& val) const
	{
		convertUnsignedToSigned(_val, val);
	}
	
	void convert(Int32& val) const
	{
		convertUnsignedToSigned(_val, val);
	}

	void convert(Int64& val) const
	{
		convertUnsignedToSigned(_val, val);
	}

	void convert(UInt8& val) const
	{
		convertToSmallerUnsigned(_val, val);
	}

	void convert(UInt16& val) const
	{
		convertToSmallerUnsigned(_val, val);
	}
	
	void convert(UInt32& val) const
	{
		convertToSmallerUnsigned(_val, val);
	}

	void convert(UInt64& val) const
	{
		val = static_cast<UInt64>(_val);
	}

	void convert(bool& val) const
	{
		val = (_val != 0);
	}

	void convert(float& val) const
	{
		val = static_cast<float>(_val);
	}

	void convert(double& val) const
	{
		val = static_cast<double>(_val);
	}

	void convert(char& val) const
	{
		UInt8 tmp;
		convert(tmp);
		val = static_cast<char>(tmp);
	}

	void convert(std::string& val) const
	{
		val = NumberFormatter::format(_val);
	}

	DynamicAnyHolder* clone() const
	{
		return new DynamicAnyHolderImpl(_val);
	}

	const unsigned long& value() const
	{
		return _val;
	}

private:
	unsigned long _val;
};


#endif // 64bit


} // namespace Poco


#endif // Foundation_DynamicAnyHolder_INCLUDED
