///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file RefTargetMessage.h
 * \brief Contains the definition of the Core::RefTargetMessage class.
 */

#ifndef __OVITO_REFTARGET_MESSAGE_H
#define __OVITO_REFTARGET_MESSAGE_H

#include <core/Core.h>
#include <core/plugins/PluginClass.h>

namespace Core {

/***************** Types of messages sent by reference targets ********************/

/// This notification message will be sent by a reference target when it has changed in some way.
#define REFTARGET_CHANGED						-1

/// This notification message will be sent by a reference target if has been deleted.
/// All RefMakers will stop observing the deleted target. This message ist automatically
/// handled by the RefMaker class.
#define REFTARGET_DELETED						-8

/// This notification message is sent by a RefMaker when the value of one
/// of its reference fields changes. This may be due to a call to SetReference()
/// or ClearReference().
/// The descriptor of the changed reference field is passed as data object with the notification message.
#define REFERENCE_FIELD_CHANGED				-10

/// This notification message is sent by a RefMaker when a new reference has
/// been added to a list reference field of the RefMaker. This may be due to a call to AddReference()
/// or InsertReference().
#define REFERENCE_FIELD_ADDED					-11

/// This notification message is sent by a RefMaker when a reference has
/// been removed from a list reference field of the RefMaker. This may be due to a call to RemoveReference().
#define REFERENCE_FIELD_REMOVED				-12

/// This notification message is sent by a RefMaker when one of the custom attribute objects
/// associated with it changes. It will also be sent when a attribute has been added or
/// removed from the RefTargets's custom attribute container.
#define CUSTOM_ATTRIBUTE_CHANGED				-13

/// This notification is sent by a RefTarget when its SchematicTitle property
/// has changed.
#define SCHEMATIC_TITLE_CHANGED				-30

/// This notification is sent by a selection set, when a node in the
/// selection set has changed.
#define NODE_IN_SELECTION_SET_CHANGED			-40

/// This notification is sent by a group node if it has been opened or closed.
#define GROUPNODE_OPENED_OR_CLOSED				-50

/// This notification is sent by a Modifier when it has been enabled or disabled.
#define MODIFIER_ENABLED						-60

/// This notification is sent by an object (e.g. a modifier) when its status has changed.
#define REFTARGET_STATUS_CHANGED				-64

class RefTarget;						// defined in RefTarget.h
class PropertyFieldDescriptor;			// defined in PropertyFieldDescriptor.h

/**
 * \brief Generic base class for messages sent by a RefTarget.
 *
 * The message type is specified by an integer number and can be checked using the type() method.
 * Negative values are reserved for system defined messages. Appropriate constants are defined in the RefMaker.h header file.
 * Positive numbers are used for custom object class specific messages.
 *
 * \author Alexander Stukowski
 *
 * \sa RefMaker::onRefTargetMessage()
 * \sa RefTarget::notifyDependents()
 */
class RefTargetMessage
{
public:
	/// \brief Constructs the message.
	/// \param sender The object that generates the message.
	/// \param msgtype Identifies the type of the notification message.
	RefTargetMessage(RefTarget* sender, const int msgtype) : reftarget(sender), messageType(msgtype) {}

	/// \brief Returns the RefTarget that has generated this message.
	/// \return The sender of this notification message.
	RefTarget* sender() const { return reftarget; }

	/// \brief Returns the type of this message.
	/// \return An identification number. Negative values are system defined messages, positive numbers
	///         are custom object class specific messages.
	int type() const { return messageType; }

private:

	/// The RefTarget that sent the message.
	RefTarget* reftarget;

	/// One of the message identifiers defined above.
	int messageType;
};

/**
 * \brief This message is sent by a RefMaker when the pointer stored in one of its reference
 *        fields has been replaced, removed or added.
 *
 * This type of message is sent by the reference system to inform RefMaker object that a reference of one of its
 * referenced object has changed. The message type is either REFERENCE_FIELD_CHANGED, REFERENCE_FIELD_ADDED or
 * REFERENCE_FIELD_REMOVED.
 *
 * Plugin developers should not send this messages because they are automatically generated by the system.
 *
 * \author Alexander Stukowski
 */
class ReferenceFieldMessage : public RefTargetMessage
{
public:
	/// Constructor.
	ReferenceFieldMessage(RefTarget* sender, const int msgtype, const PropertyFieldDescriptor& field, RefTarget* oldTarget, RefTarget* newTarget, int index = -1) :
		RefTargetMessage(sender, msgtype), reffield(field), oldvalue(oldTarget), newvalue(newTarget), vectorindex(index) {}

	/// \brief Returns the reference field that has changed.
	/// \return The descriptor of the changed reference field in the RefTarget that
	///         has generated the message.
	const PropertyFieldDescriptor& field() const { return reffield; }

	/// \brief Returns the old target that was stored in the reference field.
	RefTarget* oldTarget() const { return oldvalue; }

	/// \brief Returns the new target that is now stored in the reference field.
	RefTarget* newTarget() const { return newvalue; }

	/// \brief The index that has been added or removed if the changed reference field is a vector field.
	/// \return The index into the VectorReferenceField where the entry has been added or removed.
	///         Returns -1 if the reference field is not a vector reference field.
	int index() const { return vectorindex; }

private:

	const PropertyFieldDescriptor& reffield;

	RefTarget* oldvalue;

	RefTarget* newvalue;

	int vectorindex;
};

};

#endif // __OVITO_REFTARGET_MESSAGE_H
