
/******************************************************************************
**
**  Copyright (C) 2006 Brian Wotring.
**
**  This program is free software; you can redistribute it and/or
**  modify it, however, you cannot sell it.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
**  You should have received a copy of the license attached to the
**  use of this software.  If not, view a current copy of the license
**  file here:
**
**      http://www.hostintegrity.com/osiris/LICENSE
**
******************************************************************************/

/*****************************************************************************
**
**  File:    md_hosts.c
**  Date:    May 24, 2002
**
**  Author:  Brian Wotring
**  Purpose: interact with local file system to read in host information from
**	         hosts directory.
**
******************************************************************************/

#include "libosiris.h"
#include "libosirism.h"
#include "libosirisdb.h"
#include "libosirisctl.h"
#include "libfileapi.h"

#include "md_hosts.h"
#include "md_compare.h"
#include "md_utilities.h"
#include "md_database.h"

#include "common.h"
#include "logging.h"

extern char root_path[MAX_PATH_LENGTH];

OSI_HOST * osi_host_new()
{
    OSI_HOST *host = osi_malloc( sizeof( OSI_HOST ) );
    osi_set_host_defaults( host );

    /* initialize the list of configs and databases. */

    host->configs   = list_new();
    host->databases = list_new();
    host->logs      = list_new();

    return host;
}

void osi_host_destroy( OSI_HOST *host )
{
    if( host != NULL )
    {
        /* this assumes we only have config and database brief */
        /* structures, no complicated free methods required.   */

        if( host->configs != NULL )
        {
            list_destroy_with_function( host->configs,
                                        (void (*)(void *))&osi_free );
        }

        if( host->databases != NULL )
        {
            list_destroy_with_function( host->databases,
                                        (void (*)(void *))&osi_free );
        }

        if( host->logs != NULL )
        {
            list_destroy_with_function( host->logs,
                                        (void (*)(void *))&osi_free );
        }

        osi_free( host );
    }
}


void osi_set_host_defaults( OSI_HOST *host )
{
    if( host != NULL )
    {
        memset( host, 0, sizeof( OSI_HOST ) );
        
        host->enabled = 1;
        host->type = OSI_HOST_TYPE_GENERIC;

        host->file_log_enabled = 0;
        host->db_flags = ( OSI_DB_PURGE | OSI_DB_AUTOACCEPT );

        host->notify_enabled = 0;
        host->notify_flags = 0;

        host->schedule_start = (osi_get_time() - 61);
        host->schedule_period = 0;

        host->port = DEFAULT_SCAN_AGENT_PORT;
    }
}

osi_bool osi_remove_host( const char *name )
{
    osi_bool result = FALSE;
    char path[MAX_PATH_LENGTH];

    if( name == NULL )
    {
        goto exit_gracefully;
    }

    /* remove configs. */

    osi_set_path_to_configs( name, path, sizeof( path ) );
    osi_remove_files_from_directory( path );
    osi_remove_directory( path );

    /* remove db files. */

    osi_set_path_to_databases( name, path, sizeof( path ) );
    osi_remove_files_from_directory( path );
    osi_remove_directory( path );

    /* remove log files. */

    osi_set_path_to_logs( name, path, sizeof( path ) );
    osi_remove_files_from_directory( path );
    osi_remove_directory( path );

    /* remove host directory. */

    osi_set_path_to_host( name, path, sizeof( path ) );
    osi_remove_files_from_directory( path );
    
    if( osi_remove_directory( path ) == OSI_FILE_OK )
    {
        result = TRUE;
    }

exit_gracefully:

    return result;
}


osi_list osi_read_hosts()
{
    OSI_DIRECTORY directory;
    OSI_HOST *host;

    char path[MAX_PATH_LENGTH];

    osi_list new_hosts = NULL;
    memset( &directory, 0, sizeof( directory ) );

    new_hosts = list_new();
    osi_set_path_to_hosts( path, sizeof( path ) );        

    if( osi_open_directory( path, &directory ) )
    {
        do
        {
            if( ( strcmp( directory.filename, "." ) == 0 ) ||
                ( strcmp( directory.filename, ".." ) == 0 ) )
            {
                continue;
            }

            host = osi_read_host( directory.filename );

            if( host != NULL )
            {
                list_add( new_hosts, host );
            }

        } while( osi_get_next_file( &directory ) );

        /* close root directory. */

        osi_close_directory( &directory );
    }

    return new_hosts;
}

osi_list osi_read_host_briefs()
{
    OSI_DIRECTORY directory;
    OSI_HOST_BRIEF *host;
    
    char path[MAX_PATH_LENGTH];

    osi_list new_hosts = NULL;
    memset( &directory, 0, sizeof( directory ) );

    new_hosts = list_new();
    osi_set_path_to_hosts( path, sizeof( path ) );        

#if defined(HAVE_SCANDIR) && defined(HAVE_ALPHASORT)
    {
        int i;
        int count;
        
        struct dirent **files = NULL;
        count = scandir( path, &files, NULL, alphasort );

        if( ( count > 0 ) && ( files != NULL ) )
        {
            for( i = 0; i < count; i++ )
            {
                char *name = NULL;

                if( (struct dirent *)files[i] != NULL )
                {
                    name = ((struct dirent *)files[i])->d_name;
                }

                if( ( name != NULL ) && ( ( strcmp( name, "." ) == 0 ) || 
                    ( strcmp( name, ".." ) == 0 ) ) )
                {
                    continue;
                }

                host = osi_read_host_brief( name );

                if( host != NULL )
                {
                    list_add( new_hosts, host );
                }
            
                osi_free( files[i-1] );
            }
            
            osi_free( files );
        }
    }
#else
    if( osi_open_directory( path, &directory ) )
    {
        do
        {
            if( ( strcmp( directory.filename, "." ) == 0 ) ||
                ( strcmp( directory.filename, ".." ) == 0 ) )
            {
                continue;
            }

            host = osi_read_host_brief( directory.filename );

            if( host != NULL )
            {
                list_add( new_hosts, host );
            }

        } while( osi_get_next_file( &directory ) );

        /* close root directory. */

        osi_close_directory( &directory );
    }
#endif

    return new_hosts;
}

OSI_HOST * osi_read_host( const char *name )
{
    char path[MAX_PATH_LENGTH];
    
    OSI_HOST *host = NULL;
    OSI_HOST_BRIEF host_brief;
    
    osi_bool result;

    if( name == NULL  )
    {
        return NULL;
    }

    /* create a default host structure and copy in the name */

    if( osi_host_exists( name ) == FALSE )
    {
        return NULL;
    }

    host = osi_host_new();
    osi_set_path_to_host_config( name, path, sizeof( path ) );
    
    /* parse the given file into a config, then take what we need */
    /* out of it.                                                 */
            
    result = osi_host_brief_read_file( path, &host_brief );

    if( result )
    {
        osi_strlcpy( host->host, host_brief.host, sizeof( host->host ) );

        osi_strlcpy( host->description, host_brief.description,
                     sizeof( host->description ) );

        osi_strlcpy( host->session_key, host_brief.session_key,
                     sizeof( host->session_key ) );

        osi_strlcpy( host->base_db, host_brief.base_db,
                     sizeof( host->base_db ) );

        osi_strlcpy( host->config, host_brief.config,
                     sizeof( host->config) );

        osi_strlcpy( host->notify_email, host_brief.notify_email,
                     sizeof( host->notify_email ) );

        host->file_log_enabled  = host_brief.file_log_enabled; 
        host->db_flags          = host_brief.db_flags;

        host->notify_enabled   = host_brief.notify_enabled;
        host->notify_flags     = host_brief.notify_flags;

        host->enabled          = host_brief.enabled;
        host->type             = host_brief.type;

        host->schedule_start   = host_brief.schedule_start;
        host->schedule_period  = host_brief.schedule_period;

        host->port             = host_brief.port;
    }
            
    /* put in name, after the host has been populated. */
            
    osi_strlcpy( host->name, name, sizeof( host->name ) );
                      
    /* we don't get databases or configs, they need to be */
    /* requested later.                                   */

    return host;
}

osi_bool osi_write_host( OSI_HOST *host )
{
    osi_bool result = FALSE;

    char path[MAX_PATH_LENGTH];
    char temp_path[MAX_PATH_LENGTH];

    char config_path[MAX_PATH_LENGTH];
    char path_separator[2] = { PATH_SEPARATOR, '\0' };
    
    OSI_HOST_BRIEF host_brief;

    if( host == NULL  )
    {
        return FALSE;
    }

    /* construct full path to directory. */

    osi_set_path_to_host( host->name, path, sizeof( path ) );

    /* construct full path to config file. */

    osi_strlcpy( config_path, path, sizeof( config_path ) );
    osi_strlcat( config_path, path_separator, sizeof( config_path ) );
    osi_strlcat( config_path, HOST_CONFIG_FILE_NAME, sizeof( config_path ) );
            
    /* if host doesn't exist, create directory structure for it including */
    /* the directories, and the master config file for this host.         */

    if( osi_host_exists( host->name ) == FALSE )
    {
        if( osi_create_directory( path ) != OSI_FILE_OK )
        {
            return FALSE;
        }

        /* now create the config, log, and database directories. */

        osi_strlcpy( temp_path, path, sizeof( temp_path ) );
        osi_strlcat( temp_path, path_separator, sizeof( temp_path ) );

        osi_strlcat( temp_path, HOST_CONFIG_DIRECTORY_NAME,
                     sizeof( temp_path ) );

        if( osi_create_directory( temp_path ) != OSI_FILE_OK )
        {
            return FALSE;
        }

        osi_strlcpy( temp_path, path, sizeof( temp_path ) );
        osi_strlcat( temp_path, path_separator, sizeof( temp_path ) );
        osi_strlcat( temp_path, HOST_LOG_DIRECTORY_NAME, sizeof( temp_path ) );

        if( osi_create_directory( temp_path ) != OSI_FILE_OK )
        {
            return FALSE;
        }

        osi_strlcpy( temp_path, path, sizeof( temp_path ) );
        osi_strlcat( temp_path, path_separator, sizeof( temp_path ) );

        osi_strlcat( temp_path, HOST_DATABASE_DIRECTORY_NAME,
                     sizeof( temp_path ) );

        if( osi_create_directory( temp_path ) != OSI_FILE_OK )
        {
            return FALSE;
        }

        /* now create the host.conf file. */

        if( osi_create_file( config_path, HOST_CONFIG_PERMISSIONS ) !=
                             OSI_FILE_OK )
        {
            return FALSE;
        }
    }
        
    /* now populate our host brief structure with this host so we */
    /* can quickly and efficiently write a basic config to disk.  */

    osi_strlcpy( host_brief.host, host->host, sizeof( host_brief.host ) );

    osi_strlcpy( host_brief.description, host->description,
                 sizeof( host_brief.description ) );

    osi_strlcpy( host_brief.session_key, host->session_key,
                 sizeof( host_brief.session_key ) );

    osi_strlcpy( host_brief.base_db, host->base_db,
                 sizeof( host_brief.base_db ) );

    osi_strlcpy( host_brief.config, host->config,
                 sizeof( host_brief.config) );

    osi_strlcpy( host_brief.notify_email, host->notify_email,
                 sizeof( host_brief.notify_email ) );

    host_brief.file_log_enabled  = host->file_log_enabled; 
    host_brief.db_flags          = host->db_flags;

    host_brief.notify_enabled   = host->notify_enabled;
    host_brief.notify_flags     = host->notify_flags;

    host_brief.enabled          = host->enabled;
    host_brief.type             = host->type;

    host_brief.schedule_start   = host->schedule_start;
    host_brief.schedule_period  = host->schedule_period;

    host_brief.port             = host->port;
                
    result = osi_host_brief_write_file( config_path, &host_brief );

    return result;
}


OSI_HOST_BRIEF * osi_read_host_brief( const char *name )
{
    char path[MAX_PATH_LENGTH];
    OSI_HOST_BRIEF *host_brief = NULL;

    if( name == NULL )
    {
        return NULL;
    }

    /* create a default host structure and copy in the name */

    if( osi_host_exists( name ) )
    {
        host_brief = (OSI_HOST_BRIEF *)osi_malloc( sizeof(OSI_HOST_BRIEF) );
        osi_set_host_brief_defaults( host_brief );
            
        /* construct full path to config file. */

        osi_set_path_to_host_config( name, path, sizeof( path ) );
    
        /* now use the management library to acquire this host brief. */

        osi_host_brief_read_file( path, host_brief );
            
        /* set name after brief has been populated. */
            
        osi_strlcpy( host_brief->name, name, sizeof( host_brief->name ) );
    }

    return host_brief;
}


osi_bool osi_write_host_brief( OSI_HOST_BRIEF *host_brief )
{
    char path[MAX_PATH_LENGTH];
    
    if( host_brief == NULL )
    {            
        return FALSE;
    }

    osi_set_path_to_host_config( host_brief->name, path, sizeof( path ) );
    return osi_host_brief_write_file( path, host_brief );
}

osi_bool osi_host_exists( const char *name )
{
    osi_bool result = FALSE;
    char path[MAX_PATH_LENGTH];

    if( name == NULL )
    {
        return FALSE;
    }

    /* we verify that this host exists by verifying */
    /* that the directory exists                    */

    osi_set_path_to_host( name, path, sizeof( path ) );

    if( osi_directory_exists( path ) )
    {
        result = TRUE;
    }

    return result;
}

OSI_HOST_CONFIG * osi_host_read_host_config_with_name( const char *name )
{
    char path[MAX_PATH_LENGTH];
    
    if( name == NULL )
    {            
        return NULL;
    }

    osi_set_path_to_host_config( name, path, sizeof( path ) );
    return osi_host_config_new_from_file( path );
}

osi_bool osi_host_write_host_config_with_name( OSI_HOST_CONFIG *config,
                                               const char *name )
{
    char path[MAX_PATH_LENGTH];
    
    if( name == NULL )
    {            
        return FALSE;
    }

    osi_set_path_to_host_config( name, path, sizeof( path ) );
    return osi_host_config_write_file( path, config );
}



unsigned long osi_host_get_config_count( const char *name )
{
    unsigned long count = 0;

    char path[MAX_PATH_LENGTH];
    char path_separator[2] = { PATH_SEPARATOR, '\0' };

    OSI_DIRECTORY directory;
    memset( &directory, 0, sizeof( OSI_DIRECTORY ) );

    if( name == NULL )
    {
        return count;
    }

    osi_set_path_to_configs( name, path, sizeof( path ) );

    /* now traverse the directory, creating db brief objects and */
    /* adding them to this host's list of databases.             */

    if( osi_open_directory( path, &directory ) )
    {
        do
        {
            char file_path[MAX_PATH_LENGTH];

            /* create full path to the current filename. */

            osi_strlcpy( file_path, path, sizeof( file_path ) );
            osi_strlcat( file_path, path_separator, sizeof( file_path ) );
            osi_strlcat( file_path, directory.filename, sizeof( file_path ) );

            if( osi_file_is_regular_file( file_path ) )
            {
                count++;
            }

        } while( osi_get_next_file( &directory ) );

        osi_close_directory( &directory );
    }

    return count;
}

unsigned long osi_host_get_database_count( const char *name )
{
    unsigned long count = 0;

    char path[MAX_PATH_LENGTH];
    char path_separator[2] = { PATH_SEPARATOR, '\0' };

    OSI_DIRECTORY directory;
    memset( &directory, 0, sizeof( OSI_DIRECTORY ) );

    if( name == NULL )
    {
        return count;
    }

    osi_set_path_to_databases( name, path, sizeof( path ) );

    /* now traverse the directory, creating db brief objects and */
    /* adding them to this host's list of databases.             */

    if( osi_open_directory( path, &directory ) )
    {
        do
        {
            char file_path[MAX_PATH_LENGTH];

            /* create full path to the current filename. */

            osi_strlcpy( file_path, path, sizeof( file_path ) );
            osi_strlcat( file_path, path_separator, sizeof( file_path ) );
            osi_strlcat( file_path, directory.filename, sizeof( file_path ) );

            if( osi_file_is_regular_file( file_path ) )
            {
                count++;
            }

        } while( osi_get_next_file( &directory ) );

        osi_close_directory( &directory );
    }

    return count;
}

unsigned long osi_host_get_log_count( const char *name )
{
    unsigned long count = 0;

    char path[MAX_PATH_LENGTH];
    char path_separator[2] = { PATH_SEPARATOR, '\0' };

    OSI_DIRECTORY directory;
    memset( &directory, 0, sizeof( OSI_DIRECTORY ) );

    if( name == NULL )
    {
        return count;
    }

    osi_set_path_to_logs( name, path, sizeof( path ) );

    /* now traverse the directory, creating db brief objects and */
    /* adding them to this host's list of databases.             */

    if( osi_open_directory( path, &directory ) )
    {
        do
        {
            char file_path[MAX_PATH_LENGTH];

            /* create full path to the current filename. */

            osi_strlcpy( file_path, path, sizeof( file_path ) );
            osi_strlcat( file_path, path_separator, sizeof( file_path ) );
            osi_strlcat( file_path, directory.filename, sizeof( file_path ) );

            if( osi_file_is_regular_file( file_path ) )
            {
                count++;
            }

        } while( osi_get_next_file( &directory ) );

        osi_close_directory( &directory );
    }

    return count;
}

osi_bool osi_host_brief_set_base_db( OSI_HOST_BRIEF *host, char *dbname )
{
    osi_bool result = FALSE;

    if( ( host == NULL ) || ( dbname == NULL ) )
    {
        return result;
    }

    osi_strlcpy( host->base_db, dbname, sizeof( host->base_db ) );

    if( osi_write_host_brief( host ) )
    {
        result = TRUE;
    }

    /* if this host purges databases, we need to delete the previous */
    /* database file, if it exists.                                  */

    if( host->db_flags & OSI_DB_PURGE )
    {
        char prev_name[15];
        int previous = atoi( dbname );

        previous--;
        osi_snprintf( prev_name, sizeof( prev_name ), "%d", previous );

        if( osi_host_database_exists( host->name, prev_name ) )
        {
            osi_host_remove_db( host->name, prev_name );
        }
    }

    return result;
}

osi_bool osi_host_brief_from_host( OSI_HOST *host, OSI_HOST_BRIEF *brief )
{
    int result = FALSE;

    if( ( host == NULL ) || ( brief == NULL ) )
    {
        return result;
    }

    brief->enabled = host->enabled;
    brief->type = host->type;
    brief->file_log_enabled = host->file_log_enabled;
    brief->db_flags = host->db_flags;
    brief->notify_enabled = host->notify_enabled;
    brief->notify_flags = host->notify_flags;
    brief->port         = host->port;

    if( host->configs )
    {
        brief->config_count = list_get_size( host->configs );
    }

    if( host->databases )
    {
        brief->database_count = list_get_size( host->databases );
    }

    brief->schedule_start = host->schedule_start;
    brief->schedule_period = host->schedule_period;

    osi_strlcpy( brief->name, host->name, sizeof( brief->name ) );
    osi_strlcpy( brief->host, host->host, sizeof( brief->host ) );
    osi_strlcpy( brief->description, host->description,
                 sizeof( brief->description ) );
    osi_strlcpy( brief->session_key, host->session_key,
                 sizeof( brief->session_key ) );
    osi_strlcpy( brief->base_db, host->base_db, sizeof( brief->base_db ) );
    osi_strlcpy( brief->notify_email, host->notify_email,
                 sizeof( brief->notify_email ) );

    osi_strlcpy( brief->config, host->config,
                 sizeof( brief->config ) );

    result = TRUE;
    return result;
}

