#!/usr/bin/env python
#
#   OpenRPG Server Graphical Interface
#	GNU General Public License
#

__appname__=' OpenRPG GUI Server v0.7 '
__version__='$Revision: 1.15 $'[11:-2]
__cvsinfo__='$Id: mplay_server_gui.py,v 1.15 2002/11/11 20:00:44 posterboy Exp $'[5:-2]
__doc__="""
OpenRPG Server Graphical Interface
"""

import os
import sys
import time
import types

import orpg.dirpath 
import orpg.systempath

from wxPython.wx import *
from wxPython.lib.splashscreen import SplashScreen
from wxPython.html import *
import wxPython.lib.wxpTag

import webbrowser

from threading import Thread

from meta_server_lib import post_server_data, remove_server
from mplay_server import mplay_server


# Constants ######################################

SERVER_RUNNING = 1
SERVER_STOPPED = 0

MENU_START_SERVER = wxNewId()
MENU_STOP_SERVER = wxNewId()
MENU_EXIT = wxNewId()

MENU_REGISTER_SERVER = wxNewId()
MENU_UNREGISTER_SERVER = wxNewId()

MENU_START_PING_PLAYERS = wxNewId()
MENU_STOP_PING_PLAYERS = wxNewId()
MENU_PING_INTERVAL = wxNewId()

# Add our menu id's for our right click popup
MENU_PLAYER_BOOT = wxNewId()
MENU_PLAYER_CREATE_ROOM = wxNewId()
MENU_PLAYER_SEND_MESSAGE = wxNewId()
MENU_PLAYER_SEND_ROOM_MESSAGE = wxNewId()
MENU_PLAYER_SEND_SERVER_MESSAGE = wxNewId()


# Our new event type that gets posted from one
# thread to another
wxEVT_LOG_MESSAGE = wxNewId()*100+50
wxEVT_FUNCTION_MESSAGE = wxNewId()*100+50


# Our event connector -- wxEVT_LOG_MESSAGE
def EVT_LOG_MESSAGE( win, func ):
    win.Connect( -1, -1, wxEVT_LOG_MESSAGE, func )


# Our event connector -- wxEVT_FUNCTION_MESSAGE
def EVT_FUNCTION_MESSAGE( win, func ):
    win.Connect( -1, -1, wxEVT_FUNCTION_MESSAGE, func )



# Utils ##########################################
def format_bytes(b):
    f = ['b', 'Kb', 'Mb', 'Gb']
    i = 0
    while i < 3:
        if b < 1024:
            return str(b) + f[i]
        else:
            b = b/1024
        i += 1
    return str(b) + f[3]



# wxEVT_LOG_MESSAGE
# MessageLogEvent ###############################
class MessageLogEvent( wxPyEvent ):
    def __init__( self, message ):
        wxPyEvent.__init__( self )
        self.SetEventType( wxEVT_LOG_MESSAGE )
        self.message = message


# wxEVT_TUPLE_MESSAGE
# MessageLogEvent ###############################
class MessageFunctionEvent( wxPyEvent ):
    def __init__( self, func, message ):
        wxPyEvent.__init__( self )
        self.SetEventType( wxEVT_FUNCTION_MESSAGE )
        self.func = func
        self.message = message



# ServerConfig Object ############################
class ServerConfig:
    """ This class contains configuration
        setting used to control the server.
    """


    def __init__(self, owner ):
        """ Loads default configuration settings.
        """
        OPENRPG_PORT = 6774
        self.owner = owner
        


    def load_xml(self, xml):
        """ Load configuration from XML data.
            xml (xml) -- xml string to parse
        """
        pass



    def save_xml(self):
        """ Returns XML file representing
            the active configuration.
        """
        pass




# Server Monitor #################################

class ServerMonitor(Thread):
    """ Monitor thread for GameServer. """
    def __init__( self, cb, conf, name ):
        """ Setup the server. """
        Thread.__init__(self)

        self.cb = cb
        self.conf = conf
        self.serverName = name



    def log(self, mesg):
        if type(mesg) == types.TupleType:
            func, msg = mesg
            event = MessageFunctionEvent( func, msg )

        else:
            event = MessageLogEvent( mesg )

        wxPostEvent( self.conf.owner, event )
        del event



    def run(self):
        """ Start the server. """

        self.server = mplay_server( self.log, self.serverName )

        self.alive = 1
        while self.alive:
            time.sleep( 1 )



    def stop(self):
        """ Stop the server. """
        self.server.kill_server()
        self.alive = 0





# GUI Server #####################################
# Parent = notebook
# Main = GUI
class Connections(wxListCtrl):
    def __init__( self, parent, main ):
        wxListCtrl.__init__( self, parent, -1, wxDefaultPosition, wxDefaultSize, wxLC_REPORT|wxSUNKEN_BORDER|wxEXPAND|wxLC_HRULES )
        self.main = main
        self.roomList = { "0" : "Lobby" }

        self._imageList = wxImageList( 16, 16, false )
        img = wxImage(orpg.dirpath.dir_struct["icon"]+"player.gif", wxBITMAP_TYPE_GIF).ConvertToBitmap()
        self._imageList.Add( img )
        img = wxImage(orpg.dirpath.dir_struct["icon"]+"player-whisper.gif", wxBITMAP_TYPE_GIF).ConvertToBitmap()
        self._imageList.Add( img )
        self.SetImageList( self._imageList, wxIMAGE_LIST_SMALL )

        # Set The columns
        self.InsertColumn(0, "ID")
        self.InsertColumn(1, "Player")
        self.InsertColumn(2, "IP Address")
        self.InsertColumn(3, "Status")
        self.InsertColumn(4, "Room")
        self.InsertColumn(5, "Version")
        self.InsertColumn(6, "Role")

        # Set the column widths
        self.SetColumnWidth(0, 60)
        self.SetColumnWidth(1, 200)
        self.SetColumnWidth(2, 100)
        self.SetColumnWidth(3, 100)
        self.SetColumnWidth(4, 120)
        self.SetColumnWidth(5, 80)
        self.SetColumnWidth(6,80)

        # Build our pop up menu to do cool things with the players in the list
        self.menu = wxMenu()
        self.menu.SetTitle( "Player Menu" )
        self.menu.Append( MENU_PLAYER_BOOT, "Boot Player" )
        self.menu.AppendSeparator()
        self.menu.Append( MENU_PLAYER_SEND_MESSAGE, "Send Message" )
        self.menu.Append( MENU_PLAYER_SEND_SERVER_MESSAGE, "Broadcast Server Message" )


        # Associate our events
        EVT_RIGHT_DOWN( self, self.OnPopupMenu )
        EVT_MENU( self, MENU_PLAYER_BOOT, self.OnPopupMenuItem )
        EVT_MENU( self, MENU_PLAYER_SEND_MESSAGE, self.OnPopupMenuItem )
        EVT_MENU( self, MENU_PLAYER_SEND_ROOM_MESSAGE, self.OnPopupMenuItem )
        EVT_MENU( self, MENU_PLAYER_SEND_SERVER_MESSAGE, self.OnPopupMenuItem )



    def add(self, player):
        i = self.InsertImageStringItem( 0, player["id"], 0 )
        self.SetStringItem( i, 1, self.stripHtml( player["name"] ) )
        self.SetStringItem( i, 2, player["ip"] )
        self.SetStringItem( i, 3, "New - Idle" )
        self.SetStringItem( i, 4, self.roomList[player["group_id"]] )
        self.SetStringItem( i, 5, player["version"] )
        self.SetStringItem( i, 6, "Lurker" )
        self.SetItemData( i, int(player["id"]) )


    def remove(self, id):
        i = self.FindItemData( -1, int(id) )
        self.DeleteItem( i )
        self.Refresh()


    def update(self, player):
        i = self.FindItemData( -1, int(player["id"]) )
        self.SetStringItem(i, 1, player["name"])
        self.SetStringItem(i, 3, self.stripHtml( player["status"] ) )
        self.Refresh()


    def updateRoom( self, data ):
        (from_id, id) = data
        i = self.FindItemData( -1, int(from_id) )
        self.SetStringItem( i, 4, self.roomList[id] )
        self.SetStringItem( i, 6, "Lurker") 
        self.Refresh()


    def setPlayerRole( self, id, role ):
        i = self.FindItemData( -1, int(id) )
        self.SetStringItem( i, 6, role ) 


    def stripHtml( self, name ):
        ret_string = ""
        x = 0
        in_tag = 0

        for x in range( len(name) ):
            if name[x] == "<" or name[x] == ">" or in_tag == 1 :
                if name[x] == "<" :
                    in_tag = 1
                elif name[x] == ">" :
                    in_tag = 0
                else :
                    pass
            else :
                ret_string = ret_string + name[x]

        return ret_string



    # When we right click, cause our popup menu to appear
    def OnPopupMenu( self, event ):
        pos = wxPoint( event.GetX(), event.GetY() )
        (item, flag) = self.HitTest( pos )
        if item > -1:
            self.selectedItem = item
            self.PopupMenu( self.menu, pos )



    # Process the events associated with our popup menu
    def OnPopupMenuItem( self, event ):
        # if we are not running, the player list is empty anyways
        if self.main.STATUS == SERVER_RUNNING:
            menuItem = event.GetId()
            playerID = str( self.GetItemData( self.selectedItem ) )
            idList = self.main.server.server.groups
            groupID = 0

            if menuItem == MENU_PLAYER_BOOT:
                print "booting player: ", playerID
                self.main.server.server.del_player( playerID, groupID )
                self.main.server.server.check_group( playerID, groupID )
                self.remove( playerID )

            elif menuItem == MENU_PLAYER_SEND_MESSAGE:
                print "send a message..."
                msg = self.GetMessageInput( "Send a message to player" )
                if len(msg):
                    self.main.server.server.send( msg, playerID, str(groupID) )

#            elif menuItem == MENU_PLAYER_SEND_TO_ROOM:
#                print "Send message to room..."
#                msg = self.GetMessageInput( "Send message to room of this player")
#                if len(msg):
#                    self.main.server.server.send_to_group( 0, GroupID, msg )

            elif menuItem == MENU_PLAYER_SEND_SERVER_MESSAGE:
                print "broadcast a message..."
                msg = self.GetMessageInput( "Broadcast Server Message" )

                # If we got a message back, send it
                if len(msg):
                    self.main.server.server.broadcast( msg )


    def GetMessageInput( self, title ):
        prompt = "Please enter the message you wish to send:"
        msg = wxTextEntryDialog(self, prompt, title)
        msg.ShowModal()
        msg.Destroy()
        return msg.GetValue()




class ServerGUI(wxFrame):

    STATUS = SERVER_STOPPED

    def __init__(self, parent, id, title):
        wxFrame.__init__(self, parent, id, title, size = (760, 560) )


        if wxPlatform == '__WXMSW__':
            icon = wxIcon( orpg.dirpath.dir_struct["icon"]+'WAmisc9.ico', wxBITMAP_TYPE_ICO )

        else:
            icon = wxIcon( orpg.dirpath.dir_struct["icon"]+'connect.gif', wxBITMAP_TYPE_GIF )

        self.SetIcon(icon)
        self.serverName = "Server Name"

        # Register our events to process -- notice the custom event handler
        EVT_CLOSE( self, self.OnExit )
        EVT_LOG_MESSAGE( self, self.OnLogMessage )
        EVT_FUNCTION_MESSAGE( self, self.OnFunctionMessage )

        # Creat GUI
        self.build_menu()
        self.build_body()
        self.build_status()

        # Server Callbacks
        cb = {}
        cb["log"]        = self.Log
        cb["connect"]    = self.OnConnect
        cb["disconnect"] = self.OnDisconnect
        cb["update"]     = self.OnUpdatePlayer
        cb["data_recv"]  = self.OnDataRecv
        cb["data_sent"]  = self.OnDataSent
        cb["create_group"] = self.OnCreateGroup
        cb["delete_group"] = self.OnDeleteGroup
        cb["join_group"] = self.OnJoinGroup
        cb["role"] = self.OnSetRole
        self.callbacks = cb

        # Misc.
        self.conf = ServerConfig( self )
        self.total_messages_received = 0L
        self.total_data_received = 0L
        self.total_messages_sent = 0L
        self.total_data_sent = 0L

    ### Build GUI ############################################

    def build_menu(self):
        """ Build the GUI menu. """


        self.mainMenu = wxMenuBar()

        # File Menu
        menu = wxMenu()

        # Start
        menu.Append( MENU_START_SERVER, 'Start', 'Start server.')
        EVT_MENU(self, MENU_START_SERVER, self.OnStart)

        # Stop
        menu.Append( MENU_STOP_SERVER, 'Stop', 'Shutdown server.')
        EVT_MENU(self, MENU_STOP_SERVER, self.OnStop)

        # Exit
        menu.AppendSeparator()
        menu.Append( MENU_EXIT, 'E&xit', 'Exit application.')
        EVT_MENU(self, MENU_EXIT, self.OnExit)

        self.mainMenu.Append(menu, '&Server')

        # Registration Menu
        menu = wxMenu()

        # Register
        menu.Append( MENU_REGISTER_SERVER, 'Register', 'Register with OpenRPG server directory.')
        EVT_MENU( self, MENU_REGISTER_SERVER, self.OnRegister )

        # Unregister
        menu.Append( MENU_UNREGISTER_SERVER, 'Unregister', 'Unregister from OpenRPG server directory.')
        EVT_MENU( self, MENU_UNREGISTER_SERVER, self.OnUnregister )

        # Add the registration menu
        self.mainMenu.Append( menu, '&Registration' )

        # Server Configuration Menu
        menu = wxMenu()

        # Ping Connected Players
        menu.Append( MENU_START_PING_PLAYERS, 'Start Ping', 'Ping players to validate remote connection.' )
        EVT_MENU( self, MENU_START_PING_PLAYERS, self.PingPlayers )

        # Stop Pinging Connected Players
        menu.Append( MENU_STOP_PING_PLAYERS, 'Stop Ping', 'Stop validating player connections.' )
        EVT_MENU( self, MENU_STOP_PING_PLAYERS, self.StopPingPlayers )

        # Set Ping Interval
        menu.Append( MENU_PING_INTERVAL, 'Ping Interval', 'Change the ping interval.' )
        EVT_MENU( self, MENU_PING_INTERVAL, self.ConfigPingInterval )
        self.mainMenu.Append( menu, '&Configuration' )

        # Add the menus to the main menu bar
        self.SetMenuBar( self.mainMenu )

        # Disable register, unregister & stop server by default
        self.mainMenu.Enable( MENU_STOP_SERVER, false )
        self.mainMenu.Enable( MENU_REGISTER_SERVER, false )
        self.mainMenu.Enable( MENU_UNREGISTER_SERVER, false )

        # Disable the ping menu items
        self.mainMenu.Enable( MENU_START_PING_PLAYERS, false )
        self.mainMenu.Enable( MENU_STOP_PING_PLAYERS, false )
        self.mainMenu.Enable( MENU_PING_INTERVAL, false )



    def build_body(self):
        """ Create the ViewNotebook and logger. """
        splitter = wxSplitterWindow(self, -1, style=wxNO_3D|wxSP_3D)

        nb = wxNotebook( splitter, -1 )
        self.conns = Connections( nb, self )
        nb.AddPage( self.conns, "Players" )

        ##nb.AddPage( self.conns, "Rooms" )

        self.msgWindow = HTMLMessageWindow( nb )
        nb.AddPage( self.msgWindow, "Messages" )

        log = wxTextCtrl(splitter, -1, style = wxTE_MULTILINE|wxTE_READONLY|wxHSCROLL)
        wxLog_SetActiveTarget( wxLogTextCtrl(log) )

        splitter.SplitHorizontally(nb, log, 400)
        #splitter.SetSashPosition(400, true)
        splitter.SetMinimumPaneSize(40)

        self.nb = nb
        self.log = log


    def build_status(self):
        """ Create status bar and set defaults. """
        sb = wxStatusBar(self, -1)
        sb.SetFieldsCount(5)
        sb.SetStatusWidths([-1, 115, 115, 65, 200])
        sb.SetStatusText("Sent: 0", 1)
        sb.SetStatusText("Recv: 0", 2)
        sb.SetStatusText("Stopped", 3)
        sb.SetStatusText("Unregistered", 4)
        self.SetStatusBar(sb)
        self.sb = sb


    def show_error(self, mesg, title = "Error"):
        """ Display an error.
            mesg (str) -- error message to display
            title (str) -- title of window
        """
        dlg = wxMessageDialog(self, mesg, title, wxOK | wxICON_EXCLAMATION)
        dlg.ShowModal()
        dlg.Destroy()


    # Event handler for out logging event
    def OnLogMessage( self, event ):
        self.Log( event.message )


    # Event handler for out logging event
    def OnFunctionMessage( self, event ):
        self.callbacks[event.func]( event.message )


    ### Server Callbacks #####################################
    def Log(self, log):
        wxLogMessage(str(log))


    def OnConnect(self, player):
        self.conns.add(player)


    def OnDisconnect(self, id):
        self.conns.remove(id)


    def OnUpdatePlayer(self, data):
        self.conns.update(data)


    def OnDataSent(self, bytes):
        self.total_messages_sent += 1
        self.total_data_sent += bytes
        self.sb.SetStatusText("Sent: %s (%d)" % (format_bytes(self.total_data_sent), self.total_messages_sent), 1)


    def OnDataRecv(self, bytes):
        self.total_messages_received += 1
        self.total_data_received += bytes
        self.sb.SetStatusText("Recv: %s (%d)" % (format_bytes(self.total_data_received), self.total_messages_received), 2)


    def OnCreateGroup( self, data ):
        (id, name) = data
        self.conns.roomList[id] = name
        print "room list: ", self.conns.roomList


    def OnDeleteGroup( self, data ):
        (from_id, id) = data
        del self.conns.roomList[id]
        print "room list: ", self.conns.roomList


    def OnJoinGroup( self, data ):
        self.conns.updateRoom( data )


    def OnSetRole( self, data ):
        (id, role) = data
        self.conns.setPlayerRole( id, role )



    ### Misc. ################################################

    def OnStart(self, event = None):
        """ Start server. """
        if self.STATUS == SERVER_STOPPED:
            serverNameEntry = wxTextEntryDialog( self, "Please Enter The Server Name You Wish To Use:",
                                                 "Server's Name", self.serverName, wxOK|wxCANCEL|wxCENTRE )

            if serverNameEntry.ShowModal() == wxID_OK:
                self.serverName = serverNameEntry.GetValue()

                if len( self.serverName ):
                    wxBeginBusyCursor()
                    self.server = ServerMonitor( self.callbacks, self.conf, self.serverName )
                    self.server.start()
                    self.STATUS = SERVER_RUNNING
                    self.sb.SetStatusText("Running", 3)
                    self.SetTitle(__appname__ + "- (running) - (unregistered)")
                    self.mainMenu.Enable( MENU_START_SERVER, false )
                    self.mainMenu.Enable( MENU_STOP_SERVER, true )
                    self.mainMenu.Enable( MENU_REGISTER_SERVER, true )
                    wxEndBusyCursor()
                else:
                    self.show_error("Server is already running.", "Error Starting Server")


    def OnStop(self, event = None):
        """ Stop server. """
        if self.STATUS == SERVER_RUNNING:
            self.OnUnregister()
            self.server.stop()
            self.STATUS = SERVER_STOPPED
            self.sb.SetStatusText("Stopped", 3)
            self.SetTitle(__appname__ + "- (stopped) - (unregistered)")
            self.mainMenu.Enable( MENU_STOP_SERVER, false )
            self.mainMenu.Enable( MENU_START_SERVER, true )
            self.mainMenu.Enable( MENU_REGISTER_SERVER, false )
            self.mainMenu.Enable( MENU_UNREGISTER_SERVER, false )

            # Delete any items that are still in the player list
            self.conns.DeleteAllItems()



    def OnRegister(self, event = None):
        """ Call into mplay_server's register() function.
            This will begin registerThread(s) to keep server
            registered with configured metas
        """

        if len( self.serverName ):
            wxBeginBusyCursor()
            self.server.server.register(self.serverName)

            self.sb.SetStatusText( ("%s" % (self.serverName)), 4 )
            self.mainMenu.Enable( MENU_REGISTER_SERVER, false )
            self.mainMenu.Enable( MENU_UNREGISTER_SERVER, true )
            self.mainMenu.Enable( MENU_STOP_SERVER, false )
            self.SetTitle(__appname__ + "- (running) - (registered)")
            wxEndBusyCursor()



    def OnUnregister(self, event = None):
        """ Call into mplay_server's unregister() function.
            This will kill any registerThreads currently running
            and result in the server being de-listed
            from all metas
        """

        wxBeginBusyCursor()
        self.server.server.unregister()

        self.sb.SetStatusText( "Unregistered", 4 )
        self.mainMenu.Enable( MENU_UNREGISTER_SERVER, false )
        self.mainMenu.Enable( MENU_REGISTER_SERVER, true )
        self.mainMenu.Enable( MENU_STOP_SERVER, true )
        self.SetTitle(__appname__ + "- (running) - (unregistered)")
        wxEndBusyCursor()



    def PingPlayers( self, event = None ):
        "Ping all players that are connected at a periodic interval, detecting dropped connections."
        wxBeginBusyCursor()
        wxYield()
        wxEndBusyCursor()


    def StopPingPlayers( self, event = None ):
        "Stop pinging connected players."


    def ConfigPingInterval( self, event = None ):
        "Configure the player ping interval.  Note that all players are pinged on a single timer."


    def OnExit(self, event = None):
        """ Quit the program. """
        self.OnStop()
        self.Destroy()



class ServerGUIApp(wxApp):
    def OnInit(self):
        # Make sure our image handlers are loaded before we try to display anything
        wxInitAllImageHandlers()
        self.splash = SplashScreen(None, bitmapfile=orpg.dirpath.dir_struct["icon"]+'splash.gif',
                              duration=2000, callback=self.AfterSplash)
        self.splash.Show(true)
        wxYield()
        return true


    def AfterSplash(self):
        self.splash.Close(true)
        frame = ServerGUI(None, -1, __appname__ + "- (stopped) - (unregistered)")
        frame.Show(true)
        frame.Raise()
        self.SetTopWindow(frame)



class HTMLMessageWindow( wxHtmlWindow ):
    "Widget used to present user to admin messages, in HTML format, to the server administrator"

    # Init using the derived from class
    def __init__( self, parent ):
        wxHtmlWindow.__init__( self, parent )


    def OnLinkClicked( self, ref ):
        "Open an external browser to resolve our About box links!!!"

        href = ref.GetHref()
        webbrowser.open( href )



if __name__ == '__main__':
    app = ServerGUIApp(0)
    app.MainLoop()

