;;;-*-Mode: LISP; Package: CCL -*-
;;;
;;;   Copyright (C) 1994-2001 Digitool, Inc
;;;   This file is part of Opensourced MCL.
;;;
;;;   Opensourced MCL is free software; you can redistribute it and/or
;;;   modify it under the terms of the GNU Lesser General Public
;;;   License as published by the Free Software Foundation; either
;;;   version 2.1 of the License, or (at your option) any later version.
;;;
;;;   Opensourced MCL is distributed in the hope that it will be useful,
;;;   but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;;   Lesser General Public License for more details.
;;;
;;;   You should have received a copy of the GNU Lesser General Public
;;;   License along with this library; if not, write to the Free Software
;;;   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;;;



(in-package :ccl)




; Returns two values: the arglist & it's functions binding.
; If the second arg is NIL, there was no function binding.
(defun arglist (sym &optional include-bindings (use-help-file t))
  (%arglist sym include-bindings nil use-help-file))

(defun arglist-string (sym &optional include-bindings)
  (multiple-value-bind (res type)
                       (%arglist-internal sym include-bindings nil nil t)
    (values
     (if (stringp res)
       res
       (and res (prin1-to-string res)))
     type)))

(defun set-arglist (sym arglist)
  (let ((real-sym (arglist-sym-and-def sym)))
    (when (or real-sym (null sym))
      (if (eq arglist t)
        (remhash real-sym %lambda-lists%)
        (setf (gethash real-sym %lambda-lists%) arglist)))))

(defsetf arglist set-arglist)

; Same as ARGLIST, but has the option of using TEMP-CONS instead of CONS
; to cons up the list.
(defun %arglist (sym &optional include-bindings temp-cons-p (use-help-file t))
  (multiple-value-bind (res type)
                       (%arglist-internal
                        sym include-bindings temp-cons-p nil use-help-file)
    (when (stringp res)
      (with-input-from-string (stream res)
        (setq res nil)
        (let ((eof (list nil))
              val errorp)
          (declare (dynamic-extent eof))
          (loop
            (multiple-value-setq (val errorp)
              (ignore-errors (values (read stream nil eof))))
            (when errorp
	      #+help-file ; %HEL temporarily avoiding reference to help file
              (if use-help-file
                (return-from %arglist 
                  (%arglist sym include-bindings temp-cons-p nil)))
              (push '&rest res)
              (push ':unparseable res)
              (return))
            (when (eq val eof)
              (return))
            (push val res))
          (setq res
                (if (and (null (cdr res)) (listp (car res)))
                  (car res)
                  (nreverse res))))))
    (values res type)))

(defun %arglist-internal (sym include-bindings temp-cons-p string use-help-file 
                              &aux def type)
  (multiple-value-setq (sym def) (arglist-sym-and-def sym))
  (when (standard-generic-function-p def)
    (let ((methods (%gf-methods def)))
      (if methods
        (setq def (closure-function
                   (find-unencapsulated-definition
                     (%method-function (car methods))))
              type :analysis))))
  (let ((ll (gethash sym %lambda-lists% *eof-value*))
        (conser (if temp-cons-p #'temp-cons #'cons))
        (macrop (and (symbolp sym) (eq (macro-function sym) def))))
    (flet ((strip (f) (if (stringp f) f (strip-bindings f include-bindings conser))))
      (declare (dynamic-extent #'strip))
      (cond ((neq ll *eof-value*) (values (strip ll) :declaration))
            ((consp def)
             ;; Presumably (lambda (... arglist) ...)
             (values (strip (cadr def)) :definition))
            ((neq (setq ll (getf (%lfun-info def) 'arglist *eof-value*)) *eof-value*)
             (values ll :definition))
            ((and (not macrop) (setq ll (uncompile-function def)))
             (values (strip (cadr ll)) (or type :definition)))
            ((lfunp def)
             (multiple-value-bind (arglist gotit) 
                                  (unless macrop (arglist-from-map def conser))
               (if gotit
                 (values arglist :analysis)
                 (cond #+help-file ; %HEL temporarily avoiding reference to help file
		       ((and use-help-file
                             sym
                             (setq arglist (arglist-from-help-file sym string)))
                        (values arglist :declaration))
                       (macrop (values nil :unknown))
                       (t (values (arglist-from-compiled-def def conser) :analysis))))))
            #+help-file ; %HEL temporarily avoiding reference to help file
	    ((and use-help-file (setq def (arglist-from-help-file sym string)))
             (values def :declaration))
            (t (values nil nil))))))

#+help-file ; %HEL temporarily avoiding reference to help file
(defun arglist-from-help-file (sym &optional string)
  (flet ((string-reader (reader arg)
           (unless (and string (stringp string)
                        (array-has-fill-pointer-p string))
             (setq string (make-array 10 :fill-pointer 0 :adjustable t
                                      :element-type 'base-char)))
           (setf (fill-pointer string) 0)
           (let (char)
             (loop                         ; strip the font info
               (setq char (funcall reader arg))
               (loop
                 (unless (eql #\@ char) (return))
                 (setq char (funcall reader arg))
                 (if (eql #\@ char)
                   (return)
                   (setq char (funcall reader arg))))
               (when (or (null char) (eql #\newline char))
                 (return string))
               (vector-push-extend char string)))))
    (declare (dynamic-extent #'string-reader))
    (get-help-file-entry sym #'string-reader)))            

(defun strip-bindings (arglist include-bindings conser)
  (if include-bindings
    arglist
    (macrolet ((push (elt list)
                 `(setq ,list (funcall conser ,elt ,list))))
      (let ((res nil))
        (do ((args arglist (%cdr args)))
            ((not (consp args)) (nreconc res args))
          (let ((arg (car args)))
            (cond ((atom arg)
                   (push arg res))
                  ((atom (car arg))
                   (push (car arg) res))
                  (t (push (caar arg) res)))))))))

(defun arglist-sym-and-def (sym &aux def)
  (cond ((functionp sym)
         (setq def sym
               sym (function-name def))
         (unless (and (symbolp sym) (eq def (fboundp sym)))
           (setq sym nil)))
        ((listp sym)
         (if (eq (car sym) 'setf)
           (setq sym (setf-function-name (cadr sym))
                 def (find-unencapsulated-definition (fboundp sym)))
           (setq sym nil def nil)))
        ((standard-method-p sym)
         (setq def (closure-function 
                    (find-unencapsulated-definition (%method-function sym)))))
        ((and (macro-function sym))
         (setq def (macro-function sym)))
        ((special-operator-p sym)
         nil)
        (t (setq def (find-unencapsulated-definition (fboundp sym)))))
  (values sym (if (standard-generic-function-p def) def (closure-function def))))

(defun arglist-from-map (lfun &optional (conser #'cons))
  (multiple-value-bind (nreq nopt restp nkeys allow-other-keys
                             optinit lexprp
                             ncells nclosed)
                       (function-args lfun)
    (declare (ignore optinit ncells))
    (macrolet ((push (elt list)
                     `(setf ,list (funcall conser ,elt ,list))))
      (when (not lexprp)
        (let ((map (car (function-symbol-map lfun))))
          (when map
            (let ((total (+ nreq nopt (if restp 1 0) (or nkeys 0)))
                  (idx (- (length map) nclosed))
                  (res nil))
              (if (%izerop total)
                (values nil t)
                (progn
                  (dotimes (x nreq)
                    (declare (fixnum x))
                    (push (if (> idx 0) (elt map (decf idx)) (make-arg "REQ" x)) res))
                  (when (neq nopt 0)
                    (push '&optional res)
                    (dotimes (x (the fixnum nopt))
                      (push (if (> idx 0) (elt map (decf idx)) (make-arg "OPT" x)) res)))
                  (when restp
                    (push '&rest res)
                    (push (if (> idx 0) (elt map (decf idx)) 'the-rest) res))
                  (when nkeys
                    (push '&key res)
                    (let ((keyvect (lfun-keyvect lfun)))
                      (dotimes (i (length keyvect))
                        (push (elt keyvect i) res))))
                  (when allow-other-keys
                    (push '&allow-other-keys res))))
              (values (nreverse res) t))))))))

(defvar *make-arg-string* (make-array 6 
                                      :element-type 'base-char
                                      :fill-pointer 0
                                      :adjustable t))
(defun make-arg (prefix count)
  (without-interrupts
   (let ((string *make-arg-string*)
         (*make-arg-string* nil))
     (setf (fill-pointer string) 0)
     (format string "~a-~d" prefix count)
     (consless-intern string :ccl))))

(defun consless-intern (string &optional package)
  (if (and (stringp string) (not (simple-string-p string)))
    (with-managed-allocation
      (let* ((length (length string))
             (copy (%make-temp-uvector length arch::subtag-simple-base-string)))
        (dotimes (i length) (declare (fixnum i)) (setf (uvref copy i) (aref string i)))
        (or (find-symbol copy package)
            (intern (ensure-simple-string string) package))))
    (intern string package)))

(defun arglist-from-compiled-def (lfun &optional (conser #'cons) 
                                       &aux (res nil) argnames)
  (multiple-value-bind (nreq nopt restp nkeys allow-other-keys
                        optinit lexprp
                        ncells nclosed)
          (function-args lfun)
    (declare (ignore optinit ncells nclosed))
    (when (typep lfun 'interpreted-function)
      (setq argnames (evalenv-names (%nth-immediate lfun 0))))
    (macrolet ((push (elt list)
                     `(setf ,list (funcall conser ,elt ,list))))
      (flet ((push-various-args (prefix count)
               (dotimes (i (the fixnum count))
                 (push (make-arg prefix i) res))))
        (declare (dynamic-extent #'push-various-args))
	;; Treat &LEXPR like &REST.
	(if lexprp (setq restp t lexprp nil))
        (cond ((and (eq 0 (+ nreq nopt (or nkeys 0))) (not restp))
               nil)
              (t 
               (if argnames
                 (setq res (reverse (butlast argnames (- (length argnames) nreq))))
                 (push-various-args "ARG" nreq))
               (when (> nopt 0)
                 (push '&optional res)
                 (if argnames
                   (setq res (append (reverse (subseq argnames nreq (+ nreq nopt))) res))
                   (push-various-args "OPT" nopt)))
               (when restp
                 (push '&rest res)
                 (if argnames
                   (push (nth (+ nreq nopt) argnames) res)
                   (push 'the-rest res)))
               (when nkeys
                 (push '&key res)
                 (let ((keyvect (lfun-keyvect lfun)))
                   (dotimes (i (length keyvect))
                     (push (elt keyvect i) res))))
               (when allow-other-keys
                 (push '&allow-other-keys res))
               (nreverse res)))))))

; End of arglist.lisp
