;;;-*-Mode: LISP; Package: CCL -*-
;;;
;;;   Copyright (C) 1994-2001 Digitool, Inc
;;;   This file is part of Opensourced MCL.
;;;
;;;   Opensourced MCL is free software; you can redistribute it and/or
;;;   modify it under the terms of the GNU Lesser General Public
;;;   License as published by the Free Software Foundation; either
;;;   version 2.1 of the License, or (at your option) any later version.
;;;
;;;   Opensourced MCL is distributed in the hope that it will be useful,
;;;   but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;;   Lesser General Public License for more details.
;;;
;;;   You should have received a copy of the GNU Lesser General Public
;;;   License along with this library; if not, write to the Free Software
;;;   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;;;


; l1-clos.lisp

(in-package :ccl)





(defvar *type-system-initialized* nil)

(eval-when (eval compile)
  (require 'defstruct-macros))

(eval-when (:compile-toplevel :execute)
  (defmacro make-instance-vector (len)
    `(allocate-typed-vector :INSTANCE ,len (%unbound-marker-8)))
)

(eval-when (:compile-toplevel :execute)
  (defmacro make-structure-vector (size)
    `(%alloc-misc ,size arch::subtag-struct nil))

)
;;;;;;;;;;;;;;;;;;;;;;;;;;; defmethod support ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(%fhave 'function-encapsulation ;Redefined in encapsulate
        (qlfun bootstrapping-function-encapsulation (name)
          (declare (ignore name))
          nil))

(%fhave '%move-method-encapsulations-maybe ; Redefined in encapsulate
        (qlfun boot-%move-method-encapsulations-maybe (m1 m2)
          (declare (ignore m1 m2))
          nil))


(%fhave 'find-unencapsulated-definition  ;Redefined in encapsulate
        (qlfun bootstrapping-unenecapsulated-def (spec)
          (values
           (typecase spec
             (symbol (fboundp spec))
             (method (%method-function spec))
             (t spec))
           spec)))



(defun %inner-method-function (method)
  (let ((f (%method-function method)))
    (when (function-encapsulation f)
      (setq f (find-unencapsulated-definition f)))
    (closure-function f)))




(defun copy-method-function-bits (from to)
  (let ((new-bits (logior (logand (logior (lsh 1 $lfbits-method-bit)
                                          (ash 1 $lfbits-nextmeth-bit)
                                          (ash 1 $lfbits-nextmeth-with-args-bit)
                                          $lfbits-args-mask) 
                                  (lfun-bits from))
                          (logand (lognot (logior (lsh 1 $lfbits-method-bit)
                                                  (ash 1 $lfbits-nextmeth-bit)
                                                  (ash 1 $lfbits-nextmeth-with-args-bit)
                                                  $lfbits-args-mask))
                                  (lfun-bits to)))))
    (lfun-bits to new-bits)
    new-bits))

(defun %defmethod (function specializers qualifiers &optional method-class documentation)
  (let* ((method (%anonymous-method
                  function specializers qualifiers method-class documentation t))
         (name (%method-name method))
         (fbinding (fboundp name))
         (gf (if (functionp fbinding) fbinding))
         (already-gf (and gf (typep gf 'standard-generic-function))))
    (unless already-gf
      (when fbinding
        (unless (defmethod-congruency-override name nil)
          (cerror "Replace it with a generic-function."
                  "~S is not a generic function" (or gf name))))
      (setq gf (make-gf name (lfun-bits (closure-function function)))))    
    (let ((oldmethod (when fbinding
                       (ignore-errors 
                        (find-method gf qualifiers (mapcar #'(lambda (s)
                                                               (if (symbolp s)(find-class s) s))
                                                             specializers)
                                                   nil)))))
      (record-source-file method 'method)
      (check-defmethod-congruency gf method)
      (%add-method method gf t)
      (unless already-gf
        (if fbinding
          (forget-encapsulations name))
        (%fhave name gf))
      ; add-method is unhappy about lambda lists if we do this first
      (when oldmethod (%move-method-encapsulations-maybe oldmethod method))
      method)))


(defun forget-encapsulations (name)
  (declare (ignore name))
  nil)

(defun %anonymous-method (function specializers qualifiers &optional method-class documentation
                                   default-method-class?
                                   &aux name)
  (let ((inner-function (closure-function function)))
    (unless (%method-function-p inner-function)
      (report-bad-arg inner-function 'method-function))   ; Well, I suppose we'll have to shoot you.
    (unless (eq inner-function function)   ; must be closed over
      (copy-method-function-bits inner-function function))
    (setq name (function-name inner-function))
    (if (typep name 'standard-method)     ; method-function already installed.
      (setq name (%method-name name)))
    (setq method-class
          (if method-class
            (find-class method-class)
            (or (and default-method-class?
                     (let ((gf (fboundp name)))
                       (and gf 
                            (standard-generic-function-p gf)
                            (%gf-method-class gf))))
                *standard-method-class*)))
    (unless (memq *standard-method-class* (%class-cpl (initialize-class method-class)))
      (%badarg method-class 'standard-method))
;    (unless (member qualifiers '(() (:before) (:after) (:around)) :test #'equal)
;      (report-bad-arg qualifiers))
    (setq specializers (mapcar #'(lambda (s)
                                   (or (and (consp s)
                                            (eq (%car s) 'eql)
                                            (consp (%cdr s))
                                            (null (%cddr s))
                                            s)
                                       (and (specializer-p s) s)
                                       (find-class s)))
                               specializers))
    (let ((method (if (eq method-class *standard-method-class*)
                    (%cons-method name qualifiers specializers function method-class)
                    (make-instance method-class
                      :name name
                      :qualifiers qualifiers
                      :specializers specializers
                      :function function))))
      (lfun-name inner-function method)
      (when documentation
        (setf (documentation method t) documentation))
      method)))


(defun check-defmethod-congruency (gf method)
  (unless (congruent-lambda-lists-p gf method)
    ;(dbg 7)  ; piece of crap - got here
    (unless (defmethod-congruency-override gf method)
      (cerror (format nil
                      "Remove ~d method~:p from the generic-function and change its lambda list."
                      (length (%gf-methods gf)))
              "Incompatible lambda list in ~S for ~S" method gf))
    (loop
      (let ((methods (%gf-methods gf)))
        (if methods
          (remove-method gf (car methods))
          (return))))
    (%set-defgeneric-keys gf nil)
    (inner-lfun-bits gf (%ilogior (%ilsl $lfbits-gfn-bit 1)
                            (%ilogand $lfbits-args-mask
                                      (lfun-bits (%method-function method))))))
  gf)

; A symbol or a function of two args: the generic-function & the new method that disagree.
(defvar *defmethod-congruency-override* nil)

(defun defmethod-congruency-override (gf method)
  (let ((override *defmethod-congruency-override*))
    (if (functionp override)
      (funcall override gf method)
      override)))

(defun %method-function-method (method-function)
  (setq method-function
        (closure-function
         (if (function-encapsulation method-function)
           (find-unencapsulated-definition method-function)
           method-function)))
  (setq method-function (require-type method-function 'method-function))
  (lfun-name method-function))

(defvar %defgeneric-methods% (make-hash-table :test 'eq :weak t))
(defvar %defgeneric-keys% (make-hash-table :test 'eq :weak t))

(defun %defgeneric-methods (gf)
   (gethash gf %defgeneric-methods%))

(defun %set-defgeneric-methods (gf &rest methods)
   (if methods
     (setf (gethash gf %defgeneric-methods%) methods)
     (remhash gf %defgeneric-methods%)))

(defun %defgeneric-keys (gf)
   (gethash gf %defgeneric-keys%))

(defun %set-defgeneric-keys (gf keyvect)
   (if keyvect
     (setf (gethash gf %defgeneric-keys%) keyvect)
     (remhash gf %defgeneric-keys%)))




(defun congruent-lfbits-p (gbits mbits)
  (and (eq (ldb $lfbits-numreq gbits) (ldb $lfbits-numreq mbits))
       (eq (ldb $lfbits-numopt gbits) (ldb $lfbits-numopt mbits))
       (eq (or (logbitp $lfbits-rest-bit gbits)
               (logbitp $lfbits-restv-bit gbits)
               (logbitp $lfbits-keys-bit gbits))
           (or (logbitp $lfbits-rest-bit mbits)
               (logbitp $lfbits-restv-bit mbits)
               (logbitp $lfbits-keys-bit mbits)))))

(defun congruent-lambda-lists-p (gf method &optional
                                    error-p gbits mbits gkeys)
  (unless gbits (setq gbits (inner-lfun-bits gf)))
  (unless mbits (setq mbits (lfun-bits (%method-function method))))
  (and (congruent-lfbits-p gbits mbits)
       (or (and (or (logbitp $lfbits-rest-bit mbits)
                    (logbitp $lfbits-restv-bit mbits))
                (not (logbitp $lfbits-keys-bit mbits)))
           (logbitp $lfbits-aok-bit mbits)
           (progn
             (unless gkeys (setq gkeys (%defgeneric-keys gf)))
             (or (null gkeys)
                 (eql 0 (length gkeys))
                 (let ((mkeys (lfun-keyvect
                               (%inner-method-function method))))
                   (dovector (key gkeys t)
                     (unless (find key mkeys :test 'eq)
                       (if error-p
                         (error "~s does not specify keys: ~s" method gkeys))
                       (return nil)))))))))


(defun %add-method (method gfn &optional skip-congruency-test)
  (when (%method-gf method)
    (error "~s is already a method of ~s." method (%method-gf method)))
  (let* ((gbits (inner-lfun-bits gfn)) (mbits (lfun-bits (%method-function method))))
    (unless (or skip-congruency-test (congruent-lambda-lists-p gfn method t gbits mbits))
      (error "Incompatible lambda list in ~S for ~S" method gfn))
    (let* ((dt (%gf-dispatch-table gfn))
           (methods (%gf-dispatch-table-methods dt))
           (specializers (%method-specializers method))
           (qualifiers (%method-qualifiers method)))
      (remove-obsoleted-combined-methods method dt specializers)
      (apply #'invalidate-initargs-vector-for-gf gfn specializers)
      (block search
        (do* ((ms methods (cdr ms))
              (m (car ms) (car ms))
              (specs specializers specializers))
             ((null ms) (push method (%gf-dispatch-table-methods dt)))
          (when (equal qualifiers (%method-qualifiers m))
            (when (dolist (msp (%method-specializers m) t)
                    (let ((spec (%pop specs)))
                      (unless (if (listp msp)
                                (and (listp spec)
                                     (eql (%cadr msp) (%cadr spec)))
                                (eq msp spec))
                        (return nil))))
              (redefine-kernel-method (%car ms))
              (%remove-from-class-direct-methods (%car ms))
              (setf (%method-gf (%car ms)) nil)   ; old method no longer has a gf
              (setf (%car ms) method)   ; replace existing method
              (return-from search)))))
      (%add-to-class-direct-methods method)
      (setf (%method-gf method) gfn)
      (compute-dcode gfn dt)))
  gfn)

(defvar *standard-kernel-method-class* nil)

(defun redefine-kernel-method (method)
  (when (and *warn-if-redefine-kernel*
             (or (let ((class *standard-kernel-method-class*))
                   (and class (typep method class)))
                 (and (standard-method-p method)
                      (kernel-function-p (%method-function method)))))
    (cerror "Replace the definition of ~S."
            "The method ~S is already defined in the MCL kernel." method)))

; Called by the expansion of generic-labels
(defun %add-methods (gf &rest methods)
  (declare (dynamic-extent methods))
  (dolist (m methods)
    (add-method gf m)))

(defun methods-congruent-p (m1 m2)
  (when (and (standard-method-p m1)(standard-method-p m2))
    (when (equal (%method-qualifiers m1) (%method-qualifiers m2))
      (let ((specs (%method-specializers m1)))
        (dolist (msp (%method-specializers m2) t)
          (let ((spec (%pop specs)))
            (unless (if (listp msp)
                      (and (listp spec)
                           (eql (%cadr msp) (%cadr spec)))
                      (eq msp spec))
              (return nil))))))))

(defvar *maintain-class-direct-methods* nil)

(defun %class-direct-methods (class)
  (let ((v (%class-kernel-p.direct-methods class)))
    (and (consp v) (%cdr v))))

(defun (setf %class-direct-methods) (new-value class)
  (let ((v (%class-kernel-p.direct-methods class)))
    (if (consp v)
      (setf (%cdr v) new-value)
      (setf (%class-kernel-p.direct-methods class) (cons v new-value))))
  new-value)


(defun %class-kernel-p (class)
  (let ((v (%class-kernel-p.direct-methods class)))
    (if (consp v) (%car v) v)))

(defun (setf %class-kernel-p) (new-value class)
  (setq new-value (not (null new-value)))
  (let ((v (%class-kernel-p.direct-methods class)))
    (if (consp v) 
      (setf (%car v) new-value)
      (setf (%class-kernel-p.direct-methods class) new-value))))

; CAR is an EQL hash table for objects whose identity is not used by EQL
; (numbers and macptrs)
; CDR is a weak EQ hash table for other objects.
(defvar *eql-methods-hashes* (cons (make-hash-table :test 'eql)
                                   (make-hash-table :test 'eq :weak :key)))

(defun eql-methods-cell (object &optional addp)
  (let ((hashes *eql-methods-hashes*))
    (without-interrupts
     (let* ((hash (cond
                   ((or (typep object 'number)
                        (typep object 'macptr))
                    (car hashes))
                   (t (cdr hashes))))
            (cell (gethash object hash)))
       (when (and (null cell) addp)
         (setf (gethash object hash) (setq cell (cons nil nil))))
       cell))))


; The %class-direct-methods slot is a cons of (methods . generic-functions)
; The methods list is maintained here.
; The generic-functions list is computed lazily by CLASS-DIRECT-GENERIC-FUNCTIONS
(defun %add-to-class-direct-methods (method)
  #-bccl (setq method (require-type method 'standard-method))
  (let ((specs (%method-specializers method))
        cell)
    (dolist (spec specs)
      (when (or *maintain-class-direct-methods*
                (listp spec))
        (when (setq cell
                    (if (listp spec)    ; eql method
                      (eql-methods-cell (cadr spec) t)
                      (or (%class-direct-methods spec)
                          (setf (%class-direct-methods spec) (cons nil nil)))))
          (setf (%cdr cell) nil)          ; must recompute generic-functions list
          (setq cell
                (or (%car cell)
                    (setf (%car cell) (%cons-population nil))))
          (pushnew method (population-data cell))))))
  method)

(defun %remove-from-class-direct-methods (method)
  #-bccl (setq method (require-type method 'standard-method))
  (let ((specs (%method-specializers method))
        cell)
    (dolist (spec specs)
      (when (or *maintain-class-direct-methods*
                (listp spec))           ; eql method
        (setq cell
              (if (listp spec)
                (eql-methods-cell (cadr spec))
                (%class-direct-methods spec)))
        (setf (%cdr cell) nil)          ; must recompute generic-functions list
        (when (and cell (setq cell (%car cell)))
          (setf (population-data cell)
                (nremove method (population-data cell)))))))
  method)

(defun map-classes (function)
  (with-hash-table-iterator (m %find-classes%)
    (loop
      (multiple-value-bind (found name cell) (m)
        (declare (list cell))
        (unless found (return))
        (when (cdr cell)
          (funcall function name (cdr cell)))))))

(defun clear-specializer-direct-methods-caches ()
  (setq *maintain-class-direct-methods* nil)
  (map-classes #'(lambda (name class)
                   (declare (ignore name))
                   (when (typep class 'class)
                     (setf (%class-direct-methods class) nil)))))

(defun %class-primary-slot-accessor-info (class accessor-or-slot-name &optional create?)
  (let ((info-list (%class-get class '%class-primary-slot-accessor-info)))
    (or (car (member accessor-or-slot-name info-list
                     :key #'(lambda (x) (%slot-accessor-info.accessor x))))
        (and create?
             (let ((info (%cons-slot-accessor-info class accessor-or-slot-name)))
               (setf (%class-get class '%class-primary-slot-accessor-info)
                     (cons info info-list))
               info)))))

;; Clear the %class-primary-slot-accessor-info for an added or removed method's specializers
(defun clear-accessor-method-offsets (gf method)
  (when (or (typep method 'standard-accessor-method)
            (member 'standard-accessor-method
                    (%gf-methods gf)
                    :test #'(lambda (sam meth)
                             (declare (ignore sam))
                             (typep meth 'standard-accessor-method))))
    (labels ((clear-class (class)
               (when (typep class 'standard-class)
                 (let ((info (%class-primary-slot-accessor-info class gf)))
                   (when info
                     (setf (%slot-accessor-info.offset info) nil)))
                 (mapc #'clear-class (%class-subclasses class)))))
      (declare (dynamic-extent #'clear-class))
      (mapc #'clear-class (%method-specializers method)))))

;; Remove methods which specialize on a sub-class of method's specializers from
;; the generic-function dispatch-table dt.
(defun remove-obsoleted-combined-methods (method &optional dt
                                                 (specializers (%method-specializers method)))
  (without-interrupts
   (unless dt
     (let ((gf (%method-gf method)))
       (when gf (setq dt (%gf-dispatch-table gf)))))
   (when dt
     (if specializers
       (let* ((argnum (%gf-dispatch-table-argnum dt))
              (class (nth argnum specializers))
              (size (%gf-dispatch-table-size dt))
              (index 0))
         (clear-accessor-method-offsets (%gf-dispatch-table-gf dt) method)
         (if (listp class)                   ; eql specializer
           (setq class (class-of (second class))))
         (while (%i< index size)
           (let* ((wrapper (%gf-dispatch-table-ref dt index))
                  hash-index-0?
                  (cpl (and wrapper
                            (not (setq hash-index-0?
                                       (eql 0 (%wrapper-hash-index wrapper))))
                            (%inited-class-cpl
                             (require-type (%wrapper-class wrapper) 'class)))))
             (when (or hash-index-0? (and cpl (cpl-index class cpl)))
               (setf (%gf-dispatch-table-ref dt index) *obsolete-wrapper*
                     (%gf-dispatch-table-ref dt (%i+ index 1)) *gf-dispatch-bug*))
             (setq index (%i+ index 2)))))
       (setf (%gf-dispatch-table-ref dt 1) nil)))))   ; clear 0-arg gf cm

; SETQ'd below after the GF's exist.
(defvar *initialization-invalidation-alist* nil)

; Called by %add-method, %remove-method
(defun invalidate-initargs-vector-for-gf (gf &optional class &rest rest)
  (declare (ignore rest))
  (when (and (not (listp class))        ; no eql methods or gfs with no specializers need apply
             #-bccl (setq class (require-type class 'class)))
    (let ((indices (cdr (assq gf *initialization-invalidation-alist*))))
      (when indices
        (labels ((invalidate (class indices)
                             (when (std-class-p class)   ; catch the class named T
                               (dolist (index indices)
                                 (setf (%svref class index) nil))
                               (remove-aux-init-functions-cache class))
                             (dolist (subclass (%class-subclasses class))
                               (invalidate subclass indices))))
          (invalidate class indices))))))

;; Return two values:
;; 1) the index of the first non-T specializer of method, or NIL if
;;    all the specializers are T or only the first one is T
;; 2) the index of the first non-T specializer
(defun multi-method-index (method &aux (i 0) index)
  (dolist (s (%method-specializers method) (values nil index))
    (unless (eq s *t-class*)
      (unless index (setq index i))
      (unless (eql i 0) (return (values index index))))
    (incf i)))

(defun %remove-method (method)
  (setq method (require-type method 'standard-method))
  (let ((gf (%method-gf method)))
    (when gf
      (let* ((dt (%gf-dispatch-table gf)))
        (setf (%method-gf method) nil)
        (setf (%gf-dispatch-table-methods dt) 
              (nremove method (%gf-dispatch-table-methods dt)))
        (%remove-from-class-direct-methods method)
        (remove-obsoleted-combined-methods method dt)
        (apply #'invalidate-initargs-vector-for-gf gf (%method-specializers method))
        (compute-dcode gf dt))))
  method)

; values filled in below, after we can DEFCLASS
(defvar *accessor-method-class* nil)
(defvar *standard-reader-method-class* nil)
(defvar *standard-writer-method-class* nil)

(defparameter dcode-proto-alist
  (list (cons #'%%reader-dcode-no-lexpr  *gf-proto-one-arg*)
        (cons #'%%writer-dcode-no-lexpr *gf-proto-two-arg*)
        (cons #'%%one-arg-dcode *gf-proto-one-arg*)
        (cons #'%%1st-two-arg-dcode *gf-proto-two-arg*)))
    
(defun compute-dcode (gf &optional dt)
  (setq gf (require-type gf 'standard-generic-function))
  (unless dt (setq dt (%gf-dispatch-table gf)))
  (let* ((methods (%gf-dispatch-table-methods dt))
         (bits (inner-lfun-bits gf))
         (nreq (ldb $lfbits-numreq bits))
         (0-args? (eql 0 nreq))
         (other-args? (or (not (eql 0 (ldb $lfbits-numopt bits)))
                          (logbitp $lfbits-rest-bit bits)
                          (logbitp $lfbits-restv-bit bits)
                          (logbitp $lfbits-keys-bit bits)
                          (logbitp $lfbits-aok-bit bits)))
         multi-method-index 
	 min-index
         (only-readers t)
         (only-writers t)
         (can-optimize (if *compile-definitions* t nil))
         )
    (when methods
      (unless 0-args?
        (dolist (m methods)
          (when can-optimize
            (let ((method-class (class-of m)))
              (unless (or (if (eq method-class *standard-reader-method-class*)
                            (progn (setq only-writers nil)
                                   t)
                            (setq only-readers nil))
                          (if (eq method-class *standard-writer-method-class*)
                            t
                            (setq only-writers nil)))
                (setq can-optimize nil))))
          (multiple-value-bind (mm-index index) (multi-method-index m)
            (when mm-index
              (if (or (null multi-method-index) (< mm-index multi-method-index))
                (setq multi-method-index mm-index)))
            (when index
              (if (or (null min-index) (< index min-index))
                (setq min-index index))))))
      (let ((dcode (if 0-args?
                     #'%%0-arg-dcode
                     (or (and can-optimize 
                              (cond (only-readers #'%%reader-dcode-no-lexpr)
                                    (only-writers #'%%writer-dcode-no-lexpr)))
                         (if multi-method-index
                           #'%%nth-arg-dcode)
                         (if (null other-args?)
                           (if (eql nreq 1)
                             #'%%one-arg-dcode
                             (if (eql nreq 2)
                               #'%%1st-two-arg-dcode
                               #'%%1st-arg-dcode))                            
                             #'%%1st-arg-dcode)))))
        (setq multi-method-index
              (if multi-method-index
                (if min-index
                  (min multi-method-index min-index)
                  multi-method-index)
                0))
        (let* ((old-dcode (%gf-dcode gf))
               (encapsulated-dcode-cons (and (combined-method-p old-dcode)
                                             (eq '%%call-gf-encapsulation 
                                                 (function-name (%combined-method-dcode old-dcode)))
                                             (cdr (%combined-method-methods old-dcode)))))
          (when (or (neq dcode (if encapsulated-dcode-cons (cdr encapsulated-dcode-cons) old-dcode))
                    (neq multi-method-index (%gf-dispatch-table-argnum dt)))
            (let ((proto (or (cdr (assq dcode dcode-proto-alist)) *gf-proto*)))
              (clear-gf-dispatch-table dt)
              (setf (%gf-dispatch-table-argnum dt) multi-method-index)
              (if encapsulated-dcode-cons ; and more?
                (let ((old-gf (car encapsulated-dcode-cons)))
                  (if (not (typep old-gf 'generic-function))
                    (error "Confused"))
                  ;(setf (uvref old-gf 0)(uvref proto 0))
                  (setf (cdr encapsulated-dcode-cons) dcode))
                (progn 
                  (setf (%gf-dcode gf) dcode)
                  (setf (uvref gf 0)(uvref proto 0)))))))
        (values dcode multi-method-index)))))

(defun inherits-from-standard-generic-function-p (class)
  (memq *standard-generic-function-class*
        (%inited-class-cpl (require-type class 'class))))

;;;;;;;;;;; The type system needs to get wedged into CLOS fairly early ;;;;;;;


; Could check for duplicates, but not really worth it.  They're all allocated here
(defun new-type-class (name)
  (let* ((class (%istruct 
                 'type-class 
                 name
                 #'missing-type-method
                 nil
                 nil
                 #'(lambda (x y) (vanilla-union x y))
                 nil
                 #'(lambda (x y) (vanilla-intersection x y))
                 nil
                 #'missing-type-method
                 nil
                 #'missing-type-method)))
    (push (cons name class) *type-classes*)
    class))

;; There are ultimately about a dozen entries on this alist.
(defvar *type-classes* nil)
(declaim (special *wild-type* *empty-type* *universal-type*))
(defvar *type-kind-info* (make-hash-table :test #'equal))

(defun info-type-kind (name)
  (gethash name *type-kind-info*))

(defun (setf info-type-kind) (val name)
  (setf (gethash name *type-kind-info*) val))

(defun missing-type-method (&rest foo)
  (error "Missing type method for ~S" foo))
          
(new-type-class 'values)
(new-type-class 'function)
(new-type-class 'constant)
(new-type-class 'wild)
(new-type-class 'bottom)
(new-type-class 'named)
(new-type-class 'hairy)
(new-type-class 'unknown)
(new-type-class 'number)
(new-type-class 'array)
(new-type-class 'member)
(new-type-class 'union)
(new-type-class 'foreign)
(defparameter *class-type-class* (new-type-class 'class))


                        
;;;;;;;;;;;;;;;;;;;;;;;;  Instances and classes ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar %find-classes% (make-hash-table :test 'eq))

(defun class-cell-typep (form class-cell)
  (unless (listp class-cell)(error "puke"))
  (locally (declare (type list class-cell))
    (let ((class (cdr class-cell)))
      (when (not class)
        (setq class (find-class (car class-cell) nil))
        (when class (rplacd class-cell class)))
      (if class
        (not (null (memq class (%inited-class-cpl (class-of form)))))
        (if (fboundp 'typep)(typep form (car class-cell)) t)))))


;(defvar puke nil)

(defun %require-type-class-cell (arg class-cell)
  ; sort of weird  
  (if (or ;(not *type-system-initialized*)
          (not (listp class-cell)))  ; bootstrapping prob no longer
    arg ; (progn (pushnew class-cell puke) arg)
    (if (class-cell-typep arg class-cell)
      arg
      (%kernel-restart $xwrongtype arg (car class-cell)))))



(defun find-class-cell (name create?)
  (let ((cell (gethash name %find-classes%)))
    (or cell
        (and create?
             (setf (gethash name %find-classes%) (cons name nil))))))


(defun find-class (name &optional (errorp t) environment)
  (let* ((cell (find-class-cell name nil)))
    (declare (list cell))
    (or (cdr cell)
        (let ((defenv (and environment (definition-environment environment))))
          (when defenv
            (dolist (class (defenv.classes defenv))
              (when (eq name (%class-name class))
                (return class)))))
        (when (or errorp (not (symbolp name)))
          (error "Class named ~S not found." name)))))

(defun set-find-class (name class)
  (clear-type-cache)
  (let ((cell (find-class-cell name class)))
    (when cell
      (setf (info-type-kind name) :instance)
      (setf (cdr (the cons cell)) class))
    class))


; bootstrapping definition. real one is in "sysutils.lisp"

(defun built-in-type-p (name)
  (or (type-predicate name)
      (memq name '(signed-byte unsigned-byte mod 
                   values satisfies member and or not))
      (typep (find-class name nil) 'built-in-class)))



(defun %compile-time-defclass (name environment)
  (unless (find-class name nil environment)
    (let ((defenv (definition-environment environment)))
      (when defenv
        (push (make-instance 'compile-time-class :name name)
              (defenv.classes defenv)))))
  name)

(queue-fixup
 (without-interrupts 
  (defun set-find-class (name class)
    (setq name (require-type name 'symbol))
    (let ((cell (find-class-cell name class)))
      (declare (type list cell))
      (when *warn-if-redefine-kernel*
        (let ((old-class (cdr cell)))
          (when (and old-class (neq class old-class) (%class-kernel-p old-class))
            (cerror "Redefine ~S."
                    "~S is already defined in the CCL kernel." old-class)
            (setf (%class-kernel-p old-class) nil))))
      (when (null class)
        (when cell
          (setf (cdr cell) nil))
        (return-from set-find-class nil))
      (setq class (require-type class 'class))
      (when (built-in-type-p name)
        (unless (eq (cdr cell) class)
          (error "Cannot redefine built-in type name ~S" name)))
      (when (%deftype-expander name)
        (cerror "set ~S anyway, removing the ~*~S definition"
                "Cannot set ~S because type ~S is already defined by ~S"
                `(find-class ',name) name 'deftype)
        (%deftype name nil nil))
      (setf (info-type-kind name) :instance)
      (setf (cdr cell) class)))
  ) ; end of without-interrupts
 ) ; end of queue-fixup

#|
(defun class-name (class)
  (%class-name (require-type class 'class)))
|#

#|
; This tended to cluster entries in gf dispatch tables too much.
(defvar *class-wrapper-hash-index* 0)
(defun new-class-wrapper-hash-index ()
  (let ((index *class-wrapper-hash-index*))
    (setq *class-wrapper-hash-index*
        (if (< index (- most-positive-fixnum 2))
          ; Increment by two longwords.  This is important!
          ; The dispatch code will break if you change this.
          (%i+ index 3)                 ; '3 = 24 bytes = 6 longwords in lap.
          1))))
|#

(defvar *class-wrapper-random-state* (make-random-state))

(defun new-class-wrapper-hash-index ()
  ; mustn't be 0
  (the fixnum (1+ (the fixnum (random most-positive-fixnum *class-wrapper-random-state*)))))

; Initialized after built-in-class is made
(defvar *built-in-class-wrapper* nil)

(defun make-class-ctype (class)
  (%istruct 'class-ctype *class-type-class* nil class))

(defvar *t-class* (let ((class (%cons-built-in-class 't)))
                    (setf (%class-cpl class) (list class))
                    (setf (%class-own-wrapper class)
                          (%cons-wrapper class (new-class-wrapper-hash-index)))
                    (setf (%class-ctype class) (make-class-ctype class))
                    (setf (find-class 't) class)
                    class))

(defun compute-cpl (class)
  (flet ((%real-class-cpl (class)
           (or (%class-cpl class)
               (compute-cpl class))))
    (let* ((predecessors (list (list class))) candidates cpl)
      (dolist (sup (%class-local-supers class))
        (when (symbolp sup) (report-bad-arg sup 'class))
        (dolist (sup (%real-class-cpl sup))
          (unless (assq sup predecessors) (push (list sup) predecessors))))
      (labels ((compute-predecessors (class table)
                 (dolist (sup (%class-local-supers class) table)
                   (compute-predecessors sup table)
                   ;(push class (cdr (assq sup table)))
                   (let ((a (assq sup table))) (%rplacd a (cons class (%cdr a))))
                   (setq class sup))))
        (compute-predecessors class predecessors))
      (setq candidates (list (assq class predecessors)))
      (while predecessors
        (dolist (c candidates (error "Inconsistent superclasses for ~d" class))
          (when (null (%cdr c))
            (setq predecessors (nremove c predecessors))
            (dolist (p predecessors) (%rplacd p (nremove (%car c) (%cdr p))))
            (setq candidates (nremove c candidates))
            (setq cpl (%rplacd c cpl))
            (dolist (sup (%class-local-supers (%car c)))
              (when (setq c (assq sup predecessors)) (push c candidates)))
            (return))))
      (setq cpl (nreverse cpl))
      (do* ((tail cpl (%cdr tail))
            sup-cpl)
           ((null (setq sup-cpl (and (cdr tail) (%real-class-cpl (cadr tail))))))
        (when (equal (%cdr tail) sup-cpl)
          (setf (%cdr tail) sup-cpl)
          (return)))
      cpl)))

(defun make-built-in-class (name &rest supers)
  ;(dbg name)
  (if (null supers)
    (setq supers (list *t-class*))
    (do ((supers supers (%cdr supers)))
        ((null supers))
      (when (symbolp (%car supers)) (%rplaca supers (find-class (%car supers))))))
  (let ((class (find-class name nil)))
    (if class
      (progn
        ;Must be debugging.  Give a try at redefinition...
        (dolist (sup (%class-local-supers class))
          (setf (%class-subclasses sup) (nremove class (%class-subclasses sup)))))
      (setq class (%cons-built-in-class name)))
    (dolist (sup supers)
      (setf (%class-subclasses sup) (cons class (%class-subclasses sup))))
    (setf (%class-local-supers class) supers)
    (setf (%class-cpl class) (compute-cpl class))
    (setf (%class-own-wrapper class) (%cons-wrapper class (new-class-wrapper-hash-index)))
    (setf (%class-ctype class)  (make-class-ctype class))
    (setf (find-class name) class)
    (dolist (sub (%class-subclasses class))   ; Only non-nil if redefining
      ;Recompute the cpl.
      (apply #'make-built-in-class (%class-name sub) (%class-local-supers sub)))
    class))

;; This will be filled in below.  Need it defined now as it goes in the
;; %instance-class-wrapper of all the classes that standard-class inherits from.
(defvar *standard-class-wrapper* 
  (%cons-wrapper 'standard-class))

(defun make-standard-class (name &rest supers)
  (make-class name *standard-class-wrapper* supers))

(defun make-class (name metaclass-wrapper supers)
  (let ((class (if (find-class name nil)
                 (error "Attempt to remake standard class ~s" name)
                 (%cons-standard-class name metaclass-wrapper))))
    (if (null supers)
      (setq supers (list *standard-class-class*))
      (do ((supers supers (cdr supers))
           sup)
          ((null supers))
        (setq sup (%car supers))
        (if (symbolp sup) (setf (%car supers) (setq sup (find-class (%car supers)))))
        (unless (or (eq sup *t-class*) (std-class-p sup))
          (error "~a is not of type ~a" sup 'std-class))))
    (setf (%class-local-supers class) supers)
    (let ((cpl (compute-cpl class))
          (wrapper (%cons-wrapper class)))
      (setf (%class-cpl class) cpl
            (%wrapper-instance-slots wrapper) (vector)
            (%class-own-wrapper class) wrapper
            (%class-ctype class) (make-class-ctype class)
            (%class-instance-slotds class) (vector)
            (find-class name) class
            )
      (dolist (sup supers)
        (setf (%class-subclasses sup) (cons class (%class-subclasses sup))))
      class)))

#|
(defun standard-object-p (thing)
  ; returns thing's class-wrapper or nil if it isn't a standard-object
  (lap-inline ()
    (:variable thing)
    (move.l arg_z atemp0)
    (move.l nilreg acc)
    (if# (and (eq (ttagp ($ $t_vector) atemp0 da))
              (eq (vsubtypep ($ $v_instance) atemp0 da))
              (eq (progn (move.l (svref atemp0 %instance-class-wrapper) atemp0)
                         (ttagp ($ $t_vector) atemp0 da))))
      (move.l atemp0 acc))))
|#

(eval-when (:compile-toplevel :execute)
(declaim (inline standard-instance-p))
)



(defun standard-instance-p (i)
  (eq (typecode i) arch::subtag-instance))


(setf (type-predicate 'standard-instance) 'standard-instance-p)


(defun standard-object-p (thing)
 ; returns thing's class-wrapper or nil if it isn't a standard-object
  (if (standard-instance-p thing)
    (let* ((wrapper (%instance-class-wrapper thing)))
      (if (uvectorp wrapper)  ;; ???? - probably ok
        wrapper))))

(defun std-class-p (class)
  ; (typep class 'std-class)
  ; but works at bootstrapping time as well
  (let ((wrapper (standard-object-p class)))
    (and wrapper
         (or (eq wrapper *standard-class-wrapper*)
             (memq *std-class-class* (%inited-class-cpl (%wrapper-class wrapper) t))))))

(set-type-predicate 'std-class 'std-class-p)

(defun specializer-p (thing)
  (memq *specializer-class* (%inited-class-cpl (class-of thing))))

(defvar *standard-object-class* (make-standard-class 'standard-object *t-class*))

(defvar *metaobject-class* (make-standard-class 'metaobject *standard-object-class*))

(defvar *specializer-class* (make-standard-class 'specializer *metaobject-class*))

; The *xxx-class-class* instances get slots near the end of this file.
(defvar *class-class* (make-standard-class 'class *specializer-class*))

; an implementation class that exists so that
; standard-class & funcallable-standard-class can have a common ancestor not
; shared by anybody but their subclasses.
(defvar *std-class-class* (make-standard-class 'std-class *class-class*))

;The class of all objects whose metaclass is standard-class. Yow.
(defvar *standard-class-class* (make-standard-class 'standard-class *std-class-class*))
; Replace its wrapper and the circle is closed.
(setf (%class-own-wrapper *standard-class-class*) *standard-class-wrapper*
      (%wrapper-class *standard-class-wrapper*) *standard-class-class*
      (%wrapper-instance-slots *standard-class-wrapper*) (vector))

(defvar *built-in-class-class* (make-standard-class 'built-in-class *class-class*))
(setf *built-in-class-wrapper* (%class-own-wrapper *built-in-class-class*)
      (%instance-class-wrapper *t-class*) *built-in-class-wrapper*)

(defvar *structure-class-class* (make-standard-class 'structure-class *class-class*))
(defvar *structure-class-wrapper* (%class-own-wrapper *structure-class-class*))
(defvar *structure-object-class* 
  (make-class 'structure-object *structure-class-wrapper* (list *t-class*)))

#|
(defvar *forward-referenced-class-class*
  (make-standard-class 'forward-referenced-class *class-class*))
|#

;; Has to be a standard class because code currently depends on T being the
;; only non-standard class in the CPL of a standard class.
(defvar *function-class* (make-standard-class 'function *t-class*))

;Right now, all functions are compiled.


(defvar *compiled-function-class* *function-class*)
(setf (find-class 'compiled-function) *compiled-function-class*)

(defvar *interpreted-function-class*
  (make-standard-class 'interpreted-function *function-class*))

(defvar *compiled-lexical-closure-class* 
  (make-standard-class 'compiled-lexical-closure *function-class*))

(defvar *interpreted-lexical-closure-class*
  (make-standard-class 'interpreted-lexical-closure *interpreted-function-class*))

(defvar *funcallable-standard-object-class*
  (make-standard-class 'funcallable-standard-object
                       *standard-object-class* *function-class*))

(defvar *funcallable-standard-class-class*
  (make-standard-class 'funcallable-standard-class *std-class-class*))

(defvar *generic-function-class*
  (make-class 'generic-function
              (%class-own-wrapper *funcallable-standard-class-class*)
              (list *metaobject-class* *funcallable-standard-object-class*)))
(defvar *standard-generic-function-class*
  (make-class 'standard-generic-function
              (%class-own-wrapper *funcallable-standard-class-class*)
              (list *generic-function-class*)))

; *standard-method-class* is upgraded to a real class below
(defvar *method-class* (make-standard-class 'method *metaobject-class*))
(defvar *standard-method-class* (make-standard-class 'standard-method *method-class*))
(defvar *method-function-class* (make-standard-class 'method-function *function-class*))
(defvar *interpreted-method-function-class* 
  (make-standard-class 'interpreted-method-function *method-function-class* *interpreted-function-class*))

(defvar *combined-method-class* (make-standard-class 'combined-method *function-class*))


#+ppc-target
(defppclapfunction %class-of-instance ((i arg_z))  
  (LWZ ARG_Z (+ (* %instance-class-wrapper 4) arch::misc-data-offset) ARG_Z)
  (LWZ ARG_Z (+ (* %wrapper-class 4) arch::misc-data-offset) ARG_Z)
  (blr))

#+sparc-target
(defsparclapfunction %class-of-instance ((i %arg_z))
  (ld (i (+ (* %instance-class-wrapper 4) arch::misc-data-offset)) %arg_z)
  (retl)
    (ld (%arg_z (+ (* %wrapper-class 4) arch::misc-data-offset)) %arg_z))


#+ppc-target
(defppclapfunction class-of ((x arg_z))
  (check-nargs 1)
  (extract-fulltag imm0 x)  ; low8bits-of from here to done
  (cmpwi cr0 imm0 arch::fulltag-misc)
  (beq cr0 @misc)
  (clrlslwi imm0 x 24 arch::fixnumshift)   ; clear left 24 bits, box assume = make byte index 
  (b @done)
  @misc
  (extract-subtag imm0 x)
  (box-fixnum imm0 imm0)  
  @done
  (addi imm0 imm0 arch::misc-data-offset)
  (lwz temp1 '*class-table* nfn)
  (lwz temp1 arch::symbol.vcell temp1)
  (lwzx temp0 temp1 imm0) ; get entry from table
  (cmpw cr0 temp0 rnil)
  (beq @bad)
  ; functionp?
  (extract-typecode imm1 temp0)
  (cmpwi imm1 arch::subtag-function)
  (bne @ret)  ; not function - return entry
  ; else jump to the fn
  ;(lwz temp0 arch::function.codevector temp0) ; like jump_nfn asm macro
  (mr nfn temp0)
  (lwz temp0 arch::misc-data-offset temp0) ; get the ffing codevector
  (SET-NARGS 1) ; maybe not needed
  (mtctr temp0)
  (bctr)
  @bad
  (lwz fname 'no-class-error nfn)
  (ba .spjmpsym)
  @ret
  (mr arg_z temp0)  ; return frob from table
  (blr))

#+sparc-target
(defsparclapfunction class-of ((x %arg_z))
  (check-nargs 1)
  (extract-fulltag x %imm0)  ; low8bits-of from here to done
  (cmp %imm0 arch::fulltag-misc)
  (bne.a @done)
   (and x (1- (ash 1 8)) %imm0)
  (extract-subtag x %imm0)
  @done  
  (box-fixnum %imm0 %imm0)
  (inc arch::misc-data-offset %imm0)
  (ld (%nfn '*class-table*) %temp1)
  (ld (%temp1 arch::symbol.vcell) %temp1)
  (ld (%temp1 %imm0) %temp0) ; get entry from table
  (cmp %temp0 %rnil)
  (be @bad)
  (extract-typecode %temp0 %imm1)
  ; functionp?
  (cmp %imm1 arch::subtag-function)
  (bne @ret)  ; not function - return entry
    (nop)
  ; else jump to the fn
  (mov %temp0 %nfn)
  (ld (%temp0 arch::misc-data-offset) %temp0) ; get the ffing codevector
  (jmp %temp0 arch::misc-data-offset)
    (set-nargs 1) ; maybe not needed
  @bad
  (jump-subprim .spjmpsym)
   (ld (%nfn 'no-class-error) %fname)
  @ret
  (retl)
  (mov %temp0 %arg_z))

(let ((*dont-find-class-optimize* t))

;; The built-in classes.
(defvar *array-class* (make-built-in-class 'array))
(defvar *character-class* (make-built-in-class 'character))
(make-built-in-class 'number)
(make-built-in-class 'sequence)
(defvar *symbol-class* (make-built-in-class 'symbol))
(defvar *immediate-class* (make-built-in-class 'immediate))   ; Random immediate
;Random uvectors - these are NOT class of all things represented by a uvector
;type. Just random uvectors which don't fit anywhere else.
(make-built-in-class 'ivector)   ; unknown ivector
(make-built-in-class 'gvector)   ; unknown gvector
(defvar *istruct-class* (make-built-in-class 'internal-structure))   ; unknown istruct


(make-built-in-class 'macptr)
(make-built-in-class 'population)
(make-built-in-class 'pool)
(make-built-in-class 'package)
(make-built-in-class 'lock)
(make-built-in-class 'value-cell)

(make-built-in-class 'buffer-mark)
(make-built-in-class 'fred-record *istruct-class*)
(make-built-in-class 'buffer *istruct-class*)
(make-built-in-class 'comtab *istruct-class*)
(make-built-in-class 'restart *istruct-class*)
(make-built-in-class 'hash-table *istruct-class*)
(make-built-in-class 'lexical-environment *istruct-class*)
(make-built-in-class 'compiler-policy *istruct-class*)
(make-built-in-class 'readtable *istruct-class*)
(make-built-in-class 'pathname *istruct-class*)
(make-built-in-class 'random-state *istruct-class*)
(make-built-in-class 'xp-structure *istruct-class*)
(make-built-in-class 'process *istruct-class*)
(make-built-in-class 'process-queue *istruct-class*)
(make-built-in-class 'sgbuf)
(make-built-in-class 'resource *istruct-class*)
(make-built-in-class 'periodic-task *istruct-class*)

(make-built-in-class 'type-class *istruct-class*)

(defvar *ctype-class* (make-built-in-class 'ctype *istruct-class*))
(make-built-in-class 'key-info *istruct-class*)
(defvar *args-ctype* (make-built-in-class 'args-ctype *ctype-class*))
(make-built-in-class 'values-ctype *args-ctype*)
(make-built-in-class 'function-ctype *args-ctype*)
(make-built-in-class 'constant-ctype *ctype-class*)
(make-built-in-class 'named-ctype *ctype-class*)
(make-built-in-class 'unknown-ctype (make-built-in-class 'hairy-ctype *ctype-class*))
(make-built-in-class 'numeric-ctype *ctype-class*)
(make-built-in-class 'array-ctype *ctype-class*)
(make-built-in-class 'member-ctype *ctype-class*)
(make-built-in-class 'union-ctype *ctype-class*)
(make-built-in-class 'foreign-ctype *ctype-class*)
(make-built-in-class 'class-ctype *ctype-class*)


(make-built-in-class 'complex (find-class 'number))
(make-built-in-class 'real (find-class 'number))
(defvar *float-class* (make-built-in-class 'float (find-class 'real)))
(defvar *double-float-class* (make-built-in-class 'double-float (find-class 'float)))
(defvar *short-float-class*  (make-built-in-class 'short-float (find-class 'float)))
(make-built-in-class 'rational (find-class 'real))
(make-built-in-class 'ratio (find-class 'rational))
(make-built-in-class 'integer (find-class 'rational))
(defvar *fixnum-class* (make-built-in-class 'fixnum (find-class 'integer)))
(make-built-in-class 'bignum (find-class 'integer))

(make-built-in-class 'bit *fixnum-class*)
(make-built-in-class 'unsigned-byte (find-class 'integer))
(make-built-In-class 'signed-byte (find-class 'integer))

(setf (find-class 'short-float) #+no-sf *double-float-class*
      				#-no-sf *short-float-class*)
(setf (find-class 'single-float) *short-float-class*)
(setf (find-class 'long-float) *double-float-class*)

(make-built-in-class 'logical-pathname (find-class 'pathname))

(setf (find-class 'base-char) *character-class*)
(defvar *base-char-class* *character-class*)
(defvar *standard-char-class* (make-built-in-class 'standard-char *base-char-class*))

(defvar *keyword-class* (make-built-in-class 'keyword *symbol-class*))

(make-built-in-class 'list (find-class 'sequence))
(defvar *cons-class* (make-built-in-class 'cons (find-class 'list)))
(defvar *null-class* (make-built-in-class 'null *symbol-class* (find-class 'list)))

(defvar *vector-class* (make-built-in-class 'vector *array-class* (find-class 'sequence)))
(make-built-in-class 'simple-array *array-class*)
(make-built-in-class 'simple-1d-array *vector-class* (find-class 'simple-array))

;Maybe should do *float-array-class* etc?
;Also, should straighten out the simple-n-dim-array mess...
(make-built-in-class 'unsigned-byte-vector *vector-class*)
(make-built-in-class 'simple-unsigned-byte-vector (find-class 'unsigned-byte-vector) (find-class 'simple-1d-array))
(make-built-in-class 'unsigned-word-vector *vector-class*)
(make-built-in-class 'simple-unsigned-word-vector (find-class 'unsigned-word-vector) (find-class 'simple-1d-array))


(progn
  (make-built-in-class 'double-float-vector *vector-class*)
  (make-built-in-class 'short-float-vector *vector-class*)
  (setf (find-class 'long-float-vector) (find-class 'double-float-vector))
  (setf (find-class 'single-float-vector) (find-class 'short-float-vector))
  (make-built-in-class 'simple-double-float-vector (find-class 'double-float-vector) (find-class 'simple-1d-array))
  (make-built-in-class 'simple-short-float-vector (find-class 'short-float-vector) (find-class 'simple-1d-array))
  (setf (find-class 'simple-long-float-vector) (find-class 'simple-double-float-vector))
  (setf (find-class 'simple-single-float-vector) (find-class 'simple-short-float-vector))
)
  
(make-built-in-class 'long-vector *vector-class*)
(make-built-in-class 'simple-long-vector (find-class 'long-vector) (find-class 'simple-1d-array))
(make-built-in-class 'unsigned-long-vector *vector-class*)
(make-built-in-class 'simple-unsigned-long-vector (find-class 'unsigned-long-vector) (find-class 'simple-1d-array))
(make-built-in-class 'byte-vector *vector-class*)
(make-built-in-class 'simple-byte-vector (find-class 'byte-vector) (find-class 'simple-1d-array))
(make-built-in-class 'bit-vector *vector-class*)
(make-built-in-class 'simple-bit-vector (find-class 'bit-vector) (find-class 'simple-1d-array))
(make-built-in-class 'word-vector *vector-class*)
(make-built-in-class 'simple-word-vector (find-class 'word-vector) (find-class 'simple-1d-array))
(make-built-in-class 'string *vector-class*)
(make-built-in-class 'base-string (find-class 'string))
(make-built-in-class 'simple-string (find-class 'string) (find-class 'simple-1d-array))
(make-built-in-class 'simple-base-string (find-class 'base-string) (find-class 'simple-string))
(make-built-in-class 'general-vector *vector-class*)
(make-built-in-class 'simple-vector (find-class 'general-vector) (find-class 'simple-1d-array))







(defvar *stack-group-class* (make-built-in-class 'stack-group *function-class*))



(make-built-in-class 'hash-table-vector)
(make-built-in-class 'catch-frame)
(make-built-in-class 'code-vector)
(make-built-in-class 'creole-object)

(make-built-in-class 'xfunction)
(make-built-in-class 'xcode-vector)

(defun class-cell-find-class (class-cell errorp)
  (unless (listp class-cell)
    (setq class-cell (%kernel-restart $xwrongtype class-cell 'list)))
  (locally (declare (type list class-cell))
    (let ((class (cdr class-cell)))
      (or class
          (and 
           (setq class (find-class (car class-cell) nil))
           (when class 
             (rplacd class-cell class)
             class))
          ;(if errorp (dbg-paws (format nil "Class ~s not found." (car class-cell))))))))
          (if errorp (error "Class ~s not found." (car class-cell)) nil)))))
  




; (%wrapper-class (%instance-class-wrapper frob))



(defvar *general-vector-class* (find-class 'general-vector))

(defvar *ivector-vector-classes*
  (vector (find-class 'short-float-vector)
          (find-class 'unsigned-long-vector)
          (find-class 'long-vector)
          (find-class 'unsigned-byte-vector)
          (find-class 'byte-vector)
          (find-class 'base-string)
          (find-class 'base-string)     ;WRONG
          (find-class 'unsigned-word-vector)
          (find-class 'word-vector)
          (find-class 'double-float-vector)
          (find-class 'bit-vector)))


; Bootstrapping
(unless (fboundp 'stack-group-p)
  (%fhave 'stack-group-p #'(lambda (x) (declare (ignore x)) nil)))

(defparameter *class-table*
  (let* ((v (make-array 256 :initial-element nil)))
    ; Make one loop through the vector, initializing fixnum & list cells
    ; Set all things of arch::fulltag-imm to *immediate-class*, then special-case
    ; characters later.
    (do* ((slice 0 (+ 8 slice)))
         ((= slice 256))
      (declare (type (unsigned-byte 8) slice))
      (setf (%svref v (+ slice arch::fulltag-even-fixnum)) *fixnum-class*
            (%svref v (+ slice arch::fulltag-odd-fixnum))  *fixnum-class*
            (%svref v (+ slice arch::fulltag-cons)) *cons-class*
            (%svref v (+ slice arch::fulltag-nil)) *null-class*
            (%svref v (+ slice arch::fulltag-imm)) *immediate-class*))
    (macrolet ((map-subtag (subtag class-name)
               `(setf (%svref v ,subtag) (find-class ',class-name))))
      ; immheader types map to built-in classes.
      (map-subtag arch::subtag-bignum bignum)
      (map-subtag arch::subtag-double-float double-float)
      #-nosf
      (map-subtag arch::subtag-single-float short-float)
      (map-subtag arch::subtag-macptr macptr)
      (map-subtag arch::subtag-dead-macptr ivector)
      (map-subtag arch::subtag-code-vector code-vector)
      (map-subtag arch::subtag-creole-object creole-object)
      (map-subtag arch::subtag-xcode-vector xcode-vector)
      (map-subtag arch::subtag-xfunction xfunction)
      #-nosf
      (map-subtag arch::subtag-single-float-vector simple-short-float-vector)
      (map-subtag arch::subtag-u32-vector simple-unsigned-long-vector)
      (map-subtag arch::subtag-s32-vector simple-long-vector)
      (map-subtag arch::subtag-u8-vector simple-unsigned-byte-vector)
      (map-subtag arch::subtag-s8-vector simple-byte-vector)
      (map-subtag arch::subtag-simple-base-string simple-base-string)
      (map-subtag arch::subtag-u16-vector simple-unsigned-word-vector)
      (map-subtag arch::subtag-s16-vector simple-word-vector)
      (map-subtag arch::subtag-double-float-vector simple-double-float-vector)
      (map-subtag arch::subtag-bit-vector simple-bit-vector)
      ; Some nodeheader types map to built-in-classes; others
      ; require further dispatching.
      (map-subtag arch::subtag-ratio ratio)
      (map-subtag arch::subtag-complex complex)
      (map-subtag arch::subtag-catch-frame catch-frame)
      (map-subtag arch::subtag-sgbuf sgbuf)      ; may change to something else
      (map-subtag arch::subtag-lock lock)
      (map-subtag arch::subtag-hash-vector hash-table-vector)
      (map-subtag arch::subtag-mark buffer-mark)
      (map-subtag arch::subtag-value-cell value-cell)
      (map-subtag arch::subtag-pool pool)
      (map-subtag arch::subtag-weak population)
      (map-subtag arch::subtag-package package)
      (map-subtag arch::subtag-simple-vector simple-vector))
    (setf (%svref v arch::subtag-arrayH) *array-class*)
    ; These need to be special-cased:
    (setf (%svref v arch::subtag-character)
          #'(lambda (c) (let* ((code (%char-code c)))
                            (if (or (eq c #\NewLine)
                                    (and (>= code (char-code #\space))
                                         (<= code (char-code #\rubout))))
                              *standard-char-class*
			      *base-char-class*))))
    (setf (%svref v arch::subtag-struct)
          #'(lambda (s) (%structure-class-of s)))       ; need DEFSTRUCT
    (setf (%svref v arch::subtag-istruct)
          #'(lambda (i) (or (find-class (%svref i 0) nil) *istruct-class*)))
    (setf (%svref v arch::subtag-instance)
          #'%class-of-instance) ; #'(lambda (i) (%wrapper-class (%instance-class-wrapper i))))
    (setf (%svref v arch::subtag-symbol)
          #'(lambda (s) (if (eq (symbol-package s) *keyword-package*)
                          *keyword-class*
                          *symbol-class*)))
    (setf (%svref v arch::subtag-function)
          #'(lambda (thing)
              (let ((bits (lfun-bits thing)))
                (declare (fixnum bits))
                (if (logbitp $lfbits-trampoline-bit bits)
                  ; stack-group or closure
                  (if (stack-group-p thing)
                    *stack-group-class*
                    (if (logbitp $lfbits-evaluated-bit bits)
                      *interpreted-lexical-closure-class*
                      (let ((inner-fn (closure-function thing)))
                        (if (neq inner-fn thing)
                          (let ((inner-bits (lfun-bits inner-fn)))
                            (if (logbitp $lfbits-method-bit inner-bits)
                              *compiled-lexical-closure-class*
                              (if (logbitp $lfbits-gfn-bit inner-bits)
                                (let ((frob (%gf-instance thing)))
                                  (if (null frob) 
                                    *standard-generic-function-class*
                                    (%wrapper-class (%instance-class-wrapper frob))))
                                (if (logbitp $lfbits-cm-bit inner-bits)
                                  *combined-method-class*
                                  *compiled-lexical-closure-class*))))
                          *compiled-lexical-closure-class*))))
                  (if (logbitp $lfbits-evaluated-bit bits)
                    (if (logbitp $lfbits-method-bit bits)
                      *interpreted-method-function-class*
                      *interpreted-function-class*)
                    (if (logbitp  $lfbits-method-bit bits)
                      *method-function-class* 
                      (if (logbitp $lfbits-gfn-bit bits)
                        (let ((frob (%gf-instance thing)))
                          (if (null frob) 
                            *standard-generic-function-class*
                            (%wrapper-class (%instance-class-wrapper frob))))
                        (if (logbitp $lfbits-cm-bit bits)
                          *combined-method-class*
                          *compiled-function-class*))))))))
    (setf (%svref v arch::subtag-vectorH)
          #'(lambda (v)
              (let* ((subtype (%array-header-subtype v)))
                (declare (fixnum subtype))
                (if (eql subtype arch::subtag-simple-vector)
                  *general-vector-class*
                  (%svref *ivector-vector-classes*
                          (ash (the fixnum (- subtype arch::min-cl-ivector-subtag))
                               (- arch::ntagbits)))))))
    v))





(defun no-class-error (x)
  (error "Bug (probably): can't determine class of ~s" x))
  

  ; return frob from table




) ; end let

; Can't use typep at bootstrapping time.
(defun classp (x)
  (let ((wrapper (standard-object-p x)))
    (and wrapper
         (let ((super (%wrapper-class wrapper)))
           (memq *class-class* (%inited-class-cpl super t))))))

(set-type-predicate 'class 'classp)

(defun subclassp (c1 c2)
  (and (classp c1)
       (classp c2)
       (not (null (memq c2 (%inited-class-cpl c1 t))))))

(defun %class-get (class indicator &optional default)
  (let ((cell (assq indicator (%class-alist class))))
    (if cell (cdr cell) default)))
(defun %class-put (class indicator value)
  (let ((cell (assq indicator (%class-alist class))))
    (if cell
      (setf (cdr cell) value)
      (push (cons indicator value) (%class-alist class))))
  value)
(defsetf %class-get %class-put)
(defun %class-remprop (class indicator)
  (let* ((handle (cons nil (%class-alist class)))
         (last handle))
    (declare (dynamic-extent handle))
    (while (cdr last)
      (if (eq indicator (caar (%cdr last)))
        (progn
          (setf (%cdr last) (%cddr last))
          (setf (%class-alist class) (%cdr handle)))
        (setf last (%cdr last))))))    


(pushnew :primary-classes *features*)

(defun %class-primary-p (class)
  (%class-get class :primary-p))

(defun (setf %class-primary-p) (value class)
  (if value
    (setf (%class-get class :primary-p) value)
    (progn
      (%class-remprop class :primary-p)
      nil)))

; Returns the first element of the CPL that is primary
(defun %class-or-superclass-primary-p (class)
  (dolist (super (%inited-class-cpl class t))
    (when (and (typep super 'standard-class) (%class-primary-p super))
      (return super))))

(defun %defclass (class-name superclasses instance-slotds class-slotds doc
                  default-initargs metaclass
                  &key primary-p)
  (if (null superclasses)
    (setq superclasses (list 'standard-object))
    (setq superclasses (copy-list superclasses)))
  (if (null metaclass)
    (setq metaclass *standard-class-class*))
  (if (symbolp metaclass) (setq metaclass (find-class metaclass)))
  (unless (subclassp metaclass *std-class-class*)
    (error "~s is not a subclass of ~s" metaclass *std-class-class*))
  (let* ((old-class (find-class class-name nil))
         (class (or old-class
                    (let* ((c (if (or (eq metaclass *standard-class-class*)
                                      (eq metaclass *funcallable-standard-class-class*))
                                (%cons-standard-class class-name (%class-own-wrapper metaclass))
                                (make-instance metaclass :name class-name))))
                        (setf (%class-ctype c) (make-class-ctype c))
                        c))))
    (when (eq class *standard-object-class*)
      (error "Cannot redefine ~S" class))
    (unless (eq (class-of class) metaclass)
      (cerror (format nil "(~s '~s '~s)" 'change-class class metaclass)
              "~s is not an instance of ~s" class metaclass)
      (change-class class metaclass))   ; (s)he asked for it.
    (setf (find-class class-name) class)
    (labels ((obsolete (class)
               (dolist (sub (%class-subclasses class)) (obsolete sub))
               ;Need to save old class info in wrapper for obsolete instance access...
               (setf (%class-cpl class) nil)
               (make-instances-obsolete class)))
      (without-interrupts
       (obsolete class)
       (dolist (sup (%class-local-supers class))
         (if (typep sup 'class)          ; might be a symbol from earlier forward ref
           (setf (%class-subclasses sup) (nremove class (%class-subclasses sup)))))
       (setf (%class-local-supers class) superclasses)
       (setf (%class-local-instance-slotds class) instance-slotds)
       (setf (%old-class-local-shared-slotds class)
             (%class-local-shared-slotds class))
       (setf (%class-local-shared-slotds class) class-slotds)
       (setf (%class-local-default-initargs class) default-initargs)))
    (setf (%class-primary-p class) primary-p)
    (when doc (setf (documentation class 'type) doc))
    (record-source-file class-name 'class)
    (initialize-class class t)
    class))

; Called by the expansion of DEFCLASS.
; MUST return (find-class class-name): the value of the DEFCLASS
(defun record-accessor-methods (class-name signatures)
  (let ((class (find-class class-name)))
    (without-interrupts
     (dolist (m (%class-get class 'accessor-methods))
       (%remove-method m))
     ;(dbg 800)(dbg signatures)
     (if signatures  ; we dont do this today
       (let (methods)
         (dolist (sig signatures)
           (let* ((gf (symbol-function (pop sig)))
                  (specs (let (res)
                          (dolist (spec sig)
                            (push (find-class spec) res))
                          (nreverse res))))
             (push (find-method gf nil specs) methods)))
         (setf (%class-get class 'accessor-methods) methods))
       (%class-remprop class 'accessor-methods)))
    class))

; Called from DEFSTRUCT expansion.
(defun %define-structure-class (sd)
  (let* ((class-name (sd-name sd))
         (old-class (find-class class-name  nil))
         (class (require-type (or old-class 
                                  (let* ((c (%cons-standard-class 
                                             class-name *structure-class-wrapper*)))
                                    (setf (%class-ctype c) (make-class-ctype c))
                                    c))
                              'structure-class))
         (super-name (cadr (sd-superclasses sd)))
         (super (or (and super-name (find-class super-name))
                    *structure-object-class*)))
    (when (eq class *structure-object-class*)
      (error "Cannot redefine ~S" class))
    (labels ((obsolete (class)
               (dolist (sub (%class-subclasses class)) (obsolete sub))
               ;Need to save old class info in wrapper for obsolete instance access...
               (setf (%class-cpl class) nil)))
      (without-interrupts
       (obsolete class)
       (dolist (sup (%class-local-supers class))
         (if (typep sup 'class)          ; might be a symbol from earlier forward ref
           (setf (%class-subclasses sup) (nremove class (%class-subclasses sup)))))
       (setf (%class-local-supers class) (list super))))
    (setf (find-class class-name) class)
    (initialize-class-and-wrapper class 'structure-class)
    class))

; Bootstrapping version of union
(defun union (l1 l2)
  (dolist (e l1)
    (unless (memq e l2)
      (push e l2)))
  l2)

(defun initialize-class (class &optional can-fail (metaclass-name 'std-class)
                               subs)
  (declare (resident))
  (when (null (%class-cpl class))
    (when (null (%class-local-supers class))
      (error "No superclasses specified for ~S" class))
    (do ((supers (%class-local-supers class) (%cdr supers)))
        ((null supers))
      (when (symbolp (%car supers))
        (%rplaca supers
                 (require-type 
                  (or (find-class (%car supers) nil)
                      (if can-fail
                        (return-from initialize-class nil)
                        (error "Cannot initialize ~S because ~S doesn't exist"
                               class (%car supers))))
                  metaclass-name)))
      (when (memq class subs)
        (error "Circular class hierarchy for ~s" class))
      (let ((subs-and-me (cons class subs)))
        (declare (dynamic-extent subs-and-me))
        (unless (initialize-class (%car supers) can-fail metaclass-name subs-and-me)
          (return-from initialize-class nil))))
    ;This has a timing screw in that somebody may reset local info
    ;while we're computing...  We need a timestamp on %defclass'ing...
    (let ((cpl (compute-cpl class))
          (sup-instance-slotds nil)
          (instance-slotds nil)
          (shared-slotds nil)
          (default-initargs nil)
          (old-local-shared-slotds (%old-class-local-shared-slotds class t))
          (local-shared-slotds (%class-local-shared-slotds class)))
      (dolist (sup cpl)
        (unless (eq sup *t-class*) ;only non-standard-class sup is T...
          (do ((slotds (%class-local-instance-slotds sup))
               (instances t)
               slotd old old-instance name)
              ((and (null slotds)
                    (or (not instances)
                        (null (setq instances nil
                                    slotds (%class-local-shared-slotds sup))))))
            (setq slotd (pop slotds))
            (setq old-instance t)
            (setq name (%slotd-name slotd))
            (cond ((setq old (or (%find-slotd name instance-slotds)
                                 (setq old-instance nil)
                                 (%find-slotd name shared-slotds)))
                   (if old-instance
                     (when (null (%slotd-initform old))
                       (setf (%slotd-initform old) (%slotd-initform slotd)))
                     (if (null (cdr (%slotd-initform old)))
                       (setf (cdr (%slotd-initform old)) (%slotd-initform slotd))))
                   (setf (%slotd-type old) (type-intersect (%slotd-type slotd)
                                                           (%slotd-type old)))
                   (setf (%slotd-initargs old) (union
                                                (%slotd-initargs slotd)
                                                (%slotd-initargs old))))
                  (instances (push (%copy-slotd slotd) sup-instance-slotds))
                  (t (push (%copy-slotd slotd) shared-slotds)
                     (setf (%slotd-initform (car shared-slotds))
                           (cons (unless (and (eq sup class)
                                              (listp old-local-shared-slotds)
                                              (%find-slotd name local-shared-slotds)
                                              (not (%find-slotd name old-local-shared-slotds)))
                                   (%slotd-value (%find-slotd name
                                                       (%class-shared-slotds sup))))
                                 (%slotd-initform (car shared-slotds)))))))
          (dolist (key.form (%class-local-default-initargs sup))
            (unless (assq (%car key.form) default-initargs)
              (push key.form default-initargs))))
        (setq instance-slotds (nconc (nreverse sup-instance-slotds) instance-slotds)
              sup-instance-slotds nil))
      (setq shared-slotds (nreverse shared-slotds))
      (setq default-initargs (nreverse default-initargs))
      (dolist (slotd shared-slotds)
          (let* ((value-dot-initform (%slotd-initform slotd))
                 (value-cell (or (car value-dot-initform) (list (%unbound-marker-8))))
                 (initform (cdr value-dot-initform)))
            (setf (%slotd-value slotd) value-cell)
            (if (eq (car value-cell) (%unbound-marker-8))
              (unless (null initform)
                (setf (car value-cell)
                      (if (listp initform) (%car initform)
                          (funcall initform)))))))
      (setq instance-slotds (apply 'vector
                                   (sort-instance-slotds instance-slotds class cpl)))
      (let ((subclasses (mapcar #'(lambda (sup)
                                    (unless (memq class (%class-subclasses sup))
                                      (cons sup (%class-subclasses sup))))
                                (%class-local-supers class))))
        (without-interrupts
         (when (null (%class-cpl class)) ;(and not-changed-local-info)
           (while subclasses
             (when (%car subclasses)
               (setf (%class-subclasses (%car (%car subclasses)))
                     (%car subclasses))
               (%rplaca (%car subclasses) class))
             (setq subclasses (%cdr subclasses)))
           (setf (%class-cpl class) cpl)
           (setf (%class-instance-slotds class) instance-slotds)
           (setf (%class-shared-slotds class) shared-slotds)
           (%class-remprop class '%old-class-local-shared-slotds)
           (setf (%class-default-initargs class) default-initargs)
           (setf (%class-aux-init-functions-cache class) nil
                 (%class-make-instance-initargs class) nil
                 (%class-reinit-initargs class) nil
                 (%class-redefined-initargs class) nil
                 (%class-changed-initargs class) nil)))))
    (%clear-class-primary-slot-accessor-offsets class)
    (dolist (sub (%class-subclasses class))
      (initialize-class sub t)))
  class)

;; Stub to prevent errors when the user doesn't define types
(defun type-intersect (type1 type2)
  (cond ((and (null type1) (null type2))
         nil)
        ((equal type1 type2)
         type1)
        ((subtypep type1 type2)
         type1)
        ((subtypep type2 type1)
         type2)
        (t `(and ,type1 ,type2))
        ;(t (error "type-intersect not implemented yet."))
        ))

(defmethod find-method ((generic-function standard-generic-function)
                        method-qualifiers specializers &optional (errorp t))
  (dolist (m (%gf-methods generic-function)
             (if errorp
               (error "~s has no method for ~s ~s"
                      generic-function method-qualifiers specializers)))
    (flet ((err ()
                (error "Wrong number of specializers: ~s" specializers)))
    (let ((ss (%method-specializers m))
          (q (%method-qualifiers m))
          s)
      (when (equal q method-qualifiers)
        (dolist (spec specializers
                      (if (null ss)
                        (return-from find-method m)
                        (err)))
          (unless (setq s (pop ss))
            (err))
          (unless (if (listp s)
                    (and (listp spec) (eql (cadr s) (cadr spec)))
                    (eq s spec))
            (return))))))))





(defun %make-instance (class-cell &rest initargs)
  (declare (dynamic-extent initargs))
  (apply #'make-instance
         (or (cdr class-cell) (car (the list class-cell)))
         initargs))


(defmethod make-instance ((class symbol) &rest initargs)
  (declare (dynamic-extent initargs))
  (apply 'make-instance (find-class class) initargs))


(defmethod make-instance ((class standard-class) &rest initargs &key &allow-other-keys)
  (declare (dynamic-extent initargs))
  (%make-std-instance class initargs))

(defmethod make-instance ((class std-class) &rest initargs &key &allow-other-keys)
  (declare (dynamic-extent initargs))
  (%make-std-instance class initargs))


(defun %make-std-instance (class initargs)
  (setq initargs (default-initargs class initargs))
  (when initargs
    (apply #'check-initargs
           nil class initargs t
           #'initialize-instance #'allocate-instance #'shared-initialize
           (aux-init-functions class)))
  (let ((instance (apply #'allocate-instance class initargs)))
    ;(dbg 445)
    (apply #'initialize-instance instance initargs)
    instance))

(defun default-initargs (class initargs)
  (unless (std-class-p class)
    (setq class (require-type class 'std-class)))
  (when (null (%class-cpl class)) (initialize-class class))
  (let ((defaults ()))
    (dolist (key.form (%class-default-initargs class))
      (unless (pl-search initargs (%car key.form))
        (setq defaults
              (list* (if (listp (%cdr key.form))
                       (%cadr key.form)
                       (funcall (%cdr key.form)))
                     (%car key.form)
                     defaults))))
    (when defaults
      (setq initargs (append initargs (nreverse defaults))))
    initargs))


(defun %allocate-instance (class &aux (wrapper (%class-own-wrapper class)))
  (when (null wrapper)
    (setq wrapper
          (initialize-class-and-wrapper class)))
  ;(dbg 4373)
  ;De-lapified on general principles ...
  (let* ((len (uvsize (%wrapper-instance-slots wrapper)))
         (v (make-instance-vector (%i+ len 2))))
    (declare (fixnum len))
    (setf (%instance-class-wrapper v) wrapper
          (%forwarded-instance v) v)
    ;(dbg 4374)
    v))

(defmethod copy-instance ((instance standard-object))
  (let ((copy (copy-uvector (%maybe-forwarded-instance instance))))
    (setf (%forwarded-instance copy) copy)
    copy))

(defun initialize-class-and-wrapper (class &optional (superclass-name 'standard-class))
  ;; Initialize the class and make a new wrapper for it.
  ;; Clear the class prototype
  ;; return the new wrapper.
  ;; This is where all the mapping table stuff will eventually live.
  ;; This code will also need to change when the instance-slotds vector and the
  ;; shared-slotds list become a single list of slot descriptors.
  (unless (memq *class-class*
                (%inited-class-cpl (class-of class)))
    (report-bad-arg class 'class))
  (initialize-class class nil superclass-name)
  (let* ((wrapper (%cons-wrapper class (new-class-wrapper-hash-index)))
         (instance-slotds (%class-instance-slotds class))
         (instance-slot-count (if instance-slotds (uvsize instance-slotds) 0))
         (wrapper-instance-slots (make-array (the fixnum instance-slot-count)
                                             :initial-element nil))
         (shared-slotds (%class-shared-slotds class))
         (wrapper-class-slots))
    (dotimes (i instance-slot-count)
      (declare (fixnum i))
      (setf (%svref wrapper-instance-slots i) 
            (%slotd-name (%svref instance-slotds i))))
    (dolist (slotd shared-slotds)
      (push (cons (%slotd-name slotd) (%slotd-value slotd))
            wrapper-class-slots))
    (setf (%wrapper-instance-slots wrapper) wrapper-instance-slots
          (%wrapper-class-slots wrapper) (nreverse wrapper-class-slots))
    (setf (%class-own-wrapper class) wrapper)
    (setf (%class-prototype class) nil)
    wrapper))

(defmethod initialize-instance ((instance standard-object) &rest initargs)
  (declare (dynamic-extent initargs))
  (apply 'shared-initialize instance t initargs))


(defmethod reinitialize-instance ((instance standard-object) &rest initargs)
  (declare (dynamic-extent initargs))
  (when initargs
    (check-initargs 
     instance nil initargs t #'reinitialize-instance #'shared-initialize))
  (apply 'shared-initialize instance nil initargs))

(defmethod shared-initialize ((instance standard-object) slot-names &rest initargs)
  (declare (dynamic-extent initargs))
  (%shared-initialize instance slot-names initargs))

(defmethod shared-initialize ((instance standard-generic-function) slot-names
                              &rest initargs)
  (declare (dynamic-extent initargs))
  (let ((gf-instance
         (if (functionp instance)
           (%gf-instance instance)
           instance)))
    (when gf-instance
      (%shared-initialize gf-instance slot-names initargs))))

(defun %shared-initialize (instance slot-names initargs)
  ;(dbg 4376)
  (unless (or (listp slot-names) (eq slot-names t))
    (report-bad-arg slot-names '(or list (eql t))))
  
  (unless (plistp initargs) (report-bad-arg initargs '(satisfies plistp)))
  (let* ((wrapper (%instance-class-wrapper instance))
         (class (%wrapper-class wrapper))
         forwarded-instance)
    (when (eql 0 (%wrapper-hash-index wrapper))   ; obsolete
      (update-obsolete-instance instance)
      (setq wrapper (%instance-class-wrapper instance)))
    (setq forwarded-instance (%forwarded-instance instance))
    ;(dbg 4375)
    (dotimes (i (uvsize (%class-instance-slotds class)))
      (declare (fixnum i))
      (let* ((slotd (%svref (%class-instance-slotds class) i))
             (slot-initargs (%slotd-initargs slotd))
             (inits initargs)
             (index (%i+ i 2)))
        (while (and inits (not (memq (%car inits) slot-initargs)))
          (setq inits (%cddr inits)))
        (if inits
          (%svset forwarded-instance index (%cadr inits))
          (when (and (%slotd-initform slotd)
                     (eq (%svref forwarded-instance index) (%unbound-marker-8))
                     (or (eq slot-names t) (memq (%slotd-name slotd) slot-names)))
            (%svset forwarded-instance index
                    (if (listp (%slotd-initform slotd)) ;(value)
                      (%car (%slotd-initform slotd))
                      (funcall (%slotd-initform slotd))))))))
    (dolist (slotd (%class-shared-slotds class))
      (let ((slot-initargs (%slotd-initargs slotd))
            (inits initargs))
        (while (and inits (not (memq (%car inits) slot-initargs)))
          (setq inits (%cddr inits)))
        (when inits (setf (car (%slotd-value slotd)) (%cadr inits))))))
  instance)


(declaim (inline %slot-value))
; at least inline for slot-value - and do before slot-value
 
; from - really not too slow 6 for him, 5 for %slot-value, 4 for reader
(defun %slot-value (instance name)
  (declare (optimize (speed 3)(safety 0)))
  (unless (standard-instance-p instance)
    (unless (typep instance 'standard-generic-function)
      (if (structurep instance)
        (return-from %slot-value (structure-slot-value instance name))
        (report-bad-arg instance 'standard-object)))
    (unless (setq instance (%gf-instance instance))
      (return-from %slot-value (%slot-missing-marker))))
  (let* ((wrapper (%instance-class-wrapper instance))
         (instance-slots (%wrapper-instance-slots wrapper)))
    (when (eql 0 instance-slots)
      (update-obsolete-instance instance)
      (return-from %slot-value (%slot-value instance name)))
    (dotimes (i (uvsize instance-slots))
      (declare (fixnum i))
      (when (eq (%svref instance-slots i) name)
        (return-from %slot-value (%svref (%forwarded-instance instance) (%i+ i 2)))))
    (dolist (pair (%wrapper-class-slots wrapper))
      (when (eq (%car pair) name)
        (return-from %slot-value (%cadr pair))))
    (%slot-missing-marker)))

(defun slot-value (instance name)
  (let ((value (%slot-value instance name)))
    (if (eq (%unbound-marker-8) value)
      (slot-unbound (class-of instance) instance name)
      (if (eq (%slot-missing-marker) value)
        (slot-missing (class-of instance) instance name 'slot-value)
        value))))

(defmethod slot-unbound (class instance slot-name)
  (declare (ignore class))
  (error "Slot ~s is unbound in ~s" slot-name instance))









#|
(defun %slot-value (instance name)
  (declare (%noforcestk))
  (lap-inline (instance name)
@toptop
    (move.l arg_y arg_x)                ; save for error message
@top
    (move.l arg_x atemp0)               ; possibly forwarded instance
    (if# (or (ne (ttagp ($ $t_vector) arg_x da))
             (ne (vsubtypep ($ $v_instance) atemp0 da)))
      (movem.l #(arg_x arg_y arg_z) -@vsp)
      (ccall standard-generic-function-p arg_x)
      (if# (eq nilreg acc)
        (ccall structure-slot-value (vsp 4) (vsp))
        (add.l ($ 12) vsp)
        (bra @value))
      (ccall %gf-instance (vsp 8))
      (bif (eq nilreg acc) @slot-missing)
      (move.l arg_z arg_x)
      (vpop arg_z)
      (vpop arg_y)
      (add.l ($ 4) vsp)
      (bra @top))
    (move.l (svref atemp0 %instance-class-wrapper) atemp1)
    (move.l (svref atemp1 %wrapper-instance-slots) atemp0)         ; %wrapper-hash-index
    (move.l atemp0 da)                  ; set condition codes: movea doesn't do it.
    (if# eq
      (if# (eq (tst.l (svref atemp1 %wrapper-hash-index)))   ; obsolete?
        (vpush arg_x) (vpush arg_z)
        (ccall update-obsolete-instance arg_x)
        (vpop arg_z) (vpop arg_y)
        (set_nargs 2)
        (bra @toptop))
      (move.l arg_x atemp0)
      (move.l (svref atemp0 1) arg_x)   ; forwarded-instance
      (if# (eq (tst.l nargs))
        (ccall error '"Inconsistant forwarded instance: ~s" arg_y))
      (clr.l nargs)                     ; mark that we've been here
      (bra @top))                       ; try again.
    (getvect atemp0 da)
    (move.l atemp0 db)
    (asr.l ($ 2) da)
    (beq @tryclass)
    (sub.l ($ 1) da)
@loop
     (cmp.l (@+ atemp0) arg_z)
     @test
     (dbeq da @loop)
    (bne @tryclass)
    (sub.l db atemp0)
    (move.l (atemp0 arg_x.l $v_data) acc)
    (lfret)
@tryclass
    (move.l (svref atemp1 %wrapper-class-slots) atemp0)
    (while# (ne atemp0 nilreg)
      (move.l (car atemp0) atemp1)
      (if# (eq (car atemp1) arg_z)      ; name
        (cdr atemp1 atemp1)             ; value-cell
        (car atemp1 acc)                ; value
        (bra @value))
      (cdr atemp0 atemp0))
@slot-missing
    (move.l ($ $illegal) acc)           ; slot-missing
@value))
|#




(defun structure-slot-value (object slot-name)
  (let ((pos (structure-slot-index object slot-name)))
    (if pos
      (uvref object pos)
      (%slot-missing-marker))))

(defun set-structure-slot-value (object slot-name value)
  (let ((pos (structure-slot-index object slot-name)))
    (if pos
      (setf (uvref object pos) value)
      (%slot-missing-marker))))

(defun structure-slot-index (object slot-name)
  (let (sd)
    (unless (and (structurep object)
                 (setq sd (gethash (car (%svref object 0)) %defstructs%)))
      (%badarg object '(or standard-instance standard-generic-function
                        structure-instance)))
    (do ((i 0 (1+ i))
         (slots (sd-slots sd) (%cdr slots)))
        ((null slots) nil)
      (declare (fixnum i))
      (when (eq slot-name (caar slots))
        (return i)))))      

#|
(defun bad-slot (instance slot-name)
  (error "~s has no slot named ~s" instance slot-name))
|#

(defmethod slot-missing (class object slot-name operation &optional new-value)
  (declare (ignore class operation new-value))
  (error "~s has no slot named ~s." object slot-name))


(defun set-slot-value (instance name value)
  ;(dbg 334)
  (locally (declare (optimize (speed 3)(safety 0)))
    ;(setq name (require-type name 'symbol))
    (let ((original-instance instance))
      (unless (standard-instance-p instance) ;(eq (%type-of instance) 'standard-instance) ; same as instance-p ?
        (unless (typep instance 'standard-generic-function)
          (if (structurep instance)
            (return-from set-slot-value (set-structure-slot-value instance name value))
            (report-bad-arg instance 'standard-object)))
        (unless (setq instance (%gf-instance instance))
          (return-from set-slot-value
            (slot-missing
             (class-of original-instance) original-instance name 'setf value)))) 
      (let* ((wrapper (%instance-class-wrapper instance))
             (instance-slots (%wrapper-instance-slots wrapper)))
        (when (eql 0 instance-slots)
          (update-obsolete-instance instance)
          (return-from set-slot-value (set-slot-value instance name value)))
        (setq instance (%forwarded-instance instance))
        (dotimes (i (uvsize instance-slots))
          (declare (fixnum i))
          (when (eq (%svref instance-slots i) name)
            (return-from set-slot-value (%svset instance (%i+ i 2) value))))
        ;Not really right, but that's how PCL did it last time I looked...
        (dolist (slot (%wrapper-class-slots wrapper))
          (when (eq (%car slot) name)
            (return-from set-slot-value (setf (%cadr slot) value))))
        (slot-missing (class-of instance) original-instance name 'setf value)))))

#|
(defun set-slot-value (instance name value)
  (declare (%noforcestk))
  (lap-inline (instance name value)
    (vpush arg_x)                       ; save for error message
@top
    (move.l arg_x atemp0)
    (if# (or (ne (ttagp ($ $t_vector) arg_x da))
             (ne (vsubtypep ($ $v_instance) atemp0 da)))
      (movem.l #(arg_x arg_y arg_z) -@vsp)
      (ccall standard-generic-function-p arg_x)
      (if# (eq nilreg acc)
        (ccall structure-slot-index (vsp 8) (vsp 4))
        (move.l acc da)
        (movem.l vsp@+ #(arg_x arg_y arg_z))
        (bif (eq nilreg da) @slot-missing)
        (ccall uvset arg_x da arg_z)
        (bra @value))
      (ccall %gf-instance (vsp 8))
      (move.l acc arg_x)
      (vpop arg_z)
      (vpop arg_y)
      (add.l ($ 4) vsp)
      (bif (eq nilreg arg_x) @slot-missing)
      (bra @top))
    (move.l (svref atemp0 %instance-class-wrapper) atemp1)
    (move.l (svref atemp1 %wrapper-instance-slots) atemp0)         ; %wrapper-hash-index
    (move.l atemp0 da)                  ; set condition codes: movea doesn't do it.
    (if# eq
      (if# (eq (tst.l (svref atemp1 %wrapper-hash-index)))
        (vpush arg_y) (vpush arg_z)
        (ccall update-obsolete-instance arg_x)
        (vpop arg_z) (vpop arg_y) (move.l @vsp arg_x)
        (set_nargs 3)
        (bra @top))
      (move.l arg_x atemp0)
      (move.l (svref atemp0 1) arg_x)
      (bra @top))
    (getvect atemp0 da)
    (move.l atemp0 db)
    (asr.l ($ 2) da)
    (beq @tryclass)
    (sub.l ($ 1) da)
@loop
     (cmp.l (@+ atemp0) arg_y)
@test
     (dbeq da @loop)
    (bne @tryclass)
    (sub.l db atemp0)
    (move.l arg_z (atemp0 arg_x.l $v_data))
    (lfret)
@tryclass
    (move.l (svref atemp1 %wrapper-class-slots) atemp0)
    (while# (ne atemp0 nilreg)
      (move.l (car atemp0) atemp1)
      (if# (eq (car atemp1) arg_y)      ; name
        (cdr atemp1 atemp1)             ; value-cell
        (move.l arg_z (car atemp1))     ; replace value
        (bra @value))
      (cdr atemp0 atemp0))
@slot-missing
    (vpush arg_y)
    (vpush arg_z)
    (ccall class-of (vsp 4))
    (vpop atemp0)
    (vpop arg_y)
    (vpop arg_x)
    (lfunlk)
    (cjmp slot-missing arg_z arg_x arg_y 'setf atemp0)
@value
    (add.l ($ 4) vsp)))
|#

(defsetf slot-value set-slot-value)

(defun slot-makunbound (instance name)
  (if (slot-exists-p instance name)
    (progn
      (set-slot-value instance name (%unbound-marker-8))
      instance)
    (slot-missing (class-of instance) instance name 'slot-makunbound)))

(defun slot-boundp (instance name)
  (let ((value (%slot-value instance name)))
    (if (eq value (%slot-missing-marker))
      (slot-missing (class-of instance) instance name 'slot-boundp)
      (neq (%slot-value instance name) (%unbound-marker-8)))))

(defun slot-value-if-bound (instance name &optional default)
  (let ((value (%slot-value instance name)))
    (if (eq value (%unbound-marker-8))
      default
      value)))

(defun set-class-slot-value (class name value)
  (setq class (require-type class 'std-class))
  (when (null (%class-cpl class)) (initialize-class class))
  (dolist (slotd (%class-shared-slotds class))
    (when (eq (%slotd-name slotd) name)
      (return-from set-class-slot-value (setf (car (%slotd-value slotd)) value))))
  (error "~s has no class slot named ~s" class name))

(defun slot-exists-p (instance name)
  (and (or (standard-instance-p instance)
           (standard-generic-function-p instance)
           (structurep instance))
       (neq (%slot-value instance name) (%slot-missing-marker))))

; returns nil if (apply gf args) wil cause an error because of the
; non-existance of a method (or if GF is not a generic function or the name
; of a generic function).
(defun method-exists-p (gf &rest args)
  (declare (dynamic-extent args))
  (when (symbolp gf)
    (setq gf (fboundp gf)))
  (when (typep gf 'standard-generic-function)
    (or (null args)
        (let* ((dt (%gf-dispatch-table gf))
               (methods (%gf-dispatch-table-methods dt)))
          (dolist (m methods)
            (when (null (%method-qualifiers m))
              (let ((specializers (%method-specializers m))
                    (args args))
                (when (dolist (s specializers t)
                        (unless (cond ((listp s) (eql (second s) (car args)))
                                      (t (memq s (%inited-class-cpl
                                                  (class-of (car args))))))
                          (return nil))
                        (pop args))
                  (return-from method-exists-p m)))))
          nil))))

(defun funcall-if-method-exists (gf &optional default &rest args)
  (declare (dynamic-extent args))
  (if (apply #'method-exists-p gf args)
    (apply gf args)
    (if default (apply default args))))


(defun find-specializer (specializer)
  (if (and (listp specializer) (eql (car specializer) 'eql))
    specializer
    (find-class specializer)))

(defmethod make-instances-obsolete ((class symbol))
  (make-instances-obsolete (find-class class)))

(defmethod make-instances-obsolete ((class standard-class))
  (let ((wrapper (%class-own-wrapper class)))
    (when wrapper
      (setf (%class-own-wrapper class) nil)
      (make-wrapper-obsolete wrapper))))

(defmethod make-instances-obsolete ((class funcallable-standard-class))
  (let ((wrapper (%class-own-wrapper class)))
    (when wrapper
      (setf (%class-own-wrapper class) nil)
      (make-wrapper-obsolete wrapper))))



; A wrapper is made obsolete by setting the hash-index & instance-slots to 0
; The instance slots are saved for update-obsolete-instance
; by consing them onto the class slots.
; Method dispatch looks at the hash-index.
; slot-value & set-slot-value look at the instance-slots.
; Each wrapper may have an associated forwarding wrapper, which must also be made
; obsolete.  The forwarding-wrapper is stored in the hash table below keyed
; on the wrapper-hash-index of the two wrappers.
(defvar *forwarding-wrapper-hash-table* (make-hash-table :test 'eq))  


(defun make-wrapper-obsolete (wrapper)
  (without-interrupts
   (let ((forwarding-info
          (unless (eql 0 (%wrapper-instance-slots wrapper))   ; already forwarded or obsolete?
            (%cons-forwarding-info (%wrapper-instance-slots wrapper)
                                   (%wrapper-class-slots wrapper)))))
     (when forwarding-info
       (setf (%wrapper-hash-index wrapper) 0
             (%wrapper-instance-slots wrapper) 0
             (%wrapper-forwarding-info wrapper) forwarding-info
             #| (%wrapper-slot-mapping-tables wrapper) 0 |#
             ))))
  wrapper)

;; Give slots to the bootstrapping classes
(defun %copy-class-info (from to)
  (setf (%instance-class-wrapper to) (%instance-class-wrapper from)
        (%class-cpl to) (cons to (cdr (%class-cpl from)))
        (%class-local-supers to) (%class-local-supers from)
        (%class-direct-slots to) (%class-direct-slots from)
        (%class-slots to) (%class-slots from)
        (%class-local-default-initargs to) (%class-default-initargs from))
  (dolist (pair (%class-alist from))
    (setf (%class-get to (car pair)) (cdr pair)))
  (let ((to-w (or (%class-own-wrapper to) (initialize-class-and-wrapper to)))
        (from-w (or (%class-own-wrapper from) (initialize-class-and-wrapper from))))
    (setf (%wrapper-instance-slots to-w) (%wrapper-instance-slots from-w))))

(defun delete-temporary-class (name)
  (let* ((class (find-class name nil)))
    (when class
      (dolist (sup (%class-local-supers class))
        (setf (%class-subclasses sup) (delete class (%class-subclasses sup))))
      (setf (find-class name) nil))))

#|  ; This is now handled by the primary class mechanism.
(defvar *special-instance-slotds-alist*
  `((,*standard-method-class* . (qualifiers specializers function generic-function name))
    (,*standard-class-class* . (name precedence-list own-wrapper direct-superclasses
                                     direct-subclasses direct-methods class-ctype
                                     direct-slots slots prototype dependants
                                     local-default-initargs default-initargs
                                     aux-init-functions-cache alist
                                     make-instance-initargs reinit-initargs
                                     redefined-initargs changed-initargs))
    (,*class-class* . (name precedence-list own-wrapper direct-superclasses
                            direct-subclasses direct-methods class-ctype))))
|#

; This uses the primary class information to sort a class'es slots
(defun sort-instance-slotds (slotds class cpl)
  (let (primary-slotds
        primary-slotds-class
        (primary-slotds-length 0))
    (declare (fixnum primary-slotds-length))
    (dolist (sup (cdr cpl))
      (unless (eq sup *t-class*)      ; the only non standard-class in the CPL
        (when (%class-primary-p sup)
          (let ((sup-slotds (%class-instance-slotds sup)))
            (if (null primary-slotds-class)
              (setf primary-slotds-class sup
                    primary-slotds sup-slotds
                    primary-slotds-length (length sup-slotds))
              (let ((sup-slotds-length (length sup-slotds)))
                (dotimes (i (min sup-slotds-length primary-slotds-length))
                  (unless (eq (%slotd-name (svref sup-slotds i))
                              (%slotd-name (svref primary-slotds i)))
                    (error "While initializing ~s:~%~
                            attempt to mix incompatible primary classes:~%~
                            ~s and ~s"
                           class sup primary-slotds-class)))
                (when (> sup-slotds-length primary-slotds-length)
                  (setq primary-slotds-class sup
                        primary-slotds sup-slotds
                        primary-slotds-length sup-slotds-length))))))))
    (if (null primary-slotds-class)
      slotds
      (flet ((slotd-position (slotd)
               ; Can't use POSITION since it's not completely there at bootstrapping time
               (dotimes (i primary-slotds-length primary-slotds-length)
                 (when (eq (%slotd-name slotd)
                           (%slotd-name (svref primary-slotds i)))
                   (return i)))))
        (declare (dynamic-extent #'slotd-position))
        (sort-list slotds '< #'slotd-position)))))

(defun %clear-class-primary-slot-accessor-offsets (class)
  (let ((info-list (%class-get class '%class-primary-slot-accessor-info)))
    (dolist (info info-list)
      (setf (%slot-accessor-info.offset info) nil))))

(defun primary-class-slot-offset (class slot-name)
  (dolist (super (%class-cpl class))
    (and (typep super 'standard-class)
         (%class-primary-p super)
         (let ((pos (position slot-name
                              (%class-instance-slotds class)
                              :test 'eq
                              :key 'slot-definition-name)))
           (return (and pos (+ pos 2)))))))

; Called by the compiler-macro expansion for slot-value
; info is the result of a %class-primary-slot-accessor-info call.
; value-form is specified if this is set-slot-value.
; Otherwise it's slot-value.
(defun primary-class-slot-value (instance info &optional (value-form nil value-form-p))
  (let ((slot-name (%slot-accessor-info.slot-name info)))
    (prog1
      (if value-form-p
        (setf (slot-value instance slot-name) value-form)
        (slot-value instance slot-name))
      (setf (%slot-accessor-info.offset info)
            (primary-class-slot-offset (class-of instance) slot-name)))))

(defun primary-class-accessor (instance info &optional (value-form nil value-form-p))
  (let ((accessor (%slot-accessor-info.accessor info)))
    (prog1
      (if value-form-p
        (funcall accessor value-form instance)
        (funcall accessor instance))
      (let ((methods (compute-applicable-methods
                      accessor
                      (if value-form-p (list value-form instance) (list instance))))
            method)
        (when (and (eql (length methods) 1)
                   (typep (setq method (car methods)) 'standard-accessor-method))
          (let* ((slot-name (method-slot-name method)))
            (setf (%slot-accessor-info.offset info)
                  (primary-class-slot-offset (class-of instance) slot-name))))))))


(redefine-standard-class standard-method (method)
  ((qualifiers :initarg :qualifiers :initform nil)
   (specializers :initarg :specializers :initform nil)
   (function :initarg :function)
   (generic-function :initarg :generic-function :initform nil)
   (name :initarg :name :initform nil))
  (:primary-p t))


(redefine-standard-class class (specializer)
  ((name :initarg :name :initform nil)
   (precedence-list :initarg :precedence-list :initform nil)
   (own-wrapper :initarg :own-wrapper :initform nil)
   (direct-superclasses :initarg :direct-superclasses :initform nil )
   (direct-subclasses :initarg :direct-subclasses :initform nil)
   (direct-methods :initform nil)
   (class-ctype :initform nil))
  (:primary-p t))

;(redefine-standard-class forward-referenced-class (class) ())
(redefine-standard-class structure-class (class) ())
(redefine-standard-class built-in-class (class) ())

; This class exists only so that standard-class & funcallable-standard-class
; can inherit its slots.
(redefine-standard-class std-class (class)
  ((direct-slots :initform (list nil) :initarg :direct-slots)
   (slots :initform (list nil))
   (prototype :initform nil)
   (dependants :initform nil)
   (local-default-initargs :initform nil)
   (default-initargs :initform nil)
   (aux-init-functions-cache :initform nil)
   (alist :initform nil)
   (make-instance-initargs :initform nil)
   (reinit-initargs :initform nil)
   (redefined-initargs :initform nil)
   (changed-initargs :initform nil))
  (:primary-p t))

(redefine-standard-class standard-class (std-class) ())

(redefine-standard-class funcallable-standard-class (std-class) ())

(redefine-standard-class generic-function 
                         (metaobject funcallable-standard-object)
  ()
  (:metaclass funcallable-standard-class))

(redefine-standard-class standard-generic-function
                         (generic-function)
  ((method-combination :initarg :method-combination
                       :initform *standard-method-combination*)
   (method-class :initarg :method-class
                 :initform *standard-method-class*))
  (:metaclass funcallable-standard-class))




; Method accessors
(defmethod method-qualifiers ((m standard-method))
  (%method-qualifiers m))
(defmethod method-specializers ((m standard-method))
  (%method-specializers m))
(defmethod method-function ((m standard-method))
  (%method-function m))
(defmethod method-generic-function ((m standard-method))
  (%method-gf m))
(defmethod method-name ((m standard-method))
  (%method-name m))

; class accessors
(defmethod class-name ((c class))
  (%class-name c))
(defmethod (setf class-name) (new-name (c class))
  (setf (%class-name c) new-name))
; class-precedence-list is elsewhere in this file.
(defmethod class-own-wrapper ((c class))
  (%class-own-wrapper c))
(defmethod class-direct-superclasses ((c class))
  (%class-local-supers c))
(defmethod class-direct-subclasses ((c class))
  (%class-subclasses c))

; standard-class accessors.  Most of it is internal.
(defmethod class-direct-instance-slots ((c std-class))
  (%class-local-instance-slotds c))
(defmethod class-direct-class-slots ((c std-class))
  (%class-local-shared-slotds c))

; generic-function readers
(defmethod generic-function-method-class ((gf standard-generic-function))
  (%gf-method-class gf))

(defmethod generic-function-method-combination ((gf standard-generic-function))
  (%gf-method-combination gf))

; Reader & writer methods classes.
(setq *accessor-method-class*
      (or *accessor-method-class*
          (defclass standard-accessor-method (standard-method) ())))
(setq *standard-reader-method-class*
      (or *standard-reader-method-class*
          (defclass standard-reader-method (standard-accessor-method) ())))
(setq *standard-writer-method-class*
      (or *standard-writer-method-class*
          (defclass standard-writer-method (standard-accessor-method) ())))

(initialize-class-and-wrapper *standard-reader-method-class*)
(initialize-class-and-wrapper *standard-writer-method-class*)

(defmethod method-slot-name ((m standard-accessor-method))
  (values (%nth-immediate (%lfun-vector (%inner-method-function m)) 0)))

;;; How slot values transfer (from PCL):
;;;
;;; local  --> local        transfer 
;;; local  --> shared       discard
;;; local  -->  --          discard
;;; shared --> local        transfer
;;; shared --> shared       discard
;;; shared -->  --          discard
;;;  --    --> local        added
;;;  --    --> shared        --
;;;
;;; See make-wrapper-obsolete to see how we got here.
;;; A word about forwarding.  When a class is made obsolete, the
;;; %wrapper-instance-slots slot of its wrapper is set to 0.
;;; %wrapper-class-slots = (instance-slots . class-slots)
;;; Note: this should stack-cons the new-instance if we can reuse the
;;; old instance or it's forwarded value.
(defun update-obsolete-instance (thing)
  (let ((instance thing)
        new-instance added discarded plist)
    (without-interrupts
     (let ((wrapper (standard-object-p instance)))
       (unless wrapper
         (when (standard-generic-function-p thing)
           (when (setq instance (%gf-instance thing))
             (setq wrapper (%instance-class-wrapper instance))))
         (unless wrapper
           (report-bad-arg thing '(or standard-instance standard-generic-function))))
       (when (eql 0 (%wrapper-instance-slots wrapper))   ; is it really obsolete?
         (let* ((class (%wrapper-class wrapper))
                (new-wrapper (or (%class-own-wrapper class)
                                 (initialize-class-and-wrapper class)))
                (forwarding-info (%wrapper-forwarding-info wrapper))
                (forwarded-instance (%forwarded-instance instance))
                (old-class-slots (%forwarding-class-slots forwarding-info))
                (old-instance-slots (%forwarding-instance-slots forwarding-info))
                (new-instance-slots (%wrapper-instance-slots new-wrapper))
                (new-class-slots (%wrapper-class-slots new-wrapper)))
           (setq new-instance (unless (equal new-instance-slots old-instance-slots)
                                (%allocate-instance class)))
           (if (not new-instance)
             ; no local slots have changed.  May have to fix wrapper.
             (progn 
               (setf (%instance-class-wrapper instance) new-wrapper)
               (unless (eq instance forwarded-instance)
                 (setf (%instance-class-wrapper forwarded-instance) new-wrapper)))
             ; Lots to do.  Hold onto your hat.
             (let ((old-size (uvsize old-instance-slots))
                   (new-size (uvsize new-instance-slots)))
               ; Go through old instance slots
               (dotimes (i old-size)
	         (declare (fixnum i))
                 (let* ((slot-name (%svref old-instance-slots i))
                        (pos (%vector-member slot-name new-instance-slots))
                        (val (%svref forwarded-instance (%i+ i 2))))
                   (if pos
                     (setf (%svref new-instance (%i+ pos 2)) val)
                     (progn (push slot-name discarded)
                            (setf (getf plist slot-name) val)))))
               ; Go through old class slots
               (dolist (tuple old-class-slots)
                 (let* ((slot-name (%car tuple))
                        (val (%cadr tuple))
                        (pos (%vector-member slot-name new-instance-slots)))
                   (if pos
                     (setf (%svref new-instance (%i+ pos 2)) val)
                     (progn (push slot-name discarded)
                            (setf (getf plist slot-name) val)))))
               ; Go through new instance slots
               (dotimes (i new-size)
	         (declare (fixnum i))
                 (let* ((slot-name (%svref new-instance-slots i)))
                   (unless (or (%vector-member slot-name old-instance-slots)
                               (%find-slotd slot-name old-class-slots))
                     (push slot-name added))))
               ; Go through new class slots
               (dolist (tuple new-class-slots)
                 (let ((slot-name (%car tuple)))
                   (unless (or (%vector-member slot-name old-instance-slots)
                               (%find-slotd slot-name old-class-slots))
                     (push slot-name added))))
               
               ;Store the new data in the instance or forward it to new-instance
               (let ((instance-size (uvsize instance)))
                 (if (%i> instance-size (%i+ new-size 1))
                   ; new data will fit in old instance vector
                   (dotimes (i (%i+ new-size 2)   ; 0th element is the wrapper
                               (let ((j i))
                                 (while (%i< j instance-size)   ; zero out unused elements
                                   (setf (%svref instance j) (%unbound-marker-8))
                                   (setq j (%i+ j 1)))
                                 (setf (%forwarded-instance instance) instance)))
		     (declare (fixnum i))
                     (setf (%svref instance i) (%svref new-instance i)))
                   ;New data won't fit.  Must forward this instance.
                   (progn
                     (when (typep instance 'std-class)
                       (error "Implementation restriction.~%Can't add slots to an instance of ~s."
                              'std-class))
                     (%forward-instance instance new-instance))))))))))
    ;; run user code with interrupts enabled.
    (when new-instance
      (update-instance-for-redefined-class thing added discarded plist)))
  thing)

; Forward instance to new-instance
(defun %forward-instance (instance new-instance &optional
                                   (new-wrapper (%instance-class-wrapper new-instance))
                                   (instance-size (uvsize instance)))
  (setf (%instance-class-wrapper instance) new-wrapper
        (%forwarded-instance instance) new-instance)
  (let ((i 2))
    (while (< i instance-size)
      (setf (%svref instance i) (%unbound-marker-8)
            i (%i+ i 1)))))
            
          
(defmethod update-instance-for-redefined-class ((instance standard-object)
						added-slots
						discarded-slots
						property-list
						&rest initargs)
  (declare (ignore discarded-slots property-list))
  (when initargs
    (check-initargs
     instance nil initargs t
     #'update-instance-for-redefined-class #'shared-initialize))
  (apply #'shared-initialize instance added-slots initargs))

(defmethod update-instance-for-redefined-class ((instance standard-generic-function)
						added-slots
						discarded-slots
						property-list
						&rest initargs)
  (declare (ignore discarded-slots property-list))
  (when initargs
    (check-initargs
     instance nil initargs t
     #'update-instance-for-redefined-class #'shared-initialize))
  (apply #'shared-initialize instance added-slots initargs))

(defun check-initargs (instance class initargs errorp &rest functions)
  (declare (dynamic-extent functions))
  (declare (list functions))
  (setq class (require-type (or class (class-of instance)) 'std-class))
  (let ((initvect (initargs-vector instance class functions)))
    (when (eq initvect t) (return-from check-initargs nil))
    (do* ((tail initargs (cddr tail))
          (initarg (car tail) (car tail))
          bad-keys? bad-key)
         ((null (cdr tail))
          (if bad-keys?
            (if errorp
              (error #'(lambda (stream key name class vect)
                         (let ((*print-array* t))
                           (format stream 
                                   "~s is an invalid initarg to ~s for ~s.~%~
                                    Valid initargs: ~s."
                                   key name class vect)))
                     bad-key (function-name (car functions)) class initvect)
              (values bad-keys? bad-key))))
      (if (eq initarg :allow-other-keys)
        (if (cadr tail)
          (return))                   ; (... :allow-other-keys t ...)
        (unless (or bad-keys? (%vector-member initarg initvect))
          (setq bad-keys? t
                bad-key initarg))))))

(defun initargs-vector (instance class functions)
  (let ((index (cadr (assq (car functions) *initialization-invalidation-alist*))))
    (unless index
      (error "Unknown initialization function: ~s." (car functions)))
    (let ((initvect (%svref class index)))
      (unless initvect
        (setf (%svref class index) 
              (setq initvect (compute-initargs-vector instance class functions))))
      initvect)))


(defun compute-initargs-vector (instance class functions)
  (let ((initargs (class-slot-initargs class))
        (cpl (%inited-class-cpl class)))
    (dolist (f functions)         ; for all the functions passed
      (dolist (method (%gf-methods f))   ; for each applicable method
        (let ((spec (car (%method-specializers method))))
          (when (if (listp spec)
                  (eql instance (cadr (the list spec)))
                  (memq spec cpl))
            (let* ((func (%inner-method-function method))
                   (keyvect (if (logbitp $lfbits-aok-bit (lfun-bits func))
                              (return-from compute-initargs-vector t)
                              (lfun-keyvect func))))
              (dovector (key keyvect)
                (pushnew key initargs)))))))   ; add all of the method's keys
    (apply #'vector initargs)))



; A useful function
(defun class-make-instance-initargs (class)
  (setq class (require-type (if (symbolp class) (find-class class) class)
                            'std-class))
  (flet ((iv (class &rest functions)
           (declare (dynamic-extent functions))
           (initargs-vector (class-prototype class) class functions)))
    (let ((initvect (apply #'iv
                           class
                           #'initialize-instance #'allocate-instance #'shared-initialize
                           (aux-init-functions class))))
      (if (eq initvect 't)
        t
        (concatenate 'list initvect)))))

                                   

; This is part of the MOP
(defmethod class-slot-initargs ((class std-class))
  (let ((res nil))
    (initialize-class class)
    (dovector (slotd (%class-instance-slotds class))
      (setq res (append (%slotd-initargs slotd) res)))
    (dolist (slotd (%class-shared-slotds class))
      (setq res (append (%slotd-initargs slotd) res)))
    res))

(defun maybe-update-obsolete-instance (instance)
  (let ((wrapper (standard-object-p instance)))
    (unless wrapper
      (when (standard-generic-function-p instance)
        (setq wrapper (generic-function-wrapper instance)))
      (unless wrapper
        (report-bad-arg instance '(or standard-object standard-generic-function))))
    (when (eql 0 (%wrapper-hash-index wrapper))
      (update-obsolete-instance instance)))
  instance)


; If you ever reference one of these through anyone who might call update-obsolete-instance,
; you will lose badly.
(defun %maybe-forwarded-instance (instance)
  (maybe-update-obsolete-instance instance)
  (%forwarded-instance instance))

(defmethod change-class (instance (new-class symbol))
  (change-class instance (find-class new-class)))

(defmethod change-class ((instance standard-object) (new-class standard-class))
  (%change-class instance new-class))

(defmethod change-class ((instance standard-object) (new-class funcallable-standard-class))
  (%change-class instance new-class))

(defmethod change-class ((instance standard-generic-function) (new-class standard-class))
  (unless (inherits-from-standard-generic-function-p new-class)
    (%badarg new-class 'standard-generic-function))
  (unless (or (%gf-instance instance) 
              (eq new-class *standard-generic-function-class*))
    (let ((i (allocate-instance new-class)))
      (shared-initialize i t)
      (setf (%gf-instance instance) i)))
  (%change-class instance new-class))

(defvar *temp-gf* (make-gf nil 0))

(defun %change-class (object new-class)
  ; The spec doesn't say whether to run update-instance-for-different-class
  ; if the class isn't really different.  I choose not to.  So there.
  (unless (eq (class-of object) new-class)
    ; Try to prevent this from happenning inside of without-interrupts
    (maybe-update-obsolete-instance object)
    ; uncomment this as soon as it works.
      (let (copy
            (instance (%maybe-gf-instance object)))
        (without-interrupts
         (maybe-update-obsolete-instance object)   ; probably a nop, but you never know.
         (let* ((forwarded-instance instance)
                (new-wrapper (or (%class-own-wrapper new-class)
                                 (initialize-class-and-wrapper new-class)))
                (new-slots (%wrapper-instance-slots new-wrapper))
                old-slots
                (old-size (%i- (uvsize instance) 2))
                (new-size (uvsize new-slots))
                (eql-methods (let ((eql-specializer (list 'eql object)))
                               (declare (dynamic-extent eql-specializer))
                               (specializer-direct-methods eql-specializer))))
           ; Remove any combined-methods for the old class
           (mapc #'remove-obsoleted-combined-methods eql-methods)
           ; If it wasn't forwarded, need to copy.  This really should be stack-allocated.
           (when (eq instance (setq copy (%maybe-forwarded-instance instance)))
             (let ((size (%i+ old-size 2)))
               (setq copy (%make-temp-uvector size  arch::subtag-instance))
               (dotimes (i size)
	         (declare (fixnum i))
                 (setf (%svref copy i) (%svref instance i)))
               (setf (%forwarded-instance copy) copy)))
           (setq old-slots (%wrapper-instance-slots (%instance-class-wrapper copy)))
           ; If the new class won't fit, need to forward instance.
           (if (%i> new-size old-size)
             (progn
               (when (typep object 'std-class)
                 (error "Implementation restriction.~%Can't add slots to an instance of ~s" 'std-class))
               (setq forwarded-instance (%allocate-instance new-class))
               (%forward-instance instance forwarded-instance))
             (progn
               (setf (%instance-class-wrapper instance) new-wrapper
                     (%forwarded-instance instance) instance)
               ; Make all instance's slots unbound
               (do ((i 2 (%i+ i 1)))
                   ((%i> i new-size))
                 (setf (%svref instance i) (%unbound-marker-8)))))
           ; Copy the shared slots
           (dotimes (i new-size)
	     (declare (fixnum i))
             (let* ((slot-name (%svref new-slots i))
                    (old-index (position slot-name old-slots :test 'eq)))
               (when old-index
                 (setf (%svref forwarded-instance (%i+ i 2))
                       (%svref copy (%i+ old-index 2))))))
           ; Remove any combined-methods for the new class.
           (mapc #'remove-obsoleted-combined-methods eql-methods)))
        ; And let the user & shared-initialize do what they will
        (if (functionp object)          ; was a generic-function
          (let ((temp-gf *temp-gf*))
            (setq *temp-gf* nil)
            (unwind-protect
              (progn
                (let ((gf (or temp-gf (make-gf nil 0))))
                  (setf (%gf-instance gf) copy)
                  (update-instance-for-different-class gf object)))
              (if temp-gf (setq *temp-gf* temp-gf))))
          (update-instance-for-different-class copy object))))
  object)


(defmethod update-instance-for-different-class ((previous standard-object)
                                                (current standard-object)
                                                &rest initargs)
  (declare (dynamic-extent initargs))
  (%update-instance-for-different-class previous current initargs))


(defmethod update-instance-for-different-class ((previous standard-generic-function)
                                                (current standard-generic-function)
                                                &rest initargs)
  (declare (dynamic-extent initargs))
  (%update-instance-for-different-class previous current initargs))

(defun %update-instance-for-different-class (previous current initargs)
  (when initargs
    (check-initargs
     current nil initargs t
     #'update-instance-for-different-class #'shared-initialize))
  (let (current-slots previous-slots new-slots)
    (without-interrupts
     (setq current-slots (%wrapper-instance-slots 
                          (%instance-class-wrapper
                           (%maybe-forwarded-instance (%maybe-gf-instance current))))
           previous-slots (%wrapper-instance-slots 
                           (%instance-class-wrapper
                            (%maybe-forwarded-instance (%maybe-gf-instance previous))))))
    (dotimes (i (uvsize current-slots))
      (declare (fixnum i))
      (let ((slot-name (%svref current-slots i)))
        (unless (find slot-name previous-slots :test 'eq)
          (push slot-name new-slots))))
    (apply #'shared-initialize current new-slots initargs)))


; This is to support instance-initialize, initialize-window, fred-initialize,
; window-make-parts keywords as initialization arguments.
(defun get-aux-init-functions-cache (class)
  #-bccl (setq class (require-type class 'std-class))
  (let ((res (%class-aux-init-functions-cache class)))
    (values (if (eq res t) nil res) res)))
(defun set-aux-init-functions-cache (class value)
  #-bccl (setq class (require-type class 'std-class))
  (setf (%class-aux-init-functions-cache class) (or value t))
  value)
(defun remove-aux-init-functions-cache (class)
  #-bccl (setq class (require-type class 'std-class))
  (setf (%class-aux-init-functions-cache class) nil))


(defun aux-init-functions (class)
  (setq class (require-type class 'class))
  (multiple-value-bind (res exists) (get-aux-init-functions-cache class)
    (if exists
      res
      (dolist (super (%inited-class-cpl class)
                     (set-aux-init-functions-cache class res))
        (setq res (append (get-aux-init-functions super) res))))))

; Clear all the valid initargs caches.
(defun clear-valid-initargs-caches ()
  (map-classes #'(lambda (name class)
                   (declare (ignore name))
                   (when (std-class-p class)
                     (setf (%class-make-instance-initargs class) nil
                           (%class-reinit-initargs class) nil
                           (%class-redefined-initargs class) nil
                           (%class-changed-initargs class) nil
                           (%class-aux-init-functions-cache class) nil)))))

(defun clear-clos-caches ()
  (clear-all-gf-caches)
  (clear-valid-initargs-caches))

(defmethod allocate-instance ((class standard-class) &rest initargs)
  (declare (ignore initargs))
  (let ((it (%allocate-instance class)))
  it))



(unless *initialization-invalidation-alist*
  (setq *initialization-invalidation-alist*
        (list (list #'initialize-instance %class-make-instance-initargs)
              (list #'allocate-instance %class-make-instance-initargs)
              (list #'reinitialize-instance %class-reinit-initargs)
              (list #'shared-initialize 
                    %class-make-instance-initargs %class-reinit-initargs
                    %class-redefined-initargs %class-changed-initargs)
              (list #'update-instance-for-redefined-class
                    %class-redefined-initargs)
              (list #'update-instance-for-different-class
                    %class-changed-initargs))))


(defvar *initialization-function-lists*
  (list (list #'initialize-instance #'allocate-instance #'shared-initialize)
        (list #'reinitialize-instance #'shared-initialize)
        (list #'update-instance-for-redefined-class #'shared-initialize)
        (list #'update-instance-for-different-class #'shared-initialize)))



(unless *clos-initialization-functions*
  (setq *clos-initialization-functions*
        (list #'initialize-instance #'allocate-instance #'shared-initialize
              #'reinitialize-instance
              #'update-instance-for-different-class #'update-instance-for-redefined-class)))

(defun compute-initialization-functions-alist ()
  (let ((res nil)
        (lists *initialization-function-lists*))
    (dolist (cell *initialization-invalidation-alist*)
      (let (res-list)
        (dolist (slot-num (cdr cell))
          (push
           (ecase slot-num
             (#.%class-make-instance-initargs 
              (assq #'initialize-instance lists))
             (#.%class-reinit-initargs
              (assq #'reinitialize-instance lists))
             (#.%class-redefined-initargs
              (assq #'update-instance-for-redefined-class lists))
             (#.%class-changed-initargs
              (assq #'update-instance-for-different-class lists)))
           res-list))
        (push (cons (car cell) (nreverse res-list)) res)))
    (setq *initialization-functions-alist* res)))

(compute-initialization-functions-alist)

                  

; This stays as an alist as I don't expect there to be many entries.
(defvar *aux-init-functions* nil)
(defun get-aux-init-functions (class)
  (if (symbolp class) (setq class (find-class class)))
  (cdr (assq class *aux-init-functions*)))
(defun set-aux-init-functions (class functions)
  (if (symbolp class) (setq class (find-class class)))
  (setq functions (require-type functions 'list))
  (let* ((cell (assq class *aux-init-functions*))
         (old-functions (cdr cell)))
    (unless (equal functions old-functions)
      (if cell
        (setf (cdr cell) functions)
        (if functions
          (push (cons class functions) *aux-init-functions*)))
      (let* ((handle (cons nil *initialization-invalidation-alist*))
             (splice handle)
             elt)      
        (declare (dynamic-extent splice))
        (while (cdr splice)
          (if (and (setq elt (memq (%cadr splice) old-functions))
                   (not (memq elt functions)))
            (setf (%cdr splice) (cddr splice))
            (setq splice (%cdr splice))))
        (setq *initialization-invalidation-alist* (cdr handle)))
      ; delete doesn't exist yet.
      (dolist (f functions)
        (pushnew f *clos-initialization-functions*)
        (unless (assq f *initialization-invalidation-alist*)
          (push (list f %class-make-instance-initargs) 
                *initialization-invalidation-alist*)))
      (invalidate-initargs-vector-for-gf #'initialize-instance class)
      (compute-initialization-functions-alist)))
  functions)


;; Need to define this for all of the built-in-class'es.
(defmethod class-prototype ((class std-class))
  (or (%class-prototype class)
      (setf (%class-prototype class)
            (if (memq *standard-generic-function-class* (%inited-class-cpl class))
              (gf-class-prototype class)
              (%allocate-instance class)))))



(defun gf-class-prototype (class)
  (let ((gf (make-gf nil 0)))
    (unless (eql class *standard-generic-function-class*)
      (setf (%gf-instance gf) (%allocate-instance class)))
    gf))



(defmethod class-prototype ((class structure-class))
  (or (%class-prototype class)
      (setf (%class-prototype class)
            (funcall (sd-constructor (gethash (%class-name class) %defstructs%))))))

;; Some stubs of the MOP for the interface designer - huh?

;; this should be class-slots, but it doesn't return any :class slots.
(defmethod class-instance-slots ((class std-class))
  (concatenate 'list (%class-instance-slotds (initialize-class class))))
(defmethod class-class-slots ((class std-class))
  (%class-shared-slotds (initialize-class class)))


; We don't have any real slot-definitions objects yet.
; Here are the AMOP readers that make sense for what we do have.
(defun maybe-slot-definition-p (thing)
  (let ((sd thing))
    (and (consp sd)
         (consp (setq sd (cdr (the cons sd))))
         (consp (cdr (the cons sd))))))



(defun ensure-slotd (thing)
  (if (maybe-slot-definition-p thing)
    thing
    (require-type thing '(satisfies maybe-slot-definition-p))))

(defmethod slot-definition-name ((slot-definition list))
  (%slotd-name (ensure-slotd slot-definition)))

(defmethod slot-definition-initargs ((slotd list))
  (%slotd-initargs (ensure-slotd slotd)))


(defmethod slot-definition-initform ((slotd list))
  (let ((fun-or-form-list (%slotd-initform (ensure-slotd slotd))))
    (if (listp fun-or-form-list)
      (car fun-or-form-list)
      (let ((lambda-expr (function-lambda-expression fun-or-form-list)))
        (if lambda-expr
          (let ((body-forms (cddr lambda-expr)))
            (if (cdr body-forms)        ; shouldn't ever be true
              `(progn ,@body-forms)
              (car body-forms)))
          `(funcall ,fun-or-form-list))))))

(defmethod slot-definition-initfunction ((slotd list))
  (let ((fun-or-form-list (%slotd-initform (ensure-slotd slotd))))
    (cond ((null fun-or-form-list) nil)
          ((listp fun-or-form-list) 
           (let ((value (car fun-or-form-list)))
             #'(lambda () value)))
          (t fun-or-form-list))))

(defmethod slot-definition-type ((slotd list))
  (or (%slotd-type (ensure-slotd slotd)) t))

; The AMOP defines functions called SLOT-DEFINITION-READERS
; and SLOT-DEFINITION-WRITERS, which take a slot definition
; object as their single argument. MCL does not store the
; class in a slot definition object, so it can't work that way.
(defun slot-readers (class slot-name)
  (let ((res nil))
    (dolist (accessor (%class-get class 'accessor-methods))
      (if (and (eq slot-name (method-slot-name accessor))
               (typep accessor 'standard-reader-method))
        (push (method-name accessor) res)))
    res))

(defun slot-writers (class slot-name)
  (let ((res nil))
    (dolist (accessor (%class-get class 'accessor-methods))
      (if (and (eq slot-name (method-slot-name accessor))
               (typep accessor 'standard-writer-method))
        (push (method-name accessor) res)))
    res))


(defmethod accessor-method-slot-definition ((method standard-accessor-method))
  (let* ((name (method-slot-name method))
         (class (car (method-specializers method))))
    (or (assq name (class-direct-instance-slots class))
        (assq name (class-direct-class-slots class))
        (error "Can't find slot definition for slot named ~s of ~s"
               name class))))

(defmethod remove-method ((generic-function standard-generic-function) method)
  (when (and (typep method 'standard-method)
             (eq generic-function (%method-gf method)))
    (%remove-method method))
  generic-function)


(defmethod function-keywords ((method standard-method))
  (let ((f (%inner-method-function method)))
    (values
     (concatenate 'list (lfun-keyvect f))
     (%ilogbitp $lfbits-aok-bit (lfun-bits f)))))

(defmethod no-next-method ((generic-function standard-generic-function)
                           (method standard-method)
                           &rest args)
  (error "There is no next method for ~s~%args: ~s" method args))

(defmethod add-method ((generic-function standard-generic-function) (method method))
  (%add-method method generic-function))

(defmethod no-applicable-method (gf &rest args)
  (error "No applicable method for args:~% ~s~% to ~s" args gf))


(defmethod no-applicable-primary-method (gf methods)
  (%method-combination-error "No applicable primary methods for ~s~@
                              Applicable methods: ~s" gf methods))

(defmethod compute-applicable-methods ((gf standard-generic-function) args)
  (%compute-applicable-methods* gf args))

(defun %compute-applicable-methods+ (gf &rest args)
  (declare (dynamic-extent args))
  (%compute-applicable-methods* gf args))

(defun %compute-applicable-methods* (gf args)
  (let* ((methods (%gf-methods gf))
         (args-length (length args))
         (bits (inner-lfun-bits gf))
         arg-count res)
    (when methods
      (setq arg-count (length (%method-specializers (car methods))))
      (unless (<= arg-count args-length)
        (error "Too few args to ~s" gf))
      (unless (or (logbitp $lfbits-rest-bit bits)
                  (logbitp $lfbits-restv-bit bits)
                  (logbitp $lfbits-keys-bit bits)
                  (<= args-length 
                      (+ (ldb $lfbits-numreq bits) (ldb $lfbits-numopt bits))))
        (error "Too many args to ~s" gf))
      (let ((cpls (make-list arg-count)))
        (declare (dynamic-extent cpls))
        (do* ((args-tail args (cdr args-tail))
              (cpls-tail cpls (cdr cpls-tail)))
            ((null cpls-tail))
          (setf (car cpls-tail)
                (%class-precedence-list (class-of (car args-tail)))))
        (dolist (m methods)
          (if (%method-applicable-p m args cpls)
            (push m res)))
        (sort-methods res cpls (%gf-precedence-list gf))))))


(defun %method-applicable-p (method args cpls)
  (do* ((specs (%method-specializers method) (%cdr specs))
        (args args (%cdr args))
        (cpls cpls (%cdr cpls)))
      ((null specs) t)
    (let ((spec (%car specs)))
      (if (listp spec)
        (unless (eql (%car args) (%cadr spec))
          (return nil))
        (unless (memq spec (%car cpls))
          (return nil))))))


; Need this so that (compute-applicable-methods #'class-precedence-list ...)
; will not recurse.
(defun %class-precedence-list (class)
  (if (eq (class-of class) *standard-class-class*)
    (%inited-class-cpl class)
    (class-precedence-list class)))

(defmethod class-precedence-list ((class standard-class))
  (%inited-class-cpl class))

(defmethod class-precedence-list ((class class))
  (or (%class-cpl class)
      (error "~s has no class-precedence-list." class)))



;; Fake method-combination
(defclass method-combination (metaobject) 
  ((name :accessor method-combination-name :initarg :name)))



(defclass standard-method-combination (method-combination) ())

(setq *standard-kernel-method-class*
  (defclass standard-kernel-method (standard-method)
    ()))

(defun make-all-methods-kernel ()
  (%map-lfuns #'(lambda (f)
                  (when (typep f 'generic-function)
                    (let ((smc *standard-method-class*))
                      (dolist (method (generic-function-methods f))
                        (when (eq (class-of method) smc)
                          (change-class method *standard-kernel-method-class*))))))))

(unless *standard-method-combination*
  (setq *standard-method-combination*
        (make-instance 'standard-method-combination :name 'standard)))



(defun required-lambda-list-args (l)
  (multiple-value-bind (ok req) (verify-lambda-list l)
    (unless ok (error "Malformed lambda-list: ~s" l))
    req))

(defun ensure-generic-function (function-name &key 
                                              lambda-list
                                              argument-precedence-order
                                              declare
                                              documentation
                                              (generic-function-class *standard-generic-function-class*)
                                              (method-combination *standard-method-combination*)
                                              (method-class *standard-method-class*)
                                              environment)
  (ensure-generic-function-internal
   function-name t
   lambda-list
   argument-precedence-order
   declare
   documentation
   generic-function-class
   method-combination
   method-class
   environment))

(defun check-generic-function-lambda-list (ll &optional (errorp t))
  (multiple-value-bind (ok reqsyms opttail resttail keytail auxtail)
                       (verify-lambda-list ll)
    (declare (ignore reqsyms resttail))
    (when ok 
      (block checkit
        (when (eq (car opttail) '&optional)
          (dolist (elt (cdr opttail))
            (when (memq elt lambda-list-keywords) (return))
            (unless (or (symbolp elt)
                        (and (listp elt)
                             (non-nil-symbol-p (car elt))
                             (null (cdr elt))))
              (return-from checkit (setq ok nil)))))
        (dolist (elt (cdr keytail))
          (when (memq elt lambda-list-keywords) (return))
          (unless (or (symbolp elt)
                      (and (listp elt)
                           (or (non-nil-symbol-p (car elt))
                               (and (listp (car elt))
                                    (non-nil-symbol-p (caar elt))
                                    (non-nil-symbol-p (cadar elt))
                                    (null (cddar elt))))
                           (null (cdr elt))))
            (return-from checkit (setq ok nil))))
        (when auxtail (setq ok nil))))
    (when (and errorp (not ok))
      (signal-program-error "Bad generic function lambda list: ~s" ll))
    ok))


(defun ensure-generic-function-internal (function-name global-p
                                         lambda-list
                                         argument-precedence-order
                                         declare
                                         documentation
                                         generic-function-class
                                         method-combination
                                         method-class
                                         environment)
  ; Need to handle the environment arg here.
  (declare (ignore environment declare))
  (check-generic-function-lambda-list lambda-list)
  (if (symbolp generic-function-class)
    (setq generic-function-class (find-class generic-function-class)))
  (if (symbolp method-class)
    (setq method-class (find-class method-class)))
  (setq method-combination (require-type method-combination 'method-combination))
  (when argument-precedence-order
    (setq argument-precedence-order
          (canonicalize-argument-precedence-order 
           argument-precedence-order (required-lambda-list-args lambda-list))))
  (multiple-value-bind (bits keys) (encode-lambda-list lambda-list t)
    (when (or (null bits) (memq '&aux lambda-list))
      (error "Malformed lambda-list: ~a" lambda-list))
    (let ((bits bits)
          gf methods gbits bad-methods)
      (if (and global-p (setq gf (fboundp function-name))
               (or (typep gf 'standard-generic-function)
                   (unless (defmethod-congruency-override function-name nil)
                     (cerror "Replace it with a generic-function."
                             "~S is not a generic function" gf)
                     (forget-encapsulations function-name)
                     nil)))
        (progn 
          (setq gf (fdefinition function-name)
                gbits (inner-lfun-bits gf))
          (unless (typep gf 'generic-function)
            (error "~s is not of type ~s" gf 'generic-function))
          (setq methods (%gf-methods gf))
          (unless (or (null methods)
                      (and (congruent-lfbits-p bits gbits)
                           (dolist (m methods (null bad-methods))
                             (unless (congruent-lambda-lists-p gf m nil gbits nil keys)
                               (push m bad-methods)))))
            (cerror (if bad-methods 
                      (format nil "Ignore keyword mismatches with ~d existing method~:p."
                              (length bad-methods))
                      (format nil "Remove ~d method~:p from the generic-function and change its lambda list."
                              (length (%gf-methods gf))))
                    "Incompatible lambda list: ~S ~%for ~S.~@[~%Keys disagree with: ~s~]"
                   lambda-list gf bad-methods)
            (unless bad-methods
              (loop
                (let ((methods (%gf-methods gf)))
                  (if methods
                    (remove-method gf (car methods))
                    (return))))))
          (unless (eq (class-of gf) generic-function-class)
            (change-class gf generic-function-class))
          (clear-gf-cache gf)
          (inner-lfun-bits gf (logior (logand bits $lfbits-args-mask)
                                (logand (lognot $lfbits-args-mask) (inner-lfun-bits gf)))))
        (progn
          (setq gf (make-gf function-name bits))
          (if global-p
            (%fhave function-name gf))))
      (setf (%gf-dispatch-table-precedence-list (%gf-dispatch-table gf))
            argument-precedence-order)
      (setf (documentation gf t) documentation)
      (%set-defgeneric-keys 
       gf
       (if (logbitp $lfbits-keys-bit bits) keys nil))
      (if (and (eq generic-function-class *standard-generic-function-class*)
               (eq method-class *standard-method-class*)
               (eq method-combination *standard-method-combination*))
        (setf (%gf-instance gf) nil)
        (let ((instance (%gf-instance gf)))
          (if instance
            (let ((old-mc (slot-value instance 'method-combination)))
              (unless (or (eq old-mc method-combination)
                          (eq old-mc *standard-method-combination*))
                (unregister-gf-method-combination gf old-mc)
                (register-gf-method-combination gf method-combination))
              (setf (slot-value instance 'method-class) method-class
                    (slot-value instance 'method-combination) method-combination))
            (progn
              (unless (eq method-combination *standard-method-combination*)
                (register-gf-method-combination gf method-combination))
              (setq instance (setf (%gf-instance gf) 
                                   (%allocate-instance generic-function-class)))
              (shared-initialize gf t
                                 :method-class method-class
                                 :method-combination method-combination)))))
      gf)))

(defun canonicalize-argument-precedence-order (apo req)
  (cond ((equal apo req) nil)
        ((not (eql (length apo) (length req)))
         (error "Lengths of ~S and ~S differ." apo req))
        (t (let ((res nil))
             (dolist (arg apo (nreverse res))
               (let ((index (position arg req)))
                 (if (or (null index) (memq index res))
                   (error "Missing or duplicate arguments in ~s" apo))
                 (push index res)))))))



(defun %defgeneric (function-name lambda-list method-combination generic-function-class
                                  options)
  (setq generic-function-class (find-class generic-function-class))
  (setq method-combination 
        (find-method-combination
         (class-prototype generic-function-class)
         (car method-combination)
         (cdr method-combination)))
  (let ((gf (fboundp function-name)))
    (when gf
      (dolist (method (%defgeneric-methods gf))
        (%remove-method method))))
  (record-source-file function-name 'function)
  (record-arglist function-name lambda-list)
  (apply #'ensure-generic-function 
         function-name
         :lambda-list lambda-list
         :method-combination method-combination
         :generic-function-class generic-function-class
         options))

(defun ensure-generic-function (function-name &key 
                                              lambda-list
                                              argument-precedence-order
                                              declare
                                              documentation
                                              (generic-function-class *standard-generic-function-class*)
                                              (method-combination *standard-method-combination*)
                                              (method-class *standard-method-class*)
                                              environment)
  (ensure-generic-function-internal
   function-name t
   lambda-list
   argument-precedence-order
   declare
   documentation
   generic-function-class
   method-combination
   method-class
   environment))


(defun %generic-function (lambda-list method-combination generic-function-class
                                      options &rest methods)
  (declare (dynamic-extent methods))
  (setq generic-function-class (find-class generic-function-class))
  (setq method-combination 
        (find-method-combination
         (class-prototype generic-function-class)
         (car method-combination)
         (cdr method-combination)))
  (destructuring-bind (&key declare documentation
                            argument-precedence-order
                            (method-combination *standard-method-combination*)
                            (method-class *standard-method-class*))
                      options
    (let ((gf (ensure-generic-function-internal
               nil nil
               lambda-list
               argument-precedence-order
               declare
               documentation
               generic-function-class
               method-combination
               method-class
               nil)))
      (dolist (method methods)
        (add-method gf method))
      gf)))


; Redefined in lib;method-combination.lisp
(defmethod find-method-combination ((gf standard-generic-function) type options)
  (unless (and (eq type 'standard) (null options))
    (error "non-standard method-combination not supported yet."))
  *standard-method-combination*)


(defmethod specializer-direct-methods ((class class))
  (unless  *maintain-class-direct-methods*
    (%recache-class-direct-methods))
  (%specializer-direct-methods (%class-direct-methods class) nil))

(defmethod specializer-direct-generic-functions ((class class))
  (unless  *maintain-class-direct-methods*
    (%recache-class-direct-methods))
  (%specializer-direct-methods (%class-direct-methods class) t))

(defmethod specializer-direct-methods ((list list))
  (if (and (eql 2 (length list))
           (eq (car list) 'eql))
    (%specializer-direct-methods (eql-methods-cell (cadr list)) nil)
    (no-applicable-method #'specializer-direct-methods list)))


(defmethod specializer-direct-generic-functions ((list list))
  (if (and (eql 2 (length list))
           (eq (car list) 'eql))
    (%specializer-direct-methods (eql-methods-cell (cadr list)) t)
    (no-applicable-method #'specializer-direct-methods list)))


(defun %specializer-direct-methods (cell gfs?)
  (when cell
    (if gfs?
      (if (cdr cell)
        (population-data (%cdr cell))
        (let ((pop (%car cell))
              res)
          (setf (%cdr cell)
                (%cons-population
                 (dolist (method (and pop (population-data pop)) res)
                   (pushnew (%method-gf method) res))))
          res))
      (and (car cell) (population-data (%car cell))))))

; something is fishy here - class-of is wrong for generic functions and combined meths
; in 68k world not to worry - will be right in PPC land
(defmethod generic-function-methods ((gf standard-generic-function))
  (%gf-methods gf))

; added this for now - called from window-can-do-operation
(defmethod generic-function-methods (gf)
  (when (standard-generic-function-p gf)
    (%gf-methods gf)))

(defmethod instance-class-wrapper ((instance standard-object))
  (%instance-class-wrapper instance))

(defmethod instance-class-wrapper ((instance standard-generic-function))
  (generic-function-wrapper instance))

(defun generic-function-wrapper (gf)
  (unless (inherits-from-standard-generic-function-p (class-of gf))
    (%badarg gf 'standard-generic-function))
  (let ((i (if (functionp gf) (%gf-instance gf) gf)))
    (if i
      (%instance-class-wrapper i)
      (let ((class *generic-function-class*))
        (or (%class-own-wrapper class)
            (initialize-class-and-wrapper class))))))

(defvar *make-load-form-saving-slots-hash* (make-hash-table :test 'eq))

(defun make-load-form-saving-slots (object &key
					   (slot-names nil slot-names-p)
					   environment)
  (declare (ignore environment))
  (let* ((class (class-of object))
         (class-name (class-name class))
         (structurep (structurep object))
         (sd (and structurep (require-type (gethash class-name %defstructs%) 'vector))))
    (unless (or structurep
                (standard-instance-p object))
      (%badarg object '(or standard-object structure-object)))
    (if slot-names-p
      (dolist (slot slot-names)
        (unless (slot-exists-p object slot)
          (error "~s has no slot named ~s" object slot)))
      (setq slot-names
            (if structurep
              (let ((res nil))
                (dolist (slot (sd-slots sd))
                  (unless (fixnump (car slot))
                    (push (%car slot) res)))
                (nreverse res))
              (mapcar 'slot-definition-name
                      (class-instance-slots (class-of object))))))
    (values
     (let* ((form (gethash class-name *make-load-form-saving-slots-hash*)))
       (or (and (consp form)
                (eq (car form) 'allocate-instance)
                form)
           (setf (gethash class-name *make-load-form-saving-slots-hash*)
                 `(allocate-instance (find-class ',class-name)))))
     ;; initform is NIL when there are no slots
     (when slot-names
       `(%set-slot-values
         ',object
         ',slot-names
         ',(let ((temp #'(lambda (slot)
                           (if (slot-boundp object slot)
                             (slot-value object slot)
                             (%unbound-marker-8)))))
             (declare (dynamic-extent temp))
             (mapcar temp slot-names)))))))


    

(defmethod allocate-instance ((class structure-class) &rest initargs)
  (declare (ignore initargs))
  (let* ((class-name (class-name class))
         (sd (or (gethash class-name %defstructs%)
                 (error "Can't find structure named ~s" class-name)))
         (res (make-structure-vector (sd-size sd))))
    (setf (%svref res 0) (sd-superclasses sd))
    res))


(defun %set-slot-values (object slots values)
  (dolist (slot slots)
    (let ((value (pop values)))
      (if (eq value (%unbound-marker-8))
        (slot-makunbound object slot)
        (setf (slot-value object slot) value)))))

#|
(defmethod method-specializers ((method standard-method))
  (%method-specializers method))

(defmethod method-qualifiers ((method standard-method))
  (%method-qualifiers method))
|#

(defun %recache-class-direct-methods ()
  (let ((*maintain-class-direct-methods* t))   ; in case we get an error
    (dolist (f (population-data %all-gfs%))
      (when (standard-generic-function-p f)
        (dolist (method (%gf-methods f))
          (%add-to-class-direct-methods method)))))
  (setq *maintain-class-direct-methods* t))   ; no error, all is well

; For %compile-time-defclass
(defclass compile-time-class (class) ())



                  

; Test code
#|
(defclass foo () (x y (c :allocation :class)))
(defclass bar (foo) ())
(defclass baz (bar) ())

(defparameter fo (make-instance 'foo))
(defparameter ba (make-instance 'bar))
(defparameter bz (make-instance 'baz))
(defparameter bz2 (make-instance 'baz))

(fmakunbound 'f)

(defmethod f ((x foo))
  (format t "(method f (foo)): ~s~%" x)
  (values x 1))

(defmethod f :before ((x foo))
  (format t "(method f :before (foo)): ~s~%" x)
  x)

(defmethod f :after ((x foo))
  (format t "(method f :after (foo)): ~s~%" x)
  x)

(defmethod f :around ((x foo))
  (format t "(method f :around (foo)): ~s~%" x)
  (call-next-method))

(defmethod f ((x bar))
  (format t "(method f (bar)): ~S~%" x)
  (values (list 'bar x) 2))

(defmethod f :before ((x bar))
  (format t "(method f :before (bar)): ~S~%" x)
  (list 'bar x))

(defmethod f :around ((x bar))
   (format t "(method f :around (bar)): ~S~%" x)
   (call-next-method x)
   (values (list :around 'bar x) 2))

(defmethod f ((x baz))
   (format t "(method f (baz)): ~S~%" x)
   (call-next-method)
   (list 'baz x))

(defmethod f :after ((x baz))
   (format t "(method f :after (baz)): ~S~%" x)
   (list :after 'baz x))

(defmethod f :around ((x baz))
   (format t "(method f :around (baz)): ~s~%" x)
   (call-next-method))

(defmethod f ((x (eql bz2)))
   (format t "(method f ((eql bz2))): ~s~%" x)
   (call-next-method))

(defmethod f :before ((x (eql bz2)))
   (format t "(method f :before ((eql bz2))): ~s~%" x))

;; Multi-method testing.
(fmakunbound 'plus)
(defmethod plus (x y &rest rest)
  (declare (ignore x y rest))
  (error "non-numeric arg to plus"))

(defmethod plus ((x number) (y number) &rest rest)
  (declare (dynamic-extent rest))
  (format t "(method plus (number number))~%")
  (if rest
    (apply 'plus (+ x y) rest)
    (+ x y)))

(defconstant *half-most-positive-fixnum* (ash most-positive-fixnum -1))
(defmethod plus ((x fixnum) (y fixnum) &rest rest)
  (declare (dynamic-extent rest))
  (format t "(method plus (fixnum fixnum))~%")
  (if (and (%i<= x *half-most-positive-fixnum*)
           (%i>= x (- *half-most-positive-fixnum*))
           (%i<= y *half-most-positive-fixnum*)
           (%i>= y (- *half-most-positive-fixnum*)))
    (let ((res (%i+ x y)))
      (if rest (apply 'plus res rest) res))
    (call-next-method)))

(defmethod plus ((x (eql 0)) (y number) &rest rest)
  (declare (dynamic-extent rest))
  (format t "(method plus ((eql 0) number))~%")
  (if rest
    (apply 'plus y rest)
    y))

(defmethod plus ((x number) (y (eql 0)) &rest rest)
  (declare (dynamic-extent rest))
  (format t "(method plus (number (eql 0)))~%")
  (if rest
    (apply 'plus x rest)
    x))

(defclass modn ()
  ((value :initarg :value :reader modn-value)
   (mod :initarg :mod :reader modn-mod)
   (frob :initform "Hi, Mom.")
  ))

(defmethod print-object ((x modn) stream)
  (print-unreadable-object (x stream)
    (write-1 (modn-value x) stream)
    (pp-space stream)
    (write-1 'mod stream)
    (pp-space stream)
    (write-1 (modn-mod x) stream)))

(defmethod (setf modn-value) (value (x modn))
  (setf (slot-value x 'value) (mod value (modn-mod x))))

(defparameter x (make-instance 'modn :value 3 :mod 5))
(defparameter y (make-instance 'modn :value 4 :mod 5))

(defmethod plus ((x modn) (y number) &rest rest)
  (declare (dynamic-extent rest))
  (let ((res (make-instance 'modn :value (mod (plus (modn-value x) y)))))
    (if rest (apply 'plus res rest) res)))
(defmethod plus ((x number) (y modn) &rest rest)
  (declare (dynamic-extent rest))
  (apply 'plus y x rest))
(defmethod plus ((x modn) (y modn) &rest rest)
  (declare (dynamic-extent rest))
  (let ((mod (modn-mod x)))
    (unless (eql mod (modn-mod y))
      (error "Attempt to add numbers in different rings"))
    (let ((res (make-instance 'modn
                             :value (mod (plus (modn-value x) (modn-value y)) mod)
                             :mod mod)))
      (if rest (apply 'plus res rest) res))))
                   
               
;; test interface for Get-Nth-Arg-Combined-Method
(defun gnacm (gf &rest args)
    (if (symbolp gf)
      (setq gf (symbol-function gf)))
    (if (null args)
      (setq args '(1 2)))
    (apply 'get-nth-arg-combined-method gf 
           (if (typep gf 'standard-generic-function)
             (%gf-dispatch-table gf)
             (%combined-method-methods gf))
           args)
    )

|#
   

