;;;-*- Mode: Lisp; Package: CCL -*-
;;;
;;;   Copyright (C) 1994-2001 Digitool, Inc
;;;   This file is part of Opensourced MCL.
;;;
;;;   Opensourced MCL is free software; you can redistribute it and/or
;;;   modify it under the terms of the GNU Lesser General Public
;;;   License as published by the Free Software Foundation; either
;;;   version 2.1 of the License, or (at your option) any later version.
;;;
;;;   Opensourced MCL is distributed in the hope that it will be useful,
;;;   but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;;   Lesser General Public License for more details.
;;;
;;;   You should have received a copy of the GNU Lesser General Public
;;;   License along with this library; if not, write to the Free Software
;;;   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;;;


(eval-when (:compile-toplevel :load-toplevel :execute)
  (require "SPARC-LAP"))


(defsparclapmacro not (src &optional (dest src))
  `(xnor ,src %rzero ,dest))

(defsparclapmacro neg (src &optional (dest src))
  `(sub %rzero ,src ,dest))


(defsparclapmacro movcc (src dest)
  `(orcc %rzero ,src ,dest))

(defsparclapmacro inc (amt &optional dest)
  (unless dest (setq dest amt amt 1))
  `(add ,dest ,amt ,dest))

(defsparclapmacro inccc (amt &optional dest)
  (unless dest (setq dest amt amt 1))
  `(addcc ,dest ,amt ,dest))

(defsparclapmacro dec (amt &optional dest)
  (unless dest (setq dest amt amt 1))
  `(sub ,dest ,amt ,dest))

(defsparclapmacro deccc (amt &optional dest)
  (unless dest (setq dest amt amt 1))
  `(subcc ,dest ,amt ,dest))

(defsparclapmacro clr (dest)
  (if (atom dest)
    `(mov 0 ,dest)
    `(st %rzero ,dest)))

(defsparclapmacro bset (bits dest)
  `(or ,dest ,bits ,dest))

(defsparclapmacro btst (bits dest)
  `(andcc ,dest ,bits %rzero))

(defsparclapmacro bclr (bits dest)
  `(andn ,dest ,bits ,dest))

(defsparclapmacro btog (bits dest)
  `(xor ,dest ,bits ,dest))



#|
(defsparclapmacro dbg (&optional save-lr?)
  (if save-lr?
    `(progn
       (mflr loc-pc)
       (stw imm0 -40 sp) ; better than clobbering imm0
       (bla .SPbreakpoint)
       (lwz imm0 -40 sp)
       (mtlr loc-pc))
    `(bla .SPbreakpoint)))
|#

(defsparclapmacro set (n dest)
  (setq n (logand n #xffffffff))
  (let* ((mask #xfffff000)
         (masked (logand n mask))
         (high (ash n -10))
         (low (logand #x1fff n)))
    (if (or (= 0 masked) (= mask masked))
      `(mov  ,(if (logbitp 12 low) (- low (ash 1 12)) low) ,dest)
      (if (= low 0)
        `(sethi  ,high ,dest)
        `(progn
           (sethi ,high ,dest)
           (or ,dest ,(logand low (1- (ash 1 10))) ,dest))))))

(defsparclapmacro set-nargs (n)
  (check-type n (unsigned-byte 10))
  `(mov ',n %nargs))

(defsparclapmacro check-nargs (min &optional (max min))
  (if (eq max min)
    `(progn
      (cmp %nargs ',min)
      (tne #.sparc::trap-nargs))
    (if (null max)
      (unless (= min 0)
        `(progn
          (cmp %nargs ',min)
          (tlu #.sparc::trap-nargs)))
      (if (= min 0)
        `(progn
          (cmp %nargs ',max)
          (tgu #.sparc::trap-nargs))
        `(progn
          (cmp %nargs ',min)
          (tlu #.sparc::trap-nargs)
          (cmp %nargs ',max)
          (tgu #.sparc::trap-nargs))))))


; Event-polling involves checking to see if the value of *interrupt-level*
; is > 0.  For now, use nargs; this may change to "any register BUT nargs".
; (Note that most number-of-args traps use unsigned comparisons.)
(defsparclapmacro event-poll ()
  '(progn
    (ld (%rnil  #.(+ arch::symbol.vcell (arch::nrs-offset *interrupt-level*))) %nargs)
    (tst %nargs)
    (tg #. sparc::trap-event-poll)))

(defsparclapmacro create-lisp-frame ()
  `(progn
    (add %sp (- sparc::lisp-frame.size) %sp)
    (st %lsp (%sp 64))
    (add %sp 64 %lsp)))

(defsparclapmacro unlink (reg)
  `(ld (,reg) ,reg))

(defsparclapmacro restore-sp-from-lsp ()
  '(add %lsp (- 64) %sp))

(defsparclapmacro discard-lisp-frame ()
  `(progn
    (unlink %lsp)
    (restore-sp-from-lsp)))

; This needs to be done if we aren't a leaf function (e.g., if we clobber our
; return address or need to reference any constants.  Note that it's not
; atomic wrt a preemptive scheduler, but we need to pretend that it will be.)
; The VSP to be saved is the value of the VSP before any of this function's
; arguments were vpushed by its caller; that's not the same as the VSP register
; if any non-register arguments were received, but is usually easy to compute.

(defsparclapmacro save-lisp-context (&optional (vsp '%vsp))
  `(progn
    (create-lisp-frame)
    (st %fn (%lsp #.sparc::lisp-frame.savefn))
    (mov %nfn %fn)
    (st %ra0 (%lsp #.sparc::lisp-frame.savelr))
    (st ,vsp (%lsp #.sparc::lisp-frame.savevsp))))

; There are a few cases to deal with when restoring: whether or not to restore the
; vsp, whether we need to saved LR back in the LR or whether it only needs to get
; as far as loc-pc, etc.  This fully restores everything (letting the caller specify
; some register other than the VSP, if that's useful.)  Note that, since FN gets restored,
; it's no longer possible to use it to address the current function's constants.
(defsparclapmacro restore-full-lisp-context (&optional (vsp '%vsp))
  `(progn
    (ld (%lsp #.sparc::lisp-frame.savelr) %ra0)
    (ld (%lsp #.sparc::lisp-frame.savefn) %fn)
    (ld (%lsp #.sparc::lisp-frame.savevsp) ,vsp)
    (discard-lisp-frame)))

(defsparclapmacro stwu (src disp base)
  `(progn
    (st ,src (,base ,disp))
    (add ,base ,disp ,base)))

(defsparclapmacro push (src stack)
  `(stwu ,src -4 ,stack))

(defsparclapmacro vpush (src)
  `(push ,src %vsp))

; You typically don't want to do this to pop a single register (it's better to
; do a sequence of loads, and then adjust the stack pointer.)

(defsparclapmacro pop (dest stack)
  `(progn
    (ld (,stack) ,dest)
    (add ,stack 4 ,stack)))

(defsparclapmacro vpop (dest)
  `(pop ,dest %vsp))

(defsparclapmacro %cdr (node dest)
  `(ld (,node #.arch::cons.cdr) ,dest))

(defsparclapmacro %car (node dest)
  `(ld (,node #.arch::cons.car) ,dest))

(defsparclapmacro extract-lisptag (node dest)
  `(and ,node #.arch::tagmask ,dest))

(defsparclapmacro extract-fulltag (node dest)
  `(and ,node  #.arch::fulltagmask ,dest))

(defsparclapmacro extract-subtag (node dest)
  `(ldub (,node #.arch::misc-subtag-offset) ,dest))

(defsparclapmacro extract-typecode (node dest)
  (let* ((label (gensym)))
    `(progn
      (extract-lisptag ,node ,dest)
      (cmp ,dest arch::tag-misc)
      (be.a ,label)
       (extract-subtag ,node ,dest)
      ,label)))

(defsparclapmacro trap-unless-lisptag= (node tag &optional (immreg '%imm0))
  `(progn
    (extract-lisptag ,node ,immreg)
    (cmp ,immreg  ,tag)
    (tne #.sparc::trap-lisptag-check)))

(defsparclapmacro trap-unless-fulltag= (node tag &optional (immreg '%imm0))
  `(progn
    (extract-fulltag ,node ,immreg)
    (cmp ,immreg  ,tag)
    (tne #.sparc::trap-fulltag-check)))


(defsparclapmacro trap-unless-typecode= (node tag &optional (immreg '%imm0))
  `(progn
    (extract-typecode ,node ,immreg)
    (cmp ,immreg ,tag)
    (tne #.sparc::trap-typecode-check)))


(defsparclapmacro load-constant (constant dest)
  `(ld (%fn ',constant) ,dest))

;; This is about as hard on the pipeline as anything I can think of.
(defsparclapmacro call-symbol (function-name)
  `(progn
    (load-constant ,function-name %fname)
    (ld (%fname #.arch::symbol.fcell) %nfn)
    (ld (%nfn #.arch::misc-data-offset) %temp0)
    (jmpl %temp0 #.arch::misc-data-offset %ra0)))

(defsparclapmacro sp-call-symbol (function-name)
  `(progn
     (call-subprim .SPjmpsym)
     (load-constant ,function-name fname)))

(defsparclapmacro getvheader (src dest)
  `(ld (,src  arch::misc-header-offset) ,dest))

;; "Size" is unboxed element-count.
(defsparclapmacro header-size (vheader dest)
  `(srl ,vheader #.arch::num-subtag-bits ,dest))

;; "Length" is fixnum element-count.
(defsparclapmacro header-length (vheader dest)
  `(progn
    (andn ,vheader (1- (ash 1 arch::num-subtag-bits)) ,dest)
    (srl ,dest (- arch::num-subtag-bits arch::fixnumshift) ,dest)))

(defsparclapmacro header-subtag[fixnum] (vheader dest)
  `(progn
    (box-fixnum ,vheader ,dest)
    (and ,dest  (ash arch::subtag-mask arch::fixnumshift) ,dest)))



(defsparclapmacro vector-size (v vheader dest)
  `(progn
     (getvheader ,v ,vheader)
     (header-size ,vheader ,dest)))

(defsparclapmacro vector-length (v vheader dest)
  `(progn
     (getvheader ,v ,vheader)
     (header-length ,vheader ,dest)))


;; Reference a 32-bit miscobj entry at a variable index.
;; Make the caller explicitly designate a scratch register
;; to use for the scaled index.

(defsparclapmacro vref32 (miscobj index scaled-idx dest)
  `(progn
    (add ,index arch::misc-data-offset ,scaled-idx)
    (ld (,miscobj ,scaled-idx) ,dest)))


;; The simple (no-memoization) case.
(defsparclapmacro vset32 (src miscobj index scaled-idx)
  `(progn
    (add ,index arch::misc-data-offset ,scaled-idx)
    (st ,src (,miscobj ,scaled-idx))))

(defsparclapmacro extract-lowbyte (src dest)
  `(and ,src (1- (ash 1 arch::num-subtag-bits)) ,dest))


(defsparclapmacro unbox-fixnum (src dest)
  `(sra ,src arch::fixnumshift ,dest))

(defsparclapmacro box-fixnum (src dest)
  `(sll ,src arch::fixnumshift ,dest))

(defsparclapmacro box-signed-byte-32 (src temp dest)
  ; "temp" shouldn't be the same register as "src".
  ; "src" and "temp" should be unboxed registers; 
  ; "dest" should be a boxed register
  (let* ((lab1 (gensym))
         (lab2 (gensym)))
    `(progn
      (addcc ,src ,src ,temp)
      (bvc.a ,lab1)
      (addcc ,temp ,temp ,dest)      ; dest now a fixnum, may have lost bits
      ,lab1
      (bvs.a ,lab2)
      (uuo_box_signed ,dest ,src)
      ,lab2)))

(defsparclapmacro box-unsigned-byte-32 (src temp dest)
  ; "temp" shouldn't be the same register as "src".
  ; "src" and "temp" should be unboxed registers; 
  ; "dest" should be a boxed register
  (let* ((lab (gensym)))
    `(progn
      (sethi (ash #xe0000000 -10) ,temp)
      (andcc ,temp ,src ,temp)
      (box-fixnum ,src ,dest)
      (bne.a ,lab)
       (uuo_box_unsigned ,dest ,src)
      ,lab)))

; If check is specified, type checks src
(defsparclapmacro unbox-base-char (src dest &optional check)
  (if (null check)
    `(srl ,src #.arch::charcode-shift ,dest)
    (let ((label (gensym)))
      `(progn
        (extract-lowbyte ,src ,dest)
        (cmp ,dest #.sparc::subtag-character)
        (srl ,src #.arch::charcode-shift ,dest)
        (bne.a ,label)
        (uuo_interr arch::error-object-not-base-char ,src)
        ,label))))

(defsparclapmacro box-character (src dest)
  `(progn
     (sll ,src #.arch::charcode-shift ,dest)
     (or ,dest #.arch::subtag-character ,dest)))

(defsparclapmacro ref-global (reg sym)
  (let* ((offset (arch::%kernel-global sym)))
    `(ld (%rnil ,offset) ,reg)))

(defsparclapmacro set-global (reg sym)
  (let* ((offset (arch::%kernel-global sym)))
    `(st ,reg (%rnil ,offset))))

; Set "dest" to those bits in "src" that are other than those
; that would be set if "src" is a fixnum and of type (unsigned-byte "width").
; If no bits are set in "dest", then "src" is indeed of type (unsigned-byte "width").
(defsparclapmacro extract-unsigned-byte-bits (src width dest)
  `(progn
    (set ,(ash (1- (ash 1 width)) arch::fixnumshift) ,dest)
    (and ,src ,dest ,dest)))

; As above, but set flags according to the result.
(defsparclapmacro extract-unsigned-byte-bits. (src width dest)
  `(progn
    (set ,(ash (1- (ash 1 width)) arch::fixnumshift) ,dest)
    (andcc ,src ,dest ,dest)))


;;; from/blame slh:

; setpred depends on this
(eval-when (:compile-toplevel :execute :load-toplevel)
  (assert (= ppc::t-offset #x11)))

(defsparclapmacro setpred (dest crf cc-bit &optional (temp 'imm0))
  (let ((shift (+ (* (position crf '(:cr0 :cr1 :cr2 :cr3 :cr4 :cr5 :cr6 :cr7)) 4)
                  (position cc-bit '(:lt :gt :eq :so))
                  1)))
    `(progn
       (mfcr ,temp)
       (rlwinm ,temp ,temp ,shift 31 31)    ; get  1 bit
       (rlwimi ,temp ,temp      4 27 27)    ; get 16 bit
       (add ,dest ,temp rnil))))

; You generally don't want to have to say "mfcr": it crosses functional
; units and forces synchronization (all preceding insns must complete,
; no subsequent insns may start.)
; There are often algebraic ways of computing ppc::t-offset:

(defsparclapmacro eq0->boolean (dest src temp)
  `(progn
     (cntlzw ,temp ,src)                ; 32 leading zeros if (= rx ry)
     (srwi ,temp ,temp 5)               ; temp = (rx == ry), C-wise
     (rlwimi ,temp ,temp 4 27 27)       ; temp = ppc::t-offset or 0
     (add ,dest ppc::rnil ,temp)))      ; dest = (eq rx ry), lisp-wise

(defsparclapmacro eq->boolean (dest rx ry temp)
  `(progn
     (sub ,temp ,rx ,ry)
     (eq0->boolean ,dest ,temp ,temp)))

(defsparclapmacro get-single-float (node dest)
  `(ldf (,node #.arch::single-float.value) ,dest))

(defsparclapmacro get-double-float (node dest)
  `(lddf (,node #.arch::double-float.value) ,dest))

(defsparclapmacro put-single-float (src node)
  `(stf ,src (,node #.arch::single-float.value)))

(defsparclapmacro put-double-float (src node)
  `(stdf ,src (,node #.arch::double-float.value)))

#|
(defsparclapmacro clear-fpu-exceptions ()
  `(mtfsf #xfc #.ppc::fp-zero))
|#

#|
(defsparclapmacro get-boxed-sign (dest src)
  `(progn
     (load-constant ,dest 1)               ; assume positive
     (mtcrf ,crf ,src)
     (if (,crf :lt)
       (load-constant ,dest -1))))
|#

(defsparclapmacro digit-h (src  dest)
  `(progn
    (sethi (ash #xffff0000 -10) ,dest)
    (and ,src ,dest ,dest)
    (srl ,dest (- 16 arch::fixnumshift) ,dest)))


(defsparclapmacro digit-l (src dest)
  `(progn
    (sll ,src 16 ,dest)
    (srl ,dest (- 16 arch::fixnumshift) ,dest)))

(defsparclapmacro compose-digit (high low dest &optional (temp high))
  `(progn
     (sll ,high (- 16 arch::fixnumshift) ,temp)
     (sll ,low (- 16 arch::fixnumshift) ,dest)
     (srl ,dest 16 ,dest)
     (or ,temp ,dest ,dest)))


(defsparclapmacro macptr-ptr (macptr dest)
  `(ld (,macptr #.arch::macptr.address) ,dest))

(defsparclapmacro svref (vector index dest)
  `(ld (,vector (+ (* 4 ,index) #.arch::misc-data-offset)) ,dest))

; This evals its args in the wrong order.
; Can't imagine any code will care.
(defsparclapmacro svset (new-value index vector &optional no-memoize)
  (if no-memoize
    `(st ,new-value (,vector (+ (* 4 ,index) #.arch::misc-data-offset)))
    `(progn
      (add ,vector (+ (* 4 ,index) #.arch::misc-data-offset) %loc-g)
      (push %loc-g %memo)
      (st ,new-value (%loc-g)))))

; I don't know if anyone actually calls this with
; temp defaulting to src, but that'd be a really bad idea.
(defsparclapmacro make-int (result src &optional (temp src))
  (let ((lbl-done (gensym)))
    `(progn
       (mcrxr 0)                             ; could insist that XER[ov,so] and CR0[so] are always 0
       (addo ,temp ,src ,src)
       (addo. ,result ,temp ,temp)
       (bns+ ,lbl-done)
       (uuo_box_signed ,result ,src)
       ,lbl-done
       )))

(defsparclapmacro vpush-argregs ()
  (let* ((none (gensym))
         (two (gensym))
         (one (gensym)))
  `(progn
    (tst %nargs)
    (be ,none)
      (cmp %nargs '2)
    (be ,two)
      (nop)
    (bl ,one)
      (nop)
    (vpush %arg_x)
    ,two
    (vpush %arg_y)
    ,one
    (vpush %arg_z)
    ,none)))


#|
; Set FP-reg to 0.0 . Using (fsub fp-reg fp-reg fpreg)
; doesn't work if fp-reg contains a NaN.

(defsparclapmacro zero-fp-reg (fp-reg)
  (let* ((offset (arch::kernel-global short-float-zero)))
    `(lfs ,fp-reg ,offset rnil)))

(defsparclapmacro fp-check-binop-exception (rt ra rb)
  `(progn
     (if (:cr1 :gt)                     ; set if enabled exception has occurred
       (uuo_fpuXbinop ,rt ,ra ,rb))))

(defsparclapmacro fp-check-unaryop-exception (rt ra rb)
  `(progn
     (if (:cr1 :gt)                     ; set if enabled exception has occurred
       (uuo_fpuXbinop ,rt ,ra ,rb))))


; Functions to access exception frames

;;; These are VERY platform-specific, not suprisingly
#-eabi-target
(progn
(defsparclapmacro xp-register-image (ri xp)
  `(lwz ,ri ,(get-field-offset :ExceptionInformationPowerPC.registerImage) ,xp))

(defsparclapmacro ri-gpr (dest ri reg)
  `(lwz ,dest ,(xp-register-offset (eval reg)) ,ri))

(defsparclapmacro set-ri-gpr (source ri reg)
  `(stw ,source ,(xp-register-offset (eval reg)) ,ri))

(defsparclapmacro xp-machine-state (ms xp)
  `(lwz ,ms ,(get-field-offset :ExceptionInformationPowerPC.machineState) ,xp))

(defsparclapmacro ms-lr (dest ms)
  `(lwz ,dest ,(get-field-offset :MachineInformationPowerPC.lr.lo) ,ms))

(defsparclapmacro set-ms-lr (source ms)
  `(stw ,source ,(get-field-offset :MachineInformationPowerPC.lr.lo) ,ms))

(defsparclapmacro ms-pc (dest ms)
  `(lwz ,dest ,(get-field-offset :MachineInformationPowerPC.pc.lo) ,ms))

(defsparclapmacro set-ms-pc (source ms)
  `(stw ,source ,(get-field-offset :MachineInformationPowerPC.pc.lo) ,ms))

(defsparclapmacro ms-cr (dest ms)
  `(lwz ,dest ,(get-field-offset :MachineInformationPowerPC.cr) ,ms))

(defsparclapmacro set-ms-cr (source ms)
  `(stw ,source ,(get-field-offset :MachineInformationPowerPC.cr) ,ms))

(defsparclapmacro ms-xer (dest ms)
  `(lwz ,dest ,(get-field-offset :MachineInformationPowerPC.xer) ,ms))

(defsparclapmacro set-ms-xer (source ms)
  `(stw ,source ,(get-field-offset :MachineInformationPowerPC.xer) ,ms))

)


(defsparclapmacro linux-xp-regs (ri xp)
  `(lwz ,ri ,(get-field-offset :sigcontext.regs) ,xp))

(defsparclapmacro get-linux-regs-reg (dest ri reg)
  `(lwz ,dest ,(ash (eval reg) 2) ,ri))

(defsparclapmacro set-linux-regs-reg (source ri reg)
  `(stw ,source ,(ash (eval reg) 2) ,ri))

(eval-when (:compile-toplevel :execute)
(defconstant linux-reg-pc #.(ash (get-field-offset :pt-regs.nip) -2))
(defconstant linux-reg-lr #.(ash (get-field-offset :pt-regs.link) -2))
(defconstant linux-reg-cr #.(ash (get-field-offset :pt-regs.ctr) -2))
(defconstant linux-reg-xer #.(ash (get-field-offset :pt-regs.xer) -2))
)

(defsparclapmacro get-linux-regs-lr (dest ms)
  `(get-linux-regs-reg ,dest ,ms #.linux-reg-lr))

(defsparclapmacro set-linux-regs-lr (src ms)
  `(set-linux-regs-reg ,src ,ms #.linux-reg-lr))

(defsparclapmacro get-linux-regs-pc (dest ms)
  `(get-linux-regs-reg ,dest ,ms #.linux-reg-pc))

(defsparclapmacro set-linux-regs-pc (src ms)
  `(set-linux-regs-reg ,src ,ms #.linux-reg-pc))

(defsparclapmacro get-linux-regs-cr (dest ms)
  `(get-linux-regs-reg ,dest ,ms #.linux-reg-cr))

(defsparclapmacro set-linux-regs-cr (src ms)
  `(set-linux-regs-reg ,src ,ms #.linux-reg-cr))

(defsparclapmacro get-linux-regs-xer (dest ms)
  `(get-linux-regs-reg ,dest ,ms #.linux-reg-xer))

(defsparclapmacro set-linux-regs-xer (src ms)
  `(set-linux-regs-reg ,src ,ms #.linux-reg-xer))



 

|#

;; For consing.  While an object's being initialized, we can't
;; take an interrupt.  Note that fact by setting some bits in
;; %freeptr

(defsparclapmacro tag-freeptr (tagval)
  `(bset ,tagval %freeptr))

;; Clear low bits of the free pointer

(defsparclapmacro untag-freeptr ()
  `(bclr #.arch::fulltagmask %freeptr))



;;  Probe for a heap guard page, advance the free pointer, and mark
;;  the start of an atomic sequence.  Note that the free pointer points
;;  beyond the object that's being allocated: don't confuse that object
;;  with what the freepointer seems to be pointing to.
;;
;;  There are two variants: one for when the size is an immediate and
;;  one for when it's in a register 

(defsparclapmacro phys-alloc (n tagval)
  `(progn
    (st %rzero (%freeptr ,n))
    (inc (+ ,n ,tagval) %freeptr)))

(defsparclapmacro phys-alloc-var (regn tagval)
  `(progn
    (st %rzero (%freeptr ,regn))
    (inc ,regn %freeptr)
    (tag-freeptr ,tagval)))

(ccl::provide "SPARC-LAPMACROS")

; end of sparc-lapmacros.lisp
