/*
 * tgmb-page.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */


#ifndef lint
static const char *rcsid = "@(#) $Header: /usr/mash/src/repository/mash/mash-1/tgmb/client/tgmb-page.cc,v 1.7 2002/02/03 04:17:39 lim Exp $";
#endif


#include "tgmb-page.h"
#include "tgmb-canv.h"
#include "tgmb-app.h"

#define OFFSET(cls,field) (Word)(&((cls*)0)->field)


struct TGMB_PageChunkHdr {
	Word size, deletedSize, maxSize;
	ULong nextChunkId;
};


struct TGMB_FullObjectDescr {
	// header:
	VoidPtr objId;
	VoidPtr srcId;
	VoidPtr cmd;
	VoidPtr len;

	// descr:
	VoidPtr itemType;
	Word    npoints;
	VoidPtr points;
	VoidPtr color;
	VoidPtr fill;
	VoidPtr font;
	VoidPtr image;
	Word    imageSize;
	VoidPtr imageZoom;
	VoidPtr imageBits;
	VoidPtr width;
	VoidPtr arrowType;
	VoidPtr arrowVars;
	VoidPtr text;

	TGMB_ObjectId GetObjectId() { return *((TGMB_ObjectId*)objId); }
	SrcId *GetSrcId() { return (SrcId*) srcId; }
	DWord GetCmd() { return *((DWordPtr)cmd); }
	DWord GetLen() { return *((DWordPtr)len); }

	Word GetItemType() { return *((WordPtr)itemType); }
	Word GetNPoints() { return npoints; }
	PointType *GetPoints() { return (PointType*)points; }
	Word GetColor() { return *((WordPtr)color); }
	Word *GetFillPtr() { return ((WordPtr)fill); }
	CharPtr GetFont() { return (CharPtr) font; }
	VoidPtr GetImage() { return image; }
	Word GetImageSize() { return imageSize; }
	Word GetImageZoom() { return *((WordPtr)imageZoom); }
	Word GetImageBits() { return *((WordPtr)imageBits); }
	Word GetWidth() { return *((WordPtr)width); }
	Word GetArrowType() { return *((WordPtr)arrowType); }
	PointType *GetArrowVars() { return (PointType*)arrowVars; }
	CharPtr GetText() { return (CharPtr)text; }
};



struct CanvObj {
	Word size_;
	Word type_;

	TGMB_ObjectId thisId_, belowId_, aboveId_;

	void GetPoints(PointType *&points, Word &nPoints);
	void GetBoundingBox(AbsRectType *bbox, TGMB_Book *book);
};


struct LineObj : public CanvObj {
	Byte color_;
	Byte thickness_;
	PointType points_[0];

	Boolean Insert(TGMB_Page *page, TGMB_FullObjectDescr *descr);
	void Draw(TGMB_Page *page, WindowType *window);
	void GetPoints(Word &offset, Word &npoints) {
		offset = sizeof(LineObj);
		npoints= (size_ - offset)/sizeof(PointType);
	}
};


struct ArrowObj : public CanvObj {
	Byte color_;
	Byte thickness_;
	Word arrowType_;
	PointType vars_[2];
	PointType points_[0];

	Boolean Insert(TGMB_Page *page, TGMB_FullObjectDescr *descr);
	void Draw(TGMB_Page *page, WindowType *window);
	void GetPoints(Word &offset, Word &npoints) {
		offset = sizeof(ArrowObj);
		npoints= (size_ - offset)/sizeof(PointType);
	}
};


struct RectObj : public CanvObj {
	Byte outline_;
	Byte fill_;
	Byte shouldFill_;
	Byte thickness_;
	PointType points_[0];

	Boolean Insert(TGMB_Page *page, TGMB_FullObjectDescr *descr);
	void Draw(TGMB_Page *page, WindowType *window);
	void GetPoints(Word &offset, Word &npoints) {
		offset = sizeof(RectObj);
		npoints= 2;
	}
};


struct OvalObj : public CanvObj {
	Byte outline_;
	Byte fill_;
	Byte shouldFill_;
	Byte thickness_;
	PointType points_[0];

	Boolean Insert(TGMB_Page *page, TGMB_FullObjectDescr *descr);
	void Draw(TGMB_Page *page, WindowType *window);
	void GetPoints(Word &offset, Word &npoints) {
		offset = sizeof(OvalObj);
		npoints= 2;
	}
};


struct TextObj : public CanvObj {
	Byte color_;
	Byte font_;
	PointType coord_;
	Char text_[0];

	Boolean Insert(TGMB_Page *page, TGMB_FullObjectDescr *descr);
	void Draw(TGMB_Page *page, WindowType *window);
	void GetPoints(Word &offset, Word &npoints) {
		offset = OFFSET(TextObj, coord_);
		npoints= 1;
	}
};


struct ImageObj : public CanvObj {
	Word zoom_;
	Word bits_;
	PointType coord_;
	Byte image_[0];

	Boolean Insert(TGMB_Page *page, TGMB_FullObjectDescr *descr);
	void Draw(TGMB_Page *page, WindowType *window);
	void GetPoints(Word &offset, Word &npoints) {
		offset = OFFSET(ImageObj, coord_);
		npoints= 1;
	}
};


LineObj  proto_LineObj;
ArrowObj proto_ArrowObj;
RectObj  proto_RectObj;
OvalObj  proto_OvalObj;
TextObj  proto_TextObj;
ImageObj proto_ImageObj;


void
CanvObj::GetPoints(PointType *&points, Word &nPoints)
{
	Word offset;
	switch (type_) {
	case CanvObjType_Line:
		((LineObj*)this)->GetPoints(offset, nPoints); break;
	case CanvObjType_Arrow:
		((ArrowObj*)this)->GetPoints(offset, nPoints); break;
	case CanvObjType_Rect:
		((RectObj*)this)->GetPoints(offset, nPoints); break;
	case CanvObjType_Oval:
		((OvalObj*)this)->GetPoints(offset, nPoints); break;
	case CanvObjType_Text:
		((TextObj*)this)->GetPoints(offset, nPoints); break;
	case CanvObjType_Image:
		((ImageObj*)this)->GetPoints(offset, nPoints); break;
	}

	points = (PointType*) (BytePtr(this) + offset);
}


#define EXPAND_BBOX(bbox, x, y) \
        if ((x) < (bbox)->left) (bbox)->left = (x); \
	else if ((x) > (bbox)->right) (bbox)->right = (x); \
	if ((y) < (bbox)->top) (bbox)->top = (y); \
	else if ((y) > (bbox)->bottom) (bbox)->bottom = (y);


void
CanvObj::GetBoundingBox(AbsRectType *bbox, TGMB_Book *book)
{
	Word nPoints;
	PointType *points, p1, p2;
	GetPoints(points, nPoints);

	bbox->left = bbox->right  = points->x;
	bbox->top  = bbox->bottom = points->y;
	for (Word i=0; i < nPoints; i++) {
		/*if (points->x < bbox->left) bbox->left = points->x;
		else if (points->x > bbox->right) bbox->right = points->x;
		if (points->y < bbox->top) bbox->top = points->y;
		else if (points->y > bbox->bottom) bbox->bottom = points->y;*/
		EXPAND_BBOX(bbox, points->x, points->y);
		points++;
	}

	p1.x = bbox->left; p1.y = bbox->top;
	book->Page2Screen(&p1, &p2);
	bbox->left = p2.x; bbox->top = p2.y;
	if (type_==CanvObjType_Image) {
		BitmapPtr bmp = (BitmapPtr)((ImageObj*)this)->image_;
		Word z = ((ImageObj*)this)->zoom_;
		if (z==book->Zoom()) {
			bbox->right = bbox->left + bmp->width/2  - 1;
			bbox->bottom= bbox->top  + bmp->height   - 1;
		} else {
			bbox->right = bbox->left +
				(DWord(bmp->width/2)*book->Zoom())/z - 1;
			bbox->bottom= bbox->top +
				(DWord(bmp->height )*book->Zoom())/z - 1;
		}
	} else if (type_==CanvObjType_Text) {
		Win2_Context cxt;
		cxt.font = (FontID) ((TextObj*)this)->font_;
		Win2_GetTextBBox(&cxt, &bbox->right, &bbox->bottom,
				 ((TextObj*)this)->text_);
		bbox->bottom += bbox->top  - 1;
		bbox->right  += bbox->left - 1;
	} else {
		// convert to screen coords
		p1.x = bbox->right; p1.y = bbox->bottom;
		book->Page2Screen(&p1, &p2);
		bbox->right = p2.x; bbox->bottom = p2.y;

		if (type_==CanvObjType_Arrow) {
			// get the arrow heads in as well
			ArrowObj *a = (ArrowObj*)this;
			Word at = a->arrowType_;
			SWord x, y;
			PointType pnt;
			if (at==ArrowFirst || at==ArrowBoth) {
				book->Page2Screen(points, &pnt);
				x = pnt.x+a->vars_[0].x; y=pnt.y+a->vars_[1].y;
				EXPAND_BBOX(bbox, x, y);
				x = pnt.x+a->vars_[1].x; y=pnt.y+a->vars_[1].y;
				EXPAND_BBOX(bbox, x, y);
			}

			if (at==ArrowLast || at==ArrowBoth) {
				book->Page2Screen(points+nPoints-1, &pnt);
				x = pnt.x-a->vars_[0].x; y=pnt.y-a->vars_[1].y;
				EXPAND_BBOX(bbox, x, y);
				x = pnt.x-a->vars_[1].x; y=pnt.y-a->vars_[1].y;
				EXPAND_BBOX(bbox, x, y);
			}
		}
	}
}


TGMB_Book::TGMB_Book() : command_(this)
{
	current_ = NULL;
	zoom_ = 33;
	origin_.x = origin_.y = 0;
	onscreenWindow_ = offscreenWindow_ = NULL;
	pages_ = NULL;
	numPages_ = maxPages_ = 0;
	localPageId_ = 1;
	localObjId_ = 0x80000000;

	TGMB_Canvas::ScreenRect(&screenRect_);
}


TGMB_Book::~TGMB_Book()
{
	if (offscreenWindow_) Win2_DeleteWindow(offscreenWindow_, false);
}


void
TGMB_Book::Refresh(AbsRectType *clip)
{
	if (current_) current_->Refresh(clip);
	else {
		// clear the rectangle
		Win2_Context cxt;
		cxt.window = OnscreenWindow();
		AbsRectType oldclip = cxt.window->clippingBounds, defClip;
		if (!clip) { DefaultClip(&defClip); clip = &defClip; }
		cxt.window->clippingBounds = *clip;
		cxt.fill = 1;
		cxt.width = 0;
		cxt.fillClr = 0xFF;
		Win2_DrawRectangle(&cxt, clip->left, clip->top, clip->right,
				   clip->bottom);
		cxt.window->clippingBounds = oldclip;
	}
}


void
TGMB_Book::Open(DmOpenRef db, ULong pageDescrsId)
{
	onscreenWindow_  = Win2_GetOnscreenWindow();
	if (!offscreenWindow_)
		offscreenWindow_ = Win2_CreateOffscreenWindow(160, 160,
							      NULL, NULL);

	db_ = db;
	ht_pages_.Open(db_, sizeof(PageId));
	current_ = NULL;
	/*zoom_ = 33;
	origin_.x = origin_.y = 0;*/

	if (pageDescrsId==TGMB_INVALID) {
		UInt index;
		maxPages_ = 16;
		descrs_ = DmNewRecord(db_, &index, maxPages_ *
				      sizeof(TGMB_PageDescr));
		DmRecordInfo(db_, index, NULL, &descrsId_, NULL);
		pages_ = NULL;
		numPages_ = 0;
	} else {
		UInt index;
		TG_Pointer entry;

		descrsId_ = pageDescrsId;
		DmFindRecordByID(db_, pageDescrsId, &index);
		descrs_ = DmGetRecord(db_, index);

		maxPages_ = MemHandleSize(descrs_)/sizeof(TGMB_PageDescr);
		TGMB_PageDescr *descrPtr = (TGMB_PageDescr*)
			MemHandleLock(descrs_), *p;
		numPages_ = 0;
		p = descrPtr;
		while (numPages_<maxPages_ && p->id.uid!=TGMB_INVALID) {
			numPages_++;
			p++;
		}

		pages_ = new TGMB_Page [numPages_];
		for (Word i=0; i<numPages_; i++) {
			pages_[i].Open(db_, descrPtr+i);
			pages_[i].Book(this);
			entry.offset = i;
			ht_pages_.Insert(&descrPtr[i].id, &entry);
		}
		MemHandleUnlock(descrs_);
	}
}


ULong
TGMB_Book::Close(Boolean deleteAll)
{
	VoidPtr ptr=MemHandleLock(descrs_);
	TGMB_PageDescr descr;

	for (Word i=0; i < numPages_; i++) {
		pages_[i].Close(&descr, deleteAll);
		DmWrite(ptr, i*sizeof(TGMB_PageDescr), &descr,
			sizeof(TGMB_PageDescr));
	}
	if (numPages_ < maxPages_) {
		DmSet(ptr, numPages_*sizeof(PageId), sizeof(PageId), 0xFF);
	}
	MemHandleUnlock(descrs_);

	delete [] pages_;
	pages_ = NULL;
	numPages_ = maxPages_ = 0;
	current_ = NULL;

	UInt index;
	DmFindRecordByID(db_, descrsId_, &index);
	DmReleaseRecord(db_, index, true);
	if (deleteAll==true) DmDeleteRecord(db_, index);
	ht_pages_.Close();
	return descrsId_;
}


void
TGMB_Book::Scroll(SWord dx, SWord dy)
{
	dx = (SDWord(dx) * 100)/zoom_;
	dy = (SDWord(dy) * 100)/zoom_;
	origin_.x += dx; origin_.y += dy;
	TGMB_Application::Instance()->Book()->Refresh();
}


TGMB_Page *
TGMB_Book::FindPage(const PageId &pageId, Boolean *created)
{
	TG_Pointer entry;
	if (ht_pages_.Find((void*)&pageId, &entry)) {
		if (created) *created = false;
		return &pages_[entry.offset];
	} else {
		if (created) *created = true;
		return NewPage(pageId);
	}
}


TGMB_Page *
TGMB_Book::CreateNewPage()
{
	Boolean created=false;
	PageId pid;
	TGMB_Page *p;
	pid.sid = APP->Session()->MySrcId();

	while (!created) {
		pid.uid = localPageId_++;
		p = FindPage(pid, &created);
	}

	APP->Session()->NewPage(p);
	return p;
}


TGMB_Page *
TGMB_Book::NewPage(const PageId &pageId)
{
	TGMB_Page *newPages;
	TGMB_PageDescr descr;

	newPages = new TGMB_Page[numPages_+1];
	if (pages_ && numPages_>0) {
		if (current_) current_ = newPages + (current_ - pages_);
		MemMove(newPages, pages_, sizeof(TGMB_Page)*numPages_);
	}

	// initialize the new page
	descr.id = pageId;
	descr.firstChunkId = TGMB_INVALID;
	newPages[numPages_].Open(db_, &descr);
	newPages[numPages_].Book(this);

	// write the description of this page to the book database record
	VoidPtr descrPtr;
	if (numPages_ >= maxPages_) {
		// grow the book record
		UInt index;
		maxPages_ *= 2;
		DmFindRecordByID(db_, descrsId_, &index);
		descrs_ = DmResizeRecord(db_, index,
					 maxPages_*sizeof(TGMB_PageDescr));
	}
	descrPtr = MemHandleLock(descrs_);
	DmWrite(descrPtr, numPages_*sizeof(TGMB_PageDescr), &descr,
		sizeof(TGMB_PageDescr));
	MemHandleUnlock(descrs_);

	TG_Pointer entry;
	entry.offset = numPages_;
	ht_pages_.Insert(&descr.id, &entry);

	delete [] pages_;
	pages_ = newPages;
	numPages_++;

	return &pages_[numPages_-1];
}


Boolean
TGMB_Book::CreateObjectFromCommand(TGMB_ObjectId *idPtr)
{
	*idPtr = localObjId_++;
	TGMB_FullObjectDescr descr;
	DWord cmd = cmdCanvasCreate;
	Word outlineClr = Word(command_.cxt_.outlineClr) << 8;
	Word fillClr =  Word(command_.cxt_.fillClr) << 8;
	Word itemType = command_.GetItemType();
	Word arrowType = Word(command_.arrowType_);
	PointType arrowVars[2] = { {0,0}, {0,0} };
	command_.points_ = (PointType*)MemHandleLock(command_.handle_);

	descr.objId = idPtr;
	descr.srcId = (VoidPtr)&APP->Session()->MySrcId();
	descr.cmd = &cmd;
	descr.itemType = &itemType;
	descr.npoints = command_.nPoints_;
	descr.points = command_.points_;
	descr.color = &outlineClr;
	descr.fill = (command_.cxt_.fill ? &fillClr : NULL);
	descr.font = NULL;
	descr.width = &command_.width_;
	descr.arrowType = &arrowType;
	descr.arrowVars = arrowVars;
	descr.text = (char*)(command_.points_+1);

	Boolean retval;
	if (current_) retval = current_->CreateObject(&descr, 0);
	else retval = false;
	MemHandleUnlock(command_.handle_);
	return retval;
}


void
TGMB_Book::DefaultClip(AbsRectType *clip)
{
	*clip = screenRect_;
}


inline Boolean
TGMB_Page::IsVisible()
{
	return Boolean(book_->CurrentPage()==this);
}


void
TGMB_Page::Open(DmOpenRef db, const TGMB_PageDescr *descr)
{
	db_ = db;
	descr_ = *descr;
	topmostId_ = TGMB_INVALID;
	bottommostId_ = TGMB_INVALID;

	numChunks_ = 0;
	chunks_ = NULL;
	ht_objects_.Open(db, sizeof(TGMB_ObjectId));

	ULong chunkId = descr_.firstChunkId;
	while (chunkId!=TGMB_INVALID)
		chunkId = InitializeChunk(chunkId);
}


void
TGMB_Page::Close(TGMB_PageDescr *descr, Boolean deleteAll)
{
	ht_objects_.Close();

	Word i=0;
	UInt index;
	ULong id, nextId;

	for (id=descr_.firstChunkId, i=0; i<numChunks_; i++, id=nextId) {
		VoidPtr ptr = MemHandleLock(chunks_[i]);
		nextId = ((TGMB_PageChunkHdr*)ptr)->nextChunkId;
		MemHandleUnlock(chunks_[i]);

		DmFindRecordByID(db_, id, &index);
		DmReleaseRecord(db_, index, true);
		if (deleteAll==true) DmDeleteRecord(db_, index);
	}

	*descr = descr_;
}


#define FATAL(x) TGMB_Application::ErrorDialog("%s (%d): Could not find id " \
					       "%lu in hash table", \
					       __FILE__, __LINE__, (x))
void
TGMB_Page::Refresh(AbsRectType *clip)
{
	TG_Pointer tgmbPtr;
	TGMB_ObjectId id=bottommostId_;
	VoidHand h=(VoidHand)TGMB_INVALID;
	CanvObj *canvObj;
	VoidPtr ptr=NULL;
	AbsRectType bbox, defClip;

	if (!clip) { book_->DefaultClip(&defClip); clip = &defClip; }

	// calculate the page coordinates of the clipping region
	/*PointType p1, p2, p;
	p.x = clip->left; p.y = clip->top;
	book_->Screen2Page(&p, &p1);
	p.x = clip->right; p.y = clip->bottom;
	book_->Screen2Page(&p, &p2);
	pageClip.left = p1.x; pageClip.right = p2.x;
	pageClip.top  = p1.y; pageClip.bottom= p2.y;*/

	// clear the off-screen drawing area first
	Win2_Context cxt;
	cxt.window = book_->OffscreenWindow();
	cxt.width  = 0;
	cxt.fill   = 1;
	cxt.fillClr= 0xFF;
	AbsRectType oldclip = cxt.window->clippingBounds;
	cxt.window->clippingBounds = *clip;
	Win2_DrawRectangle(&cxt, clip->left, clip->top, clip->right,
			   clip->bottom);

	// loop thru all the objects on the page
	while (id!=TGMB_INVALID) {
		if (!ht_objects_.Find(&id, &tgmbPtr)) {FATAL(id); break;}
		if (h!=tgmbPtr.handle) {
			if (h!=(VoidHand)TGMB_INVALID) MemHandleUnlock(h);
			ptr = MemHandleLock(tgmbPtr.handle);
			h = tgmbPtr.handle;
		}
		canvObj =((CanvObj*) (ptr + tgmbPtr.offset));
		canvObj->GetBoundingBox(&bbox, book_);
		if (bbox.right >= clip->left && bbox.left<=clip->right &&
		    bbox.bottom>= clip->top  && bbox.top <=clip->bottom)
			DrawLockedObject(canvObj, book_->OffscreenWindow(),
					 clip);
		id = canvObj->aboveId_;
	}
	if (h!=(VoidHand)TGMB_INVALID) MemHandleUnlock(h);

	// draw the current local object if any
	if (book_->Command()->IsActive()) book_->Command()->Draw(cxt.window);
	cxt.window->clippingBounds = oldclip;

	// tranfer the offscreen window to the screen
	Win2_CopyRectangle(book_->OffscreenWindow(), book_->OnscreenWindow(),
			   clip, clip->left, clip->top, scrCopy);
}


void
TGMB_Page::DrawLockedObject(CanvObj *obj, WindowType *window,AbsRectType *clip)
{
	AbsRectType oldclip = window->clippingBounds, defClip;
	if (!clip) { book_->DefaultClip(&defClip); clip = &defClip; }
	window->clippingBounds = *clip;
	switch (obj->type_) {
	case CanvObjType_Line:
		((LineObj*)obj)->Draw(this, window); break;
	case CanvObjType_Arrow:
		((ArrowObj*)obj)->Draw(this, window); break;
	case CanvObjType_Rect:
		((RectObj*)obj)->Draw(this, window); break;
	case CanvObjType_Oval:
		((OvalObj*)obj)->Draw(this, window); break;
	case CanvObjType_Text:
		((TextObj*)obj)->Draw(this, window); break;
	case CanvObjType_Image:
		((ImageObj*)obj)->Draw(this, window); break;
	}
	window->clippingBounds = oldclip;
}


ULong
TGMB_Page::InitializeChunk(ULong chunkId)
{
	VoidHand *newChunks = new VoidHand [numChunks_+1];
	for (Word i=0; i<numChunks_; i++) newChunks[i] = chunks_[i];

	UInt index;
	DmFindRecordByID(db_, chunkId, &index);
	newChunks[numChunks_] = DmGetRecord(db_, index);
	TGMB_PageChunkHdr *hdr = (TGMB_PageChunkHdr*)
			MemHandleLock(newChunks[numChunks_]);
	chunkId = hdr->nextChunkId;

	// run thru the chunk and insert all objects into the ht_objects_
	// table
	Word size=0;
	CanvObj *obj;
	TG_Pointer ptr;
	ptr.handle = newChunks[numChunks_];
	while (size < hdr->size) {
		ptr.offset = sizeof(TGMB_PageChunkHdr) + size;
		obj = (CanvObj*)(((BytePtr)hdr) + ptr.offset);
		ht_objects_.Insert(&obj->thisId_, &ptr);
		size += obj->size_;

		if (obj->aboveId_==TGMB_INVALID) topmostId_ = obj->thisId_;
		if (obj->belowId_==TGMB_INVALID) bottommostId_ = obj->thisId_;
	}

	MemHandleUnlock(newChunks[numChunks_]);
	chunks_ = newChunks;
	numChunks_++;
	return chunkId;
}


Boolean
TGMB_Page::CreateChunk()
{
	VoidHand *newChunks = new VoidHand [numChunks_+1];
	for (Word i=0; i<numChunks_; i++) newChunks[i] = chunks_[i];

	Word size=32768;
	ULong id;
	UInt index;
	newChunks[numChunks_] = NULL;
	while (size > 1024) {
		newChunks[numChunks_] = DmNewRecord(db_, &index,size);
		if (newChunks[numChunks_]!=0) break;
		size /= 2;
	}

	if (newChunks[numChunks_]==0) {
		delete [] newChunks;
		return false;
	}

	DmRecordInfo(db_, index, NULL, &id, NULL);
	if (chunks_) {
		delete [] chunks_;

		VoidPtr pointer = MemHandleLock(newChunks[numChunks_-1]);
		DmWrite(pointer, OFFSET(TGMB_PageChunkHdr, nextChunkId),
			&id, sizeof(ULong));
		MemHandleUnlock(newChunks[numChunks_-1]);
	} else {
		// this is the first chunk; record it in the descr_ object
		descr_.firstChunkId = id;
	}
	TGMB_PageChunkHdr hdr;
	hdr.size = 0;
	hdr.deletedSize = 0;
	hdr.maxSize = size;
	hdr.nextChunkId = TGMB_INVALID;
	VoidPtr pointer = MemHandleLock(newChunks[numChunks_]);
	DmWrite(pointer, 0, &hdr, sizeof(TGMB_PageChunkHdr));

	chunks_ = newChunks;
	numChunks_++;
	return true;
}


void *
TGMB_Page::FindAndLockObject(TGMB_ObjectId id, TG_Pointer *tgmbPtr)
{
	if (!ht_objects_.Find(&id, tgmbPtr)) return NULL;
	return MemHandleLock(tgmbPtr->handle);
}


void *
TGMB_Page::CreateAndLockObject(TGMB_ObjectId id, Word size, Word type,
			       TG_Pointer *tgmbPtr,
			       Boolean setBelowAboveIds)
{
	void *ptr=NULL;
	if (numChunks_ > 0)
		ptr = CreateAndLockObject(chunks_[numChunks_-1], id, size,
					  type, tgmbPtr, setBelowAboveIds);
	if (!ptr) {
		// try creating a new chunk
		if (!CreateChunk()) return NULL;
		ptr = CreateAndLockObject(chunks_[numChunks_-1], id, size,type,
					  tgmbPtr, setBelowAboveIds);
	}
	return ptr;
}


void *
TGMB_Page::CreateAndLockObject(VoidHand handle, TGMB_ObjectId id, Word size,
			       Word type, TG_Pointer *tgmbPtr,
			       Boolean setBelowAboveIds)
{
	VoidPtr ptr = MemHandleLock(handle);
	TGMB_PageChunkHdr hdr = *((TGMB_PageChunkHdr*)ptr);
	if (sizeof(TGMB_PageChunkHdr) + hdr.size + size > hdr.maxSize) {
		// object is too big to fit in this chunk
		MemHandleUnlock(handle);
		return NULL;
	}

	tgmbPtr->handle = handle;
	tgmbPtr->offset = hdr.size + sizeof(TGMB_PageChunkHdr);
	CanvObj obj;
	obj.size_   = size;
	obj.type_   = type;
	obj.thisId_ = id;
	obj.aboveId_= TGMB_INVALID;
	obj.belowId_= TGMB_INVALID;

	DmWrite(ptr, tgmbPtr->offset, &obj, sizeof(CanvObj));
	if (setBelowAboveIds) {
		RaiseLockedObject(tgmbPtr, ptr, topmostId_);
		ht_objects_.Insert(&id, tgmbPtr);
	}

	hdr.size += size;
	DmWrite(ptr, OFFSET(TGMB_PageChunkHdr, size), &hdr.size,
		sizeof(hdr.size));

	return ptr;
}


Boolean
TGMB_Page::RaiseLockedObject(TG_Pointer *tgmbPtr, VoidPtr ptr,
			     TGMB_ObjectId after)
{
	CanvObj *obj = (CanvObj*)(((BytePtr)ptr)+tgmbPtr->offset);
	TG_Pointer otherTgmbPtr;
	VoidPtr otherPtr;
	TGMB_ObjectId aboveId;

	if (after!=TGMB_INVALID) {
		if (ht_objects_.Find(&after, &otherTgmbPtr)==false) {
			FATAL(after);
			return false;
		}
		if (otherTgmbPtr.handle != tgmbPtr->handle) {
			otherPtr = MemHandleLock(otherTgmbPtr.handle);
		} else {
			otherPtr = ptr;
		}

		CanvObj *otherObj = (CanvObj*)(((BytePtr)otherPtr)
					       + otherTgmbPtr.offset);
		aboveId = otherObj->aboveId_;

		DmWrite(otherPtr,
			otherTgmbPtr.offset + OFFSET(CanvObj, aboveId_),
			&obj->thisId_, sizeof(TGMB_ObjectId));
	} else {
		aboveId = bottommostId_;
		otherTgmbPtr = *tgmbPtr;
	}

	DmWrite(ptr, tgmbPtr->offset + OFFSET(CanvObj, belowId_),
		&after, sizeof(TGMB_ObjectId));
	if (otherTgmbPtr.handle != tgmbPtr->handle)
		MemHandleUnlock(otherTgmbPtr.handle);

	if (topmostId_==after) topmostId_ = obj->thisId_;
	if (after==TGMB_INVALID) bottommostId_ = obj->thisId_;

	if (aboveId==TGMB_INVALID|| !ht_objects_.Find(&aboveId,&otherTgmbPtr)){
		if (aboveId!=TGMB_INVALID) FATAL(aboveId);
		aboveId = TGMB_INVALID;
		DmWrite(ptr, tgmbPtr->offset + OFFSET(CanvObj, aboveId_),
			&aboveId, sizeof(TGMB_ObjectId));
		return true;
	}

	if (otherTgmbPtr.handle != tgmbPtr->handle) {
		otherPtr = MemHandleLock(otherTgmbPtr.handle);
	} else {
		otherPtr = ptr;
	}

	DmWrite(otherPtr, otherTgmbPtr.offset + OFFSET(CanvObj, belowId_),
		&obj->thisId_, sizeof(TGMB_ObjectId));
	DmWrite(ptr, tgmbPtr->offset + OFFSET(CanvObj, aboveId_),
		&aboveId, sizeof(TGMB_ObjectId));
	if (otherTgmbPtr.handle != tgmbPtr->handle)
		MemHandleUnlock(otherTgmbPtr.handle);
	return true;
}


void
TGMB_Page::MarkAsDeleted(void *ptr, Word offset)
{
	Word word = CanvObjType_Deleted;
	DmWrite(ptr, offset + OFFSET(CanvObj, type_),
		&word, sizeof(Word));

	TGMB_PageChunkHdr hdr = *((TGMB_PageChunkHdr*)ptr);
	hdr.deletedSize -= ((CanvObj*)(((BytePtr)ptr)+offset))->size_;
	DmWrite(ptr, 0, &hdr, sizeof(TGMB_PageChunkHdr));

	// FIXME: do an optimization where we check for a set of contiguous
	// deleted records at the end of the page and just move all pointers
	// back accordingly
}


void
Union(AbsRectType *r1, const AbsRectType *r2)
{
	if (r1->left==-1 && r1->right==-1 && r1->top==-1 && r1->bottom==-1)
		*r1 = *r2;
	else {
		if (r2->left  < r1->left)   r1->left  = r2->left;
		if (r2->top   < r1->top)    r1->top   = r2->top;
		if (r2->right > r1->right)  r1->right = r2->right;
		if (r2->bottom> r1->bottom) r1->bottom= r2->bottom;
	}
}


Boolean
Intersect(AbsRectType *r1, const AbsRectType *r2)
{
	if (r1->left  < r2->left)   r1->left  = r2->left;
	if (r1->top   < r2->top)    r1->top   = r2->top;
	if (r1->right > r2->right)  r1->right = r2->right;
	if (r1->bottom> r2->bottom) r1->bottom= r2->bottom;
	if (r1->left > r1->right || r1->top > r1->bottom) return false;
	else return true;
}


Boolean
TGMB_Page::ParsePacket(VoidPtr data, ULong len, DWord requestId)
{
	Boolean retval=true;
	ULong skip;
	AbsRectType refreshBox = { -1, -1, -1, -1 };
	while (len > 0) {
		skip = ParseFragment(data, len, requestId, &refreshBox);
		if (skip==Word(-1)) { retval=false; break; }
		len  -= skip;
		data += skip;
	}

	if (IsVisible() && Intersect(&refreshBox, book_->ScreenRect()))
		Refresh(&refreshBox);
	return retval;
}


Word
TGMB_Page::ParseFragment(VoidPtr data, ULong len, DWord requestId,
			 AbsRectType *refreshBox)
{
	BytePtr start=BytePtr(data);
	Word itemDescr, itemSize;
	AbsRectType bbox;
	TGMB_FullObjectDescr descr;
	MemSet(&descr, sizeof(descr), 0);

	// parse header
	descr.objId = data;
	data = LongPtr(data) + 1;

	descr.srcId = data;
	data = BytePtr(data) + sizeof(SrcId);

	descr.cmd = data;
	data = LongPtr(data) + 1;

	descr.len = data;
	data = LongPtr(data) + 1;

	Word skip = (BytePtr(data)-start) + descr.GetLen();
	if (len < skip) return Word(-1);
	len = descr.GetLen();

	while (len > 0) {
		itemDescr = *(WordPtr(data)++);
		itemSize  = *(WordPtr(data)++);
		itemSize += itemSize % 2;
		if (itemSize > len - sizeof(Word[2])) return Word(-1);

		switch(itemDescr) {
		case descrItemType:  descr.itemType = data; break;
		case descrCoords:
			descr.npoints = itemSize/sizeof(PointType);
			descr.points  = data;
			break;
		case descrColor:     descr.color = data; break;
		case descrFill:      descr.fill  = data; break;
		case descrFont:      descr.font  = data; break;
		case descrImage:
			descr.imageSize = itemSize;
			descr.image     = data;
			break;
		case descrImageZoom: descr.imageZoom = data; break;
		case descrImageBits: descr.imageBits = data; break;
		case descrWidth:     descr.width     = data; break;
		case descrArrow:
			descr.arrowType = data;
			descr.arrowVars = WordPtr(data)+1;
			break;
		case descrText: descr.text = data; break;
		}

		len -= sizeof(Word[2]) + itemSize;
		data+= itemSize;
	}

	switch (descr.GetCmd()) {
	case cmdCanvasCreate:
		if (!CreateObject(&descr, requestId)) return Word(-1);
		break;

	case cmdCanvasMove:
		if (!MoveObject(descr.GetObjectId(), descr.GetPoints(), &bbox))
			return Word(-1);
		Union(refreshBox, &bbox);
		break;

	case cmdCanvasDelete:
		if (!DeleteObject(descr.GetObjectId(), &bbox)) return Word(-1);
		Union(refreshBox, &bbox);
		break;

	case cmdCanvasConfigText:
		if (!ConfigTextObject(descr.GetObjectId(), descr.GetText()))
			return Word(-1);
		break;

#ifdef NOTYET
	case cmdCanvasRaise:
		if (!RaiseObject(descr.GetObjectId(), )) return Word(-1);
		break; // Ignore
#endif

	default:
		return Word(-1);
	}

	return skip;
}


Boolean
TGMB_Page::CreateObject(TGMB_FullObjectDescr *descr, DWord requestId)
{
	switch (descr->GetItemType()) {
	case PgItemMLine:
	case PgItemLine:
		if (descr->GetArrowType()==ArrowNone) {
			if (!proto_LineObj.Insert(this, descr)) return false;
		} else {
			if (!proto_ArrowObj.Insert(this, descr)) return false;
		}
		break;

	case PgItemRect:
		if (!proto_RectObj.Insert(this, descr)) return false;
		break;

	case PgItemOval:
		if (!proto_OvalObj.Insert(this, descr)) return false;
		break;

	case PgItemText:
		if (!proto_TextObj.Insert(this, descr)) return false;
		break;

	case PgItemImage:
		if (!proto_ImageObj.Insert(this, descr)) return false;
		break;

	default:
		return false;
	}

	if (requestId > 0) {
		// this is in respone to an object that was originally
		// generated locally
		// we must delete the temp object now
		AbsRectType bbox;
		TGMB_Application::ErrorDialog("deleting local object %lu",
					      (DWord)requestId);
		DeleteObject((TGMB_ObjectId)requestId, &bbox);
		if (Intersect(&bbox, book_->ScreenRect()))
			Refresh(&bbox);
	}

	return true;
}


Boolean
TGMB_Page::MoveObject(TGMB_ObjectId id, const PointType *delta,
		      AbsRectType *bbox)
{
	TG_Pointer tgmbPtr;
	VoidPtr ptr;
	Word nPoints, offset;
	PointType *points, pnt;

	if (!ht_objects_.Find(&id, &tgmbPtr)) {FATAL(id); return false;}
	ptr = MemHandleLock(tgmbPtr.handle);
	CanvObj *canvObj = (CanvObj*) (ptr + tgmbPtr.offset);
	canvObj->GetPoints(points, nPoints);
	offset = ((BytePtr)points) - ((BytePtr)canvObj);

	canvObj->GetBoundingBox(bbox, book_);
	for (Word i=0; i < nPoints; i++) {
		pnt = *points++;
		// write the new coordinates
		pnt.x += delta->x;
		pnt.y += delta->y;
		DmWrite(ptr, tgmbPtr.offset + offset, &pnt, sizeof(PointType));
		offset += sizeof(PointType);
	}

	if (IsVisible())
		DrawLockedObject((CanvObj*)(ptr+tgmbPtr.offset),
				 book_->OnscreenWindow());
	MemHandleUnlock(tgmbPtr.handle);
	return true;
}


Boolean
TGMB_Page::DeleteObject(TGMB_ObjectId id, AbsRectType *bbox)
{
	TG_Pointer tgmbPtr;
	VoidPtr ptr, other;

	if (!ht_objects_.Find(&id, &tgmbPtr)) {FATAL(id); return false;}
	ptr = MemHandleLock(tgmbPtr.handle);

	// update the linked list
	CanvObj *obj = (CanvObj*)(((BytePtr)ptr)+tgmbPtr.offset);
	TG_Pointer aboveTgmbPtr, belowTgmbPtr;
	aboveTgmbPtr.handle = belowTgmbPtr.handle = (VoidHand)TGMB_INVALID;
	aboveTgmbPtr.offset = belowTgmbPtr.offset = (Word)TGMB_INVALID;

	if (obj->aboveId_!=TGMB_INVALID)
		ht_objects_.Find(&obj->aboveId_, &aboveTgmbPtr);
	if (obj->belowId_!=TGMB_INVALID)
		ht_objects_.Find(&obj->belowId_, &belowTgmbPtr);

	if (aboveTgmbPtr.offset!=(Word)TGMB_INVALID) {
		if (tgmbPtr.handle!=aboveTgmbPtr.handle)
			other = MemHandleLock(aboveTgmbPtr.handle);
		else
			other = ptr;
		DmWrite(other, aboveTgmbPtr.offset + OFFSET(CanvObj, belowId_),

			&obj->belowId_, sizeof(TGMB_ObjectId));
		if (tgmbPtr.handle!=aboveTgmbPtr.handle)
			MemHandleUnlock(aboveTgmbPtr.handle);
	}
	if (belowTgmbPtr.offset!=(Word)TGMB_INVALID) {
		if (tgmbPtr.handle!=belowTgmbPtr.handle)
			other = MemHandleLock(belowTgmbPtr.handle);
		else
			other = ptr;
		DmWrite(other, belowTgmbPtr.offset + OFFSET(CanvObj, aboveId_),
			&obj->aboveId_, sizeof(TGMB_ObjectId));
		if (tgmbPtr.handle!=belowTgmbPtr.handle)
			MemHandleUnlock(belowTgmbPtr.handle);
	}

	if (obj->aboveId_==TGMB_INVALID) topmostId_ = obj->belowId_;
	if (obj->belowId_==TGMB_INVALID) bottommostId_ = obj->aboveId_;

	// figure out the part of the screen to refresh
	// do this *before* deleting the object, otherwise GetBoundingBox()
	// could get confused
	obj->GetBoundingBox(bbox, book_);

	// delete this object
	MarkAsDeleted(ptr, tgmbPtr.offset);
	ht_objects_.Delete(&id);

	// FIXME: do an optimization where we check for a set of contiguous
	// deleted records at the end of the page and just move all pointers
	// back accordingly
	MemHandleUnlock(tgmbPtr.handle);
	return true;
}


Boolean
TGMB_Page::GrowLockedObject(VoidPtr &ptr, TG_Pointer &tgmbPtr, Word newSize)
{
	CanvObj *obj = (CanvObj*) (((BytePtr)ptr) + tgmbPtr.offset);
	Word type = obj->type_;
	Word size = obj->size_;
	TGMB_ObjectId id = obj->thisId_;
	VoidPtr newPtr;
	TG_Pointer newTgmbPtr;
	MarkAsDeleted(ptr, tgmbPtr.offset);
	MemHandleUnlock(tgmbPtr.handle);

	newPtr = (BytePtr) CreateAndLockObject(id, newSize, type,
					       &newTgmbPtr, 0);
	if (!newPtr) return false;
	if (tgmbPtr.handle != newTgmbPtr.handle) {
		// these are on two different chunks
		// we should lock the original guy as well
		ptr = (BytePtr)MemHandleLock(tgmbPtr.handle);
	} else ptr = newPtr;

	/* write the original below and above id's */
	DmWrite(newPtr, newTgmbPtr.offset+OFFSET(CanvObj, belowId_),
		BytePtr(obj) + OFFSET(CanvObj, belowId_),
		2*sizeof(TGMB_ObjectId));

	/* write the rest of the object */
	DmWrite(newPtr, newTgmbPtr.offset+sizeof(CanvObj),
		BytePtr(obj) + sizeof(CanvObj), size - sizeof(CanvObj));
	if (tgmbPtr.handle != newTgmbPtr.handle)
		MemHandleUnlock(tgmbPtr.handle);

	/* update the hash table with the new location */
	ht_objects_.Update(&id, &newTgmbPtr);
	ptr = newPtr;
	tgmbPtr = newTgmbPtr;
	return true;
}


Boolean
TGMB_Page::ConfigTextObject(TGMB_ObjectId id, const char *text)
{
	TG_Pointer tgmbPtr;
	BytePtr ptr;
	Word strLen, textLen, oldLen, newSize;
	AbsRectType bbox1, bbox2;
	Boolean needRefresh=false;

	if (!ht_objects_.Find(&id, &tgmbPtr)) {FATAL(id); return false;}
	ptr = (BytePtr) MemHandleLock(tgmbPtr.handle);
	TextObj *textObj = (TextObj*) (ptr + tgmbPtr.offset);

	strLen = StrLen(text)+1;
	textLen= textObj->size_ - sizeof(TextObj);
	oldLen = StrLen(textObj->text_);
	if (strLen <= oldLen || StrNCompare(textObj->text_, text, oldLen)!=0) {
		needRefresh = true;
		textObj->GetBoundingBox(&bbox1, book_);
	}

	if (textLen >= strLen) {
		DmWrite(ptr, tgmbPtr.offset + OFFSET(TextObj, text_),
			(VoidPtr)text, strLen);
	} else {
		// we need to make more space!

		while (textLen < strLen) textLen *= 2;
		newSize = sizeof(TextObj) + textLen;

#if 0
		MarkAsDeleted(ptr, tgmbPtr.offset);
		/*Word word = CanvObjType_Deleted;
		DmWrite(ptr, tgmbPtr.offset + OFFSET(CanvObj, type),
			&word, sizeof(Word));*/
		MemHandleUnlock(tgmbPtr.handle);

		newPtr = (BytePtr) CreateAndLockObject(id, newSize,
						       CanvObjType_Text,
						       &newTgmbPtr, 0);
		if (tgmbPtr.handle != newTgmbPtr.handle) {
			// these are on two different chunks
			// we should lock the original guy as well
			ptr = (BytePtr)MemHandleLock(tgmbPtr.handle);
		} else ptr = newPtr;

		textObj = (TextObj*) (ptr + tgmbPtr.offset);
		/* write the original below and above id's */
		DmWrite(newPtr, newTgmbPtr.offset+OFFSET(CanvObj, belowId_),
			BytePtr(textObj) + OFFSET(CanvObj, belowId_),
			2*sizeof(TGMB_ObjectId));
		/* write the rest of the TextObj */
		DmWrite(newPtr, newTgmbPtr.offset+sizeof(CanvObj),
			BytePtr(textObj) + sizeof(CanvObj),
			sizeof(TextObj)-sizeof(CanvObj));
		/* write the new string */
		DmWrite(newPtr, newTgmbPtr.offset + sizeof(TextObj),
			(VoidPtr)text, strLen);
		if (tgmbPtr.handle != newTgmbPtr.handle)
			MemHandleUnlock(tgmbPtr.handle);

		/* update the hash table with the new location */
		ht_objects_.Update(&id, &newTgmbPtr);
		ptr = newPtr;
		tgmbPtr = newTgmbPtr;
#endif
		VoidPtr tmp=ptr;
		if (!GrowLockedObject(tmp, tgmbPtr, newSize))
			return false;
		ptr = (BytePtr)tmp;
		/* write the new string */
		DmWrite(ptr, tgmbPtr.offset + sizeof(TextObj),
			(VoidPtr)text, strLen);
	}

	if (IsVisible()) {
		if (needRefresh) textObj->GetBoundingBox(&bbox2, book_);
		else DrawLockedObject((CanvObj*)(ptr+tgmbPtr.offset),
				      book_->OnscreenWindow());
		MemHandleUnlock(tgmbPtr.handle);
		if (needRefresh) {
			Union(&bbox1, &bbox2);
			if (Intersect(&bbox1, book_->ScreenRect()))
				Refresh(&bbox1);
		}
	} else MemHandleUnlock(tgmbPtr.handle);
	return true;
}


void
TGMB_Page::ResendImages()
{
	TGMB_ObjectId id=bottommostId_;
	TG_Pointer tgmbPtr;
	VoidHand h=(VoidHand)TGMB_INVALID;
	CanvObj *canvObj;
	VoidPtr ptr=NULL;
	Word zoom=book_->Zoom();

	// loop thru all the objects on the page
	while (id!=TGMB_INVALID) {
		if (!ht_objects_.Find(&id, &tgmbPtr)) {FATAL(id); break;}
		if (h!=tgmbPtr.handle) {
			if (h!=(VoidHand)TGMB_INVALID) MemHandleUnlock(h);
			ptr = MemHandleLock(tgmbPtr.handle);
			h = tgmbPtr.handle;
		}
		canvObj =((CanvObj*) (ptr + tgmbPtr.offset));
		if (canvObj->type_==CanvObjType_Image &&
		    ((ImageObj*)canvObj)->zoom_!=zoom) {
			APP->Session()->ResendImage(Id(), id);
		}
		id = canvObj->aboveId_;
	}
	if (h!=(VoidHand)TGMB_INVALID) MemHandleUnlock(h);
}


Boolean
LineObj::Insert(TGMB_Page *page, TGMB_FullObjectDescr *descr)
{
	Word npoints = descr->GetNPoints();
	TG_Pointer tgmbPtr;
	VoidPtr ptr;

	color_ = descr->GetColor() >> 8;
	thickness_ = descr->GetWidth();

	ptr = page->CreateAndLockObject(descr->GetObjectId(),
					sizeof(LineObj) + npoints*
					sizeof(PointType),
					CanvObjType_Line, &tgmbPtr);
	if (!ptr) return NULL;
	DmWrite(ptr, tgmbPtr.offset + sizeof(CanvObj),
		BytePtr(this)+sizeof(CanvObj),sizeof(LineObj)-sizeof(CanvObj));
	DmWrite(ptr, tgmbPtr.offset + sizeof(LineObj), descr->GetPoints(),
		npoints*sizeof(PointType));

	if (page->IsVisible())
		page->DrawLockedObject((CanvObj*)(ptr+tgmbPtr.offset),
				       page->Book()->OnscreenWindow());
	MemHandleUnlock(tgmbPtr.handle);
	return true;
}


void
LineObj::Draw(TGMB_Page *page, WindowType *window)
{
	PointType *points, pnt1, pnt2;
	Word nPoints, dummy;
	Win2_Context cxt;

	Word zoom = page->Book()->Zoom();
	cxt.window = window;
	cxt.width = (Word(thickness_) * zoom)/100;
	if ((Word(thickness_) * zoom) % 100) cxt.width++;
	cxt.outlineClr = color_;

	GetPoints(dummy, nPoints);
	points = points_;
	page->Book()->Page2Screen(points++, &pnt1);
	page->Book()->Page2Screen(points++, &pnt2);
	Win2_DrawLine(&cxt, pnt1.x, pnt1.y, pnt2.x, pnt2.y);
	pnt1 = pnt2;

	for (Word i=2; i<nPoints; i++) {
		page->Book()->Page2Screen(points++, &pnt2);
		if (pnt2.x!=pnt1.x || pnt2.y!=pnt1.y) {
			Win2_DrawLine(&cxt, pnt1.x, pnt1.y, pnt2.x, pnt2.y);
			pnt1 = pnt2;
		}
	}
}


Boolean
ArrowObj::Insert(TGMB_Page *page, TGMB_FullObjectDescr *descr)
{
	Word npoints = descr->GetNPoints();
	TG_Pointer tgmbPtr;
	VoidPtr ptr;

	// the line color is actually in the fill descr (!!!)
	color_ = descr->GetColor() >> 8;
	thickness_ = descr->GetWidth();
	arrowType_ = descr->GetArrowType();
	MemMove(vars_, descr->GetArrowVars(), sizeof(vars_));

	ptr = page->CreateAndLockObject(descr->GetObjectId(),
					sizeof(ArrowObj) + npoints*
					sizeof(PointType),
					CanvObjType_Arrow, &tgmbPtr);
	if (!ptr) return NULL;
	DmWrite(ptr, tgmbPtr.offset + sizeof(CanvObj),
		BytePtr(this)+sizeof(CanvObj),
		sizeof(ArrowObj)-sizeof(CanvObj));
	DmWrite(ptr, tgmbPtr.offset + sizeof(ArrowObj), descr->GetPoints(),
		npoints*sizeof(PointType));

	if (page->IsVisible())
		page->DrawLockedObject((CanvObj*)(ptr+tgmbPtr.offset),
				       page->Book()->OnscreenWindow());
	MemHandleUnlock(tgmbPtr.handle);
	return true;
}


void
ArrowObj::Draw(TGMB_Page *page, WindowType *window)
{
	PointType *points, pnt1, pnt2;
	Word nPoints, dummy;
	Win2_Context cxt;

	Word zoom = page->Book()->Zoom();
	cxt.window = window;
	cxt.width = (Word(thickness_) * zoom)/100;
	if ((Word(thickness_) * zoom) % 100) cxt.width++;
	cxt.outlineClr = color_;

	// draw the line
	GetPoints(dummy, nPoints);
	points = points_;
	page->Book()->Page2Screen(points++, &pnt1);
	for (Word i=1; i<nPoints; i++) {
		page->Book()->Page2Screen(points++, &pnt2);
		Win2_DrawLine(&cxt, pnt1.x, pnt1.y, pnt2.x, pnt2.y);
		pnt1 = pnt2;
	}

	// draw the arrowheads
	/*
	 * First arrow: (x1+a1, y1+b1) - (x1,y1) - (x1+a2,y1+b2)
	 * Last arrow:  (x2-a1, y2-b1) - (x2,y2) - (x2-a2,y2-b2)
	 */
	Boolean head=false, tail=false;
	switch(arrowType_) {
	case ArrowFirst: head = true; break;
	case ArrowLast: tail = true; break;
	case ArrowBoth: head = tail = true; break;
	default: break;
	}

	if (head) {
		page->Book()->Page2Screen(points_, &pnt1);
		Win2_DrawLine(&cxt, pnt1.x+vars_[0].x,
			      pnt1.y+vars_[0].y, pnt1.x, pnt1.y);
		Win2_DrawLine(&cxt, pnt1.x, pnt1.y, pnt1.x+vars_[1].x,
			      pnt1.y+vars_[1].y);
	}

	if (tail) {
		page->Book()->Page2Screen(points_+nPoints-1, &pnt1);
		Win2_DrawLine(&cxt, pnt1.x-vars_[0].x,
			      pnt1.y-vars_[0].y, pnt1.x, pnt1.y);
		Win2_DrawLine(&cxt, pnt1.x, pnt1.y, pnt1.x-vars_[1].x,
			      pnt1.y-vars_[1].y);
	}
}


Boolean
RectObj::Insert(TGMB_Page *page, TGMB_FullObjectDescr *descr)
{
	Word npoints = descr->GetNPoints();
	TG_Pointer tgmbPtr;
	VoidPtr ptr;

	outline_ = descr->GetColor() >> 8;
	if (descr->GetFillPtr()) {
		shouldFill_ = 1;
		fill_ = (*(descr->GetFillPtr())) >> 8;
	} else {
		shouldFill_ = 0;
		fill_ = 0;
	}

	thickness_ = descr->GetWidth();

	ptr = page->CreateAndLockObject(descr->GetObjectId(),
					sizeof(RectObj) + npoints*
					sizeof(PointType),
					CanvObjType_Rect, &tgmbPtr);
	if (!ptr) return NULL;
	DmWrite(ptr, tgmbPtr.offset + sizeof(CanvObj),
		BytePtr(this)+sizeof(CanvObj),sizeof(RectObj)-sizeof(CanvObj));
	DmWrite(ptr, tgmbPtr.offset + sizeof(RectObj), descr->GetPoints(),
		npoints*sizeof(PointType));

	if (page->IsVisible())
		page->DrawLockedObject((CanvObj*)(ptr+tgmbPtr.offset),
				       page->Book()->OnscreenWindow());
	MemHandleUnlock(tgmbPtr.handle);
	return true;
}


void
RectObj::Draw(TGMB_Page *page, WindowType *window)
{
	PointType pnt1, pnt2;
	Win2_Context cxt;

	Word zoom = page->Book()->Zoom();
	cxt.window = window;
	cxt.width = (Word(thickness_) * zoom)/100;
	if ((Word(thickness_) * zoom) % 100) cxt.width++;
	cxt.outlineClr = outline_;
	cxt.fillClr = fill_;
	cxt.fill = shouldFill_;

	// draw the rectangle
	page->Book()->Page2Screen(&points_[0], &pnt1);
	page->Book()->Page2Screen(&points_[1], &pnt2);
	Win2_DrawRectangle(&cxt, pnt1.x, pnt1.y, pnt2.x, pnt2.y);
}


Boolean
OvalObj::Insert(TGMB_Page *page, TGMB_FullObjectDescr *descr)
{
	Word npoints = descr->GetNPoints();
	TG_Pointer tgmbPtr;
	VoidPtr ptr;

	outline_ = descr->GetColor() >> 8;
	if (descr->GetFillPtr()) {
		shouldFill_ = 1;
		fill_ = (*(descr->GetFillPtr())) >> 8;
	} else {
		shouldFill_ = 0;
		fill_ = 0;
	}

	thickness_ = descr->GetWidth();


	ptr = page->CreateAndLockObject(descr->GetObjectId(),
					sizeof(OvalObj) + npoints*
					sizeof(PointType),
					CanvObjType_Oval, &tgmbPtr);
	if (!ptr) return NULL;
	DmWrite(ptr, tgmbPtr.offset + sizeof(CanvObj),
		BytePtr(this)+sizeof(CanvObj),
		sizeof(OvalObj)-sizeof(CanvObj));
	DmWrite(ptr, tgmbPtr.offset + sizeof(OvalObj), descr->GetPoints(),
		npoints*sizeof(PointType));

	if (page->IsVisible())
		page->DrawLockedObject((CanvObj*)(ptr+tgmbPtr.offset),
				       page->Book()->OnscreenWindow());
	MemHandleUnlock(tgmbPtr.handle);
	return true;
}


void
OvalObj::Draw(TGMB_Page *page, WindowType *window)
{
	PointType pnt1, pnt2;
	Win2_Context cxt;

	Word zoom = page->Book()->Zoom();
	cxt.window = window;
	cxt.width = (Word(thickness_) * zoom)/100;
	if ((Word(thickness_) * zoom) % 100) cxt.width++;
	cxt.outlineClr = outline_;
	cxt.fillClr = fill_;
	cxt.fill = shouldFill_;

	// draw the oval
	page->Book()->Page2Screen(&points_[0], &pnt1);
	page->Book()->Page2Screen(&points_[1], &pnt2);
	Win2_DrawOval(&cxt, pnt1.x, pnt1.y, pnt2.x, pnt2.y);
}


Boolean
TextObj::Insert(TGMB_Page *page, TGMB_FullObjectDescr *descr)
{
	TG_Pointer tgmbPtr;
	VoidPtr ptr;

	color_ = descr->GetColor() >> 8;
	font_ = 0; // FIXME: ignore the font for now
	MemMove(&coord_, descr->GetPoints(), sizeof(PointType));

#define TEXT_STRING_CHUNK 64
	Word textLen = TEXT_STRING_CHUNK, strLen = StrLen(descr->GetText())+1;
	while (textLen < strLen) textLen *= 2;

	ptr = page->CreateAndLockObject(descr->GetObjectId(),
					sizeof(TextObj) + textLen,
					CanvObjType_Text, &tgmbPtr);
	if (!ptr) return NULL;
	DmWrite(ptr, tgmbPtr.offset + sizeof(CanvObj),
		BytePtr(this)+sizeof(CanvObj),
		sizeof(TextObj)-sizeof(CanvObj));
	DmWrite(ptr, tgmbPtr.offset + sizeof(TextObj),descr->GetText(),strLen);

	if (page->IsVisible())
		page->DrawLockedObject((CanvObj*)(ptr+tgmbPtr.offset),
				       page->Book()->OnscreenWindow());
	MemHandleUnlock(tgmbPtr.handle);
	return true;
}


void
TextObj::Draw(TGMB_Page *page, WindowType *window)
{
	PointType pnt;
	Win2_Context cxt;

	cxt.window = window;
	cxt.outlineClr = color_;

	// draw the text
	page->Book()->Page2Screen(&coord_, &pnt);
	Win2_DrawText(&cxt, pnt.x, pnt.y, text_);
}


Boolean
ImageObj::Insert(TGMB_Page *page, TGMB_FullObjectDescr *descr)
{
	TG_Pointer tgmbPtr;
	VoidPtr ptr;

	// we could get a new object for the same ID, because of
	// the ResendImage functionality; so first check if we
	// already have an image for this ID
	ptr = page->FindAndLockObject(descr->GetObjectId(), &tgmbPtr);
	if (ptr) {
		// we found an object
		ImageObj *obj = (ImageObj*) (((BytePtr)ptr)+tgmbPtr.offset);
		if (obj->type_ != CanvObjType_Image) {
			TGMB_Application::ErrorDialog("Expected image object; "
						      "found object of type %d"
						      " (ID %lu)", obj->type_,
						      descr->GetObjectId());
			MemHandleUnlock(tgmbPtr.handle);
			return false;
		}

		if (obj->zoom_==descr->GetImageZoom() ||
			obj->zoom_==page->Book()->Zoom()) {
			// this object already has the zoom level we want
			// ignore the new object
			MemHandleUnlock(tgmbPtr.handle);
			return true;
		}

		if (obj->size_ < sizeof(ImageObj)+descr->GetImageSize()) {
			// we need to grow this object
			if (!page->GrowLockedObject(ptr, tgmbPtr,
						    sizeof(ImageObj)+
						    descr->GetImageSize()))
				return false;
		}
		// don't use the 'obj' variable below, coz the value of
		// ptr might have changed
		MemMove(&coord_,
			&((ImageObj*)(((BytePtr)ptr)+tgmbPtr.offset))->coord_,
			sizeof(PointType));
	} else {
		// this is a brand new object
		ptr = page->CreateAndLockObject(descr->GetObjectId(),
						sizeof(ImageObj)+
						descr->GetImageSize(),
						CanvObjType_Image, &tgmbPtr);
		if (!ptr) return false;
		MemMove(&coord_, descr->GetPoints(), sizeof(PointType));
	}

	// now write the new object
	zoom_ = descr->GetImageZoom();
	bits_ = descr->GetImageBits();

	DmWrite(ptr, tgmbPtr.offset + sizeof(CanvObj),
		BytePtr(this)+sizeof(CanvObj),
		sizeof(ImageObj)-sizeof(CanvObj));
	DmWrite(ptr, tgmbPtr.offset + sizeof(ImageObj),
		descr->GetImage(), descr->GetImageSize());

	if (page->IsVisible()) {
		if (page->TopmostId()==descr->GetObjectId()) {
			page->DrawLockedObject((CanvObj*)(ptr+tgmbPtr.offset),
					       page->Book()->OnscreenWindow());
			MemHandleUnlock(tgmbPtr.handle);
		} else {
			AbsRectType bbox;
			((CanvObj*)(ptr+tgmbPtr.offset))->
				GetBoundingBox(&bbox, page->Book());
			MemHandleUnlock(tgmbPtr.handle);

			if (Intersect(&bbox, page->Book()->ScreenRect()))
				page->Refresh(&bbox);
		}
		// if the zoom level on this image is incorrect
		// request a ResendImage from the RMX
		if (zoom_!=page->Book()->Zoom())
			APP->Session()->ResendImage(page->Id(),
						    descr->GetObjectId());
	}
	return true;
}


void
ImageObj::Draw(TGMB_Page *page, WindowType *window)
{
	PointType pnt;
	Win2_Context cxt;

	page->Book()->Page2Screen(&coord_, &pnt);

	// FIXME: always assume 2bit
	Word zoom = page->Book()->Zoom();
	if (zoom_ != zoom || bits_!=2) {
		Word x2, y2;
		BitmapPtr bmp = (BitmapPtr)image_;
		Win2_Context cxt;
		x2 = pnt.x + (DWord(bmp->width/2)*zoom)/zoom_ - 1;
		y2 = pnt.y + (DWord(bmp->height )*zoom)/zoom_ - 1;
		cxt.window = window;
		cxt.fillClr= 0x40;
		Win2_GrayoutRectangle(&cxt, pnt.x, pnt.y, x2, y2);
		return;
	}

	// draw the image
	cxt.window = window;
	Win2_DrawBitmap(&cxt, pnt.x, pnt.y, (BitmapPtr)image_);
}
