# http-server.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 1998-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
#  @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/net/http-server.tcl,v 1.7 2002/02/03 04:28:05 lim Exp $


import TCP MTrace

#
# The HTTP_Server class is an <b>abstract base class</b> that handles
# general HTTP requests.  Usage:<p>
# set myServer [new TCP/Server]<br>
# $myServer open [port] TCP/HTTP_Server/[child class]
#
Class TCP/HTTP_Server -superclass TCP

#
# TCP/HTTP_Server constructor
#
TCP/HTTP_Server public init { http_server } {
	$self next
	$self set http_server_ $http_server
	$self init_vars
}


#
# The init_vars method initializes class variables; this method
# should be invoked prior to each HTTP request received.
#
TCP/HTTP_Server private init_vars { } {

	$self instvar headers_done_ num_data_bytes_ headers_ data_

	# Set the headers_done_ boolean to false indicating that
	# the headers have not yet been completely read.
	set headers_done_ 0

	# Initialize num_data_bytes_ to 0.
	set num_data_bytes_ 0

	# Initialize / reset the headers_ and data_ variables.
	if [info exists headers_] { unset headers_ }
	set data_ ""
}


#
# The readable method reads input from a channel, and invokes
# either TCP::readable to handle header information or
# HTTP_Server::recv to handle data. When the entire request has
# been received, this method invokes the handle_request method.
#
TCP/HTTP_Server private readable { } {

	$self instvar chan_ headers_done_ num_data_bytes_

	# Configure the socket to binary, non-blocking mode.
	#fconfigure $chan_ -blocking 0
	#fconfigure $chan_ -translation binary

	if { $headers_done_ == 0 } {

		# Call TCP::readable, which calls recv to read in the headers.
		$self next

	} elseif { $num_data_bytes_ > 0 } {

		# Call recv to get the data from the HTTP request.
		set socket [read $chan_ $num_data_bytes_]

		# Check for eof; note that in handling headers, TCP::readable
		# does this check.
		if { [string length $socket] == 0 } {
			if { [eof $chan_] } {
				mtrace trcNet "-> eof reached"
				mtrace trcNet $socket
				$self close
				$self shutdown
			}
			return
		} else {
			$self recv $socket
		}

	} else {
		mtrace trcNet "** Error: readable called with no data to read."
		mtrace trcNet "** closing this socket"
		catch {
			$self close
			$self shutdown
		}
	}
}


#
# The recv method parses the portion of an HTTP request that is
# currently on the socket.
#
TCP/HTTP_Server private recv { sock_data } {

	$self instvar num_data_bytes_ headers_done_ headers_ data_

	# Handle the next line of headers if they have not all
	# been read yet.
	if { $headers_done_ == 0 } {
		if { [string compare "" [string trim $sock_data] ] == 0 } {
			# If this is an empty line indicating the end of the
			# headers, set headers_done_ to true and return.
			set headers_done_ 1
			mtrace trcNet "-> End of headers"
		} else {
			# Otherwise, this line contains a normal header;
			# parse the header.
			mtrace trcNet $sock_data
			$self parse_header $sock_data
		}
	} elseif { $num_data_bytes_ > 0 } {
		# Append data received to data_ and update num_bytes_read_.
		set num_bytes_read [string length $sock_data]
		set num_data_bytes_ [expr $num_data_bytes_ - $num_bytes_read]
		append data_ $sock_data
		mtrace trcNet "-> Bytes read / bytes left: $num_bytes_read /\
				$num_data_bytes_"
	} else {
		mtrace "-> Nothing received"
	}

	# If the headers have been read and there is no more data to
	# read, then call handle_request.
	if { [expr $headers_done_ == 1 && $num_data_bytes_ == 0] } {
		mtrace trcNet "-> Calling handle_request"
		$self instvar http_server_

		set delete_sock 1
		if [info exists headers_] {
			# we must have at least the first header line that
			# contains the method, url, and version fields
			# note that we are sending the headers_ array by
			# reference
			set delete_sock [$http_server_ handle_request $self \
					headers_ $data_]
			if { $delete_sock == {} } { set delete_sock 1 }
		}

		if $delete_sock {
			$self init_vars

			# since we are assuming HTTP/1.0, we will just close the
			# socket here
			catch {$self close}
			catch {$self shutdown}
			delete $self
		}
	}
}


#
# The parse_header method extracts the title and value of a header,
# handles special information headers, and appends the title/value
# pair to the list, headers_.
#
TCP/HTTP_Server private parse_header { header } {
	$self instvar headers_

	# Check if header contains the request line.  This line must
	# be handled specially since it does not follow the standard
	# format, <header-title>: <header-value>.
	if { ![info exists headers_] } {
		# this is the first header line
		# it is of the form "METHOD URL HTTP/VER"
		set header [split $header]
		set headers_(method) [lindex $header 0]
		set headers_(url) [lindex $header 1]
		set headers_(version) [lindex $header 2]
	} else {
		# this line is of the form "<header-title>: <header-value>"
		set colon_pos [string first ":" $header]

		# Extract the header title, remove whitespace, if any,
		# and convert the string to lower-case.
		set title [string range $header 0 [expr $colon_pos - 1] ]
		set title [string tolower [string trim $title] ]

		# Extract the header value and take out whitespace at the ends.
		set value [string range $header [expr $colon_pos + 1] end]
		set value [string trim $value]

		# Handle special information headers.
		if { $title  == "content-length" } {
			# If this line contains the Content-Length header,
			# set num_data_bytes_
			$self instvar num_data_bytes_
			set num_data_bytes_ $value
			mtrace trcNet "-> Number of data bytes expected:\
					$num_data_bytes_"
		}

		# Append the title/value pair to the list, headers_.
		mtrace trcNet "-> Header title: $title"
		mtrace trcNet "-> Header value: $value"
		set headers_($title) $value
	}
}

# -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

import Class

Class HTTP_Server -configuration {
	server_port 4444
}



HTTP_Server public init { } {

	$self instvar status_table_

	# Create the status table.
	set status_table_(200) "OK"
	set status_table_(204) "No Content"
	set status_table_(400) "Bad Request"
	set status_table_(404) "Not found"
	set status_table_(500) "Internal Server Error"
}


HTTP_Server public destroy { } {
	$self close
	$self next
}


HTTP_Server public open { port } {
	$self instvar server_
	set server_ [new TCP/Server]
	$server_ open $port "$self create_channel"
	$self set port_ $port
	puts stderr "opened server at port '$port'"
}


HTTP_Server public close { } {
	$self instvar server_
	if [info exists server_] {
		delete $server_
		unset server_
	}
}


HTTP_Server public port { } {
	return [$self set port_]
}


HTTP_Server private create_channel { chan } {
	set socket [new TCP/HTTP_Server $self]
	$socket open $chan

	# set the socket to binary mode
	# it will automatically get set to non-blocking inside tcp.tcl
	$socket set_binary
	return $socket
}


#
# The handle_request method is a pure virtual function; it is
# called when an entire HTTP request has been received by
# readable. Thus, this method can assume that the header_ and
# data_ variables have been set accordingly prior to its
# invokation.  Note that the implementor of this method has
# the responsibility of closing the socket.
#
HTTP_Server public handle_request { socket headers data } {
	mtrace trcNet "Error: HTTP_Server is an abstract base class."
	error "HTTP_Server is an abstract base class."
	$self close
	$self shutdown
}



# #
# # sets members of an array 'result':
# #   result(method)   GET/PUT etc
# #   result(url)      the url
# #   all other headers are stored as elements of this array
# HTTP_Server public headers_to_array { headers result_var } {
#	upvar $result_var result
#	array set result [lrange $headers 2 end]
#	set result(method) [lindex $headers 0]
#	set result(url) [lindex $headers 1]
# }


# HTTP_Server private extract_url { headers } {
#	# Convert headers into an associative array and extract the
#	# GET header/value pair.
#	array set hdr_array $headers
#	set get_pair [array get hdr_array "get"]
#	set url ""
#
#	# The SDP objects only care about GET requests with certain
#	# "magic" URLs.
#	if { [llength $get_pair] > 0 } {
#		# Extract the URL from the GET request.
#		set url [lindex $get_pair 1]
#		mtrace trcNet "-> URL received: $url"
#	}
#	return $url
# }


#
# The send_html method sends a OK reply with text/html data.
#
HTTP_Server public send_html { socket data } {
	mtrace trcNet "-> Sending generic reply"
	set reply(data) $data
	set reply(headers) [list content-type text/html]
	set reply(status) 200
	$self send_reply $socket reply
}


#
# The send_reply method sends a general reply with the given error
# code and text/html data.
#
HTTP_Server public send_reply { socket reply_var } {
	mtrace trcNet "-> Sending reply"
	# This whole section is a hack since otcl cannot handle binary data.
	upvar $reply_var reply
	set status_msg [$self get_status_msg $reply(status)]
	set status "HTTP/1.0 $reply(status) $status_msg"
	set h $reply(headers)
	lappend h content-length [string length $reply(data)]
	catch {
		$socket send [$self construct_headers $status $h]
		set chan [$socket channel]
		puts -nonewline $chan $reply(data)
		flush $chan
	}
}


#
# The construct_headers method takes a list of header pairs and
# status to construct the headers to be returned by the server.
#
HTTP_Server private construct_headers { status header_list } {

	set headers $status
	append headers "\r\n"

	foreach {title value} $header_list {
		append headers "$title: $value\r\n"
	}

	append headers "\r\n"
	mtrace trcNet "-> Constructed headers:"
	mtrace trcNet $headers
	return $headers
}


HTTP_Server private get_status_msg { status_code } {
	$self instvar status_table_
	if [info exists status_table_($status_code)] {
		return $status_table_($status_code)
	} else {
		return "Unknown error"
	}
}

