/*
 * transcoder.h --
 *
 *      Transcoder header file
 *
 * Copyright (c) 1996-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef vgw_transcoder_h
#define vgw_transcoder_h

#include <sys/types.h>
#ifndef WIN32
#include <sys/time.h>
#endif
#include "tclcl.h"
#include "config.h"
#include "source.h"
#include "crdef.h"
#include "module.h"

class Transcoder;

/* FIXME should not be derived from packetHandler*/
class Transcoder : public PacketHandler, public MultiInputManager {
 public:
	virtual int command(int argc, const char*const* argv);
	inline double gettimeofday() {
		timeval tv;
		::gettimeofday(&tv, 0);
		return (tv.tv_sec * 1e6 + tv.tv_usec);
	}
	virtual void recv(pktbuf*) {
		fprintf(stderr, "Transcoder::recv called!\n");
		abort();
	}
	virtual void recv_data(pktbuf* pb) = 0;
	virtual void recv_ctrl(pktbuf* pb);
 protected:
	Transcoder(int hdrlen);
	void bps(int b);

	EncoderModule* encoder_;

	DataInputHandler* dh_;
	ControlInputHandler* ch_;

	Source* source_;

	u_int32_t opkts_;	/* output packet */
	u_int32_t obytes_;	/* output bytes */
	u_int32_t ofrms_;	/* output frames */

	int bps_;		/* output bit rate */

	double lastfc_;		/* last frame clock */
	double fc_;		/* frame clock */
	int forward_;		/* pass-thru */
	int txonly_; 		/* transcode only with no rate control */
	virtual int build_app(rtcphdr* rr, Source& ls, int sr);
private:
	void parse_bye(rtcphdr* rh, int flags, u_char* ep, Source* ps);
	void parse_sr(rtcphdr* rh, int flags, u_char* ep, Source* ps,
		      int layer);
	void parse_rr(rtcphdr* rh, int flags, u_char* ep, Source* ps,
		      int layer);
	void parse_rr_records(rtcp_rr* rr, int cnt, const u_char* ep,
			      Source* ps, int layer);
};

class VideoTranscoder : public Transcoder, public ConditionalReplenisher {
 public:
	virtual int command(int argc, const char*const* argv);
	inline int width() const { return (inw_); }
	inline int height() const { return (inh_); }
	inline int decimation() const { return (decimation_); }
	inline int have_frame() const { return (have_frame_); }
 protected:
	inline void color(int) {
#ifdef notdef
		if (encoder_ != 0)
			encoder_->color(v);
		send_all();
#endif
	}
	VideoTranscoder(int hdrlen);
        ~VideoTranscoder();
	void pixel_crinit(int w, int h);
	void blk_to_mb_cr();
	int decimation_;	/* 411 or 422 */
	int inw_;		/* input image width */
	int inh_;		/* input image height */
	int outw_;		/* output image width */
	int outh_;		/* output image height */

	int mark_;
	int have_frame_;

	u_char* mb_crvec_;

	virtual int build_app(rtcphdr* rr, Source& ls, int sr);
	u_char* build_rtcp_thumbnail(rtcphdr* rr, Source& ls);
	int rtcp_thumbnail_;
};

#define PLAYO_FILTER 5
#define VAR_FILTER 5
#define VAR_MULT 2

class AudioTranscoder : public Transcoder {
    protected:
	AudioTranscoder(int hdrlen);
    public:
	virtual int command(int argc, const char*const* argv);
    protected:
	int adapt(const rtphdr*, u_int32_t sclock);
	inline u_int32_t playout() const { return (playout_ >> PLAYO_FILTER); }
	/*
	 * Playback point estimation state.
	 */
	int32_t var_;		// variance in this host's interarrival time
	u_int32_t hostoffset_;	// timestamp offset this host's clock
	u_int32_t predicted_drop_;	// ts of last drop + blocksize
	u_int32_t playout_;	// playout delay (in media units)
	u_int32_t lastrecv_;	// local time (in media time) of last recv
	int lecture_mode_;	/*FIXME*/
	int framesize_;		/*FIXME*/
	int block_size_;
	int maxdel_;
};

class NullTranscoder : public Transcoder {
 public:
	NullTranscoder() : Transcoder(0) {}
	virtual void recv_data(pktbuf*);
	virtual void recv_ctrl(pktbuf*);
};

#endif /* vgw_transcoder_h */

