/*
 * lts.h --
 *
 *      Logical Time System header file
 *
 * Copyright (c) 1997-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * @(#) $Header: /usr/mash/src/repository/mash/mash-1/archive/lts.h,v 1.9 2002/02/03 03:09:26 lim Exp $
 */


#ifndef MASH_LTS_H
#define MASH_LTS_H


#include <tclcl.h>
#include <timer.h>
#include <misc/all-types.h>
#include <misc/linkedlist.h>

#ifndef WIN32
#include <sys/time.h>
#else
#include <time.h>
#endif

/*
 * different types of LTS triggers
 */
#define LTS_SPEED     0x0001
#define LTS_REFERENCE 0x0002
class LTS;

class LTSTimer : private Timer {
public:
	LTSTimer() : lts_(NULL) { }
	virtual ~LTSTimer() { }

	void sched(timeval logical);
	void sched_system(timeval system);
	void cancel() { Timer::cancel(); }
	LTS* LTS_() { return lts_; }
	void LTS_(LTS *lts) { lts_ = lts; }

protected:
	virtual void timeout()=0;
	LTS *lts_;
};


class LTSTrigger {
public:
	LTSTrigger(u_int32_t triggerType) : lts_(NULL), type_(triggerType) { }
	virtual ~LTSTrigger() { }

	u_int32_t type() { return type_; }
	LTS* LTS_() { return lts_; }
	void LTS_(LTS *lts) { lts_ = lts; }

protected:
	virtual void callback(u_int32_t triggerType)=0;
	LTS *lts_;
	u_int32_t type_;

	friend class LTS;
};


class LTS : public TclObject {
public:
	LTS();
	~LTS() { }

	double Speed() { return speed_; }
	void Speed(double newValue) {
		// must reset the logical to system mapping so there is no
		// sudden jump in logical time when the speed is changed!
		timeval system = NowSystem();
		SetReference(LogicalTimeOf(system), system);
		speed_ = newValue;
	}

	timeval NowLogical() {
		return LogicalTimeOf(NowSystem());
	}

	void    NowLogical(timeval logical) {
		SetReference(logical, NowSystem());
	}

	virtual timeval NowSystem() {
		timeval tv;
		gettimeofday(&tv, NULL);
		return tv;
	}

	void    SetReference(timeval logical, timeval system) {
		refLogical_ = logical;
		refSystem_  = system;
	}

	Bool SetTrigger(LTSTrigger *trigger) {
		return triggers_.InsertAtTail(trigger);
	}
	void UnsetTrigger(LTSTrigger *trigger) {
		triggers_.Remove(trigger);
	}

	void AdjustLogicalReference(double delta);

	int speed(int argc, const char *const *argv);
	int now_logical(int argc, const char *const *argv);
	int now_system (int argc, const char *const *argv);
	int set_reference(int argc, const char *const *argv);

	void ActivateTriggers(u_int32_t triggerType);
private:
	timeval LogicalTimeOf(timeval system);
	timeval SystemTimeOf (timeval logical);

	double  speed_;
	timeval refLogical_, refSystem_;

	List<LTSTrigger> triggers_;
	friend class LTSTimer;
};



#endif /* MASH_LTS_H */

