# matrix-lib.tcl --
#
#       Library abstractions for controlling video matrix switcher through amxd
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

# these functions must be called from a program with direct access to the
#   amx-lib functions (i.e., on the AMX Daemon server)
#
# this library assumes that the AMX has already been setup with
#   setup-AMX-control

source amx-lib.tcl
source callback-lib.tcl

# these are defined in amx-lib.tcl
#global g_amxDevices g_amxChannels

global g_waitStatus g_matrixString
global g_in1 g_out1 g_in2 g_out2 g_inputs g_outputs

# level 1
set g_in1(mbonePC) 1
set g_in1(sgiPC) 2
set g_in1(frontPC) 3
#set g_in1() 4
#set g_in1() 5
set g_in1(liveboard) 6
set g_in1(laptop) 7
#set g_in1() 8

set g_inputs(1,1) "mbonePC"
set g_inputs(1,2) "sgiPC"
set g_inputs(1,3) "frontPC"
#set g_inputs(1,4)
#set g_inputs(1,5)
set g_inputs(1,6) "liveboard"
set g_inputs(1,7) "laptop"
#set g_inputs(1,8)

set g_out1(projectorRGB) 1
set g_out1(scanConverter) 2
#set g_out1() 3
#set g_out1() 4
#set g_out1() 5
#set g_out1() 6
#set g_out1() 7
#set g_out1() 8

set g_outputs(1,1) "projectorRGB"
set g_outputs(1,2) "scanConverter"
#set g_outputs(1,3)
#set g_outputs(1,4)
#set g_outputs(1,5)
#set g_outputs(1,6)
#set g_outputs(1,7)
#set g_outputs(1,8)

# level 2
set g_in2(rackVCR) 1
set g_in2(frontVCR) 2
set g_in2(speakerCamera) 3
set g_in2(wideCamera) 4
set g_in2(audienceCamera) 5
#set g_in2() 6
set g_in2(scanConverter) 7
set g_in2(elmo) 8

set g_inputs(2,1) "rackVCR"
set g_inputs(2,2) "frontVCR"
set g_inputs(2,3) "speakerCamera"
set g_inputs(2,4) "wideCamera"
set g_inputs(2,5) "audienceCamera"
#set g_inputs(2,6)
set g_inputs(2,7) "scanConverter"
set g_inputs(2,8) "elmo"


#set g_out2() 1
set g_out2(video2) 2
set g_out2(video4) 3
set g_out2(video5) 4
set g_out2(video6) 7
set g_out2(testMonitor) 6
#set g_out2(htsr) 7
set g_out2(projector) 8

#set g_outputs(2,1)
set g_outputs(2,2) "video2"
set g_outputs(2,3) "video4"
set g_outputs(2,4) "video5"
set g_outputs(2,7) "video6"
set g_outputs(2,6) "testMonitor"
#set g_outputs(2,7) "htsr"
set g_outputs(2,8) "projector"


# returns the string name of the input for the specified input level and channel
proc matrix_inputLookup {level channel} {
    global g_inputs
    
    if {[info exists g_inputs($level,$channel)]} {
	return $g_inputs($level,$channel)
    }
    # if we got here, it's not a recognized channel
    return $channel
}

# returns the string name of the output for the specified output level and channel
proc matrix_outputLookup {level channel} {
    global g_outputs
    
    if {[info exists g_outputs($level,$channel)]} {
	return $g_outputs($level,$channel)
    }
    # if we got here, it's not a recognized channel
    return $channel
}

# FIXME - do we need to switch the scan converter?
proc matrix_switchVideoStream {input output {projectorScanConverted 0}} {
    global g_amxDevices g_amxChannels
    global g_in1 g_out1 g_in2 g_out2

#    puts stdout "matrix_switchVideoStream: input=$input output=$output"
    
    switch -exact -- $output {
	video2 -
	video4 -
	video5 -
	video6 -
	realNetworks -
	projector -
	testMonitor {
	    set out2 $g_out2($output)
	}
	default {
	    return -code error "Error: output \"$output\" not supported"
	}
    }
    
    switch -exact -- $input {
	mbonePC -
	sgiPC -
	frontPC -
	liveboard -
	laptop {
	    set in1 $g_in1($input)
	    set out1 $g_out1(scanConverter)
	    set in2 $g_in2(scanConverter)
	    if {$output != "projector"} {
		matrix_rawSwitchVideoStream 1 $in1 $out1
		matrix_rawSwitchVideoStream 2 $in2 $out2
	    } else {
		# output is projector -  check if they want scan converted or direct RGB
		if {$projectorScanConverted} {
		    matrix_rawSwitchVideoStream 1 $in1 $out1
		    matrix_rawSwitchVideoStream 2 $in2 $out2
		    send-AMX-command 10 $g_amxDevices(projector) $g_amxChannels(projector,video1) 
		} else {
		    set out1 $g_out1(projectorRGB)
		    matrix_rawSwitchVideoStream 1 $in1 $out1
		    send-AMX-command 10 $g_amxDevices(projector) $g_amxChannels(projector,data2)
		}
	    }
	}
	rackVCR -
	frontVCR -
	speakerCamera -
	wideCamera -
	audienceCamera -
	scanConverter -
	elmo {
	    set in2 $g_in2($input)
	    matrix_rawSwitchVideoStream 2 $in2 $out2
	    # the projector must be told which input to display
	    if {$output == "projector"} {
		send-AMX-command 10 $g_amxDevices(projector) $g_amxChannels(projector,video1)
	    }
	}
	default {
	    return -code error "Error: input \"$input\" not supported"
	}
    }

    # if we got here, we've switched the video
    set eventInfo [list "matrixSwitch"]
    set eventData [list "$input" "$output"]
    set amxMsg [list $eventInfo $eventData]
    puts stdout "matrix_switchVideoStream, doing callbacks with: $amxMsg"
    callback_doCallbacks $amxMsg
}

proc matrix_rawSwitchVideoStream {level input output} {
    global g_amxDevices
    
    if {($input < 1) || ($input > 8)} {
	return -code error "input must be an integer in the range \[1:8\]"
    }
    if {($output < 1) || ($output > 8)} {
	return -code error "output must be an integer in the range \[1:8\]"
    }
    
    switch -exact -- $level {
	1 -	   
	2 {
	    send-AMX-command 4 $g_amxDevices(matrix) "CL${level}I${input}O${output}T"
	}
	default {
	    return -code error "level must be an integer in the range \[1:2\]"
	}
    }
}

# FIXME - should have a matrix_getOutputs which returns a list of
#    the outputs that the specified input is routed to
#

# returns which input source is routed to the specified output
proc matrix_getInputSource {output} {
    global g_in1 g_out1 g_in1 g_out2
    
    switch -exact -- $output {
	video2 -
	video4 -
	video5 -
	video6 -
	realNetworks -
	projector -
	testMonitor {
	    set retList [matrix_getStatus 2 "output" $g_out2($output)]

	    puts stdout "2nd level returns $retList"
	    
	    if {$retList == "scanConverter"} {
		set retList [matrix_getStatus 1 "output" $g_out1(scanConverter)]
	    }
	}
	projectorRGB {
	    set retList [matrix_getStatus 1 "output" $g_out1($output)]
	}
	default {
	    return -code error "Error: output \"$output\" not supported"
	}
    }

    return $retList
}

proc matrix_getStatus {level mode channel} {
    set rawList [matrix_getRawStatus $level $mode $channel]

    set retList [list]
    if {$mode == "input"} {
	foreach item $rawList {
	    lappend retList [matrix_outputLookup $level $item]
	}
    }
    if {$mode == "output"} {
	foreach item $rawList {
	    lappend retList [matrix_inputLookup $level $item]
	}
    }
    
    return $retList
}

# level is [1,2]
# mode is [input,output] - tells you whether channel is an input or output
# channel is [1,8]
proc matrix_getRawStatus {level mode channel} {
    global g_amxDevices g_waitStatus g_matrixString
    
    if {$mode != "input" && $mode != "output"} {
	return -code error "Error: invalid mode - mode must be input OR output"
    }
    if {($channel < 1) || ($channel > 8)} {
	return -code error "channel must be an integer in the range \[1:8\]"
    }
    switch -exact -- $level {
	1 -
	2 {
	    # do nothing, it's fine
	    set level
	}
	default {
	    return -code error "level must be an integer in the range \[1:2\]"
	}
    }

    set result [send-AMX-command 5 $g_amxDevices(matrix) "RXON"]

    # clear out the matrixString and wait for the stuff that we generate to
    #   show up
    set g_matrixString ""
    
    if {$mode == "input"} {
	set mask "SL${level}I${channel}T"
	set result [send-AMX-command 4 $g_amxDevices(matrix) "SL${level}I${channel}T"]
    } else {
	# mode must be output
	set mask "SL${level}O${channel}T"
	set result [send-AMX-command 4 $g_amxDevices(matrix) "SL${level}O${channel}T"]
    }

    #   sometimes the AMX doesn't wait for the matrix to finish reporting
    #   status before sending the message, so if that's the case, try again
    #   until it works
    set valid 0
    while {!$valid} {
	puts stdout "retrying"
	
	set g_waitStatus 1

	# this will wait until the callback is called - since it's not
	#   multithreaded, it won't do the callback until we vwait here, so
	#   we never have to worry about missing it
	vwait g_waitStatus

	# wait until the matrixString matches the SL?I?T(...)
	set str $g_matrixString
	set pat "$mask"
	append pat {\([0-9 ]*\)}
	set valid [regexp $pat $str match]
    }
    # we have a valid string in match now, so parse it to get the results
    regexp {\([0-9 ]*\)} $match str
    # strip off leading ( and space
    set str [string range $str 2 end]
    # strip off trailing )
    set str [string range $str 0 [expr [string length $str] - 2]]
    # at this point, str is a list of space-separated numbers or empty
    set retList [list]
    set chr [string index $str 0]
    while {$chr != ""} {
	lappend retList $chr
	set str [string range $str 1 end]
	# get rid of whitespace to left
	set str [string trimleft $str]
	set chr [string index $str 0]
    }
    
    # turn off RX so that Change commands won't get reflected
    set result [send-AMX-command 5 $g_amxDevices(matrix) "RXOFF"]

    return $retList
}

proc matrix_callback {retList} {
    global g_waitStatus g_matrixString

#    puts stdout "matrix_callback called"

    set eventInfo [lindex $retList 0]
    # see if it's a remote command or a response from an amx device
    set type [lindex $eventInfo 0]
    set eventData [lindex $retList 1]
    set cmd [lindex $eventData 0]
    set dev [lindex $eventData 1]
    set str [lindex $eventData 2]
    if {$type != "amxResponse"} {
	# we don't care about remote commands here
	return
    }
    if {($cmd != "sendString") || ($dev != "matrix")} {
	return
    }
    append g_matrixString $str
    set len [string length $g_matrixString]
    if {$len > 50} {
	set begin [expr $len - 50]
	set g_matrixString [string range $g_matrixString $begin end]
    }

    #puts stdout "##############g_matrixString is now $g_matrixString"

    # signal the get_status function that new data has arrived
    set g_waitStatus 0
}

# this initializes the library
set result [callback_register matrix_callback]
#puts stdout "registering returns: $result"
callback_enable matrix_callback

