/*
 *  Nextview EPG PI timescale queue
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License Version 2 as
 *  published by the Free Software Foundation. You find a copy of this
 *  license in the file COPYRIGHT in the root directory of this release.
 *
 *  THIS PROGRAM IS DISTRIBUTED IN THE HOPE THAT IT WILL BE USEFUL,
 *  BUT WITHOUT ANY WARRANTY; WITHOUT EVEN THE IMPLIED WARRANTY OF
 *  MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *
 *  Description: see according C source file.
 *
 *  Author: Tom Zoerner
 *
 *  $Id: epgtscqueue.h,v 1.2 2002/02/13 21:03:04 tom Exp tom $
 */

#ifndef __EPGTSCQUEUE_H
#define __EPGTSCQUEUE_H


// ----------------------------------------------------------------------------
// Declaration of PI timescale queue data types
//

// the mode describes on which occasion the data was generated
typedef enum
{
   PI_TSC_MODE_INITIAL,        // data was generated by the AddAll function, i.e. initial fill or AI version change
   PI_TSC_MODE_INCREMENTAL,    // data comes from AddPi, i.e. live acquisition
   PI_TSC_MODE_COUNT
} PI_TSC_MODE;

// these flags describe each element in the timescale buffer
#define PI_TSC_MASK_IS_EXPIRED       0x01
#define PI_TSC_MASK_IS_MISSING       0x02
#define PI_TSC_MASK_IS_LAST          0x04
#define PI_TSC_MASK_IS_DEFECTIVE     0x08
#define PI_TSC_MASK_IS_CUR_VERSION   0x10
#define PI_TSC_MASK_IS_STREAM_1      0x20
#define PI_TSC_MASK_HAS_SHORT_I      0x40
#define PI_TSC_MASK_HAS_LONG_I       0x80

// this element describes a given range on a timescale
// it covers one or more PI (of the same network), if they have equal flags
typedef struct
{
   ushort  startOffMins;       // start time, relative to base time (as offset to save 16 bit)
   ushort  durationMins;       // absolute difference between start and stop time in minutes
   uchar   netwop;             // index in the AI network table of the current provider
   uchar   flags;              // logical OR of flags as defined above
   uchar   blockIdx;           // index in the PI block tange table in the AI block (lowest index in case of concatenation)
   uchar   concatCount;        // number of PI which are covered by this range (required to recover all block indices)
} EPGDB_PI_TSC_ELEM;

// maximum length of the range array; must be appropriate for both the
// initial scale generation (where all PI are scanned) and normal acquisition.
// Initially there usually are 200-500 entries required; during acquisition
// depending on the provider max. 6-11 per second
#define PI_TSC_BUF_LEN_INITIAL            150
#define PI_TSC_BUF_LEN_INCREMENTAL         25

typedef struct EPGDB_PI_TSC_BUF_STRUCT
{
   ushort       provCni;                      // provider the data in the buffer belongs to
   uchar        mode;                         // mode in which the data was created
   bool         locked;                       // locked until pending EPG blocks are processed
   ushort       fillCount;                    // number of valid entries in pi array
   ushort       popIdx;                       // number of 'popped' entries
   time_t       baseTime;                     // base for start and stop time offsets
   struct EPGDB_PI_TSC_BUF_STRUCT  * pNext;   // pointer to next buffer in the queue
   struct EPGDB_PI_TSC_BUF_STRUCT  * pPrev;   // pointer to previous buffer in the queue
   EPGDB_PI_TSC_ELEM  pi[1];                  // array with the actual data; size depends on mode
                                              // array must be last to allow partial transmission
} EPGDB_PI_TSC_BUF;

// macros to calculate sizes of a timescale buffer
// (in network acq mode only the used fraction of the pi array is transmitted)
#define PI_TSC_GET_BUF_COUNT(M)    (((M) == PI_TSC_MODE_INITIAL) ? PI_TSC_BUF_LEN_INITIAL : PI_TSC_BUF_LEN_INCREMENTAL)
#define PI_TSC_GET_BUF_SIZE(C)     ( sizeof(EPGDB_PI_TSC_BUF) + \
                                    (sizeof(EPGDB_PI_TSC_ELEM) * ((C) - 1)))

// this type holds root of the queue
// other modules pass a pointer to this as a handle
typedef struct
{
   EPGDB_PI_TSC_BUF   * pHead;                // pointer to first buffer in the queue (newest data)
   EPGDB_PI_TSC_BUF   * pTail;                // pointer to last buffer in the queue (oldest data)
   uint          writeProvCni;                // provider CNI for additions
   PI_TSC_MODE   writeMode;                   // mode for additions: incremental or initial
   uint          readProvCni;                 // provider CNI for pop
} EPGDB_PI_TSC;


// ----------------------------------------------------------------------------
// Declaration of service interface functions
//
void EpgTscQueue_Init( EPGDB_PI_TSC * tsc );
void EpgTscQueue_Clear( EPGDB_PI_TSC * tsc );

void EpgTscQueue_SetProvCni( EPGDB_PI_TSC * pQueue, uint provCni );
void EpgTscQueue_AddPi( EPGDB_PI_TSC * tsc, EPGDB_CONTEXT * dbc, const PI_BLOCK *pPi, uchar stream );
void EpgTscQueue_AddAll( EPGDB_PI_TSC * tsc, EPGDB_CONTEXT * dbc );

// interface to network server
bool EpgTscQueue_HasElems( EPGDB_PI_TSC * tsc );
void EpgTscQueue_UnlockBuffers( EPGDB_PI_TSC * pQueue );
bool EpgTscQueue_PushBuffer( EPGDB_PI_TSC * tsc, EPGDB_PI_TSC_BUF * pTscBuf, uint size );
const EPGDB_PI_TSC_BUF * EpgTscQueue_PopBuffer( EPGDB_PI_TSC * tsc, uint * pSize );

// interface to GUI
const EPGDB_PI_TSC_ELEM * EpgTscQueue_PopElem( EPGDB_PI_TSC * tsc, time_t * pBaseTime );
const EPGDB_PI_TSC_ELEM * EpgTscQueue_PeekTail( EPGDB_PI_TSC * pQueue, uint provCni );
bool EpgTscQueue_IsIncremental( EPGDB_PI_TSC * pQueue );
void EpgTscQueue_ClearUnprocessed( EPGDB_PI_TSC * pQueue );


#endif  // __EPGTSCQUEUE_H
