/* conf.c - configuration handlers for upsd

   Copyright (C) 2001  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "upsd.h"
#include "conf.h"
#include "upsconf.h"

#include "parseconf.h"

	extern	int	maxage;
	extern	char	*statepath;
	extern	upstype	*firstups;
	extern	acltype	*firstacl;
	extern	acctype	*firstacc;
	ups_t	*upstable = NULL;
	int	reload_flag;

	/* TODO: remove this when ups.conf is required */	
	int	conv_warn;

void upsd_conf_args(int numargs, char **arg)
{
	/* everything below here uses up through arg[1] */
	if (numargs < 2)
		return;

	/* MAXAGE <seconds> */
	if (!strcmp(arg[0], "MAXAGE"))
		maxage = atoi(arg[1]);

	/* STATEPATH <dir> */
	if (!strcmp(arg[0], "STATEPATH")) {
		if (statepath)
			free(statepath);

		statepath = xstrdup(arg[1]);
	}

	/* everything below here uses up through arg[2] */
	if (numargs < 3)
		return;

	/* ACL <aclname> <ip block> */
	if (!strcmp(arg[0], "ACL"))
		addacl(arg[1], arg[2]);

	/* UPS <name> <path> */

	/* this dies soon ... */
	if (!strcmp(arg[0], "UPS")) {
		if ((!arg[1]) || (!arg[2])) {
			upslogx(LOG_ERR, "upsd.conf: UPS directive needs 2 arguments");
			return;
		}

		/* if a UPS exists, update it, else add it as new */
		if ((reload_flag) && (findups(arg[1]) != NULL))
			redefine_ups(arg[2], arg[1]);
		else
			addups(arg[2], arg[1]);

		if (!conv_warn) {
			upslogx(LOG_WARNING, "Warning: UPS definitions in upsd.conf are deprecated - switch to ups.conf");
			conv_warn = 1;
		}
	}

	if (numargs < 4)
		return;

	/* ACCESS <action> <level> <aclname> [<password>] */
	if (!strcmp(arg[0], "ACCESS")) {

		/* don't reference arg[4] unless we have it */
		if (numargs >= 5)
			addaccess(arg[1], arg[2], arg[3], arg[4]);
		else
			addaccess(arg[1], arg[2], arg[3], NULL);
	}
}

void upsd_conf_error(int linenum, char *errtext)
{
	if (linenum)
		upslogx(LOG_ERR, "upsd.conf: %s", errtext);
	else
		upslogx(LOG_ERR, "upsd.conf line %d: %s", linenum, errtext);
}

void read_upsdconf(int reload)
{
	int	ret;
	char	fn[SMALLBUF];

	snprintf(fn, sizeof(fn), "%s/upsd.conf", CONFPATH);

	reload_flag = reload;

	/* TODO: remove when ups.conf is required */
	conv_warn = 0;

	ret = parseconf(fn, upsd_conf_args, upsd_conf_error);

	if (ret != 0) {
		if (!reload)
			fatal("Can't open %s", fn);
		else {
			upslog(LOG_ERR, "Reload failed: can't open %s", fn);
			return;
		}
	}
}

/* callback during parsing of ups.conf */
void do_upsconf_args(char *upsname, char *var, char *val)
{
	ups_t	*tmp, *last;

	/* no "global" stuff for us */
	if (!upsname)
		return;

	last = tmp = upstable;

	while (tmp) {
		last = tmp;

		if (!strcmp(tmp->upsname, upsname)) {
			if (!strcmp(var, "driver")) 
				tmp->driver = xstrdup(val);
			if (!strcmp(var, "port")) 
				tmp->port = xstrdup(val);
			return;
		}

		tmp = tmp->next;
	}

	tmp = xmalloc(sizeof(ups_t));
	tmp->upsname = xstrdup(upsname);
	tmp->driver = NULL;
	tmp->port = NULL;
	tmp->next = NULL;

	if (!strcmp(var, "driver"))
		tmp->driver = xstrdup(val);
	if (!strcmp(var, "port"))
		tmp->port = xstrdup(val);

	if (last)
		last->next = tmp;
	else
		upstable = tmp;
}

/* add valid UPSes from ups.conf to the internal structures */
void upsconf_add(int reloading)
{
	ups_t	*tmp = upstable, *next;
	char	statefn[SMALLBUF];

	/* nonfatal for now - eventually this will be required */
	if (!tmp) {
		upslogx(LOG_WARNING, "Warning: no UPS definitions in ups.conf");
		return;
	}

	while (tmp) {

		/* save for later, since we delete as we go along */
		next = tmp->next;

		/* this should always be set, but better safe than sorry */
		if (!tmp->upsname) {
			tmp = tmp->next;
			continue;
		}

		/* don't accept an entry that's missing items */
		if ((!tmp->driver) || (!tmp->port)) {
			upslogx(LOG_WARNING, "Warning: ignoring incomplete configuration for UPS [%s]\n", 
				tmp->upsname);
		} else {
			snprintf(statefn, sizeof(statefn), "%s-%s", tmp->driver, xbasename(tmp->port));

			/* if a UPS exists, update it, else add it as new */
			if ((reloading) && (findups(tmp->upsname) != NULL))
				redefine_ups(statefn, tmp->upsname);
			else
				addups(statefn, tmp->upsname);
		}

		/* free tmp's resources */

		if (tmp->driver)
			free(tmp->driver);
		if (tmp->port)
			free(tmp->port);
		if (tmp->upsname)
			free(tmp->upsname);
		free(tmp);

		tmp = next;
	}

	/* upstable should be completely gone by this point */
	upstable = NULL;
}

/* remove a UPS from the linked list */
void delete_ups(upstype *target)
{
	upstype	*ptr, *last;

	if (!target)
		return;

	ptr = last = firstups;

	while (ptr) {
		if (ptr == target) {
			upslogx(LOG_NOTICE, "Deleting UPS [%s]", target->name);

			/* make sure nobody stays logged into this thing */
			kick_login_clients(target->name);

			/* about to delete the first ups? */
			if (ptr == last)
				firstups = ptr->next;
			else
				last->next = ptr->next;

			/* release memory */
			if (ptr->fn)
				free(ptr->fn);
			if (ptr->name)
				free(ptr->name);
			free(ptr);

			return;
		}

		last = ptr;
		ptr = ptr->next;
	}

	/* shouldn't happen */
	upslogx(LOG_ERR, "delete_ups: UPS not found");
}			

void delete_acls(void)
{
	acltype	*ptr, *next;

	ptr = firstacl;

	while (ptr) {
		next = ptr->next;
		
		if (ptr->name)
			free(ptr->name);
		free(ptr);

		ptr = next;
	}

	firstacl = NULL;
}

void delete_access(void)
{
	acctype	*ptr, *next;

	ptr = firstacc;

	while (ptr) {
		next = ptr->next;

		if (ptr->aclname)
			free(ptr->aclname);
		if (ptr->password)
			free(ptr->password);
		free(ptr);

		ptr = next;
	}

	firstacc = NULL;
}	

/* see if we can open a file */
static int check_file(char *fn)
{
	char	chkfn[SMALLBUF];
	FILE	*f;

	snprintf(chkfn, sizeof(chkfn), "%s/%s", CONFPATH, fn);

	f = fopen(chkfn, "r");

	if (!f) {
		upslog(LOG_ERR, "Reload failed: can't open %s", chkfn);
		return 0;	/* failed */
	}

	fclose(f);
	return 1;	/* OK */
}

/* called after SIGHUP */
void reload_config(void)
{
	upstype	*upstmp, *upsnext;

	upslogx(LOG_INFO, "SIGHUP: reloading configuration");

	/* see if we can access upsd.conf before blowing away the config */
	if (!check_file("upsd.conf"))
		return;

	/* reset retain flags on all known UPS entries */
	upstmp = firstups;
	while (upstmp) {
		upstmp->retain = 0;
		upstmp = upstmp->next;
	}

	/* reload from ups.conf */
        read_upsconf(0);		/* 0 = not required (yet) */
	upsconf_add(1);			/* 1 = reloading */

	/* flush ACL/ACCESS definitions */
	delete_acls();
	delete_access();

	/* now reread upsd.conf */
	read_upsdconf(1);

	/* now delete all UPS entries that didn't get reloaded */

	upstmp = firstups;

	while (upstmp) {
		/* upstmp may be deleted during this pass */
		upsnext = upstmp->next;

		if (upstmp->retain == 0)
			delete_ups(upstmp);

		upstmp = upsnext;
	}

	/* did they actually delete the last UPS? */
	if (firstups == NULL)
		upslogx(LOG_WARNING, "Warning: no UPSes currently defined!");

	/* and also make sure upsd.users can be read... */
	if (!check_file("upsd.users"))
		return;

	/* delete all users */
	user_flush();

	/* and finally reread from upsd.users */
	user_load();
}
