/* upscommon.c - functions used in more than one model support module

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "upscommon.h"
#include "common.h"

#include <pwd.h>
#include <ctype.h>
#include <sys/file.h>
#include <sys/ioctl.h>
#include "timehead.h"
#include <sys/socket.h>
#include <sys/un.h>

#ifdef HAVE_SYS_SHM_H
#include <sys/ipc.h>
#include <sys/shm.h>
#else
#define shmget(a,b,c) (-1)
#define shmctl(a,b,c) /* nop */
struct shmid_ds {
        void *junk;
};
#endif

#ifdef HAVE_UU_LOCK
#include <libutil.h>
#endif

	int	upsfd, shmid = -1, upsc_debug = 0;
static	int statefd = -1;
        char    statefn[SIZE_OF_STATEFN], *upsport, *pidfn = NULL;
	char	initfn[SIZE_OF_STATEFN];

static	char	sock_fn[SIZE_OF_STATEFN];
static	int	sock_fd = -1;
static	fd_set	sock_rfds;
static	int	sock_maxfd;

	itype	*info = NULL;
static	itype	*shared_info = NULL;
static	size_t info_size = 0;
static	itype *info_mtime = NULL;
	int	infoused = 0, infomax = 0;

	struct	ups_handler	upsh;

	int	do_lock_port = 1;

	/* main sets this, and nobody else does */
	/* TODO: remove when all old style drivers are dead */
	int	use_init_fn = 0;

/* need to pick a sane default.  maybe this should be set by open_serial */
	unsigned int upssend_delay = 0;
	char	upssend_endchar = '\0';
	
	int	flag_timeoutfailure = 0;

/* signal handler for SIGALRM when opening a serial port */
void openfail(int sig)
{
	fatal("Fatal error: serial port open timed out");
}

/* try whatever method(s) are available to lock the port for our use */
void lockport (int upsfd, const char *port)
{
	if (do_lock_port == 0) {
		upslogx(LOG_INFO, "Serial port locking disabled.");
		return;
	}

#ifdef HAVE_UU_LOCK
	if (upsport)		/* already locked? */
		return;

	/* save for later in case we need it at shutdown */
	upsport = xstrdup(xbasename(port));

        {
                int res = uu_lock(upsport);
 
                if (res != 0)
                        fatalx("Can't uu_lock %s: %s", upsport,
                               uu_lockerr(res));
        }
#elif defined(HAVE_FLOCK)
        if (flock(upsfd, LOCK_EX | LOCK_NB) != 0)
                fatalx("%s is locked by another process", port);
#elif defined(HAVE_LOCKF)
        lseek(upsfd, 0L, SEEK_SET);
        if (lockf(upsfd, F_TLOCK, 0L))
                fatalx("%s is locked by another process", port);
#endif
}

/* explain in full detail what's wrong when the open fails */
static void open_error(const char *port)
{
	struct	passwd	*user;
	struct	stat	fs;

	/* see if port even exists first - typo check */
	if (stat(port, &fs)) {
		upslogx(LOG_NOTICE, "Can't stat() %s - did you specify a nonexistent /dev file?",
		       port);
		fatal("Unable to open %s", port);
	}

	user = getpwuid(getuid());
	if (user)
		upslogx(LOG_NOTICE, "This program is currently running as %s (UID %d)",
		       user->pw_name, (int)user->pw_uid);

	user = getpwuid(fs.st_uid);
	if (user)
		upslogx(LOG_NOTICE, "%s is owned by user %s (UID %d), mode %04o", port, 
		       user->pw_name, (int)user->pw_uid,
                       (int)(fs.st_mode & 07777));

	upslogx(LOG_NOTICE, "Change the port name, or fix the permissions or ownership"
	       " of %s and try again.", port);

	fatal("Unable to open %s", port);
}

/* open the port, but don't touch the tty details (except CLOCAL and speed) */
void open_serial_simple(const char *port, speed_t speed, int flags)
{
	struct	termios	tio;

	signal(SIGALRM, openfail);
	alarm(3);

	if ((upsfd = open(port, O_RDWR | O_NONBLOCK)) == -1)
		open_error(port);

	alarm(0);

	lockport(upsfd, port);

	tcgetattr(upsfd, &tio);
	tio.c_cflag |= CLOCAL;
	if (speed) {
#ifndef HAVE_CFSETISPEED
#error This system lacks cfsetispeed() and has no other means to set the speed
#endif
		cfsetispeed(&tio, speed);
		cfsetospeed(&tio, speed);
	}
	tcsetattr(upsfd, TCSANOW, &tio);

	/* set "normal" flags - cable power, or whatever */
	if (ioctl(upsfd, TIOCMSET, &flags))
		fatal("ioctl");
}

/* open a serial port and setup the flags properly at a given speed */
void open_serial(const char *port, speed_t speed)
{
	struct	termios	tio;
	int	fcarg;

	/* open port for normal use */

	signal (SIGALRM, openfail);
	alarm (3);

	upsfd = open (port, O_RDWR | O_NOCTTY | O_EXCL | O_NONBLOCK);
	alarm (0);

	if (upsfd < 0)
		open_error(port);

	fcarg = fcntl(upsfd, F_GETFL, 0);
	if (fcarg < 0 || fcntl(upsfd, F_SETFL, fcarg & ~O_NONBLOCK) < 0)
		fatal("Unable to clear O_NONBLOCK 2 %s", port);

	signal (SIGALRM, SIG_IGN);

	lockport (upsfd, port);

	tcgetattr (upsfd, &tio);
	tio.c_cflag = CS8 | CLOCAL | CREAD;
	tio.c_iflag = IGNPAR;
	tio.c_oflag = 0;
	tio.c_lflag = 0;
	tio.c_cc[VMIN] = 1;
	tio.c_cc[VTIME] = 0;

#ifdef HAVE_CFSETISPEED
	cfsetispeed (&tio, speed);
	cfsetospeed (&tio, speed);
#else
#error This system lacks cfsetispeed() and has no other means to set the speed
#endif

	tcflush (upsfd, TCIFLUSH);
	tcsetattr (upsfd, TCSANOW, &tio);
}

/* put a notice in the syslog */
void notice (const char *msg)
{
	upslogx(LOG_NOTICE, "Notice: %s", msg);
}

/* function for erasing "timeout"-conditions */
void nolongertimeout(void)
{
	/* if override enabled, then return without changing anything */
	if (flag_timeoutfailure == -1)
		return;

	if (flag_timeoutfailure == 1)
		upslogx(LOG_NOTICE, "Serial port read ok again");
	
	flag_timeoutfailure = 0;
	return;
}

/* signal handler for SIGALRM when trying to read */
void timeout(int sig)
{
	struct sigaction sa;
	sigset_t sigmask;

	sa.sa_handler = SIG_DFL;
	sigemptyset (&sigmask);
	sa.sa_mask = sigmask;
	sa.sa_flags = 0;
	sigaction (SIGALRM, &sa, NULL);

	/* if override enabled, then return without changing anything */
	if (flag_timeoutfailure == -1)
		return;

	if (flag_timeoutfailure == 0)
		upslogx(LOG_NOTICE, "Serial port read timed out");
	
	flag_timeoutfailure = 1;
	return;
}

/* wait for an answer in the form <data><endchar> and store in buf */
int upsrecv (char *buf, int buflen, char endchar, const char *ignchars)
{
	char	recvbuf[512], *ptr, in;
	int	ret, cnt;
	struct 	sigaction sa;
	sigset_t sigmask;

	strcpy (recvbuf, "");

	sa.sa_handler = timeout;
	sigemptyset (&sigmask);
	sa.sa_mask = sigmask;
	sa.sa_flags = 0;
	sigaction (SIGALRM, &sa, NULL);

	alarm (3);

	ptr = recvbuf;
	*ptr = 0;
	cnt = 0;
	for (;;) {
		ret = read (upsfd, &in, 1);
		if (ret > 0) {

			if (in == endchar) {
				alarm (0);
				signal (SIGALRM, SIG_IGN);
				strlcpy (buf, recvbuf, buflen);
				if (upsc_debug > 0) {
					int i;
					printf("upsrecv: read %d bytes [", cnt);
					for (i = 0; i < cnt; ++i)
						printf(isprint((unsigned char)buf[i]) ? "%c" : "\\%03o",
							   buf[i]);
					printf ("]\n");
				}
				return (buflen);
			}

			if (strchr(ignchars, in) == NULL) {
				*ptr++ = in;
				*ptr = 0; /* tie off end */
				cnt++;
			}
			
			nolongertimeout();
		}
		else {
			alarm (0);
			signal (SIGALRM, SIG_IGN);
			strlcpy (buf, recvbuf, buflen);
			return (-1);
		}

		/* keep from overrunning the buffer - lame hack */
		if (cnt > (sizeof(recvbuf) - 4)) {
			notice ("UPS is spewing wildly");
			return(-1);
		}
	}

	return (-1);	/* not reached */
}

/* read buflen chars and store in buf */
int upsrecvchars (char *buf, int buflen)
{
	int ret,count;
	char* bufptr;
	struct 	sigaction sa;
	sigset_t sigmask;

	sa.sa_handler = timeout;
	sigemptyset (&sigmask);
	sa.sa_mask = sigmask;
	sa.sa_flags = 0;
	sigaction (SIGALRM, &sa, NULL);

	alarm (3);
	
	bufptr=buf;
	count=buflen;
	while (count>0) {
		ret=read(upsfd,bufptr,count);
		if (ret>0) {
			bufptr+=ret;
			count-=ret;
		} else {
			alarm (0);
			signal (SIGALRM, SIG_IGN);
			return (-1);
		}
	}
	alarm (0);
	signal (SIGALRM, SIG_IGN);
	return buflen;
}

/* send a single byte to the UPS */
int upssendchar (char data)
{
	if (upsc_debug > 0) {
		printf ("upssendchar: sending [");
		printf (isprint((unsigned char)data) ? "%c" : "\\%03o", data);
		printf ("]\n");
	}

	tcflush (upsfd, TCIFLUSH);
	return (write (upsfd, &data, 1));
}

int upssend(const char *fmt, ...)
{
	char buf[1024], *p;
	int bytes_sent = 0;
	va_list ap;

	va_start(ap, fmt);

	if (vsnprintf(buf, sizeof(buf), fmt, ap) >= sizeof(buf))
		; /* truncated */

	va_end(ap);

	tcflush(upsfd, TCIFLUSH);

	for (p = buf; *p; p++) {
		if (upsc_debug > 0) {
/*			printf ("upssendchar: sending [");
 *			printf (isprint(data) ? "%c" : "\\%03o", data);
 *			printf ("]\n");
 */		}

		if (write(upsfd, p, 1) != 1)
			return -1;
		bytes_sent++;
		usleep(upssend_delay);
	}

	if (upssend_endchar) {
		if (write(upsfd, &upssend_endchar, 1) != 1)
			return -1;
		bytes_sent++;
		usleep(upssend_delay);
	}

	return bytes_sent;
}


/* read in any pending characters.  If expected is set then
 * just drop them, otherwise log an error unles they are
 * in ignore chars
 */
void upsflushin (int expected, int debugit, const char *ignchars)
{
	fd_set readfds;
	struct timeval timeout;
	char logstr[256];
	char * ptr;
	int left, rc;

	ptr = logstr;
	*ptr = 0;
	left = 250;		/* somewhat smaller than buffer */

	FD_ZERO(&readfds);
	FD_SET(upsfd, &readfds);

	timeout.tv_usec = 0;
	timeout.tv_sec = 0;

	while ((rc = select(upsfd+1,
			    &readfds,
			    NULL,
			    NULL,
			    &timeout)) > 0) {
		char in;

		read(upsfd, &in, 1);
		if (debugit) {
			printf ("upsflushin: read char ");
			printf(isprint(in) ? "%c" : "\\%03o",
			       in);
		}
		if ((!expected) &&
		    (strchr(ignchars, in) == NULL)) {
			/* log the excess characters */
			if (isprint(in)) {
				*ptr++ = in;
				*ptr = 0; /* tie of string */
				left--;
			} else {
				sprintf(ptr, "\\%03o", in);
				ptr += 4;
				left--;
			}
			/* check if buffer full - if so moan */
			if (left <= 0) {
				upslogx(LOG_INFO,
					"Unexpected UPS jabber [%s]",
					logstr);
				ptr = logstr;
				left = 250;
			}
		}
	}

	/* check for error returns from select */
	if (rc != 0) {
		upslogx(LOG_NOTICE, "select() on input flush returned error");
	}

	if (ptr != logstr) {
		upslogx(LOG_INFO,
			"Unexpected UPS output [%s]",
			logstr);
	}

	return;
}

/* get data from the UPS and install it in the data array */
void installinfo (int infotype, char reqchar, char endchar, const char *ignchars)
{
	int	pos;

	pos = findinfo(info, infomax, 0, infotype);

	if (pos == -1)		/* not found, probably not supported */
		return;

	upssendchar (reqchar);
	upsrecv (info[pos].value, sizeof(info[pos].value), endchar, ignchars);
}

/* store data into the array */
void setinfo(int infotype, const char *fmt, ...)
{
	int	pos;
	va_list ap;

	pos = findinfo(info, infomax, 0, infotype);

	if (pos == -1) {	/* not found, probably debugging? */
		upslogx(LOG_ERR, "setinfo: can't find type %i", infotype);
		return;
	}

	va_start(ap, fmt);
	if (vsnprintf(info[pos].value, sizeof(info[pos].value), fmt, ap) >= sizeof(info[pos].value))
		; /* truncated */
	va_end(ap);
}

/* set a flag on an existing member of the array */
void setflag (int infotype, int newflags)
{
	int	pos;

	pos = findinfo(info, infomax, 0, infotype);

	if (pos == -1) {	/* not found, probably debugging? */
		upslogx(LOG_ERR, "setflag: can't find type %i", infotype);
		return;
	}

	info[pos].flags = newflags;
}

/* find data of a given type in the info array */
char *getdata (int infotype)
{
	int	i;

	for (i = 0; i < infomax; i++)
		if (info[i].type == infotype)
			return (info[i].value);
	
	return (NULL);
}

#ifdef HAVE_SHMAT

/* create state file with a pointer to the SHM struct */
static void writeshminfo(itype *writeinfo)
{
	if (writeinfo[0].type != INFO_SHMID)
		fatalx("shouldn't be here");

	if (statefd >= 0) {
		int ret = write(statefd, writeinfo, sizeof(itype));
		if (ret == sizeof(itype))
			return;
	}

	/* ssfd < 0 || ret < 0 */
	fatal("Can't write shminfo to %s", statefn);
}
#endif	/* HAVE_SHMAT */

void writeinfo(void)
{
	int	ret, tries = 0;
	struct	flock	lock;

	/* if data is stale, don't write it so the information ages */
	if (flag_timeoutfailure == 1)
		return;

	if (info[0].type == INFO_SHMID)
		fatalx("info[0].type == INFO_SHMID in a shm segment");

	snprintf(info_mtime->value, VALSIZE, "%ld", time(NULL));

	lock.l_start = 0;
	lock.l_len = 0;
	lock.l_type = F_WRLCK;
	lock.l_whence = SEEK_SET;

	/* try to get the lock for up to 250 ms */
	while (fcntl(statefd, F_SETLK, &lock) == -1) {
		tries++;

		if (tries == 10) {
			upslogx(LOG_WARNING, "Could not lock data for writing");
			return;
		}

		usleep(25000);
	}

	memcpy(shared_info, info, info_size);

	if (shmid >= 0) {
		struct	shmid_ds shmbuf;

		/* this updates the ctime that upsd wants to see */
		shmctl (shmid, IPC_STAT, &shmbuf);
		shmctl (shmid, IPC_SET, &shmbuf);
	} else {
#ifndef HAVE_MMAP
		if (lseek(statefd, 0, SEEK_SET))
			fatal("lseek statefd");
		if (write(statefd, info, info_size) != info_size)
			fatal("Can't write to %s", statefn);
#endif /* HAVE_MMAP */
	}

	lock.l_start = 0;
	lock.l_len = 0;
	lock.l_type = F_UNLCK;
	lock.l_whence = SEEK_SET;

	ret = fcntl(statefd, F_SETLK, &lock);

	if (ret == -1)
		upslog(LOG_INFO, "Error while unlocking UPS data");
}

static void create_socket(void)
{
	struct sockaddr_un sa;
	int ret;

	snprintf(sock_fn, sizeof(sock_fn), "%s.sock", statefn);

	memset(&sa, '\0', sizeof(sa));
	sa.sun_family = AF_UNIX;
	if (strlcpy(sa.sun_path, sock_fn, sizeof(sa.sun_path)) >= sizeof(sa.sun_path))
		fatalx("sockfile path too long: %s", sock_fn);

	/*	sa.sun_len = strlen(sock_fn); */	/* not portable */

	if ((sock_fd = socket(AF_UNIX, SOCK_STREAM, 0)) == -1)
		fatal("socket");
	if (bind(sock_fd, (struct sockaddr *) &sa, sizeof(sa)))
		fatal("bind %s", sock_fn);
	if (listen(sock_fd, 5))
		fatal("listen %s", sock_fn);

	if ((ret = fcntl(sock_fd, F_GETFL, 0)) == -1)
		fatal("fcntl(get)");
	if (fcntl(sock_fd, F_SETFL, ret | O_NDELAY) == -1)
		fatal("fcntl(set)");

	FD_ZERO(&sock_rfds);
	FD_SET(sock_fd, &sock_rfds);
	sock_maxfd = sock_fd + 1;
}

/* try to create the state file before entering the background */
static void test_writeinfo(void) 
{
	int	mode;
	char	*fn;

	/* upsd needs to be able to write to the file in mmap mode */
#ifdef HAVE_MMAP
	mode = S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP;
	umask(0117);
#else
	mode = S_IRUSR | S_IWUSR | S_IRGRP;
#endif

	/* TODO: remove when all old style drivers are dead */
	/* for now - instead of adding info_ready to all the old drivers */

	if (use_init_fn) {
		snprintf(initfn, sizeof(initfn), "%s.init", statefn);
		unlink(initfn);
		fn = initfn;
	} else {

		/* OK, this is the end of the line for the old drivers...
		 *
		 * this is a place only old drivers get to, so it's the
		 * perfect place to warn the user of the impending doom
		 */

		upslogx(LOG_WARNING, "*** Warning: old style driver detected                            ***");
		upslogx(LOG_WARNING, "*** This driver will be removed from the tree in the next version ***");
		upslogx(LOG_WARNING, "*** See the FAQ for more information                              ***");		

		unlink(statefn);
		fn = statefn;
	}

        statefd = open(fn, O_RDWR | O_CREAT | O_NOCTTY, mode);

	if (statefd < 0) {
		int tmperr = errno;
		upslog(LOG_NOTICE, "Can't open %s", fn);

		if (tmperr == EACCES) {
			struct	passwd	*user;
			user = getpwuid(getuid());

			if (user)
				upslogx(LOG_NOTICE, "This program is currently running as %s (UID %d)",
					 user ? user->pw_name : "(UNKNOWN)", user->pw_uid);
		}
		exit (1);
	}
}

/* rename from <fn>.init to <fn> when it's ready to be used */
void info_ready(void)
{
	int	ret;

	if (use_init_fn != 1) {
		upslogx(LOG_ERR, "Programming error: info_ready() called without use_init_fn set");
		return;
	}

	ret = rename(initfn, statefn);

	if (ret != 0)
		fatal("rename %s to %s", initfn, statefn);
}

static void cleanup(void)
{
	if (pidfn)
		unlink(pidfn);

	unlink(statefn);

	/* just in case this lingers... */
	unlink(initfn);

	if (sock_fd != -1) {
		close(sock_fd);
		unlink(sock_fn);
	}

	if (statefd != -1)
		close(statefd);

#ifdef HAVE_SHMAT
	/* mark for deletion after total detachment */
	if (shmid != -1) {
		shmctl (shmid, IPC_RMID, NULL);                
		shmdt ((char *) info);	/* detach */
	}
#endif	/* HAVE_SHMAT */

#ifdef HAVE_UU_LOCK
	uu_unlock (upsport);
#endif
}

void sigdie(int sig)
{
	upslogx(LOG_INFO, "Signal %d: Shutting down", sig);
	exit (0);
}

/* install handler for sigterm */
void catch_sigterm(void)
{
	struct sigaction sa;
	sigset_t sigmask;

	sa.sa_handler = sigdie;
	sigemptyset (&sigmask);
	sa.sa_mask = sigmask;
	sa.sa_flags = 0;
	sigaction (SIGTERM, &sa, NULL);
}

/* clean up the local info storage for future use */
static void info_init()
{
	int	i;

	/* initialize the array */
	for (i = 0; i < infomax; i++)
		info[i].type = INFO_UNUSED;

	info[0].type = INFO_MEMBERS;
	snprintf(info[0].value, VALSIZE, "%i", infomax);
	infoused++;

	info[1].type = INFO_MTIME;
	snprintf(info[1].value, VALSIZE, "%ld", time(NULL));
	infoused++;

	info_mtime = &info[1];

	writeinfo();
}

#ifdef HAVE_SHMAT

/* create a shared memory struct for our info */
static int shm_create(void)
{
	key_t	shmkey = IPC_PRIVATE;
	itype	shminfo;

	/* get some shared memory */
	shmid = shmget(shmkey, sizeof(itype) * infomax, IPC_MODE);

	if (shmid == -1) {
		upslog(LOG_ERR, "shmget");
		shmid = -1;
		return 0;	/* failed */
	}

	/* got a good ID, so attach to it */
	shared_info = (itype *) shmat(shmid, 0, 0);

	if (shared_info == (itype *) (-1)) {
		upslog(LOG_ERR, "shmat");
		shmid = -1;
		return 0;	/* failed */
	}

	shminfo.type = INFO_SHMID;
	snprintf(shminfo.value, sizeof(shminfo.value), "%d", shmid);

	/* create state file with pointer to shm struct */
	writeshminfo(&shminfo);

	/* use the same mtime number in both places */
	info_init();

	return 1;	/* OK */
}	

#endif	/* HAVE_SHMAT */

/* start the state file, and possibly the shared memory struct too */
void create_info(int numinfo, int shmok)
{
	infomax = numinfo;
	info_size = infomax * sizeof(itype);

	info = xcalloc(1, info_size);

	/* make sure we clean up when exiting */
	catch_sigterm();
	signal(SIGINT, sigdie);
	signal(SIGQUIT, sigdie);
	if (atexit(cleanup))
		fatal("atexit");

	/* see if the state file can be created */
	test_writeinfo();

#ifdef HAVE_SHMAT

	/* SHM failures fall through and try normal mode */
	if ((shmok) && (shm_create()))
		return;

#endif

#ifdef HAVE_MMAP
	if (ftruncate(statefd, info_size))
		fatal("ftruncate");

	if ((shared_info = (itype *) mmap(NULL, info_size, PROT_READ|PROT_WRITE, MAP_SHARED|MAP_NOSYNC, statefd, 0)) == MAP_FAILED)
		fatal("mmap fd=%d size=%u", statefd, info_size);
#else /* HAVE_MMAP */

	shared_info = xcalloc(1, info_size);

#endif /* HAVE_MMAP */

	/* clean out some space for later */
	info_init();
}

/* add a member to the info struct */
void addinfo (int type, const char *value, int flags, int auxdata)
{
	int	i;

	for (i = 0; i < infomax; i++) {
		if (info[i].type == INFO_UNUSED) {	/* found open spot */
			infoused++;
			info[i].type = type;
			strlcpy(info[i].value, value, sizeof(info[i].value));
			info[i].flags = flags;
			info[i].auxdata = auxdata;
			return;
		}
	}

	upslogx(LOG_ERR, "Unable to add new member to info array!");

	return;
}

/* add a new ENUM info entry and do other related housekeeping */
void addenum (int basetype, const char *value)
{
	int	i;

	/* first find the basetype in the struct */
	for (i = 0; i < infomax; i++)
		if (info[i].type == basetype) {

			/* add the enum value, then increment the enum count */
			addinfo (INFO_ENUM, value, 0, basetype);
			info[i].auxdata++;
			return;
		}

	upslogx(LOG_ERR, "Unable to add enumerated type for base 0x%04x", basetype);

	return;
}

void createmsgq(void)
{
	/* init handlers early */
	memset (&upsh, '\0', sizeof(upsh));

	create_socket();
}

struct client_conn {
	int fd;
	struct sockaddr_un sa;
	struct client_conn *next;
	msgtype msg;
	size_t msg_len;
};

static struct client_conn *client_conn = NULL;

void close_client_conn(struct client_conn *cc)
{
	struct client_conn *t1, *t2;

	upsdebugx(1, "closing client_conn %p fd=%d", cc, cc->fd);

	for (t1 = client_conn, t2 = NULL; t1; t2 = t1, t1 = t1->next)
		if (cc == t1)
			break;

	assert(t1 != NULL);

	if (t2)
		t2->next = cc->next;
	if (cc == client_conn)
		client_conn = cc->next;

	close(cc->fd);
	FD_CLR(cc->fd, &sock_rfds);

	if (cc->fd + 1 == sock_maxfd) {
		int maxfd = sock_fd;

		for (t1 = client_conn; t1; t1 = t1->next) {
			if (t1->fd > maxfd)
				maxfd = t1->fd;
		}

		sock_maxfd = maxfd + 1;
	}

	free(cc);
}

/* get a message if one's waiting in the queue */
int getupsmsg(int wait)
{
	struct timeval tv;
	fd_set	rfds;
	time_t	start_time;
	int	ret;
	struct client_conn *cc;

	memcpy(&rfds, &sock_rfds, sizeof(rfds));

	start_time = time(NULL);

	do {
		upsdebugx(3, "select loop maxfd=%d", sock_maxfd);

		tv.tv_sec = wait;
		tv.tv_usec = 0;
		ret = select(sock_maxfd, &rfds, NULL, NULL, &tv);

		if (ret == 0)
			break;
		if (ret == -1) {
			if (errno == EINTR)
				continue;
			else
				fatal("select");
		}

		for (cc = client_conn; cc; cc = cc->next) {
			if (FD_ISSET(cc->fd, &rfds)) {
				switch ((ret = read(cc->fd, (char *) &cc->msg, sizeof(cc->msg) - cc->msg_len))) {
				case 0:
				case -1:
					if (errno == 0 || errno == EINTR || errno == EAGAIN) /* try again the next time around */
						break;

					upsdebug(1, "read %d bytes from fd %d", ret, cc->fd);
					close_client_conn(cc);
					break;
				default:
					upsdebugx(1, "read %d bytes from fd %d", ret, cc->fd);

					cc->msg_len += ret;
					upsdebugx(1, "msg_len=%d", cc->msg_len);

					if (cc->msg_len != sizeof(cc->msg)) /* buffer not full yet */
						break;

					if (cc->msg.dlen >= UPSMSG_MAXLEN) {
						upslogx(LOG_ERR, "client sent bad data - possible buffer overflow");
						close_client_conn(cc);
						break;
					}

					switch (cc->msg.cmdtype) {
					case UPSMSG_CMDSET:
						if (upsh.setvar)
							upsh.setvar(cc->msg.auxcmd, cc->msg.dlen, cc->msg.data);
						else
							upslogx(LOG_INFO, "No handler for SET command");
						break;
					case UPSMSG_INSTCMD:
						if (upsh.instcmd)
							upsh.instcmd(cc->msg.auxcmd, cc->msg.dlen, cc->msg.data);
						else
							upslogx(LOG_INFO, "No handler for INSTCMD command");
						break;
					default: 
						upslogx(LOG_INFO, "Unknown msgcmd 0x%04x", cc->msg.cmdtype);
					}		

					cc->msg_len = 0;

					upsdebugx(1, "finished message");
				}
			}
		}

		if (FD_ISSET(sock_fd, &rfds)) {
			int sa_len;

			cc = xcalloc(1, sizeof(struct client_conn));
			sa_len = sizeof(cc->sa);

			if ((cc->fd = accept(sock_fd, (struct sockaddr *) &cc->sa, &sa_len)) != -1) {
				upsdebugx(1, "accept sock_fd fd=%d", cc->fd);

				cc->next = client_conn;
				client_conn = cc;

				FD_SET(cc->fd, &sock_rfds);
				if (cc->fd >= sock_maxfd)
					sock_maxfd = cc->fd + 1;
			} else {
				upslog(LOG_NOTICE, "accept sock_fd");
				free(cc);
			}
		}
	} while (start_time + wait < time(NULL));

	return 0;
}

/* add these functions back when a driver is capable of testing them.
void sendupsmsg(struct client_conn *cc, const msgtype *msg)
{
	if (write(cc->fd, msg, sizeof(msgtype)) != sizeof(msgtype)) {
		upslog(LOG_NOTICE, "write failed");
		close_client_conn(cc);
	}
}

void msgreply(struct client_conn *cc, int reptype)
{
	msgtype	msg;

	msg.cmdtype = reptype;
	msg.auxcmd = 0;
	msg.dlen = 0;
	msg.data[0] = '\0';

	sendupsmsg(cc, &msg);
}
*/

/* modify in - strip all trailing instances of <sep> */
void rtrim(char *in, char sep)
{
	char	*p = NULL;

	p = &in[strlen(in) - 1];

	while (p >= in) {
		if (*p != sep)
			return;

		*p-- = '\0';
	}
}

/* try to unlock the port using any methods that are available */
void unlockport (int upsfd, const char *port)
{
#ifdef HAVE_UU_LOCK
        int res;
    
        if (!upsport)		/* not already locked? */
		return;

	res = uu_unlock(upsport);

	/* free up allocated memory */
	free(upsport);

	if (res != 0)
		fatalx("Can't uu_unlock %s: %s", upsport, uu_lockerr(res));

#elif defined(HAVE_FLOCK)
        if (flock(upsfd, LOCK_UN) != 0)
                fatalx("can't unlock %s!", port);
#elif defined(HAVE_LOCKF)
        lseek(upsfd, 0L, SEEK_SET);
        if (lockf(upsfd, F_ULOCK, 0L))
                fatalx("can't unlock %s!", port);
#endif
}

/* close the state fd - used by the exiting parent */
void close_statefd(void)
{
	close(statefd);
}
