/*
 * newpowercom.c - model specific routines for following units:
 *  -Trust 425/625
 *  -Powercom
 *  -Advice Partner/King PR750
 *    See http://www.advice.co.il/product/inter/ups.html for its specifications.
 *    This model is based on PowerCom (www.powercom.com) models.
 *
 * Read docs/powercom.txt for other models and manufactures
 *
 * $Id: - will be filled in on next CVS add/update $
 *
 * Status:
 *  20011022/Revision 0.1 - Peter Bieringer <pb@bieringer.de>
 *   - porting done, looks like working
 *  20011208/Revision 0.2 - Peter Bieringer <pb@bieringer.de>
 *   - add some debug code
 *   - add new option "subtype"
 *   - add support for a 16 byte sending UPS (KP625AP)
 *   - shutdown function checked, but 'Trust' wakes up after a few seconds...
 *
 *   	
 * Copyrights:
 * (C) 2001 Peter Bieringer <pb@bieringer.de>
 *  Porting old style "powercom" to new style "newpowercom"
 *  
 * (C) 2000  Shaul Karl <shaulk@israsrv.net.il>
 *  Creating old style "powercom"
 *   Heavily based on
 *    ups-trust425+625.c - model specific routines for Trust UPS 425/625
 *    Copyright (C) 1999  Peter Bieringer <pb@bieringer.de>
 *                              
 * A technical description of the interface to the UPS that this program
 *  implements is found in docs/\*trust* files.
 *                                     
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *                            
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *    
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 */ 

#include "main.h"
#include "newpowercom.h"

#define POWERCOM_DRIVER_VERSION      "$Revision: 0.2 $"

/* Not used
static unsigned int  powercom_raw_data_bytes = NUM_OF_BYTES_FROM_UPS_MAX;
*/

/* variables used by module */
static unsigned char powercom_raw_data[NUM_OF_BYTES_FROM_UPS_MAX]; /* raw data reveived from UPS */
static unsigned int  powercom_linevoltage = 230U; /* line voltage, can be defined via command line option */
static char *powercom_modelname    = "Unknown";
static char *powercom_serialnumber = "Unknown";
static char *powercom_subtype_name = "Trust";
static unsigned int powercom_subtype = 0;


/* used external variables */
extern int sddelay;          /* shutdown delay, set by "-d $delay" in main.c */
extern int do_forceshutdown; /* shutdown delay, set by "-k" in main.c */


/*
 * local used functions
 */

static void powercom_shutdown(void)
{
        printf ("Initiating UPS shutdown!\n");
        
        upssendchar ((char)SHUTDOWN);
	upssendchar (SEC_FOR_POWERKILL);
			
        exit (0);
}


/* registered instant commands */
static void instcmd (int auxcmd, int dlen, char *data)
{
        /* TODO: reply to upsd? */

        switch (auxcmd) {
                case CMD_BTEST1:        /* start battery test */
                        upssendchar (BATTERY_TEST);
                        break;
                default:
                        upslogx(LOG_INFO, "instcmd: unknown type 0x%04x", auxcmd);
        }
}


/* wait for buflen bytes from upsfd and store in buf */
static int read_raw_data (int upsfd, unsigned char *buf, int buflen)
{
        int     counter = 0;
        struct  sigaction sa;
        sigset_t sigmask;

	sa.sa_handler = timeout;
        sigemptyset (&sigmask);
        sa.sa_mask = sigmask;
        sa.sa_flags = 0;
        sigaction (SIGALRM, &sa, NULL);

        alarm (3);

        for (counter = 0; counter < buflen; counter++) 

		if (!(read (upsfd, buf + counter, 1))) {

			alarm (0);
		        signal (SIGALRM, SIG_IGN);
			return counter;
			
		}

	nolongertimeout();
		
	return counter;
}
			

/* set DTR and RTS lines on a serial port to supply a passive
 * serial interface: DTR to 0 (-V), RTS to 1 (+V)
 */
static void powercom_init_serialDTR0RTS1 ()
{
	int dtr_bit = TIOCM_DTR;
	int rts_bit = TIOCM_RTS;

	/* set DTR to low and RTS to high */
	ioctl(upsfd, TIOCMBIC, &dtr_bit);
	ioctl(upsfd, TIOCMBIS, &rts_bit);
}


/* install pointers to functions for msg handlers called from msgparse */
static void setuphandlers(void)
{
        upsh.instcmd = instcmd;
}


/*
 * global used functions
 */

/* update information */
void upsdrv_updateinfo(void)
{
	char	val[32];
	float	tmp;
	int	i, c;
	int	flag;

	/* send trigger char to UPS */
	if (upssendchar (SEND_DATA) != 1) {
		upslogx(LOG_NOTICE, "writing error");
		return;
	} else {
		upsdebugx(5, "Num of bytes requested for reading from UPS: %d", subtypes_num_of_bytes[powercom_subtype]);

		c = read_raw_data(upsfd, powercom_raw_data, subtypes_num_of_bytes[powercom_subtype]);
	
		if (c != subtypes_num_of_bytes[powercom_subtype]) {
			upslogx(LOG_NOTICE, "data receiving error (%d instead of %d bytes)", c, subtypes_num_of_bytes[powercom_subtype]);
			return;
		};

		flag = 0;	
		if (subtypes_num_of_bytes[powercom_subtype] == 0) {
			/* Trust */	
			if (	(powercom_raw_data[5] != 0) 
				|| (powercom_raw_data[7] != 0) 
				|| (powercom_raw_data[8] != 0)
			) { flag = 1; };
		} else if (subtypes_num_of_bytes[powercom_subtype] == 1) {
			/* KP625AP */	
			if (	(powercom_raw_data[5] != 0x80) 
				|| (powercom_raw_data[7] != 0) 
				|| (powercom_raw_data[8] != 0)
			) { flag = 1; };
		};
		if (flag != 0) {
			upslogx(LOG_NOTICE, "data receiving error (validation check)");
			return;
		};
	};
	if (nut_debug_level > 4) {
		printf("Raw data from UPS: ");
		for (i = 0; i < subtypes_num_of_bytes[powercom_subtype]; i++) {
	 		printf("%d:%02x ", i, powercom_raw_data[i]);
		};
 		printf("\n");
	};

	/* set ACFREQ */
	upsdebugx(3, "ACFREQ   (raw data): [raw: %u]", powercom_raw_data[IN_AC_FREQ]);
	setinfo(INFO_ACFREQ, "%02.2f", 1.0 / (powercom_raw_data[IN_AC_FREQ] * 0.00020997 + 0.00020928));
	upsdebugx(2, "ACFREQ: %s", getdata(INFO_ACFREQ));

	/* set LOADPCT */	
	upsdebugx(3, "LOADPCT  (raw data): [raw: %u]", powercom_raw_data[LOAD_LEVEL]);
	setinfo(INFO_LOADPCT, "%03.1f", tmp = powercom_raw_data[STATUS_A] & MAINS_FAILURE ?
	 		powercom_raw_data[LOAD_LEVEL] * 6.13426854 - 0.38076152 :	
			powercom_raw_data[LOAD_LEVEL] * 4.311 + 0.1811);
	upsdebugx(2, "LOADPCT: %s", getdata(INFO_LOADPCT));

	/* set BATTPCT */
	upsdebugx(3, "BATTPCT (raw data): [raw: %u]", powercom_raw_data[BAT_LEVEL]);
	setinfo(INFO_BATTPCT, "%03.1f", powercom_raw_data[STATUS_A] & MAINS_FAILURE ?
			powercom_raw_data[BAT_LEVEL] * 5.0 + tmp * 0.3268 - 825 :
			powercom_raw_data[BAT_LEVEL] * 4.5639 - 835.82);
	upsdebugx(2, "BATTPCT: %s", getdata(INFO_BATTPCT));

	/* set UTILITY */	
	upsdebugx(3, "UTILITY (raw data): [raw: %u]", powercom_raw_data[IN_AC_VOLT]);
	setinfo(INFO_UTILITY, "%03.1f", tmp =  powercom_linevoltage >= 220 ?
			powercom_raw_data[IN_AC_VOLT] * 1.9216 - 0.0977 :
			powercom_raw_data[IN_AC_VOLT] * 0.9545);
	upsdebugx(2, "UTILITY: %s", getdata(INFO_UTILITY));
	
	*val = 0;
	if (!(powercom_raw_data[STATUS_A] & MAINS_FAILURE))
		!(powercom_raw_data[STATUS_A] & OFF) ? 
			strcat(val, "OL ") : strcat(val, "OFF ");
	else strcat(val, "OB ");
	if (powercom_raw_data[STATUS_A] & LOW_BAT)
		strcat(val, "LB ");
	if (powercom_raw_data[STATUS_A] & AVR_ON)
		tmp < powercom_linevoltage ?
			strcat(val, "BOOST ") : strcat(val, "TRIM ");
	if (powercom_raw_data[STATUS_A] & OVERLOAD)
		strcat (val, "OVER ");
	if (powercom_raw_data[STATUS_B] & BAD_BAT)
		strcat (val, "RB ");
	if (powercom_raw_data[STATUS_B] & TEST)
		strcat (val, "TEST ");
	*(val + strlen(val)) = 0;
	setinfo(INFO_STATUS, "%s", val);
	upsdebugx(2, "STATUS: %s", getdata(INFO_STATUS));
	
	writeinfo();
}


/* shutdown UPS */
void upsdrv_shutdown(void)
{
	if (do_forceshutdown == 1) {
		/* power down the attached load immediately */
		printf("Forced UPS shutdown triggered, do it...\n");
		powercom_shutdown();
	} else {
		/* power down the attached load after the given delay */
		printf("UPS shutdown with '%d' seconds delay triggered, wait now...\n", sddelay);
		sleep(sddelay);
		powercom_shutdown();
	};
}


/* initialize UPS */
void upsdrv_initups(void)
{
	int tmp, i;
	int flag;

	/* check model name from arguments */
	if (getval("modelname") != NULL) {
		powercom_modelname = getval("modelname");
	};
	
	/* check serial number from arguments */
	if (getval("serialnumber") != NULL) {
		powercom_serialnumber = getval("serialnumber");
	};
	
	/* get and check subtype */
	if (getval("subtype") != NULL) {
		powercom_subtype_name = getval("subtype");
		flag = 0;
		for (i = 0; i < NUM_OF_SUBTYPES; i++) {
			if (strcmp(subtypes_name[i], powercom_subtype_name) == 0) {
				flag = 1;
				break;
			};
		};
		if (flag != 1) {
			printf("Given UPS subtype '%s' isn't valid!\n", powercom_subtype_name);
			exit (1);
		} else {
			powercom_subtype = i;	
		};
	};
	
	/* check line voltage from arguments */
	if (getval("linevoltage") != NULL) {
		tmp = atoi(getval("linevoltage"));
		if (! ( (tmp >= 220 && tmp <= 240) || (tmp >= 110 && tmp <= 120) ) ) {
			printf("Given line voltage '%d' is out of range (110-120 or 220-240 V)\n", tmp);
			exit (1);
		};
		powercom_linevoltage  = (unsigned int) tmp;
	};

	upsdebugx(1, "Values of arguments:");
	upsdebugx(1, " model name   : '%s'", powercom_modelname);
	upsdebugx(1, " serial number: '%s'", powercom_serialnumber);
	upsdebugx(1, " line voltage : '%u'", powercom_linevoltage);
	upsdebugx(1, " subtype      : '%s'", powercom_subtype_name);

	/* open serial port */
	open_serial (device_path, B1200);

	/* initialize serial port */	
	powercom_init_serialDTR0RTS1();
}


/* display help */
void upsdrv_help(void)
{
	printf("\n");
	return;
}


/* display banner */
void upsdrv_banner(void)
{
	printf ("Network UPS Tools (version %s) - PowerCom and similars protocol driver\n",
		UPS_VERSION);
	printf ("\tDriver %s\n",
		POWERCOM_DRIVER_VERSION);
}


/* tell main how many entries we need */
int upsdrv_infomax(void)
{
	        return (16);
}


/* initialize information */
void upsdrv_initinfo(void)
{
	/* static void initinfo (char *ident, char *model, char *serial_num) */

        /* write constant data for this model */
	addinfo (INFO_MFR, "PowerCom", FLAG_STRING, 8);
	addinfo (INFO_MODEL, powercom_modelname, FLAG_STRING, strlen(powercom_modelname));
	addinfo (INFO_SERIAL, powercom_serialnumber, FLAG_STRING, strlen(powercom_serialnumber));

        /* add other things to monitor */
	addinfo (INFO_ACFREQ, "", 0, 0);
	addinfo (INFO_BATTPCT, "", 0, 0);
	addinfo (INFO_LOADPCT, "", 0, 0);
	addinfo (INFO_STATUS, "", 0, 0);
	addinfo (INFO_UTILITY, "", 0, 0);

	/* now add the instant commands */
        addinfo (INFO_INSTCMD, "", 0, CMD_BTEST1);

	setuphandlers();
}


/* define possible arguments */
void upsdrv_makevartable(void)
{
	addvar(VAR_VALUE, "linevoltage",  "Specify line voltage (110-120 or 220-240 V), because it cannot detect automagically (default: 230 V)");
	addvar(VAR_VALUE, "modelname",    "Specify model name, because it cannot detect automagically (default: Unknown)");
	addvar(VAR_VALUE, "serialnumber", "Specify serial number, because it cannot detect automagically (default: Unknown)");
	addvar(VAR_VALUE, "subtype", "Subtype of UPS like 'KP625AP' or 'Trust' (default: 'Trust')");
}
