/* ipt-anzen.c - model specific routines for IPT Anzen units

   Copyright (C) 2000 Travis Tabbal <bigboss@xmission.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/termios.h>

#include "config.h"
#include "proto.h"
#include "shared.h"
#include "version.h"
#include "upscommon.h"
#include "common.h"

#define UPSDELAY 5
#define MAXTRIES 10
#define INFOMAX 16
#define ENDCHAR 13	/* replies end with CR */

	int	shmok = 1, cap_upstemp = 0;
	float	lowvolt = 0, voltrange;
	int	lownorm, highnorm;
	int	sddelay = 90;	/* wait 90 seconds for shutdown */

void initinfo (void)
{
	create_info(INFOMAX, shmok);

	/* setup variables that should always exist */

	addinfo (INFO_MFR, "", 0, 0);
	addinfo (INFO_MODEL, "", 0, 0);
	addinfo (INFO_UTILITY, "", 0, 0);
	addinfo (INFO_BATTPCT, "", 0, 0);
	addinfo (INFO_STATUS, "", 0, 0);
	addinfo (INFO_ACFREQ, "", 0, 0);
	addinfo (INFO_LOADPCT, "", 0, 0);
	addinfo (INFO_LOWXFER, "", 0, 0);
	addinfo (INFO_HIGHXFER, "", 0, 0);
	addinfo (INFO_UPSTEMP, "", 0, 0);
}

/* Send a whole string, I hate sending a character at a time */

void xsend (char *str) 
{
	int i, len;

	len = strlen (str);

	for (i = 0; i < len; i++) 
		upssendchar (str[i]);

	upssendchar (13);
}
	
void ups_sync(void)
{
        char    buf[256];
        int     tries = 0;

        printf ("Syncing with UPS: ");
        fflush (stdout);

        for (;;) {
                tries++;
                if (tries > MAXTRIES) {
                        printf ("\nFailed - giving up...\n");
                        exit (1);
                }
                xsend ("\rIM\r");
                printf (".");
                fflush (stdout);
                sleep (UPSDELAY);
                printf (".");
                fflush (stdout);
                upsrecv (buf, sizeof(buf), ENDCHAR, "");
                printf (".");
                fflush (stdout);
                if (buf[0] == 'I')
                        break;
        }

        printf (" done\n");
 }

void getbaseinfo (const char *port)
{
	char	temp[256], model[64];

	ups_sync ();

	upssendchar ('I');
	upssendchar ('M');
	upssendchar (13);
	upsrecv (temp, sizeof(temp), ENDCHAR, "");

	/* Set manufacturer name */

	setinfo(INFO_MFR, "%s", temp);

	upssendchar ('I');
	upssendchar ('O');
	upssendchar (13);
	upsrecv (model, sizeof(model), ENDCHAR, "");

	/* Set the model name */

	setinfo(INFO_MODEL, "%s %s", getdata(INFO_MFR), model);

	/* Set low and high transfer points */
	/* These are defaults, they can be configured */
	/* I don't have specs on how to get these yet */

	setinfo(INFO_LOWXFER, "%s", "93");
	setinfo(INFO_HIGHXFER, "%s", "145");

	/* now add instant command support info */
        addinfo (INFO_INSTCMD, "", 0, CMD_ON);
        addinfo (INFO_INSTCMD, "", 0, CMD_OFF);

	printf ("Detected %s on %s\n", getdata(INFO_MODEL), port);
}

/* normal idle loop - keep up with the current state of the UPS */

void updateinfo (void)
{
	char	temp[256], acfreq[16],
		stat[16], battstat[16];
	double	bvoltp, freq;

	/* Incoming line voltage */

	xsend ("NV\r");
	upsrecv (temp, sizeof(temp), ENDCHAR, "");
	setinfo(INFO_UTILITY, "%i", atoi(temp));		

	/* Battery percentage */

	xsend ("BV\r");
	upsrecv (temp, sizeof(temp), ENDCHAR, "");

	/* UPS Sends the voltage in .1 Volt DC */
	/* Battery is rated at 24V */

	bvoltp = ((atof(temp) / 10) / 24) * 100;
	if (bvoltp > 100) bvoltp = 100;
	setinfo(INFO_BATTPCT, "%02.1f", bvoltp);

	/* Set the status flags */

	xsend ("OS\r");
	upsrecv (stat, sizeof(stat), ENDCHAR, "");
	xsend ("BS\r");
	upsrecv (battstat, sizeof(battstat), ENDCHAR, "");

	strcpy (temp, "");

	if (stat[0] == '3')
		strcat (temp, "OL ");		/* on line */

	if (stat[0] == '4')
		strcat (temp, "OB ");		/* on battery */

	if (battstat[0] == '3' || battstat[0] == '4')
		strcat (temp, "LB ");		/* low battery */
	
	if (stat[0] == '5')  		   /* Supposed to be "BYPASS" */
		strcat (temp, "OB ");  /* But it uses it for on batt */

	if (stat[0] == '6')
		strcat (temp, "BOOST ");

	if (stat[0] == '7')
		strcat (temp, "BUCK ");

	/* lose trailing space if present */
	if (temp[strlen(temp)-1] == ' ')
		temp[strlen(temp)-1] = 0;

	setinfo(INFO_STATUS, "%s", temp);

	/* Set the UPS temp (using the battery temp) */

	xsend ("BT\n");
	upsrecv (temp, sizeof(temp), ENDCHAR, "");
	setinfo(INFO_UPSTEMP, "%i", atoi(temp));

	/* Set the AC Line Freqency */

	xsend ("NF\r");
	upsrecv (acfreq, sizeof(acfreq), ENDCHAR, "");

	freq = atof(acfreq) / 10; 

	setinfo(INFO_ACFREQ, "%02.1f", freq);

	/* Set the load percentage */

	xsend ("OL\r");
	upsrecv (temp, sizeof(temp), ENDCHAR, "");
	setinfo(INFO_LOADPCT, "%i", atoi(temp));

	writeinfo();
}

/* power down the attached load immediately */

void forceshutdown(char *port)
{
	upslogx(LOG_INFO, "Initiating UPS shutdown\n");
	printf ("Initiating forced UPS shutdown!\n");

	open_serial (port, B2400);

	/* Set the UPS for automatic restart */

	xsend ("Ca1\r");

	/* Command the UPS to shut down the output now */

	xsend ("Ct1\r");
	xsend ("Cs0000000\r");

	if (sddelay > 0) {
		printf ("Waiting for poweroff...\n");
		sleep (sddelay);
		printf ("Hmm, did the shutdown fail?  Oh well...\n");
		exit (1);
	}
	exit(0);
}

void instcmd (int auxcmd, int dlen, char *data)
{
	/* TODO: reply to upsd? */

	switch (auxcmd) {
		case CMD_OFF:			/* Shut down load now */
			xsend ("Ct0\r");
			xsend ("Cs0000000\r");
			break;
		case CMD_ON:			/* Turn on the load now */
			xsend ("Cu0000000\r");
			break;
		default:
			upslogx(LOG_INFO, "instcmd: unknown type 0x%04x\n",
		        	auxcmd);
	}
}

/* install pointers to functions for msg handlers called from msgparse */
void setuphandlers(void)
{
	upsh.instcmd = instcmd;
	/* TODO: future */
}

void usage (char *prog)
{
	printf ("usage: %s [-h] [-d <num>] [-k] <device>\n", prog);
	printf ("Example: %s /dev/ttyS0\n", prog);
	exit (1);
}

void help (char *prog)
{
	printf ("usage: %s [-h] [-d <num>] [-k] <device>\n", prog);
	printf ("\n");
	printf ("-d <num> - wait <num> seconds after sending shutdown command\n");
	printf ("-h       - display this help\n");
	printf ("-k       - force shutdown\n");
	printf ("<device> - /dev entry corresponding to UPS port\n");
}

int main (int argc, char **argv)
{
	char	*portname, *prog;
	int	i;

	printf ("Network UPS Tools - IPT Anzen UPS driver 0.01 (%s)\n",UPS_VERSION);
	openlog ("ipt-anzen", LOG_PID, LOG_FACILITY);

	prog = argv[0];

	while ((i = getopt(argc, argv, "+hd:k:")) != EOF) {
		switch (i) {
			case 'd':
				sddelay = atoi(optarg);
				break;
			case 'k':
				forceshutdown(optarg);
				break;
			case 'h':
				help(prog);
				break;
			default:
				usage(prog);
				break;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc != 1) {
		help (prog);
		exit (1);
	}

	droproot(); 

	portname = NULL;
	for (i = strlen(argv[0]); i >= 0; i--)
		if (argv[0][i] == '/') {
			portname = &argv[0][i+1];
			break;
		}

	if (portname == NULL) {
		printf ("Unable to abbreviate %s\n", argv[0]);
		exit (1);
	}

	snprintf (statefn, sizeof(statefn), "%s/ipt-anzen-%s", STATEPATH,
	          portname);

	open_serial (argv[0], B2400); 

	initinfo();

	createmsgq();	/* try to create IPC message queue */

	getbaseinfo(argv[0]);

	setuphandlers();

	background(); 

	for (;;) {
		updateinfo();

		/* wait up to 2 seconds for a message from the upsd */
		if (getupsmsg(2))
			upslogx(LOG_INFO, "Received a message from upsd\n");
	}
}
