/* belkin.c - model specific routines for Belkin Smart-UPS units.

   Copyright (C) 2000 Marcus Mller <marcus@ebootis.de>
     
   based on:

   apcsmart.c - model specific routines for APC smart protocol units

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "main.h"
#include "belkin.h"

int init_communication(void)
{
	int	i;
	int	res;
	char	temp[SMALLBUF];

	res = -1;
	for (i = 1; i <= 10 && res == -1; i++) {
		send_belkin_command(STATUS,MANUFACTURER,"");
    		res = get_belkin_reply(temp);
	}

	if (res == -1 || strcmp(temp,"BELKIN")) 
		return res;

	return 0;
}

void do_status(void)
{
	char	temp[SMALLBUF], itemp[SMALLBUF], st[SMALLBUF];
	int	res;

	send_belkin_command(STATUS,STAT_STATUS,"");
	res = get_belkin_reply(temp);
	if (res == -1) 
		return;

	strcpy(itemp, "");
	get_belkin_field(temp, st, 6);
	if (*st == '1')
		strlcat(itemp, "OFF ", sizeof(itemp));

	get_belkin_field(temp, st, 2);
	if (*st == '1') {
		strlcat(itemp, "OB ", sizeof(itemp));

		send_belkin_command(STATUS,STAT_BATTERY,"");
		res = get_belkin_reply(temp);
		if (res == -1)
			return;

		get_belkin_field(temp, st, 10);
		res = atoi(st);
		get_belkin_field(temp, st, 2);

		if (*st == '1' || res < LOW_BAT) 
			strlcat(itemp, "LB ", sizeof(itemp)); /* low battery */
	}
	else
		strlcat(itemp, "OL ", sizeof(itemp));	/* on line (*/

	if (itemp[strlen(itemp)-1] == ' ') 
		itemp[strlen(itemp)-1] = 0;

	setinfo(INFO_STATUS, "%s", itemp);
}

int init_ups_data(void)
{
	int	res;
	double	low, high;
	char	temp[SMALLBUF], st[SMALLBUF];

	send_belkin_command(STATUS, MODEL, "");
	res = get_belkin_reply(temp);
	if (res == -1)
		return res;

	addinfo(INFO_MODEL, temp, 0, 0);

	send_belkin_command(STATUS, VERSION, "");
	res = get_belkin_reply(temp);
	if (res == -1)
		return (res);

	addinfo(INFO_FIRMREV, temp, 0, 0);

	send_belkin_command(STATUS, RATING, "");
	res = get_belkin_reply(temp);

	get_belkin_field(temp, st, 8);
	low = atof(st) / 0.88;

	get_belkin_field(temp, st, 9);
	high = atof(st) * 0.88;

	tcflush(upsfd,TCIOFLUSH);

	addinfo(INFO_STATUS, "", 0, 0);
	addinfo(INFO_UTILITY, "", 0, 0);
	addinfo(INFO_BATTPCT, "", 0, 0);
	addinfo(INFO_ACFREQ, "", 0, 0);
	addinfo(INFO_UPSTEMP, "", 0, 0);
	addinfo(INFO_LOADPCT, "", 0, 0);
	addinfo(INFO_BATTVOLT, "", 0, 0);
	addinfo(INFO_OUTVOLT, "", 0, 0);

	snprintf(st, sizeof(st), "%03.1f", low);
	addinfo(INFO_LOWXFER, st, 0, 0);

	snprintf(st, sizeof(st), "%03.1f", high);
	addinfo(INFO_HIGHXFER, st, 0, 0);

	addinfo(INFO_INSTCMD, "", 0, CMD_OFF);
	addinfo(INFO_INSTCMD, "", 0, CMD_ON);
	upsdrv_updateinfo();
	return 0;
}

/* normal idle loop - keep up with the current state of the UPS */
void upsdrv_updateinfo(void)
{
	int	res;
	double	val;
	char	temp[SMALLBUF], st[SMALLBUF];

	do_status();

	send_belkin_command(STATUS, STAT_INPUT, "");
	res = get_belkin_reply(temp);
	if (res == -1)
		return;

	get_belkin_field(temp, st, 3);
	val = atof(st) / 10;
	setinfo(INFO_UTILITY, "%05.1f", val);

	send_belkin_command(STATUS,STAT_BATTERY, "");
	res = get_belkin_reply(temp);
	if (res == -1)
		return;

	get_belkin_field(temp, st, 10);
	val = atof(st);
	setinfo(INFO_BATTPCT, "%03.0f", val);

	get_belkin_field(temp, st, 7);
	val = atof(st) / 10;
	setinfo(INFO_BATTVOLT, "%4.1f", val);

	get_belkin_field(temp, st, 9);
	val = atof(st);
	setinfo(INFO_UPSTEMP, "%03.0f", val);

	send_belkin_command(STATUS, STAT_OUTPUT, "");
	res = get_belkin_reply(temp);
	if (res == -1)
		return;

	get_belkin_field(temp, st, 2);
	val = atof(st) / 10;
	setinfo(INFO_ACFREQ, "%.1f", val);

	get_belkin_field(temp, st, 4);
	val = atof(st) / 10;
	setinfo(INFO_OUTVOLT, "%05.1f", val);

	get_belkin_field(temp, st, 7);
	val = atof(st);
	setinfo(INFO_LOADPCT, "%03.0f", val);

	writeinfo();
}


void get_belkin_field(char *temp, char *data, int n)
{
	char	st[SMALLBUF];
	int	i = 0, f = 1;

	strlcpy(st, temp, sizeof(st));

	while (f < n) {
		while (st[i] && st[i] != ';') 
			i++;

		st[i++] = 0;
		f++;
	}

	f = i;

	while (st[i] && st[i] != ';') 
		i++;

	st[i] = 0;

	strcpy(data, st+f);
}

int get_belkin_reply(char *buf)
{
	int	res, cnt;

	res = recvbinary(buf, 7);

	if (res)
		return res;

	buf[7] = 0;
	cnt = atoi(buf + 4);
	res = recvbinary(buf, cnt);
	buf[cnt] = 0;

	return res;
}

void send_belkin_command(char command, const char *subcommand, const char *data)
{
	upssend("~00%c%03d%s%s", command, strlen(data) + 3, subcommand, data);
}

/* power down the attached load immediately */
void upsdrv_shutdown(void)
{
	int	res;
	char	temp[SMALLBUF], st[SMALLBUF];

	res = init_communication();
	if (res == -1) {
		printf("Detection failed.  Trying a shutdown command anyway.\n");
		send_belkin_command(CONTROL, POWER_OFF, "1;1");
		exit(-1);
	}

	send_belkin_command(STATUS, STAT_STATUS, "");
	res = get_belkin_reply(temp);
	get_belkin_field(temp, st, 2);

        if (*st == '1') {
		printf("On battery - sending shutdown comand...\n");
		send_belkin_command(CONTROL,POWER_OFF,"1;1");
		return;
	}

	printf("Power restored - sending shutdown+return command...\n");

	/* turn off the outlet in 5 seconds */
	send_belkin_command(CONTROL, POWER_OFF, "1;5");

	/* ... and turn it back on 10 seconds after that */ 
	send_belkin_command(CONTROL, POWER_ON, "1;15");
}

/* handle the CMD_OFF with some paranoia */
void do_off(void)
{
	static	time_t lastcmd = 0;
	time_t	now, elapsed;
#ifdef CONFIRM_DANGEROUS_COMMANDS
	time(&now);
	elapsed = now - lastcmd;

	/* reset the timer every call - this means if you call it too      *
	 * early, then you have to wait MINCMDTIME again before sending #2 */
	lastcmd = now;

	if ((elapsed < MINCMDTIME) || (elapsed > MAXCMDTIME)) {

		/* FUTURE: tell the user (via upsd) to try it again */
		/* msgreply(UPSMSG_REPAGAIN); */
		return;
	}
#endif

	upslogx(LOG_INFO, "Sending powerdown command to UPS\n");
        send_belkin_command(CONTROL,POWER_OFF,"1;1");
	usleep(1500000);
        send_belkin_command(CONTROL,POWER_OFF,"1;1");
}

void instcmd(int auxcmd, int dlen, char *data)
{
	switch(auxcmd) {
		case CMD_OFF:		/* power off the load */
			do_off();
			break;
		case CMD_ON:
		        send_belkin_command(CONTROL,POWER_ON,"1;1");
			break;
		default:
			upslogx(LOG_INFO, "instcmd: unknown type 0x%04x\n",
				auxcmd);
	}
}

/* install pointers to functions for msg handlers called from msgparse */
void setuphandlers(void)
{
	upsh.instcmd = instcmd;
}

int send_string(char *data)
{
	tcflush(upsfd, TCIFLUSH);
	return(write(upsfd, data, strlen(data)));
}

int recvbinary(char *buf, int buflen)
{
	unsigned char	in;
	int	ret, counter = 0, retval = 0;
	struct	sigaction sa;
	sigset_t	sigmask;
 
	sa.sa_handler = timeout;
	sigemptyset(&sigmask);
	sa.sa_mask = sigmask;
	sa.sa_flags = 0;
	sigaction(SIGALRM, &sa, NULL);

	alarm(3);

	while (counter < buflen) {
		ret = read(upsfd, &in, 1);

		if (ret > 0) {
			buf[counter] = in;
			counter++;
			nolongertimeout();	   
		}
		else {
			upslogx(LOG_DEBUG, "error reading from serial device!");
			retval = -1;
			break;
		}
	}
	
	alarm(0);
	signal(SIGALRM, SIG_IGN);

	return retval;
}

void set_serialDTR0RTS1(void)
{
	int dtr_bit = TIOCM_DTR;
	int rts_bit = TIOCM_RTS;

	/* set DTR to low and RTS to high */
	ioctl(upsfd, TIOCMBIC, &dtr_bit);
	ioctl(upsfd, TIOCMBIS, &rts_bit);
}

void upsdrv_banner(void)
{
	printf("Network UPS Tools - Belkin Smart protocol driver 0.10 (%s)\n", UPS_VERSION);
}

void upsdrv_help(void)
{
}

void upsdrv_makevartable(void)
{
}

/* prep the serial port */
void upsdrv_initups(void)
{
	open_serial(device_path, B2400);
	
	set_serialDTR0RTS1();

	sleep(1);
  	tcflush(upsfd,TCIOFLUSH);
}

void upsdrv_initinfo(void)
{
	int	res;

	res = init_communication();
	if (res == -1) {
		printf("Unable to detect an Belkin Smart protocol UPS on port %s\n", 
			device_path);
		printf("Check the cabling, port name or model name and try again\n");
		exit(1);
	}

	/* manufacturer ID - hardcoded in this particular module */
	addinfo(INFO_MFR, "BELKIN", 0, 0);

	/* see what's out there */
	init_ups_data();

	printf("Detected %s on %s\n", getdata(INFO_MODEL), 
		device_path);

	setuphandlers();
}

/* tell main how many entries we need */
int upsdrv_infomax(void)
{
	return 45;
}
