// file kernel/n/x86-64/add.S: addition/subtraction of natural integers
/*-----------------------------------------------------------------------+
 |  Copyright 2005-2006, Michel Quercia (michel.quercia@prepas.org)      |
 |                                                                       |
 |  This file is part of Numerix. Numerix is free software; you can      |
 |  redistribute it and/or modify it under the terms of the GNU Lesser   |
 |  General Public License as published by the Free Software Foundation; |
 |  either version 2.1 of the License, or (at your option) any later     |
 |  version.                                                             |
 |                                                                       |
 |  The Numerix Library is distributed in the hope that it will be       |
 |  useful, but WITHOUT ANY WARRANTY; without even the implied warranty  |
 |  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU  |
 |  Lesser General Public License for more details.                      |
 |                                                                       |
 |  You should have received a copy of the GNU Lesser General Public     |
 |  License along with the GNU MP Library; see the file COPYING. If not, |
 |  write to the Free Software Foundation, Inc., 59 Temple Place -       |
 |  Suite 330, Boston, MA 02111-1307, USA.                               |
 +-----------------------------------------------------------------------+
 |                                                                       |
 |                         Addition/soustraction                         |
 |                                                                       |
 +-----------------------------------------------------------------------*/


               
                               # +------------+
                               # |  Addition  |
                               # +------------+

# sn_fadd
# -------
# entre :
#   a = naturel de longueur la     rsi = &a, rdx = la
#   b = naturel de longueur lb     rbx = &b, rcx = lb
#   c = naturel de longueur la     rdi = &c, peut tre confondu avec a ou b
# contraintes : la >= lb > 0
#
# sortie :
#   c <- a+b
#
# registres modifis :
#   rsi = &a[la],  rbx = &b[lb],   rdi = &c[la]
#   rcx = 0
#   rax,rdx,rbp,r8 = ind.
#   CF  = retenue sortante
#
# sn_finc
# -------
# entre :
#   a = naturel de longueur la     rsi = &a, rdx = la
#   b = naturel de longueur lb     rbx = &b, rcx = lb
# contraintes : la >= lb > 0
#
# sortie :
#   a <- a+b
#
# registres modifis :
#   rsi = &a[lb],  rbx = &b[lb]
#   rcx = 0
#   rax,rdx,rbp,rdi,r8 = ind.
#   CF  = retenue sortante
#
# sn_fadd_1
# ---------
# entre :
#   a = naturel de longueur l      rsi = &a, rcx = l
#   b = naturel de longueur l      rbx = &b
#   c = naturel de longueur l      rdi = &c, peut tre confondu avec a ou b
# contraintes : l > 0
#
# sortie :
#   c <- a+b
#
# registres modifis :
#   rsi = &a[l],  rbx = &b[l],   rdi = &c[l]
#   rcx = 0
#   rax,rdx,rbp = ind.
#   CF  = retenue sortante
#
# remarque :
#  sn_fadd_1 peut tre utilis pour diviser par 1 - BASE^p si p >= 2
        
#undef L                
#define L(x) .Lsn_fadd_##x

        ALIGN(32)
.Lsn_fadd:

        subq   %rcx,    %rdx            # rdx <- la - lb
        jz     .Lsn_fadd_1
        movq   %rdx, %r8                # sauve la - lb
        call   .Lsn_fadd_1              # additionne les parties communes
        movq   %r8,     %rcx            # rcx <- la - lb
        not    %r8
        incq   %r8                      # r8 <- lb - la
        cld;   rep movsq                # copie la fin de a
1:
        adcq   %rcx,    (%rdi,%r8,8)    # propage la retenue
        jnc    2f
        incq   %r8
        jne    1b
2:
        ret

        ALIGN(32)
.Lsn_finc:

	movq   %rsi,    %rdi            # rdi <- &a
        subq   %rcx,    %rdx            # rdx <- la - lb
        jz     .Lsn_fadd_1
        movq   %rdx, %r8                # sauve la - lb
        call   .Lsn_fadd_1              # additionne les parties communes
	jnc    2f
1:
	adcq   %rcx, (%rdi)             # propage la retenue
	jnc    2f
	leaq   8(%rdi), %rdi
	decq   %r8
	jnz    1b
2:
	ret

        ALIGN(32)
.Lsn_fadd_1:
        
        movq   (%rsi),  %rax            # rax <- a[0]
        leaq   (%rsi,%rcx,8), %rsi      # rsi <- &a[l]
        leaq   (%rbx,%rcx,8), %rbx      # rbx <- &b[l]
        leaq   (%rdi,%rcx,8), %rdi      # rdi <- &c[l]

        # calcule l adresse de saut dans la boucle
        decq   %rcx
        negq   %rcx
        jz     L(last)
        leaq   L(begin)(%rip), %rbp
        movq   %rcx,    %rdx
        andq   $15,     %rdx            # rdx <- (1-l) mod 16
	subq   %rdx,    %rbp            # rbp <- L(begin) + 15*rdx
	salq   $4,      %rdx
	addq   %rdx,    %rbp
        movq   %rax,    %rdx
        andq   $-16,    %rcx            # rcx <- 16*((1-l)/16)
        jmp    *%rbp
        
        # corps de boucle  drouler. taille du code = 15 octets
        # entrer avec rax = rdx = 1er chiffre de a, CF = 0
	# on code les instructions en hexadcimal pour tre sr qu'elles
	# totaliseront 15 octets pour un chiffre (les versions rcentes
	# de GAS liminent les dplacement nuls ce qui racourcit le code
	# associ au premier chiffre).
#undef BODY
#if 0
#define BODY(x,y,z) \
        adcq  x(%rbx,%rcx,8), %rax ;\
        movq  y(%rsi,%rcx,8), %rdx ;\
        movq  %rax, x(%rdi,%rcx,8) ;\
	                           ;\
        adcq  y(%rbx,%rcx,8), %rdx ;\
        movq  z(%rsi,%rcx,8), %rax ;\
        movq  %rdx, y(%rdi,%rcx,8)
#else
#define BODY(x,y,z) \
	.byte 0x48,0x13,0x44,0xcb, x ;\
	.byte 0x48,0x8b,0x54,0xce, y ;\
	.byte 0x48,0x89,0x44,0xcf, x ;\
	                             ;\
	.byte 0x48,0x13,0x54,0xcb, y ;\
	.byte 0x48,0x8b,0x44,0xce, z ;\
	.byte 0x48,0x89,0x54,0xcf, y
#endif

        # boucle d addition droule pour 16 chiffres
        ALIGN(8)
L(begin):
        BODY(-8,0,8);	BODY(8,16,24);	BODY(24,32,40);  BODY(40,48,56)
        BODY(56,64,72); BODY(72,80,88); BODY(88,96,104); BODY(104,112,120)

        leaq   15(%rcx), %rcx
        incq   %rcx
        jne    L(begin)
L(last):
        adcq   -8(%rbx), %rax           # termine la dernire addition
        movq   %rax,    -8(%rdi)
        ret
        
#  chiffre xn(add)(chiffre *a, long la, chiffre *b, long lb, chiffre *c)
# 
#  entre :
#  a = naturel de longueur la
#  b = naturel de longueur lb <= la
#  c = naturel de longueur la
# 
#  sortie :
#  c <- a + b
#  retourne la retenue

#ifdef assembly_sn_add
#undef L
#define L(x) .Lsn_add_##x

ENTER(sn_add)

        movq   %rdx,    %rbx            # rbx <- &b  
        movq   %rsi,    %rdx            # rdx <- la 
        movq   %rdi,    %rsi            # rsi <- &a  
        movq   %r8,     %rdi            # rdi <- &c
        jrcxz  L(b_nul)                 # si b = 0, recopie a
        call   .Lsn_fadd                # sinon, effectue l addition
        setc   %al                      # rax <- retenue
        movzx  %al,     %rax
        RETURN_WITH_SP

L(b_nul):
        movq   %rdx,    %rcx            # c <- a
        jrcxz  L(a_nul)
        cld;   rep movsq
L(a_nul):
        xorq   %rax,    %rax            # rax <- 0 (retenue)
        RETURN_WITH_SP

#endif /* assembly_sn_add */

#  chiffre xn(inc)(chiffre *a, long la, chiffre *b, long lb)
#  entre :
#  a = naturel de longueur la
#  b = naturel de longueur lb avec lb <= la
#
#  sortie :
#  a <- a + b
#  retourne la retenue

#ifdef assembly_sn_inc
#undef L
#define L(x) .Lsn_inc_##x

ENTER(sn_inc)

	testq  %rcx,    %rcx            # lb = 0 ?
	jz     1f
	movq   %rdx,    %rbx            # rbx <- &b
	movq   %rsi,    %rdx            # rdx <- la
	movq   %rdi,    %rsi            # rsi <- &a
	call   .Lsn_finc                # effectue l incrmentation
1:
        setc   %al                      # rax <- retenue
        movzx  %al,     %rax
        RETURN_WITH_SP
	
#endif /* assembly_sn_inc */
	
             
                             # +----------------+
                             # |  Soustraction  |
                             # +----------------+

# sn_fsub
# -------
# entre :
#   a = naturel de longueur la     rsi = &a, rdx = la
#   b = naturel de longueur lb     rbx = &b, rcx = lb
#   c = naturel de longueur la     rdi = &c, peut tre confondu avec a ou b
# contraintes : la >= lb > 0
#
# sortie :
#   c <- a-b
#
# registres modifis :
#   rsi = &a[la],  rbx = &b[lb],   rdi = &c[la]
#   rcx = 0
#   rax,rdx,rbp,r8 = ind.
#   CF  = retenue sortante
#
# sn_fdec
# -------
# entre :
#   a = naturel de longueur la     rsi = &a, rdx = la
#   b = naturel de longueur lb     rbx = &b, rcx = lb
# contraintes : la >= lb > 0
#
# sortie :
#   a <- a-b
#
# registres modifis :
#   rsi = &a[la],  rbx = &b[lb]
#   rcx = 0
#   rax,rdx,rbp,rdi,r8 = ind.
#   CF  = retenue sortante
#
# sn_fsub_1
# ---------
# entre :
#   a = naturel de longueur l      rsi = &a, rcx = l
#   b = naturel de longueur l      rbx = &b
#   c = naturel de longueur l      rdi = &c, peut tre confondu avec a ou b
# contraintes : l > 0
#
# sortie :
#   c <- a-b
#
# registres modifis :
#   rsi = &a[l],  rbx = &b[l],   rdi = &c[l]
#   rcx = 0
#   rax,rdx,rbp = ind.
#   CF  = retenue sortante
        
#undef L                
#define L(x) .Lsn_fsub_##x

        ALIGN(32)
.Lsn_fsub:
     
        subq   %rcx,    %rdx            # rdx <- la - lb
        jz     .Lsn_fsub_1
        movq   %rdx, %r8                # sauve la - lb
        call   .Lsn_fsub_1              # soustrait les parties communes
        movq   %r8,     %rcx            # rcx <- la - lb
        not    %r8
        incq   %r8                      # r8 <- lb - la
        cld;   rep movsq                # copie la fin de a
1:
        sbbq   %rcx,    (%rdi,%r8,8)    # propage la retenue
        jnb    2f
        incq   %r8
        jne    1b
2:
        ret

        ALIGN(32)
.Lsn_fdec:

	movq   %rsi,    %rdi            # rdi <- &a
        subq   %rcx,    %rdx            # rdx <- la - lb
        jz     .Lsn_fsub_1
        movq   %rdx, %r8                # sauve la - lb
        call   .Lsn_fsub_1              # soustrait les parties communes
	jnb    2f
1:
	sbbq   %rcx,   (%rdi)           # propage la retenue
	jnb    2f
	leaq   8(%rdi), %rdi
	decq   %r8
	jnz    1b
2:
	ret


        ALIGN(32)
.Lsn_fsub_1:
        
        movq   (%rsi),  %rax            # rax <- a[0]
        leaq   (%rsi,%rcx,8), %rsi      # rsi <- &a[l]
        leaq   (%rbx,%rcx,8), %rbx      # rbx <- &b[l]
        leaq   (%rdi,%rcx,8), %rdi      # rdi <- &c[l]

        # calcule l adresse de saut dans la boucle
        decq   %rcx
        negq   %rcx
        jz     L(last)
        leaq   L(begin)(%rip), %rbp
        movq   %rcx,    %rdx
        andq   $15,     %rdx            # rdx <- (1-l) mod 16
	subq   %rdx,    %rbp            # rbp <- L(begin) + 15*rdx
	salq   $4,      %rdx
	addq   %rdx,    %rbp
        movq   %rax,    %rdx
        andq   $-16,    %rcx            # rcx <- 16*((1-l)/16)
        jmp    *%rbp
        
        # corps de boucle  drouler. taille du code = 15 octets
        # entrer avec rax = rdx = 1er chiffre de a, CF = 0
	# on code les instructions en hexadcimal pour tre sr qu'elles
	# totaliseront 15 octets pour un chiffre (les versions rcentes
	# de GAS liminent les dplacement nuls ce qui racourcit le code
	# associ au premier chiffre).
#undef BODY
#if 0
#define BODY(x,y,z) \
        sbbq  x(%rbx,%rcx,8), %rax ;\
        movq  y(%rsi,%rcx,8), %rdx ;\
        movq  %rax, x(%rdi,%rcx,8) ;\
	                           ;\
        sbbq  y(%rbx,%rcx,8), %rdx ;\
        movq  z(%rsi,%rcx,8), %rax ;\
        movq  %rdx, y(%rdi,%rcx,8)
#else
#define BODY(x,y,z) \
	.byte 0x48,0x1b,0x44,0xcb, x ;\
	.byte 0x48,0x8b,0x54,0xce, y ;\
	.byte 0x48,0x89,0x44,0xcf, x ;\
	                             ;\
	.byte 0x48,0x1b,0x54,0xcb, y ;\
	.byte 0x48,0x8b,0x44,0xce, z ;\
	.byte 0x48,0x89,0x54,0xcf, y
#endif

        # boucle de soustraction droule pour 16 chiffres
        ALIGN(8)
L(begin):
        BODY(-8,0,8);	BODY(8,16,24);	BODY(24,32,40);  BODY(40,48,56)
        BODY(56,64,72); BODY(72,80,88); BODY(88,96,104); BODY(104,112,120)

        leaq   15(%rcx), %rcx
        incq   %rcx
        jne    L(begin)
L(last):
        sbbq   -8(%rbx), %rax           # termine la dernire addition
        movq   %rax,    -8(%rdi)
        ret
        
#  chiffre xn(sub)(chiffre *a, long la, chiffre *b, long lb, chiffre *c)
# 
#  entre :
#  a = naturel de longueur la
#  b = naturel de longueur lb <= la
#  c = naturel de longueur la
# 
#  sortie :
#  c <- a - b
#  retourne la retenue

#ifdef assembly_sn_sub
#undef L
#define L(x) .Lsn_sub_##x

ENTER(sn_sub)

        movq   %rdx,    %rbx            # rbx <- &b  
        movq   %rsi,    %rdx            # rdx <- la 
        movq   %rdi,    %rsi            # rsi <- &a  
        movq   %r8,     %rdi            # rdi <- &c
        jrcxz  L(b_nul)                 # si b = 0, recopie a
        call   .Lsn_fsub                # sinon, effectue la soustraction
        setc   %al                      # rax <- retenue
        movzx  %al,     %rax
        RETURN_WITH_SP

L(b_nul):
        movq   %rdx,    %rcx            # c <- a
        jrcxz  L(a_nul)
        cld;   rep movsq
L(a_nul):
        xorq   %rax,    %rax            # rax <- 0 (retenue)
        RETURN_WITH_SP

#endif /* assembly_sn_sub */

#  chiffre xn(dec)(chiffre *a, long la, chiffre *b, long lb)
#  entre :
#  a = naturel de longueur la
#  b = naturel de longueur lb avec lb <= la
#
#  sortie :
#  a <- a - b
#  retourne la retenue

#ifdef assembly_sn_dec
#undef L
#define L(x) .Lsn_dec_##x

ENTER(sn_dec)

	testq  %rcx,    %rcx            # lb = 0 ?
	jz     1f
	movq   %rdx,    %rbx            # rbx <- &b
	movq   %rsi,    %rdx            # rdx <- la
	movq   %rdi,    %rsi            # rsi <- &a
	call   .Lsn_fdec                # effectue la dcrmentation
1:
        setc   %al                      # rax <- retenue
        movzx  %al,     %rax
        RETURN_WITH_SP
	
#endif /* assembly_sn_dec */
	
# sn_fasub
# -------
# entre :
#   a = naturel de longueur la     rsi = &a, rdx = la
#   b = naturel de longueur lb     rbx = &b, rcx = lb
#   c = naturel de longueur la     rdi = &c, peut tre confondu avec a ou b
# contraintes : la >= lb > 0
#
# sortie :
#   c <- |a-b|
#
# registres modifis :
#   rcx = 0
#   rax,rdx,rsi,rbx,rdi,rbp = ind.
#   CF  = 0 si a >= b, 1 si a < b

#undef L                
#define L(x) .Lsn_fasub_##x

        ALIGN(32)
.Lsn_fasub:
        
        cmpq  %rcx,     %rdx
        je    L(compare)
        
        # si la > lb, recopie la fin de a dans c en testant s il y a
        # un chiffre non nul
        xorq   %rbp,    %rbp
        ALIGN(8)
1:
        movq  -8(%rsi,%rdx,8), %rax
        orq   %rax,     %rbp
        movq  %rax,     -8(%rdi,%rdx,8)
        decq  %rdx
        cmpq  %rdx,     %rcx
        jne   1b
        testq %rbp,     %rbp            # pfort(a) = 0 ?
        jz    L(compare)

        # ici la > lb et pfort(a) > 0, donc a > b
        call  .Lsn_fsub_1               # soustrait les chiffres communs
1:
        sbbq  %rcx,     (%rdi)          # propage la retenue
        leaq  8(%rdi),  %rdi
        jb    1b
        ret

        # si la = lb, soustrait les chiffres de poids fort jusqu 
        # trouver une diffrence
        ALIGN(8)
L(compare):
        movq  -8(%rsi,%rcx,8), %rax
        subq  -8(%rbx,%rcx,8), %rax
        ja    .Lsn_fsub_1
        jb    1f
        movq  %rax, -8(%rdi,%rcx,8)
        loop  L(compare)
        ret
        ALIGN(8)

        # ici la = lb et a[al-1] < b[la-1], donc a < b
1:
        xchgq  %rsi,    %rbx
        call   .Lsn_fsub_1              # c <- b - a
        stc                             # retour avec CF = 1
        ret


