/*
 * -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
 * command.c - a very rudimental command loop engine.
 *
 * Luca Deri     <deri@ntop.org>
 * Rocco Carbone <rocco@ntop.org>
 * -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


/*
 * ntop header file(s)
 */
#include "ntop.h"

#include "intop.h"

#if defined(HAVE_READLINE)
# define HISTORY_FILE  "intop_history"
#endif /* HAVE_READLINE */


/*
 * Function and Pointer To Function returning integer
 */
#if !defined(PTFRINULL)
typedef int Fri (int argc, char * argv []);
typedef Fri * Ptfri;
# define PTFRINULL (Ptfri) NULL
#endif /* ! PTFRINULL */


/*
 * The structure which contains information
 * on the commands the application can understand.
 */
typedef struct
{
  char * name;        /* user printable name of the command         */
  int implemented;    /* true if command is implemented             */
  char * help;        /* help string                                */
  char * doc;         /* documentation for the command              */
  int theBell;           /* flag to ring bell after command execution  */
  Fri * handler;      /* function to call to execute the command    */
  int retcode;        /* return code                                */
} Command;


#define NULLCMD      ((Command *) 0)
#define AMBIGUOUSCMD ((Command *) -1)


/*
 * Builtin
 */
static int intop_help         (int argc, char * argv []);
static int intop_quit         (int argc, char * argv []);

/*
 * C functions elsewhere defined in differents modules
 */
#if defined(HAVE_READLINE)
int     history (int argc, char * argv []);
void    initialize_gnu_readline (char * prompt);
int     read_history (char * filename);
int     write_history (char * filename);
char  * readline (char * prompt);
void    add_history (char * string);
#endif /* HAVE_READLINE */


/*
 * C functions defined in this module and referenced by differents modules
 */
char  * command_by_index (int i);
void    commandloop (int fromatty, char * progname);


/*
 * Placeholder for ideas rather than available commands.
 */
static int not_yet_available (int argc, char * argv [])
{
  printf ("%s: command not yet implemented!\n", argv [0]);
  fflush (stdout);
  return (-1);
}


/*
 * The table of all commands.
 */
static Command table [] =
{
  /*
   * Section with commands all interactive program should have.
   */

  { "?",        1, "No help yet", "No documentation yet", 1, intop_help, 0 },
  { "help",     1, "No help yet", "No documentation yet", 1, intop_help, 0 },
  { "exit",     1, "No help yet", "No documentation yet", 1, intop_quit, 0 },
  { "quit",     1, "No help yet", "No documentation yet", 1, intop_quit, 0 },
  { "prompt",   1, "No help yet", "No documentation yet", 1, intop_prompt, 0 },
  { "warranty", 0, "No help yet", "No documentation yet", 1, not_yet_available, 0 },
  { "copying",  0, "No help yet", "No documentation yet", 1, not_yet_available, 0 },

#if defined(HAVE_READLINE)
  { "history", 1, "No help yet", "No documentation yet", 1, history, 0 },
#endif /* HAVE_READLINE */


  /*
   * Section specific to this program begins here.
   */

  {
    "open", 1, "No help yet",
    "Open a network interface",
    1, intop_open, 0,
  },

  { "close", 0, "No help yet",
    "Close a network interface",
     1, intop_close, 0 },

  { "filter", 1, "No help yet",
    "Apply a filter to the a network interface handler",
     1, intop_set_filter, 0 },

  {
    "sniff", 1, "No help yet",
    "Start collecting and processing packets",
    1, intop_sniff, 0,
  },

  {
    "uptime", 1, "No help yet",
    "Tell how long the program has been running",
    0, intop_uptime, 0,
  },

  {
    "hash", 0, "No help yet",
    "Toggle printing `#' for each packet captured",
    0, intop_hash, 0,
  },

  {
    "info", 1, "No help yet",
    "Tell interface information",
    1, intop_info, 0,
  },

  {
    "swap", 1, "No help yet",
    "Switch to interface",
    1, intop_swap, 0,
  },

  {
    "top", 1, "No help yet",
    "Tell network usage information",
    0, intop_top, 0,
  },

  {
    "lsdev", 1, "No help yet",
    "List all available network interfaces",
    0, intop_lsdev, 0,
  },

  {
    "hosts", 1, "No help yet",
    "Query the ntop's HOST cache",
    1, intop_hosts, 0
  },

  {
    "arp", 1, "No help yet",
    "Query the ntop's ARP cache",
    1, intop_arp, 0
  },

  {
    "nbt", 1, "No help yet",
    "Query the ntop's NBT cache",
    1, intop_nbt, 0
  },

  {
    "dump", 0, "No help yet",
    "Enable packets header dump",
    1, intop_dump, 0,
  },


  { "last",     0, "No help yet", "No documentation yet", 1, not_yet_available, 0 },
  { "nslookup", 0, "No help yet", "No documentation yet", 1, not_yet_available, 0 },

  { 0 },  /* EOT (End Of Table) */
};


/*
 * Functions to implement the commands.
 */

static Command * command_table (void)
{
  return (table);
}


static int command_table_size (void)
{
  return (sizeof (table) / sizeof (table [0]));
}


static int command_max_length (void)
{
  Command * command;
  int current_command_length = 0;
  int max_command_length = 0;

  for (command = command_table ();
       command < command_table () + command_table_size ();
       command ++)
    {
      current_command_length = command -> name ? strlen (command -> name) : 0;
      if (max_command_length < current_command_length)
	max_command_length = current_command_length;
    }
  return (max_command_length);
}


/*
 * The `list help' command.
 */
static void list_help_commands (int argc, char * argv [])
{
  Command * command;
  int i, j, w, k;
  int theColumns, theWidth, theLines;

  /*
   * width is a multiple of 8
   *
   * it is increased of 2 to show commands
   * not yet implemented enclosed between <>
   */

#define INCREASED 2

  theWidth = (command_max_length () + INCREASED + 8) &~ 7;
  theColumns = 80 / theWidth;
  if (theColumns == 0)
    theColumns = 1;
  theLines = (command_table_size () + theColumns - 1) / theColumns;

  printf ("Commands enclosed in '<>' are not yet implemented.\n");
  printf ("Commands may be abbreviated. Commands are:\n\n");

  for (i = 0; i < theLines; i ++)
    {
      for (j = 0; j < theColumns; j ++)
	{
	  command = command_table () + j * theLines + i;
	  if (command -> name)
	    printf ("%c%s%c", command -> implemented ? ' ' : '<',
		    command -> name, command -> implemented ? ' ' : '>');
	  else
	    {
	      for (k = 0; k < /* strlen (command -> name) + */ INCREASED; k ++)
		(void) putchar (' ');
	    }
	  if (command + theLines >= command_table () + command_table_size ())
	    {
	      printf ("\n");
	      break;
	    }
	  w = strlen (command -> name) + INCREASED;
	  while (w < theWidth)
	    {
	      w = (w + 8) &~ 7;
	      (void) putchar ('\t');
	    }
	}
    }
}


char * command_by_index (int i)
{
  if (i < 0 || i > command_table_size ())
    return (NULL);
  return ((command_table () + i) -> name);
}


/*
 * Look up NAME as the name of a command, and return
 * a pointer to that command.
 * Return a NULL pointer if NAME isn't a command name.
 */
static Command * lookup (char * name)
{
  char * p;
  char * q;
  Command * command;
  Command * found;

  int nmatches = 0;
  int longest = 0;


  found = NULLCMD;
  for (command = command_table (); (p = command -> name); command ++)
    {
      for (q = name; * q == * p ++; q ++)
	if (* q == 0)		/* exact match ? */
	  return (command);
      if (! * q)
	{			/* the name was a prefix */
	  if ((q - name) > longest)
	    {
	      longest = q - name;
	      nmatches = 1;
	      found = command;
	    }
	  else if (q - name == longest)
	    nmatches ++;
	}
    }

  switch (nmatches)
    {
    case 0:
      return (NULLCMD);
    case 1:
      return ((Command *) -1);
      break;
    default:
      return (found);
    }
}


#define HELPINDENT 10

/*
 * The `help' function.
 */
static int intop_help (int argc, char * argv [])
{
  Command * command;

  if (argc == 1)
    {
      list_help_commands (argc, argv);
      return (0);
    }

  while (-- argc)
    {
      command = lookup (argv [argc]);
      if (command == (Command *) -1)
	{
	  printf ("Command (%s) : ambiguous help\n", argv [argc]);
	  continue;
	}
      else if (command == (Command *) 0)
	{
	  printf ("Command (%s) : invalid help\n", argv [argc]);
	  continue;
	}
      else
	printf ("%-*s\t%s\n", HELPINDENT, command -> name, command -> help);
    }
  return (0);
}


/*
 * The `quit' function.
 */
static int intop_quit (int argc, char * argv [])
{
  printf ("Thanks for using '%s'.\n", progname);
#if defined(HAVE_READLINE)
  write_history (HISTORY_FILE);
#endif /* HAVE_READLINE */
  exit (0);
}


/*
 * Advance a char pointer beyond current white space.
 * Function will return a NULL pointer upon reaching end of line.
 */
static char * skip_white_space (char * p)
{
  if (! p)
    return (NULL);
  while (* p && (* p == ' ' || * p == '\t' || * p == '\n'))
    p ++;
  if (* p == '\n' || * p == '\0')
    return (NULL);
  return (p);
}


/*
 * Change the prompt.
 */
void userprompt (char * interface)
{
  if (interface)
    {
      if (prompt)
	free (prompt);
      prompt = malloc (strlen (pcolor) + strlen (progname) +
		       strlen (interface) + strlen (INTOP_NORMAL) + 5);
      sprintf (prompt, "%s%s@%s>%s ", pcolor, progname, interface, INTOP_NORMAL);
    }
  else
    {
      prompt = malloc (strlen (pcolor) + strlen (progname) + strlen (INTOP_NORMAL) + 3);
      sprintf (prompt, "%s%s>%s ", pcolor, progname, INTOP_NORMAL);
    }
}


/*
 * Loop reading lines and executing commands until the user quits.
 */
void commandloop (int fromatty, char * progname)
{
  char * line = NULL;
  char * start;
  int argc;
  char ** argv;
  char ** q;
  Command * command;
#if !defined(HAVE_READLINE)
  char * room;
#endif /* HAVE_READLINE */

#if (0)
  extern FILE * rl_instream;
  int nready;
  fd_set readmask;
  struct timeval timeout;
  int time_out;
#endif


  /*
   * Need dynamic memory because of user prompt of variable length.
   */
  if (! prompt)
    {
      if (fromatty)
	userprompt (NULL);
      else
	prompt = strdup ("");
    }

#if defined(HAVE_READLINE)
  initialize_gnu_readline (prompt);
  read_history (HISTORY_FILE);
#endif /* HAVE_READLINE */

  (void) putchar ('\n');

#if (0)
  if (rl_instream)
    printf ("ROCCO %d\n", fileno (rl_instream));

  /*
   * I want stdin set to asynchronous (non-blocking) mode of operation
   */
  if (rl_instream)
    fcntl (fileno (rl_instream), F_SETFL, O_NONBLOCK);
#endif

  /*
   * forever do...
   */
  for (;;)
    {
      /*
       * If the buffer has already been allocated
       * return memory to the free pool
       */
      if (line)
	free (line);

      /*
       * Get a line from the user
       */

#if defined(HAVE_READLINE)
      line = readline (prompt);
#else
      room = line = malloc (1024);
      * line = '0';
      fputs (prompt, stdout);
      line = fgets (room, 1024 - 2, stdin);
      if (line)
	line [strlen (line) - 1] = '\0'; /* strip \n */
#endif /* HAVE_READLINE */


      if (! line)               /* EOF encountered at top level */
	{
	  printf ("^D disabled. Use \"quit\" to leave %s\n", progname);
	  continue;
	}

      /*
       * Look for empty lines
       */
      if (! * line)
	continue;

      start = skip_white_space (line);
      if (! start)
	continue;

#if defined(HAVE_READLINE)
      /*
       * If the line has any text in it, save it on the history.
       */
      add_history (start);
#endif /* HAVE_READLINE */

      argv = buildargv (start);
      if (! argv)
	{
	  printf ("Not enough memory !!!\n");
	  continue;
	}

      /*
       * Look for the command
       */
      command = lookup (argv [0]);
      if (command == AMBIGUOUSCMD)
	{
	  printf ("%s: command ambiguous\n", argv [0]);
	  (void) putchar ('\007');
	  freeargv (argv);
	  continue;
	}
      else if (command == NULLCMD)
	{
	  printf ("%s: command not found\n", argv [0]);
	  (void) putchar ('\007');
	  freeargv (argv);
	  continue;
	}

      if (! command -> implemented)
	{
	  printf ("%s: command not yet implemented\n", argv [0]);
	  (void) putchar ('\007');
	  freeargv (argv);
	  continue;
	}

      for (argc = 0, q = argv; (* q) && (* q [0]); q ++)
	argc ++;

      command -> retcode = (* command -> handler) (argc, argv);
      if (command -> theBell && command -> retcode)
	(void) putchar ('\007');
      freeargv (argv);
    }
  (void) putchar ('\n');
#if defined(HAVE_READLINE)
  write_history (HISTORY_FILE);
#endif /* HAVE_READLINE */
}
