/*
 * -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
 * arp.c - Tell the ntop ARP cache.
 *
 * Luca Deri     <deri@ntop.org>
 * Rocco Carbone <rocco@ntop.org>
 * -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


/*
 * ntop header file(s)
 */
#include "ntop.h"

#include "intop.h"

/*
 * It queries the ntop's ARP cache in various ways.
 * For debugging purposes, it also allows a complete dump
 * of the ntop's ARP cache.
 */
int intop_arp (int argc, char * argv [])
{
  /*
   * Notice the command name.
   */
  char * commandname = argv [0];

  int c;
#define USAGE(xxx) \
  printf ("Usage: %s [-h] [-i interface] [-v vendor] [-u] [-l] [-f filename] [hostname [hostname]]\n", xxx)
  char * optstring = "hi:v:ulf:";

  /*
   * Reserve here space for the local variables.
   */
  FILE * fd = stdout;
  char * interface = NULL;
  char * vendor = NULL;
  char * name = NULL;
  int unknown = 0;
  int ipless = 0;
  int current;
  HostTraffic * ht = NULL;
  int i;

  optind = 0;
  optarg = NULL;

  /*
   * Parse command line options to the application via standard system calls.
   */
  while ((c = getopt (argc, argv, optstring)) != -1)
    {
      switch (c)
	{
	case 'h':
	  USAGE (commandname);
	  return (0);

	default:
	  USAGE (commandname);
	  return (-1);

	case 'i':
	  interface = optarg;
	  break;

	case 'v':
	  if (unknown)
	    {
	      printf ("WARNING: -v is incompatible with -u\n");
	      USAGE (commandname);
	      return (-1);
	    }
	  vendor = optarg;
	  break;

	case 'u':
	  if (vendor)
	    {
	      printf ("WARNING: -u is incompatible with -v\n");
	      USAGE (commandname);
	      return (-1);
	    }
	  unknown = 1;
	  break;

	case 'l':
	  if (unknown)
	    {
	      printf ("WARNING: -l is incompatible with -u\n");
	      USAGE (commandname);
	      return (-1);
	    }
	  ipless = 1;
	  break;

	case 'f':
	  fd = fopen (optarg, "w");
	  if (! fd)
	    {
	      fprintf (stdout, "Cannot open %s. stdout will be used.\n", optarg);
	      fd = stdout;
	    }
	  break;
	}
    }

  /*
   * Safe to play with the 'active' interface (if any)
   * in case no specific one was chosen by the user.
   */
  if (! interface && ! (interface = intop_interface (active)))
    {
      printf ("No interface is currently enabled for packet sniffing.\n");
      return (-1);
    }

  /*
   * Lookup for the given name in the table of enabled interfaces.
   */
  current = intop_lookup_interface (interface);
  if (current == -1)
    {
      printf ("%s: unknown interface %s.\n", commandname, interface);
      return (-1);
    }

  /*
   * Avoid to print no available information.
   */
  if (get_intop_flags (current) != INTERFACE_ENABLED)
    {
      printf ("%s: no interface is currently enabled for packet sniffing.\n", commandname);
      return (-1);
    }


  if (fd != stdout)
    {
      fprintf (fd, "%s\n", intop_report (interface));
      fflush (fd);
    }

  /*
   * Go for fun!
   */

  if (! vendor && ! ipless && optind < argc)
    {
      fprintf (fd, "\n");
      fprintf (fd, "| MAC Address    | IP Address       | Hostname                 | Vendor\n");
      fprintf (fd, "+================+==================+==========================+==================\n");

      while (optind < argc)
	{
	  /*
	   * Print the ntop's ARP entry for the given name.
	   */
	  for (i = 1; i < device [current] . actualHashSize; i ++)
	    {
	      /*
	       * Skip unused entries.
	       */
	      if (! (ht = device [current] . hash_hostTraffic [i]))
		continue;

	      /*
	       * Skip IP-less entries.
	       */
	      if (! strcmp (ht -> ethAddressString, "0.0.0.0"))
		continue;

	      /*
	       * Look for symbolic IP address first.
	       */
	      if (ht -> hostSymIpAddress && * ht -> hostSymIpAddress &&
		  ! strcmp (ht -> hostSymIpAddress, argv [optind]))
		{
		  fprintf (fd, "| %14.14s | %-16.16s | %-24.24s | %s\n",
			   ht -> ethAddressString,
			   intoa (ht -> hostIpAddress),
			   ht -> hostSymIpAddress,
			   getVendorInfo (ht -> ethAddress, 0));
		  break;
		}
	      /*
	       * Then look for numeric IP address.
	       */
	      else if (ht -> hostNumIpAddress && * ht -> hostNumIpAddress &&
		  ! strcmp (ht -> hostNumIpAddress, argv [optind]))
		{
		  fprintf (fd, "| %14.14s | %-16.16s | %-24.24s | %s\n",
			   ht -> ethAddressString,
			   intoa (ht -> hostIpAddress),
			   ht -> hostSymIpAddress,
			   getVendorInfo (ht -> ethAddress, 0));
		  break;
		}
	    }
	  optind ++;
	}
    }
  else
    {
      /*
       * Print the ntop's ARP cache.
       */
      fprintf (fd, "\n");
      fprintf (fd, "| HW Address     | IP Address       | Hostname                 | Vendor\n");
      fprintf (fd, "+================+==================+==========================+==================\n");


      for (i = 1; i < device [current] . actualHashSize; i ++)
	{
	  name = NULL;

	  /*
	   * Skip unused entries.
	   */
	  if (! (ht = device [current] . hash_hostTraffic [i]))
	    continue;

	  /*
	   * Skip empty entries.
	   */
	  if (! * ht -> ethAddressString || ! strcmp (ht -> ethAddressString, "0.0.0.0") ||
	      ! strcmp (ht -> ethAddressString, "FF:FF:FF:FF:FF:FF"))
	    continue;

	  /*
	   * Skip IP-less entries (unless the user is really interested in IP-less addresses).
	   */
	  if (! * ht -> hostSymIpAddress || ! strcmp (ht -> hostSymIpAddress, "0.0.0.0") ||
	      ! strcmp (intoa (ht -> hostIpAddress), "0.0.0.0"))
	    {
	      if (! ipless)
		continue;
	    }
	  else if (ipless)
	    continue;

	  /*
	   * Skip unwanted vendors.
	   */
	  if (vendor)
	    {
	      name = getVendorInfo (ht -> ethAddress, 0);
	      if (! strcmp (name, "") ||
		  strncasecmp (vendor, name, min (strlen (vendor), strlen (name))))
		continue;
	    }

	  /*
	   * Skip unknown vendors.
	   */
	  if (unknown)
	    {
	      if (! strcmp (getVendorInfo (ht -> ethAddress, 0), ""))
		fprintf (fd, "| %14.14s | %-16.16s | %-24.24s | %s\n",
			 ht -> ethAddressString,
			 ipless ? "IP-less         " : intoa (ht -> hostIpAddress),
			 ht -> hostSymIpAddress,
			 name ? name : getVendorInfo (ht -> ethAddress, 0));
	    }
	  else
	    fprintf (fd, "| %14.14s | %-16.16s | %-24.24s | %s\n",
		     ht -> ethAddressString,
		     ipless ? "IP-less         " : intoa (ht -> hostIpAddress),
		     ht -> hostSymIpAddress,
		     name ? name : getVendorInfo (ht -> ethAddress, 0));
	}
    }


  fprintf (fd, "\n");
  fflush (fd);

  return (0);
}
