/*
 * Copyright (c) 1996 University College London
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Computer Science
 *      Department at University College London
 * 4. Neither the name of the University nor of the Department may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
   File: text_pane.c
   Author: mhandley
   Date: Mon Jul  5 1994
   Purpose: Functions for converting between internal format and Tk text format
 */

#include <tcl.h>

#include <stdlib.h>
#include <string.h>
#include "prototypes.h"

#define PAGE_EXTENSION 66u
extern Tcl_Interp *interp;
extern int rx_fd;

int ds_to_text(text, p)
page_text *text;
page *p;
{
    block *b;
    line *l;
    u_int16 lmax = 0;

    /*Find out the max line in document */
    b = p->first_block;
    while (b != NULL) {
	if (lmax < ((b->ypos) + (b->no_of_lines)))
	    lmax = (b->ypos) + (b->no_of_lines);
	b = b->next_block;
    }

    if (text->no_of_lines < lmax + PAGE_EXTENSION)
	text->data = (char *) realloc(text->data, (lmax + PAGE_EXTENSION) * MAX_LINE_LEN);

    b = p->first_block;

    while (b != NULL) {
	u_int8 bx;
	u_int16 by, lnum;
	bx = b->xpos;
	by = b->ypos;
	lnum = by;
	l = b->first_line;
	while (l != NULL) {
	    if (l->no_of_chars > 0) {
		strncpy(&text->data[(MAX_LINE_LEN * lnum) + bx],
			l->line_data,
			MAX_LINE_LEN - bx);
	    }
	    l = l->next_line;
	    lnum++;
	}
	b = b->next_block;
    }
    return 0;
}



line *find_line(b, y)
block *b;
int y;
{
    int i;
    line *l;
    if (b == NULL) {
	fprintf(stderr, "Trying to access null block2\n");
	return NULL;
    }
    l = b->first_line;
    for (i = 0; i < y; i++) {
	if (l == NULL) {
	    fprintf(stderr, "Trying to access null line\n");
	    if (b != NULL)
		fprintf(stderr, "Blockid=%s\n", b->blockid);
	    return NULL;
	}
	i += l->missing_lines;
	l = l->next_line;

    }
    return l;
}

u_int16 find_line_num(b, lineid)
block *b;
char *lineid;
{
    line *l;
    u_int16 lnum = 0;
    if (b == NULL) {
	fprintf(stderr, "Trying to access null block1\n");
	return (u_int16) 0;
    }
    l = b->first_line;
    if (l == NULL) {
	fprintf(stderr, "Trying to access null line\n");
	return (u_int16) 0;
    }
    while ((strncmp(l->lineid, lineid, ID_LEN) != 0) && (l != NULL)) {
	lnum++;
	lnum += l->missing_lines;
	l = l->next_line;
	if (l == NULL)
	    return (u_int16) 0;
    }
    if (strncmp(l->lineid, lineid, ID_LEN) == 0)
	return lnum;
    else
	return (u_int16) 0;
}


int renumber_block(b)		/* ***** Changed by Jim to stop it renumbering deleted lines and screwing */
block *b;			/*   them up .. *****  (deleted lines store their deleted status in their */
								/*   line number field...) */
{
    line *l;
    u_int16 lnum = 0;
    if (b == NULL) {		/* if no block */
	debug("Trying to access null block\n");
	return (u_int16) 0;
    }
    l = b->first_line;

    if (l == NULL) {		/* if no lines in block */
	debug("Trying to access a null line as first line of block.\n");
	return (u_int16) 0;
    }
    /* number from 0 to whatever's needed. */

    while (l != NULL) {		/* repeat for each line */
	lnum += l->missing_lines;

	if (l->linenum != LINE_DELETED) {	/* only renumber those lines that haven't been deleted */
	    l->linenum = lnum;	/* ok, so each line will now be numbered correctly taking */
	    /*   into account missing lines. */

	} else {
	    debug("I'm not changing the line number of a deleted line\n");


	}			/* end else */

	lnum++;			/* increment line number for the next line */
	l = l->next_line;
	if (l == NULL)
	    return 0;		/* if we have finished..  (THIS is DODGY coding style) */
    }
    return 0;
}


u_int8 find_line_len(b, y)
block *b;
int y;
{
    line *l;
    l = find_line(b, y);
    return l->no_of_chars;
}

u_int16 find_block_len(b)
block *b;
{
    return b->no_of_lines;
}

block *findblock(char *blockid, page * p, u_int8 x, u_int16 y)
{
    block *b;
/*printf("findblock %d %d\n", x, y); */
    b = p->last_block;
    while (b != NULL) {
/*    printf("blockid: %s, %d, %d, %d\n", b->blockid, b->xpos, b->ypos, (b->ypos+b->no_of_lines)); */
	if ((b->xpos <= x) &
	    (b->ypos <= y) &
	    ((b->ypos + b->no_of_lines) >= y)) {
	    /*possible match - need to check correct line */
	    line *l;
	    u_int16 i;
	    debug("provisional match\n");
	    l = b->first_line;
	    for (i = 0; i < (y - b->ypos); i++) {
		if (l != NULL)
		    l = l->next_line;
		else {
		    fprintf(stderr, "findblock: trying to move too far down block\n");
		    fprintf(stderr, "y=%d, i=%d", y, i);
		    strncpy(blockid, "blank", ID_LEN);
		    return NULL;
		}
	    }
	    if (l == NULL) {
		fprintf(stderr, "findblock: trying to move too far down block\n");
		fprintf(stderr, "y=%d, i=%d", y, i);
		strncpy(blockid, "blank", ID_LEN);
		return NULL;
	    }
	    if ((l->no_of_chars) >= (x - b->xpos)) {
		/*this is the one */
		strncpy(blockid, b->blockid, ID_LEN);
		return b;
	    }
	}
	debug("doesn't match\n");
	b = b->prev_block;
    }
    strncpy(blockid, "blank", ID_LEN);
    return NULL;
}

void redisplay_block(b)
block *b;
{

    int i;
    line *l;
    char command[MAX_LINE_LEN + 32];
    int code, lnum = 0;
    static char str[100000];	/*should redo this later *//* possible error? */
    if (b == NULL)
	return;

    if (b->first_line == NULL) {
	debug("redisplaying a block and have not found a first line..\nline no's :");
	return;
    }
    b->new_data = 0;

    str[0] = '\0';
    memset(str, 0, 10000);	/*definitely a hack *//* possible error? Perhaps change 10000->100000 */


    l = b->first_line;
    debug("redisplaying a block and have found first line\n");
    while (l != NULL) {
	debug("%d:", l->linenum);
	if (l->linenum != LINE_DELETED) {	/* if line hasn't been deleted */
	    if (l->missing_lines > 0) {		/* if there are missing lines -before- this line *//*   and after the line previously seen */
		for (i = 0; i < l->missing_lines; i++)
		    strcat(str, "**\n");	/* add a ** for each missing line */
		/*   before this one */
		lnum += l->missing_lines;	/* increment the number of lines in this */
		/*   block by the number of missing lines */
		/*   that we just made up for */
	    }
	    lnum++;		/* increment number of lines counter */
	    strcat(str, l->line_data);	/* add line to output string */

	    if ((l->next_line != NULL) && (l->next_line->linenum != LINE_DELETED))	/* if next line exists and isn't */
		strcat(str, "\n");	/* deleted, concatenate it to string. */
	} else {
	    /* line is deleted */
	}
	l = l->next_line;	/* now display the next line in the block */

    }
    debug("block has %d lines\n", b->no_of_lines);
    debug("i found %d lines\n", lnum);
    if (lnum < b->no_of_lines) {	/* if the number of lines that we found is smaller *//* than what's supposed to be in the block */
	strcat(str, "\n");	/* add a line feed first (as next pointer is */
	/*   a null pointer) */
	for (i = lnum; i < b->no_of_lines; i++)
	    strcat(str, "**\n");	/* add extra '**' lines for any lines we */
	/* found to be missing */
    }
    Tcl_SetVar(interp, "c_str", str, 0);
    sprintf(command, "set_block %s $c_str %d %d %d %d",
      b->blockid, b->xpos, b->ypos, b->face.fontindex, b->face.colindex);
    code = Tcl_Eval(interp, command);	/* display block on screen display */

    if (code != TCL_OK) {	/* if TCL is unhappy, do error messages */
	debug("%s\n", interp->result);
	debug("Orig request was [1]:\n--> %s\n", command);
    }
}

void delayed_redisplay_block(b)
block *b;
{
    struct timeval t;
    fd_set fdset;
    static block *prev_block = NULL;

    /* ** jim's hack ** */
    /* I'm sure that this is probably doing something quite clever but it doesn't seem to work */
    /* properly in that it doesn't display valid blocks - until I work out (or need to work out) */
    /* what the unhandled case is, I'll just comment it out and display the block immediately. */
    /* seems to be a problem with the pc version, instead of the unix version... */

    /*redisplay_block(b); */

    /* ** end jim's hack ** */

    /* I have replaced this hack with the original code because it was doing more good than bad */
    /*            - it prevented NTE from hanging (nte still slows down when receiving large amounts */
    /*                    of data though) */
    /*            - it wasted less CPU time on screen updates.. */
    /* Care must be taken that sometimes, changes to a line/block will be received but not displayed */
    /* In that case, editing the block/line usually triggers the re-display.. */


    if ((prev_block != b) && (prev_block != NULL)) {	/* tried two&& instead of 1 *//*  but no difference */
	debug("Displaying block right now..\n");
	redisplay_block(prev_block);
    } else {

	debug("not displaying right now - therefore, have already tried to display this block\n");
	debug("   - and it's previous block is null.");

    }				/* endif */
    FD_ZERO(&fdset);
    FD_SET(rx_fd, &fdset);
    t.tv_sec = 0;
    t.tv_usec = 0;
    select(rx_fd + 1, &fdset, (fd_set *) NULL, (fd_set *) NULL, &t);

    if (FD_ISSET(rx_fd, &fdset) != 0) {
	debug("Displaying block later...fd_isset=%d\n", FD_ISSET(rx_fd, &fdset));
	prev_block = b;
	return;
    } else {
	debug("Displaying block now ...\n");
	prev_block = NULL;
	redisplay_block(b);
    }

}
