/*

Copyright (C) 2000, 2001 Christian Kreibich <kreibich@aciri.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifndef __nd_packet_h
#define __nd_packet_h

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#ifdef LINUX
#define __FAVOR_BSD
#endif
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>
#include <net/ethernet.h>
#include <glib.h>
#include <pcap.h>

/* Protocol layers, see ND_Packet below */
typedef enum
{
  ND_NULL,
  ND_LINK,
  ND_NET,
  ND_TRANSP,
  ND_APP
}
ND_Layer;


/* Protocol types: */
typedef enum
{
  ND_PROT_NULL          = 0,
  ND_PROT_ETHER         = 1,
  ND_PROT_FDDI          = 2,
  ND_PROT_SLIP          = 3,
  ND_PROT_PPP           = 4,
  ND_PROT_ARP           = 5,
  ND_PROT_RARP          = 6,
  ND_PROT_IP            = 7,
  ND_PROT_TCP           = 8,
  ND_PROT_UDP           = 9,
  ND_PROT_ICMP          = 10,

  /* For IP headers in ICMP error packets,
     use this field. It's not set in the
     transp_data pointer in ND_Packet below.
  */
  ND_PROT_ICMP_ERROR_IP = 11
}
ND_Protocol;


/* Netdude's view of a packet: */

typedef struct nd_packet
{
  /* The pcap header */
  struct pcap_pkthdr    ph;

  /* Offsets at various layers into the packet */
  u_char               *link_data;
  u_char               *net_data;
  u_char               *transp_data;
  u_char               *app_data;

  /* What protocol is at a layer */
  ND_Protocol           link_prot;
  ND_Protocol           net_prot;
  ND_Protocol           transp_prot;
  ND_Protocol           app_prot;

  /* Whether the packet is currently hidden
     in the GUI display. */ 
  int                   is_hidden     : 1;

  /* Whether the packet is completely captured */
  int                   is_complete   : 1;

  /* Pointers to the next/previous selected
     packets, for quicker iteration. */
  struct nd_packet     *sel_next;
  struct nd_packet     *sel_prev;

  /* Linked list pointers. */
  struct nd_packet     *next;
  struct nd_packet     *prev;
}
ND_Packet;


/* Create, destroy, copy packets: */
ND_Packet      *nd_packet_new(void);
void            nd_packet_free(ND_Packet *p);
ND_Packet      *nd_packet_duplicate(ND_Packet *p);

/* Initializes a packet, its data offset pointers,
   and the protocol types for the protcol stack.
 */
void            nd_packet_init(ND_Packet *p);

ND_Packet      *nd_packet_sel_next(ND_Packet *p);
u_char         *nd_packet_get_data(ND_Packet *p, ND_Layer *l);

/* Various functions to adjust the GUI to a packet */
void            nd_packet_set(ND_Packet *p);
void            nd_packet_set_link_level(ND_Packet *p, int upward);
void            nd_packet_set_ip(ND_Packet *p, int upward);
void            nd_packet_set_icmp(ND_Packet *p);
void            nd_packet_set_icmp_error_ip(ND_Packet *p);
void            nd_packet_set_tcp(ND_Packet *p, int upward);
void            nd_packet_set_udp(ND_Packet *p, int upward);
void            nd_packet_set_arp(ND_Packet *p);
void            nd_packet_set_rarp(ND_Packet *p);

guint           nd_packet_hash(gconstpointer p);
gint            nd_packet_cmp(gconstpointer p1, gconstpointer p2);

/* Access to various protocol headers in the packet. These do
   *NOT* check for valid casts -- use the nd_packet_is_xxx()
   functions below for testing.
*/
struct ip      *nd_packet_ip(ND_Packet *p);
struct udphdr  *nd_packet_udp(ND_Packet *p);
struct tcphdr  *nd_packet_tcp(ND_Packet *p);
struct icmp    *nd_packet_icmp(ND_Packet *p);
struct ip      *nd_packet_icmp_error_ip(ND_Packet *p);
struct udphdr  *nd_packet_icmp_error_udp(ND_Packet *p);
struct arphdr  *nd_packet_arp(ND_Packet *p);
struct ether_header *nd_packet_ethernet(ND_Packet *p);

/* Returns pointer to last byte of captured data: */
u_char         *nd_packet_end(ND_Packet *p);

/* Packet content information: */

/* Returns true when a packet contains a complete header
   of the requested protocol, FALSE otherwise.
*/
gboolean        nd_packet_has_complete_header(ND_Packet *p, ND_Protocol prot);

/* Returns TRUE when a packet contains at least parts
   of the requested protocol. It doesn't check if the
   protocol header is complete. After a packet is
   initialized, this function also returns FALSE for
   all incomplete headers.
*/
gboolean        nd_packet_has_protocol(ND_Packet *p, ND_Protocol proto);


#endif
