--------------------------------------------------------------------------
--                                                                      --
--           Copyright: Copyright (C) 2000-2010 CNRS/IN2P3              --
--                                                                      --
-- Narval framework is free  software; you can redistribute  it and/or  --
-- modify  it   under  terms  of  the  GNU General  Public  License as  --
-- published  by  the  Free Software Foundation; either version  2, or  --
-- (at your option) any later version. Narval framework is distributed  --
-- in the hope  that  they will  be useful, but  WITHOUT ANY WARRANTY;  --
-- without even the implied warranty of  MERCHANTABILITY or FITNESS FOR --
-- A PARTICULAR PURPOSE. See the  GNU. General Public License for more  --
-- details. You should have received  a copy of the GNU General Public  --
-- License distributed with Narval; see file COPYING. If not, write to  --
-- the Free Software  Foundation,  Inc., 51 Franklin St,  Fifth Floor,  --
-- Boston, MA 02110-1301 USA.                                           --
--------------------------------------------------------------------------
with Ada.Text_IO;
with Ada.Strings.Unbounded;
with Ada.IO_Exceptions;

with Exec_Command;
with Echo_Command;
with Exception_Message;
with Error_Message;

with Narval.Environment;
with McKae.XML.EZ_Out.String_Stream;

with GNAT.OS_Lib;

package body Load_Command is

   ----------
   -- Load --
   ----------

   function Replace (Shell_Line : String;
                     Arguments : Shell_Commands.Arguments_Array) return String;

   function Load (Shell_Line : String) return String is
      use Ada.Strings.Unbounded;
      use McKae.XML.EZ_Out.String_Stream.String_Buffering;
      use McKae.XML.EZ_Out.String_Stream.XML_String_Buffer;
      Xml_Buffer : String_Buffer;
   begin
      Clear (Xml_Buffer);
      declare
         Arguments : constant Shell_Commands.Arguments_Array :=
           Shell_Commands.Arguments (Shell_Line);
      begin
         if Arguments'Length >= 1 then
            declare
               File : Ada.Text_IO.File_Type;
               Scripts_Directory : constant GNAT.OS_Lib.String_Access :=
                 Narval.Environment.Scripts_Directory;
            begin
               begin
                  Ada.Text_IO.Open (File,
                                    Ada.Text_IO.In_File,
                                    Ada.Strings.Unbounded.To_String
                                    (Arguments (1)));
               exception
                  when Ada.IO_Exceptions.Name_Error =>
                     Ada.Text_IO.Open (File,
                                       Ada.Text_IO.In_File,
                                       Scripts_Directory.all & "/" &
                                       Ada.Strings.Unbounded.To_String
                                       (Arguments (1)));
               end;

               Start_Element (Xml_Buffer, "result", ("status" = "ok",
                                                     "cmd" = "load"));
               while not Ada.Text_IO.End_Of_File (File) loop
                  declare
                     Local_Shell_Line : constant String :=
                       Ada.Text_IO.Get_Line (File);
                     use type Echo_Command.Echo_Value;
                  begin
                     if Echo_Command.Echo = Echo_Command.On then
                        Ada.Text_IO.Put_Line (Local_Shell_Line);
                     end if;
                     if Local_Shell_Line (Local_Shell_Line'First) = '#' then
                        null;
                     else
                        declare
                           New_Shell_Line : constant String :=
                             Replace (Local_Shell_Line, Arguments);
                        begin
                           declare
                              Result_Exec_Command : constant String :=
                                Exec_Command (New_Shell_Line);
                           begin
                              if Echo_Command.Echo = Echo_Command.On then
                                 Ada.Text_IO.Put_Line (Result_Exec_Command);
                              end if;
                              Put (Xml_Buffer, Result_Exec_Command);
                           end;
                        end;
                     end if;
                  end;
               end loop;
               Ada.Text_IO.Close (File);
               End_Element (Xml_Buffer, "result");
               declare
                  String_To_Return : constant String :=
                    Get_String (Xml_Buffer);
               begin
                  Full_Clear (Xml_Buffer);
                  return String_To_Return;
               end;
            exception
               when Ada.IO_Exceptions.Name_Error =>
                  return Error_Message ("load",
                                        "WARNING",
                                        "error: file name not valid : " &
                                        To_String (Arguments (1)));
            end;
         else
            return Syntax_String;
         end if;
      end;
   exception
      when Shell_Commands.No_Argument =>
         return Syntax_String;
      when E : others =>
         return Exception_Message ("load", E);
   end Load;

   function Replace (Shell_Line : String;
                     Arguments : Shell_Commands.Arguments_Array)
                    return String is
      use Ada.Strings.Unbounded;
   begin
      if Arguments'Length = 1 then
         return Shell_Line;
      end if;
      declare
         Arguments_A_Remplacer : constant Shell_Commands.Arguments_Array :=
           Shell_Commands.Arguments (Shell_Line);
         Command : constant Shell_Commands.Command_Type :=
           Shell_Commands.Find (Shell_Line);
         Chaine_Retour : Unbounded_String := Null_Unbounded_String;
      begin
         Chaine_Retour := Chaine_Retour & To_Unbounded_String (Command'Img);
         for I in Arguments_A_Remplacer'Range loop
            if Arguments_A_Remplacer (I) = "$1" then
               Chaine_Retour := Chaine_Retour & " " & Arguments (2);
            elsif Arguments_A_Remplacer (I) = "$2" and Arguments'Last >= 3 then
               Chaine_Retour := Chaine_Retour & " " & Arguments (3);
            elsif Arguments_A_Remplacer (I) = "$3" and Arguments'Last >= 4 then
               Chaine_Retour := Chaine_Retour & " " & Arguments (4);
            elsif Arguments_A_Remplacer (I) = "$4" and Arguments'Last >= 5 then
               Chaine_Retour := Chaine_Retour & " " & Arguments (5);
            else
               Chaine_Retour := Chaine_Retour & " " &
                 Arguments_A_Remplacer (I);
            end if;
         end loop;
         return To_String (Chaine_Retour);
      end;
   end Replace;

   function Syntax_String
     (Status : Shell_Commands.Status_Type := Shell_Commands.Syntax_Error)
     return String is
      use McKae.XML.EZ_Out.String_Stream.String_Buffering;
      use McKae.XML.EZ_Out.String_Stream.XML_String_Buffer;
      Xml_Buffer : String_Buffer;
   begin
      Clear (Xml_Buffer);
      Start_Element (Xml_Buffer, "result",
                     ("cmd" = "load", "status" = Status'Img));
      Start_Element (Xml_Buffer, "message", "type" = "help");
      Output_Element (Xml_Buffer, "syntax", "load file_name");
      End_Element (Xml_Buffer, "message");
      End_Element (Xml_Buffer, "result");
      declare
         String_To_Return : constant String := Get_String (Xml_Buffer);
      begin
         Full_Clear (Xml_Buffer);
         return String_To_Return;
      end;
   end Syntax_String;

end Load_Command;
