--------------------------------------------------------------------------
--                                                                      --
--           Copyright: Copyright (C) 2000-2010 CNRS/IN2P3              --
--                                                                      --
-- Narval framework is free  software; you can redistribute  it and/or  --
-- modify  it   under  terms  of  the  GNU General  Public  License as  --
-- published  by  the  Free Software Foundation; either version  2, or  --
-- (at your option) any later version. Narval framework is distributed  --
-- in the hope  that  they will  be useful, but  WITHOUT ANY WARRANTY;  --
-- without even the implied warranty of  MERCHANTABILITY or FITNESS FOR --
-- A PARTICULAR PURPOSE. See the  GNU. General Public License for more  --
-- details. You should have received  a copy of the GNU General Public  --
-- License distributed with Narval; see file COPYING. If not, write to  --
-- the Free Software  Foundation,  Inc., 51 Franklin St,  Fifth Floor,  --
-- Boston, MA 02110-1301 USA.                                           --
--------------------------------------------------------------------------
with Ada.Characters.Handling;
with System.Storage_Elements;
with GNAT.Traceback.Symbolic;

with Narval.Configurator.Abstract_Actors_Coordination;
with Narval.Local_Configuration;
use Narval.Local_Configuration;

package body Narval.Actors.Actives.Producers.Generics is

   use Log4ada.Loggers;
   use Ada.Strings.Unbounded;

   procedure Dummy (Algo_Data : System.Address;
                    Error_Code : out Error_Code_Type);
   pragma Convention (C, Dummy);
   procedure Dummy (Algo_Data : System.Address;
                    Error_Code : out Error_Code_Type) is
      pragma Unreferenced (Algo_Data);
   begin
      Error_Code := 0;
   end Dummy;

   procedure Test_Error_Code (Object : access Generic_Producer_Type;
                              Error_Code : Error_Code_Type;
                              Procedure_Name : String);

   procedure Set (Object : access Generic_Producer_Type;
                  Parameter : String;
                  Value : String) is
      Parameter_Low_Case : constant String :=
        Ada.Characters.Handling.To_Lower (Parameter);
      Error_Code : aliased Error_Code_Type;
      Library_Problem : exception;
      Max_Size : Integer_Access;
      Algo_Path : Unbounded_String;
      Dl_Close_Return : Integer;
   begin
      Actives.Set (Active_Actor_Access (Object),
                   Parameter, Value);
      if Parameter_Low_Case = "library" then
         if Value /= "clear" then
            Object.Library_Reference := Shared_Library.Open_Library
              (Value, Shared_Library.RTLD_NOW);
            Object.Library_Is_Loaded := True;
            Info_Out (Object.Logger'Access,
                      "library loaded");
            Object.Config := Library_Symbol (Object.Library_Reference,
                                             "process_config");
            if Object.Config = null then
               Fatal_Out (Object.Logger'Access, "process_config undefined");
               raise Library_Problem;
            end if;
            Algo_Path := To_Unbounded_String
              (Actives.Get_Image
               (Active_Actor_Access (Object),
                "algo_path"));
            if Algo_Path = Null_Unbounded_String then
               Warn_Out (Object.Logger'Access,
                         "algo_path not set");
            end if;
            Object.Config (To_String (Algo_Path) & ASCII.NUL, Error_Code);
            Test_Error_Code (Object, Error_Code, "process_config");
            Object.Block_Producer := Library_Symbol (Object.Library_Reference,
                                                     "process_block");
            if Object.Block_Producer = null then
               Fatal_Out (Object.Logger'Access,
                          "process_block undefined");
               raise Library_Problem;
            end if;
            Object.Register := Library_Symbol (Object.Library_Reference,
                                               "process_register");
            if Object.Register = null then
               Fatal_Out (Object.Logger'Access,
                          "process_register undefined");
               raise Library_Problem;
            else
               Object.Common_Reference :=
                 Object.Register (Error_Code'Unchecked_Access);
               Test_Error_Code (Object, Error_Code, "process_register");
            end if;
            Max_Size := Library_Symbol
              (Object.Library_Reference, "max_size");
            if Max_Size = null then
               Info_Out (Object.Logger'Access,
                         "max event size set to 0");
            else
               Object.Max_Size := Max_Size.all;
            end if;
            Object.Timeout := Library_Symbol
              (Object.Library_Reference, "library_timeout");
            Object.Handle_Timeout := Object.Timeout /= null;
            Object.Initialise := Library_Symbol
              (Object.Library_Reference,
               "process_initialise");
            if Object.Initialise = null then
               Object.Initialise := Dummy'Access;
            end if;
            Object.Reset := Library_Symbol
              (Object.Library_Reference, "process_reset");
            if Object.Reset = null then
               Object.Reset := Dummy'Access;
            end if;
            Object.Start := Library_Symbol
              (Object.Library_Reference, "process_start");
            if Object.Start = null then
               Object.Start := Dummy'Access;
            end if;
            Object.Stop := Library_Symbol
              (Object.Library_Reference, "process_stop");
            if Object.Stop = null then
               Object.Stop := Dummy'Access;
            end if;
            Object.Pause_Ptr := Library_Symbol
              (Object.Library_Reference, "process_pause");
            if Object.Pause_Ptr = null then
               Object.Pause_Ptr := Dummy'Access;
            end if;
            Object.Resume := Library_Symbol
              (Object.Library_Reference, "process_resume");
            if Object.Resume = null then
               Object.Resume := Dummy'Access;
            end if;
            Object.Unload := Library_Symbol
              (Object.Library_Reference, "process_unload");
            if Object.Unload = null then
               Object.Unload := Dummy'Access;
            end if;
         else
            if Object.Library_Is_Loaded then
               Dl_Close_Return := Shared_Library.Close_Library
                 (Object.Library_Reference);
               if Dl_Close_Return /= 0 then
                  Warn_Out (Object.Logger'Access,
                            "close library returned non zero value :" &
                            Dl_Close_Return'Img);
               end if;
               Object.Config := null;
               Object.Register := null;
               Object.Block_Producer := null;
               Object.Initialise := null;
               Object.Reset := null;
               Object.Start := null;
               Object.Stop := null;
               Object.Pause_Ptr := null;
               Object.Resume := null;
               Object.Unload := null;
               Object.Max_Size := 0;
               Object.Library_Is_Loaded := False;
            else
               Warn_Out (Object.Logger'Access, "no library to clear");
            end if;
         end if;
      end if;
   exception
      when E : Shared_Library.Library_Loading_Failed =>
         Error_Out (Object.Logger'Access,
                    "set library :" &
                    Shared_Library.Library_Error,
                    E);
         raise;
   end Set;

   -----------------
   -- Initialiser --
   -----------------

   procedure Initialise (Object : access Generic_Producer_Type;
                         Actor_Name : String) is
      Parameter : Parameters.Parameter_Access;
      use Parameters;
   begin
      Actives.Initialise (Active_Actor_Access (Object), Actor_Name);
      Parameter := new Parameter_Type'(Container_Kind => String_Type,
                                       Name => To_Unbounded_String
                                         ("algo_path"),
                                       Mode => Read_Write,
                                       Monitor => Request,
                                       Run_Parameter => False,
                                       Editor => None,
                                       String_Value =>
                                         Null_Unbounded_String);
      Parameter_Vector_Package.Append (Object.Parameters_List,
                                       Parameter);
      Parameter := new Parameter_Type'(Container_Kind => Unsigned_32_Type,
                                       Name => To_Unbounded_String
                                         ("watchdog_ms"),
                                       Mode => Read_Write,
                                       Monitor => Request,
                                       Run_Parameter => False,
                                       Editor => None,
                                       Unsigned_32_Value => 1000);
      Parameter_Vector_Package.Append (Object.Parameters_List,
                                       Parameter);
      Object.Watchdog_Parameter := Parameter;
      Parameter := new Parameter_Type'(Container_Kind => Unsigned_32_Type,
                                       Name => To_Unbounded_String
                                         ("timeout_inner_counter"),
                                       Mode => Read_Write,
                                       Monitor => Request,
                                       Run_Parameter => False,
                                       Editor => None,
                                       Unsigned_32_Value => 0);
      Parameter_Vector_Package.Append (Object.Parameters_List,
                                       Parameter);
      Object.Timeout_Inner_Counter := Parameter;
      Parameter := new Parameter_Type'(Container_Kind => String_Type,
                                       Name => To_Unbounded_String
                                         ("library"),
                                       Mode => Read_Write,
                                       Monitor => Request,
                                       Run_Parameter => False,
                                       Editor => None,
                                       String_Value =>
                                         Null_Unbounded_String);
      Parameter_Vector_Package.Append (Object.Parameters_List,
                                       Parameter);
      Object.Buffer_Handling_Switchoff := False;
   end Initialise;

   procedure On_Initialise
     (Object : access Generic_Producer_Type) is
      use Shared_Library;
      use Configurator.Abstract_Actors_Coordination;
      Name : constant String := To_String (Object.Name);
      No_Library : exception;
      Error_Code : Error_Code_Type;
   begin
      Producers.On_Initialise
        (Producer_Access (Object));
      Object.Waiting_Delay := 0.1;
      if not Object.Library_Is_Loaded then
         Fatal_Out (Object.Logger'Access,
                    "one need to set a library to the " & Name & " actor");
         raise No_Library;
      end if;
      Object.Initialise
        (Object.Common_Reference, Error_Code);
      Test_Error_Code (Object, Error_Code, "process_initialise");
   end On_Initialise;

   ----------------------------
   -- Travail_En_Acquisition --
   ----------------------------

   procedure Buffer_Handling
     (Object : access Generic_Producer_Type) is
      Adresse_Sortie : System.Address;
      Available_Memory_Size : System.Storage_Elements.Storage_Count;
      use type System.Storage_Elements.Storage_Count;
      Error_Code : Error_Code_Type := 0;
      Used_Size_Of_Output_Buffer : Interfaces.Unsigned_32 := 0;
      Watchdog_Timeout : Duration;
      use type Interfaces.Unsigned_32;
   begin
      if Object.Buffer_Handling_Switchoff then
         delay 0.01;
         return;
      end if;
      select
         Object.Outputs (1).Memory.Get_Memory
           (Adresse_Sortie, Available_Memory_Size);
      or
         delay 1.0;
         Info_Out (Object.Logger'Access,
                   "generic producer : waiting for free memory");
         return;
      end select;
      if Object.Max_Size /= 0 then
         if Available_Memory_Size < System.Storage_Elements.Storage_Count
           (Object.Max_Size) then
            Object.Outputs (1).Memory.Release_Memory (0);
            Object.Outputs (1).Memory.Check;
            return;
         end if;
      end if;
      if Object.Handle_Timeout then
         Watchdog_Timeout := Duration
           (Object.Watchdog_Parameter.Unsigned_32_Value / 1000);
         Watchdog_Timeout := Watchdog_Timeout +
           0.001 * Duration
           (Object.Watchdog_Parameter.Unsigned_32_Value mod 1000);
         select
            Object.Watchdog_Task.Stop;
         else
            null;
         end select;
         Object.Timeout.all := 0;
         Object.Watchdog_Task.Start (Watchdog_Timeout);
      end if;
      Object.Block_Producer
        (Object.Common_Reference,
         Adresse_Sortie,
         Interfaces.Unsigned_32 (Available_Memory_Size),
         Used_Size_Of_Output_Buffer,
         Error_Code);
      Test_Error_Code (Object, Error_Code, "process_block");
      if Error_Code >= 100 then
         Object.Buffer_Handling_Switchoff := True;
         if Error_Code < 1000 then
            Object.Stop_Task.Send_Signal;
         end if;
      end if;
      Object.Outputs (1).Memory.Release_Memory
        (System.Storage_Elements.Storage_Count (Used_Size_Of_Output_Buffer),
         Object.Duplicate);
   exception
      when E : others =>
         Put_Sub_System_In_Error (Object, "buffer_handling");
         Object.Buffer_Handling_Switchoff := True;
         Error_Out (Object.Logger'Access,
                    "process_block generic producer", E);
         Error_Out (Object.Logger'Access, "trace " &
                    GNAT.Traceback.Symbolic.Symbolic_Traceback (E));
   end Buffer_Handling;

   procedure On_Unload
     (Object : access Generic_Producer_Type) is
      use Shared_Library;
      Function_Return : Integer;
      Error_Code : Error_Code_Type;
   begin
      Producers.On_Unload
        (Producer_Access (Object));
      if Object.Unload /= null then
         Object.Unload
           (Object.Common_Reference,
            Error_Code);
         Test_Error_Code (Object, Error_Code, "process_unload");
      end if;
      Function_Return := Close_Library
        (Object.Library_Reference);
      if Function_Return /= 0 then
         Warn_Out (Object.Logger'Access,
                   "close library return non null value :" &
                     Function_Return'Img);
      end if;
   exception
      when E : others =>
         Put_Sub_System_In_Error (Object, "on_unload");
         Fatal_Out (Object.Logger'Access,
                    "on_unload", E);
         raise;
   end On_Unload;

   procedure On_Reset_Com
     (Object : access Generic_Producer_Type) is
      Error_Code : Error_Code_Type := 0;
   begin
      Producers.On_Reset_Com
        (Producer_Access (Object));
      Object.Reset
        (Object.Common_Reference,
         Error_Code);
      Test_Error_Code (Object, Error_Code, "process_reset");
   exception
      when E : others =>
         Put_Sub_System_In_Error (Object, "on_reset");
         Fatal_Out (Object.Logger'Access,
                    "on_reset", E);
         raise;
   end On_Reset_Com;

   procedure On_Start
     (Object : access Generic_Producer_Type) is
      Error_Code : Error_Code_Type;
   begin
      Object.Timeout_Inner_Counter.Unsigned_32_Value := 0;
      Object.Start
        (Object.Common_Reference, Error_Code);
      Producers.On_Start
        (Producer_Access (Object));
      Test_Error_Code (Object, Error_Code, "process_start");
   exception
      when E : others =>
         Put_Sub_System_In_Error (Object, "on_start");
         Fatal_Out (Object.Logger'Access,
                    "Sur_Demarrer", E);
         raise;
   end On_Start;

   procedure On_Stop
     (Object : access Generic_Producer_Type) is
      Error_Code : Error_Code_Type;
   begin
      Object.Stop
        (Object.Common_Reference, Error_Code);
      Test_Error_Code (Object, Error_Code, "process_stop");

      Producers.On_Stop
        (Producer_Access (Object));
   exception
      when E : others =>
         Put_Sub_System_In_Error (Object, "on_stop");
         Fatal_Out (Object.Logger'Access,
                    "on_stop", E);
         raise;
   end On_Stop;

   procedure On_Suspend
     (Object : access Generic_Producer_Type) is
      Error_Code : Error_Code_Type;
   begin
      Producers.On_Suspend
        (Producer_Access (Object));
      Object.Pause_Ptr
        (Object.Common_Reference, Error_Code);
      Test_Error_Code (Object, Error_Code, "process_suspend");
   exception
      when E : others =>
         Put_Sub_System_In_Error (Object, "on_suspend");
         Fatal_Out (Object.Logger'Access,
                    "on_suspend", E);
         raise;
   end On_Suspend;

   procedure On_Resume
     (Object : access Generic_Producer_Type) is
      Error_Code : Error_Code_Type;
   begin
      Producers.On_Resume
        (Producer_Access (Object));
      Object.resume
        (Object.Common_Reference, Error_Code);
      Test_Error_Code (Object, Error_Code, "process_resume");
   exception
      when E : others =>
         Put_Sub_System_In_Error (Object, "on_resume");
         Fatal_Out (Object.Logger'Access,
                    "on_resume", E);
         raise;
   end On_Resume;

   task body Stop_Signaling_Task is
   begin
      loop
         select
            accept Send_Signal;
            Configurator.Abstract_Actors_Coordination.Set
              (Local_Configuration.Config_Local,
               "action",
               "stop");
         or
            terminate;
         end select;
      end loop;
   end Stop_Signaling_Task;

   procedure Test_Error_Code (Object : access Generic_Producer_Type;
                              Error_Code : Error_Code_Type;
                              Procedure_Name : String) is
   begin
      if Error_Code in 1 .. 999 then
         Warn_Out (Object.Logger'Access,
                   "warning in " & Procedure_Name &
                   " call :" & Error_Code'Img);
      elsif Error_Code >= 1000 then
         Put_Sub_System_In_Error
           (Object,
            "error in " & Procedure_Name & " call :" & Error_Code'Img);
         Error_Out (Object.Logger'Access,
                    "error in " & Procedure_Name & " call :" & Error_Code'Img);
      end if;
   end Test_Error_Code;

   task body Watchdog_Task_Type is
      Local_Duration : Duration;
      use type Interfaces.Unsigned_32;
   begin
      loop
         select
            accept Start (Timeout : Duration) do
               Local_Duration := Timeout;
            end Start;
            select
               accept Stop;
            or
               delay Local_Duration;
               Wrapper.Timeout_Inner_Counter.Unsigned_32_Value :=
                 Wrapper.Timeout_Inner_Counter.Unsigned_32_Value + 1;
               Wrapper.Timeout.all :=
                 Wrapper.Timeout_Inner_Counter.Unsigned_32_Value;
            end select;
         or
            terminate;
         end select;
      end loop;
   end Watchdog_Task_Type;

end Narval.Actors.Actives.Producers.Generics;
