/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_canvas.cpp 
 * @brief Generic canvas main class and entry point.
 * 
 */

#ifdef _WINDOWS
  #define  WIN32_LEAN_AND_MEAN
  #include <windows.h>
#endif // ifdef _WINDOWS

#include <assert.h>

#ifdef __APPLE__
  #include <OpenGL/gl.h>
  #include <OpenGL/glu.h>
#else
  #include <GL/gl.h>
  #include <GL/glu.h>
#endif // #ifdef __APPLE__

#include "myx_gc_figure.h"
#include "myx_gc_layer.h"
#include "myx_gc_model.h"
#include "myx_gc_canvas.h"
#include "myx_gc_gl_helper.h"
#include "myx_gc_font_manager.h"
#include "myx_gc_texture.h"
#include "myx_gc_svgparser.h"
#include "myx_gc_figure_parser.h"

//----------------------------------------------------------------------------------------------------------------------

// Factory function for a canvas.
GENERIC_CANVAS_API CGenericCanvas* CreateGenericCanvas(GCContext Context)
{
	return new CGenericCanvas(Context);
}

//----------------- CHitResults ----------------------------------------------------------------------------------------

CHitResults::CHitResults(void)
{
  FCurrentEntry = FEntries.end();
}

//----------------------------------------------------------------------------------------------------------------------

CHitResults::~CHitResults(void)
{
  for (THitEntryIterator Iterator = FEntries.begin(); Iterator != FEntries.end(); ++Iterator)
    delete (*Iterator).second;
}

//----------------------------------------------------------------------------------------------------------------------

void CHitResults::AddHit(CFigureInstance* Instance, double Min, double Max)
{
  THitEntry* Entry = new THitEntry;
  Entry->Instance = Instance;
  Entry->ZMax = Max;
  Entry->ZMin = Min;
  FEntries[Instance] = Entry;
}

//----------------------------------------------------------------------------------------------------------------------

int CHitResults::Count(void)
{
  return FEntries.size();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Deletes the given entry from the internal list.
 *
 * @param Instance The entry to be deleted.
 * @return If the entry was in the list then true is returned, otherwise false.
 */
bool CHitResults::Delete(CFigureInstance* Instance)
{
  THitEntryIterator Iterator = FEntries.find(Instance);
  bool Result = Iterator != FEntries.end();
  if (Result)
  {
    if (FCurrentEntry == Iterator)
      ++FCurrentEntry;
    FEntries.erase(Iterator);
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

void CHitResults::DeleteCurrent(void)
{
  if (FCurrentEntry != FEntries.end())
  {
    FEntries.erase(FCurrentEntry);
    ++FCurrentEntry;
  };
}

//----------------------------------------------------------------------------------------------------------------------

bool CHitResults::HasNext(void)
{
  return FCurrentEntry != FEntries.end();
}

//----------------------------------------------------------------------------------------------------------------------

THitEntry* CHitResults::Next(void)
{
  if (FCurrentEntry == FEntries.end())
    return NULL;
  else
  {
    THitEntryIterator Temp = FCurrentEntry;
    ++FCurrentEntry;
    return Temp->second;
  };
}

//----------------------------------------------------------------------------------------------------------------------

void CHitResults::Release(void)
{
  delete this;
}

//----------------------------------------------------------------------------------------------------------------------

void CHitResults::Reset(void)
{
  FCurrentEntry = FEntries.begin();
}

//----------------- CFigureInstanceEnumerator --------------------------------------------------------------------------

/**
 * Constructor of the enumerator class.
 *
 * @param Canvas The canvas which contains the layers which are to be enumerated.
 */
CFigureInstanceEnumerator::CFigureInstanceEnumerator(CGenericCanvas* Canvas)
{
  FCanvas = Canvas;
  FLayerIterator = FCanvas->FLayers.end();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Determines if there is a next figure instance to enumerate.
 *
 * @return True if there is still a figure instance otherwise false.
 */
bool CFigureInstanceEnumerator::HasNext(void)
{
  return FLayerIterator != FCanvas->FLayers.end();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the next figure instance in the sequence.
 *
 * @return The next figure instance.
 */
CFigureInstance* CFigureInstanceEnumerator::Next(void)
{
  CFigureInstance* Result = *FFigureInstanceIterator;

  // Advance to next instance.
  ++FFigureInstanceIterator;
  while (true)
  {
    CLayer* Layer = *FLayerIterator;
    if (FFigureInstanceIterator != Layer->FInstances.end())
      break;

    // This layer is exhausted. Get the next one.
    ++FLayerIterator;
    if (FLayerIterator == FCanvas->FLayers.end())
    {
      // No more layers.
      break;
    };

    if (!(*FLayerIterator)->FEnabled)
      FFigureInstanceIterator = (*FLayerIterator)->FInstances.end();
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Frees this enumerator instance. Usually called by non-C++ languages as memory is managed by the C++ runtime.
 */
void CFigureInstanceEnumerator::Release(void)
{
  delete this;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Resets the enumerator to the first figure instance in the canvas.
 */
void CFigureInstanceEnumerator::Reset(void)
{
  FLayerIterator = FCanvas->FLayers.begin();
  while (FLayerIterator != FCanvas->FLayers.end())
  {
    if ((*FLayerIterator)->FEnabled)
    {
      CLayer* Layer = *FLayerIterator;
      FFigureInstanceIterator = Layer->FInstances.begin();
      if (FFigureInstanceIterator == Layer->FInstances.end())
      {
        ++FLayerIterator;
      }
      else
        break; // Found the first instance.
    }
    else
      ++FLayerIterator;
  };
}

//----------------- CGenericCanvas -------------------------------------------------------------------------------------

CGenericCanvas::CGenericCanvas(GCContext Context)
{ 
  FIsPicking = false;
  FContext = Context;
  FViewport.Left = 0;
  FViewport.Top = 0;
  FViewport.Width = 100;
  FViewport.Height = 100;
  FUpdateCount = 0;
  FZoomX = 1;
  FZoomY = 1;
  FOffsetX = 0;
  FOffsetY = 0;
  FNearPlane = -100;
  FFarPlane = 100;

  FModel = new CGCModel(this);
  FFeedbackLayer = new CFeedbackLayer(this);
  // Set a new lock on the font manager (and create it by the way if not yet done).
  LockFontManager();

#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
  FBackgroundColor = 0;
#elif defined(__APPLE__)
  {
	GCColor zero= {0,0,0};
	FBackgroundColor= zero;
  }
#else
  {
    XColor zero= {0, 0, 0, 0, DoRed | DoGreen | DoBlue, 0};
    FBackgroundColor = zero;
  }
#endif
  glClearColor(0, 0, 0, 1);

  // TODO: make this dynamically configurable (where applicable).
  glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
  glFrontFace(GL_CW);
  glDisable(GL_AUTO_NORMAL);
  glDisable(GL_CULL_FACE);
  glDisable(GL_DITHER);
  glDisable(GL_DEPTH_TEST);

  // glEnable(GL_POINT_SMOOTH);
  // glHint(GL_POINT_SMOOTH_HINT, GL_NICEST);
  glEnable(GL_LINE_SMOOTH);
  glHint(GL_LINE_SMOOTH_HINT, GL_NICEST);
  glEnable(GL_POLYGON_SMOOTH);
  glHint(GL_POLYGON_SMOOTH_HINT, GL_NICEST);
  
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
}

//----------------------------------------------------------------------------------------------------------------------

CGenericCanvas::~CGenericCanvas(void)
{
  // Free any dynamically allocated memory. Explicitely increase update count. We don't want any recursive
  // notifcations anymore.
  ++FUpdateCount;
  FFeedbackLayer->BeginUpdate();
  for (CLayers::iterator Iterator = FLayers.begin(); Iterator != FLayers.end(); ++Iterator)
    delete *Iterator;

  delete FModel;
  delete FFeedbackLayer;

  // Release the lock we placed in the constructor.
  // If this is the last GC instance then also the font manager can be released.
  UnlockFontManager();
}

//----------------------------------------------------------------------------------------------------------------------

void CGenericCanvas::ApplyViewport(void)
{
  glViewport(FViewport.Left, FViewport.Top, FViewport.Width, FViewport.Height);
}

//----------------------------------------------------------------------------------------------------------------------

void CGenericCanvas::ClearBuffers(void)
{
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief Adds the given layer at the end of the layer list. The new layer becomes the top layer in the view then.
 * @param Layer The layer to add.
 */
void CGenericCanvas::AddLayer(CLayer* Layer)
{
  FLayers.push_back(Layer);
  if (FUpdateCount == 0)
    Invalidate();
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Reads the layout info stored in the given (XML) file and creates figure templates.
 * Existing templates remain in where they are but are replaced if a new definition with an existing name is found.
 *
 * @param Filename The name of the file to load.
 * @return Returns GC_NO_ERROR if everything was ok, otherwise an error code.
 */
TGCError CGenericCanvas::AddLayoutsFromFile(const char* Filename)
{
  TGCError Result = GC_NO_ERROR;
  xmlNodePtr Root, Current;
  CFigureParser Parser;

  string CurrentDir = GetCurrentDir();
  xmlDocPtr Document = xmlParseFile(Utf8ToANSI(string(Filename)).c_str());

  if (Document == NULL)
    return GC_XML_PARSE_ERROR;

  Root = xmlDocGetRootElement(Document);

  if (Root == NULL)
  {
    xmlFreeDoc(Document);
    Error("XML Error: Template file is empty.");
    return GC_XML_EMPTY_DOCUMENT;
  }
  
  if (!XML_IS(Root, "gc-layouts"))
  {
    xmlFreeDoc(Document);
    Error("XML Error: Template file invalid.");
    return GC_XML_INVALID_DOCUMENT;
  }

  // Switch to the directory of the given file. This is necessary to make relative file names working.
  string Path = ExtractFilePath(Filename);
  SetCurrentDir(Path);

  // Parse description elements.
  glMatrixMode(GL_MODELVIEW);
  Current = Root->children;
  while (Current != NULL)
  {
    // Be flexible, ignore any unknown layout entries.
    if (XML_IS(Current, "layout-definition"))
      Parser.ParseLayoutDefinition(Current, FModel);
    Current = Current->next;
  }

  SetCurrentDir(CurrentDir);
  xmlFreeDoc(Document);

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief Adds a listener to the internal list.
 *
 * @param Listener The listener to add.
 */
void CGenericCanvas::AddListener(CGCListener* Listener)
{
  FListeners.push_back(Listener);
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Reads the style template info stored in the given (XML) file and creates style templates (OpenGL display lists).
 * Existing templates remain where they are but are replaced if a new definition with an existing name is found.
 *
 * @param Filename The name of the file to load.
 * @return Returns GC_NO_ERROR if everything was ok, otherwise an error code.
 */
TGCError CGenericCanvas::AddStylesFromFile(const char* Filename)
{
  TGCError Result = GC_NO_ERROR;
  xmlNodePtr Root, Current;
  CSVGParser Parser;

  string CurrentDir = GetCurrentDir();
  xmlDocPtr Document = xmlParseFile(Utf8ToANSI(string(Filename)).c_str());

  if (Document == NULL)
    return GC_XML_PARSE_ERROR;

  Root = xmlDocGetRootElement(Document);

  if (Root == NULL)
  {
    xmlFreeDoc(Document);
    Error("XML Error: Template file is empty.");
    return GC_XML_EMPTY_DOCUMENT;
  }
  
  if (!XML_IS(Root, "gc-styles"))
  {
    xmlFreeDoc(Document);
    Error("XML Error: Template file invalid.");
    return GC_XML_INVALID_DOCUMENT;
  }

  // Switch to the directory of the given file. This is necessary to make relative file names working.
  string Path = ExtractFilePath(Filename);
  SetCurrentDir(Path);

  // Parse description elements.
  glMatrixMode(GL_MODELVIEW);
  Current = Root->children;
  while (Current != NULL)
  {
    // Be flexible, ignore any unknown layout entries.
    if (XML_IS(Current, "style-definition"))
    {
      Parser.ParseDefinition(Current, FModel);
    }
    else
      if (XML_IS(Current, "texture"))
      {
        // Adds a new texture to the texture list.
        ParseTextureEntry(Current);
        CheckError();
      };
    Current = Current->next;
  }

  SetCurrentDir(CurrentDir);
  xmlFreeDoc(Document);

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Adds the given figure instance to the current selection. This call is simply forwared to the selection layer.
 * 
 * @param Instance The instance to be added to the selection. It is taken care that an instance is only added once.
 */
void CGenericCanvas::AddToSelection(CFigureInstance* Instance)
{
  FFeedbackLayer->AddToSelection(Instance);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Increases the update count by 1 to stop any recursive update until (@see EndUpdate()) was called.
 */
void CGenericCanvas::BeginUpdate(void)
{
  ++FUpdateCount;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Notifies all registered listeners that a change has occured.
 *
 * @param AObject The reference to an object for which the change event is triggered. Can be anything like a figure, figure
 *                instance, the canvas itself, a layer etc.
 * @param Reason Indicates what change actually happend.
 */
void CGenericCanvas::Change(void* AObject, TGCChangeReason Reason)
{
  CListenerIterator Iterator = FListeners.begin(); 
  while (Iterator != FListeners.end())
  {
    CListenerIterator Next = Iterator + 1;
    try
    {
      CGCListener* Listener = *Iterator;
      Listener->OnChange(AObject, Reason);
    }
    catch(...)
    {
      // If there was an exception while executing the method the listener is removed from our list
      // to avoid further harm.
      FListeners.erase(Iterator);
    };
    Iterator = Next;
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Checks if there is an OpenGL error registered and triggers the error method if so.
 */
void CGenericCanvas::CheckError(void)
{
  GLenum OGLError = glGetError();
  if (OGLError != GL_NO_ERROR)
  {
    char Buffer[100];
    sprintf(Buffer, "OpenGL error encountered (0x%x).", (int)OGLError);
    Error(Buffer);
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes all GC content.
 */
void CGenericCanvas::ClearContent(void)
{
  FModel->ClearFigures();
  Invalidate();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes all layout info.
 */
void CGenericCanvas::ClearLayouts(void)
{
  FModel->ClearLayouts();
  Invalidate();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes all currently selected figure instances from the selection set. This call is simply forwareded to the
 * selection layer.
 */
void CGenericCanvas::ClearSelection(void)
{
  FFeedbackLayer->ClearSelection();
  Invalidate();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Causes the model to clear its style definitions.
 */
void CGenericCanvas::ClearStyles(void)
{
  FModel->ClearStyles();
  Invalidate();
}

//----------------------------------------------------------------------------------------------------------------------

CFigure* CGenericCanvas::CreateFigure(wchar_t* Type)
{
  return FModel->CreateFigure(Type);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates a new layer with the given name and returns it to the caller. The new layer is added to this canvas.
 *
 * @param Name The layer identification, encoded in UTF-8.
 * @param Type The type of layer that is requested.
 */
CLayer* CGenericCanvas::CreateLayer(const char* Name, TGCLayerType Type)
{
  CLayer* Layer = NULL;
  switch (Type) 
  {
    case GC_LAYER_NORMAL:
      {
        Layer = new CLayer(this);
        break;
      };
    case GC_LAYER_GRID:
      {
        Layer = new CGridLayer(this);
        break;
      };
  };

  if (Layer != NULL)
  {
    Layer->FName = Utf8ToUtf16(Name);
    AddLayer(Layer);
  };

  return Layer;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * The counterpart to (@see BeginUpdate). It releases one update lock and invalidates the canvas if the count drops to 0.
 */
void CGenericCanvas::EndUpdate(void)
{
  if (FUpdateCount > 0)
    FUpdateCount--;
  if (FUpdateCount == 0)
    Invalidate();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief Notifies all registered listeners about an error that occured.
 */
void CGenericCanvas::Error(const char* Message)
{
  CListenerIterator Iterator = FListeners.begin(); 
  while (Iterator != FListeners.end())
  {
    CListenerIterator Next = Iterator + 1;
    try
    {
      CGCListener* Listener = *Iterator;
      Listener->OnError(Message);
    }
    catch(...)
    {
      // If there was an exception while executing the method the listener is removed from our list
      // to avoid further harm.
      FListeners.erase(Iterator);
    };
    Iterator = Next;
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates and returns a new figure instance enumerator instance.
 *
 * @return The new enumerator.
 */
CFigureInstanceEnumerator* CGenericCanvas::GetFigureInstanceEnumerator(void)
{
  return new CFigureInstanceEnumerator(this);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Takes the given coordinates and tries to find a figure that was rendered at this position.
 *
 * @param X The horizontal window (viewer) coordinate.
 * @param Y The vertical coordinate. 
 * @return A hit result class is returned regardless of the actual number of hits. It must be freed by the caller.
 */
CHitResults* CGenericCanvas::GetHitTestInfoAt(const int X, const int Y)
{
  FIsPicking = true;

  GLint Viewport[4];
  glGetIntegerv(GL_VIEWPORT, Viewport);

  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPickMatrix(X, Y, 4, 4, Viewport);
  glOrtho(-0.25, Viewport[2] - 0.25, Viewport[3] - 0.25, -0.25, FNearPlane, FFarPlane);

  GLsizei BufferSize = 500;
  GLuint* Buffer = NULL;
  int Hits = 0;
  glMatrixMode(GL_MODELVIEW);
  do
  {
    Buffer = (GLuint*) realloc(Buffer, BufferSize * sizeof(GLuint));
    glSelectBuffer(BufferSize, Buffer);

    glRenderMode(GL_SELECT);
    glLoadIdentity();

    glInitNames();
    glPushName(0);

    // Render the scene (in select mode nothing is drawn).
    glTranslated(FOffsetX, FOffsetY, 0);
    glScaled(FZoomX, FZoomY, 1);
    for (CLayers::iterator Iterator = FLayers.begin(); Iterator != FLayers.end(); ++Iterator)
    {
      CLayer* Layer = *Iterator;
      if (Layer->FEnabled)
        Layer->Render();
    };

    Hits = glRenderMode(GL_RENDER);
    BufferSize <<= 1;
  }
  while (Hits < 0);

  CheckError();
  FIsPicking = false;

  CHitResults* Result = new CHitResults();

  GLuint* Run = Buffer;
  for (int I = 0; I < Hits; I++)
  {
    int Count = *Run++;
    // The depth values are normalized so that the largest value corresponds to 1, while the smallest one is 0.
    // To store this as integer a mapping is applied to make 1 <=> 0xFFFFFFFF.
    double ZMin = (double) *Run++ / 0xFFFFFFFF;
    double ZMax = (double) *Run++ / 0xFFFFFFFF;
    CFigureInstance* Instance = (CFigureInstance*) (*Run++);
    Result->AddHit(Instance, ZMin, ZMax);

    // Usually we have only one entry per hit record.
    // That's the way it works when we only use glLoadName when rendering figures.
    // In order to be error tolerant we skip unused entries accordingly.
    Run += Count - 1;
  };

  free(Buffer);
  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Retrieves the value of the given property, if it is a property of this class.
 *
 * @param Property The property to retrieve.
 * @param Value [out] The value of the property.
 * @return True if the asked property is known in this class, false otherwise (in this case Value is not changed).
 * @note Implicit datatype conversion is performed if datatypes of property and Value parameter do not match.
 *       This can result in lost precision or even conversion errors. So make sure to use the right datatype for the call.
 */
bool CGenericCanvas::GetProperty(TProperty Property, double& Value)
{
  bool Result = true;
  switch (Property)
  {
    case GC_PROPERTY_HANDLE_SIZE:
      {
        break;
      };
    default:
      {
        // Property is not known here so try the special classes.
        Result = FFeedbackLayer->GetProperty(Property, Value);
        break;
      };
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Retrieves the value of the given property, if it is a property of this class.
 *
 * @param Property The property to retrieve.
 * @param Value [out] The value of the property.
 * @return True if the asked property is known in this class, false otherwise (in this case Value is not changed).
 * @note Implicit datatype conversion is performed if datatypes of property and Value parameter do not match.
 *       This can result in lost precision or even conversion errors. So make sure to use the right datatype for the call.
 */
bool CGenericCanvas::GetProperty(TProperty Property, int& Value)
{
  bool Result = true;
  switch (Property)
  {
    case GC_PROPERTY_HANDLE_SIZE:
      {
        break;
      };
    default:
      {
        // Property is not known here so try the special classes.
        Result = FFeedbackLayer->GetProperty(Property, Value);
        break;
      };
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Determines whether the given position corresponds to any of the parts (body, handles) of a selection decoration.
 * This test is quite fast and can be used for cursor feedback and such. This method is simply forwarded to
 * CFeedbackLayer::GetSelectionInfo after modelview and projection matrix are constructed with the current settings.
 *
 * @param X The horizontal mouse position in window coordinates.
 * @param Y The vertical mouse position in window coordinate.
 * @return One of the direction flags.
 */
TGCDirection CGenericCanvas::GetSelectionInfo(const int X, const int Y)
{
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glOrtho(-0.25, FViewport.Width - 0.25, FViewport.Height - 0.25, -0.25, FNearPlane, FFarPlane);

  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  glTranslated(FOffsetX, FOffsetY, 0);
  glScaled(FZoomX, FZoomY, 1);

  return FFeedbackLayer->GetSelectionInfo(X, Y);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the current zoom factors.
 *
 * @param X [out] The horizontal zoom factor.
 * @param Y [out] The vertical zoom factor.
 */
void CGenericCanvas::GetZoom(double* X, double* Y)
{
  *X = FZoomX;
  *Y = FZoomY;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Notifies all registered listeners that a change has occured and they need to invalidate their viewers.
 */
void CGenericCanvas::Invalidate(void)
{
  CListenerIterator Iterator = FListeners.begin(); 
  while (Iterator != FListeners.end())
  {
    CListenerIterator Next = Iterator + 1;
    try
    {
      CGCListener* Listener = *Iterator;
      Listener->OnInvalidate();
    }
    catch(...)
    {
      // If there was an exception while executing the method the listener is removed from our list
      // to avoid further harm.
      FListeners.erase(Iterator);
    };
    Iterator = Next;
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Tells the selection layer to recompute the bounds of the selection decoration for the given figure instance.
 *
 * @param Instance The figure instance for which to recompute the selection decoration.
 */
void CGenericCanvas::InvalidateSelectionBounds(CFigureInstance* Instance)
{
  FFeedbackLayer->InvalidateBounds(Instance);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Tells the caller whether the canvas is currently being updated.
 */
bool CGenericCanvas::IsUpdating(void)
{
  return FUpdateCount != 0;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Destroys this canvas instance. The release method is used by non-C++ languages access this code in order to avoid
 * releasing memory in an environment where it wasn't allocated.
 */
void CGenericCanvas::Release(void)
{
  delete this;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes the given figure instance from the current selection. This call is simply forwared to the selection layer.
 *
 * @param Instance The figure instance to be removed from the selection. Nothing happens if it isn't selected.
 */
void CGenericCanvas::RemoveFromSelection(CFigureInstance* Instance)
{
  FFeedbackLayer->RemoveFromSelection(Instance);
  Invalidate();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes the given layer from the internal layer list. The layer itself will not be destroed, just removed.
 *
 * @param Layer The layer to be removed.
 */
void CGenericCanvas::RemoveLayer(CLayer* Layer)
{
  for (CLayers::iterator Iterator = FLayers.begin(); Iterator != FLayers.end(); ++Iterator)
    if (*Iterator == Layer)
    {
      FLayers.erase(Iterator);
      Layer->FCanvas = NULL;
      Invalidate();
      break;
    };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes a listener from the internal list. If the listener is not in the list then this call has no effect.
 *
 * @param Listener The listener to remove.
 */
void CGenericCanvas::RemoveListener(CGCListener* Listener)
{
  for (CListenerIterator Iterator = FListeners.begin(); Iterator != FListeners.end(); ++Iterator)
    if (*Iterator == Listener)
    {
      FListeners.erase(Iterator);
      break;
    };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * This is the main paint routine. It must be called by the viewer holding the reference to this canvas (e.g. when a 
 * window must be redrawn).
 */
void CGenericCanvas::Render(void)
{
  // No display if the canvas is currently being updated.
  if (FUpdateCount == 0)
  {
    ClearBuffers();

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    // In order to get better result for pixel-exact rendering it is recommended to offset the glOrtho
    // values by a small amount.
    glOrtho(-0.25, FViewport.Width - 0.25, FViewport.Height - 0.25, -0.25, FNearPlane, FFarPlane);

    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    glTranslated(FOffsetX, FOffsetY, 0);
    glScaled(FZoomX, FZoomY, 1);

    for (CLayers::iterator Iterator = FLayers.begin(); Iterator != FLayers.end(); ++Iterator)
      (*Iterator)->Render();
    FFeedbackLayer->Render();

    CheckError();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Preparation function for CFeedbackLayer::ResizeFiguresStart.
 */
void CGenericCanvas::ResizeFiguresStart(int X, int Y, TGCDirection Direction)
{
  // Initialize our transformation queue for convertion of the given screen coordinates into layer
  // coordinates (which happens in the selection layer).
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glOrtho(-0.25, FViewport.Width - 0.25, FViewport.Height - 0.25, -0.25, FNearPlane, FFarPlane);

  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  glTranslated(FOffsetX, FOffsetY, 0);
  glScaled(FZoomX, FZoomY, 1);

  FFeedbackLayer->ResizeFiguresStart(X, Y, Direction);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Preparation function for CFeedbackLayer::ResizeFiguresStop.
 */
void CGenericCanvas::ResizeFiguresStop(void)
{
  FFeedbackLayer->ResizeFiguresStop();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Preparation function for CFeedbackLayer::ResizeFiguresTo.
 */
void CGenericCanvas::ResizeFiguresTo(int X, int Y)
{
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  glTranslated(FOffsetX, FOffsetY, 0);
  glScaled(FZoomX, FZoomY, 1);

  FFeedbackLayer->ResizeFiguresTo(X, Y);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Preparation function for CFeedbackLayer::RubberbandResize.
 */
void CGenericCanvas::RubberbandResize(int X, int Y, TRBSelectionAction Action)
{
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  glTranslated(FOffsetX, FOffsetY, 0);
  glScaled(FZoomX, FZoomY, 1);

  FFeedbackLayer->RubberbandResize(X, Y, Action);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Preparation function for CFeedbackLayer::RubberbandStart.
 */
void CGenericCanvas::RubberbandStart(TRubberbandStyle Style, int X, int Y, bool ClearSelection)
{
  // Initialize our transformation queue for convertion of the given screen coordinates into layer
  // coordinates (which happens in the feedback layer).
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glOrtho(-0.25, FViewport.Width - 0.25, FViewport.Height - 0.25, -0.25, FNearPlane, FFarPlane);

  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  glTranslated(FOffsetX, FOffsetY, 0);
  glScaled(FZoomX, FZoomY, 1);

  FFeedbackLayer->RubberbandStart(Style, X, Y, ClearSelection);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Preparation function for CFeedbackLayer::RubberbandStop.
 */
void CGenericCanvas::RubberbandStop(void)
{
  FFeedbackLayer->RubberbandStop();
}

//----------------------------------------------------------------------------------------------------------------------

void CGenericCanvas::SetBackgroundColor(GCColor NewColor)
{
#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
  if (NewColor != FBackgroundColor)
  {
    FBackgroundColor = NewColor;
    glClearColor((float) GetRValue(FBackgroundColor) / 255, (float) GetGValue(FBackgroundColor) / 255, 
      (float) GetBValue(FBackgroundColor) / 255, 1);
  }
#else
  if (NewColor.red != FBackgroundColor.red && NewColor.green != FBackgroundColor.green && NewColor.blue != FBackgroundColor.blue)
  {
    FBackgroundColor = NewColor;
    glClearColor((float) FBackgroundColor.red / 0xffff, (float) FBackgroundColor.green / 0xffff,
      (float) FBackgroundColor.blue / 0xffff, 1);
  }
#endif

}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the current display offsets.
 *
 * @param X The horizontal display offset.
 * @param Y The vertical display offset.
 */
void CGenericCanvas::SetOffset(double X, double Y)
{
  FOffsetX = X;
  FOffsetY = Y;
  Invalidate();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the the value of the given property, if it is a property of this class.
 *
 * @param Property The property to set.
 * @param Value The new value of the property.
 * @return True if the asked property is known in this class, false otherwise (in this case the property is not changed).
 * @note Implicit datatype conversion is performed if datatypes of property and Value parameter do not match.
 *       This can result in lost precision or even conversion errors. So make sure to use the right datatype for the call.
 */
bool CGenericCanvas::SetProperty(TProperty Property, double Value)
{
  bool Result = true;
  switch (Property)
  {
    case GC_PROPERTY_HANDLE_SIZE:
      {
        break;
      };
    default:
      {
        // Property is not known here so try the special classes.
        Result = FFeedbackLayer->SetProperty(Property, Value);
        break;
      };
  };

  if (Result)
    Invalidate();

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the the value of the given property, if it is a property of this class.
 *
 * @param Property The property to set.
 * @param Value The new value of the property.
 * @return True if the asked property is known in this class, false otherwise (in this case the property is not changed).
 * @note Implicit datatype conversion is performed if datatypes of property and Value parameter do not match.
 *       This can result in lost precision or even conversion errors. So make sure to use the right datatype for the call.
 */
bool CGenericCanvas::SetProperty(TProperty Property, int Value)
{
  bool Result = true;
  switch (Property)
  {
    case GC_PROPERTY_HANDLE_SIZE:
      {
        break;
      };
    default:
      {
        // Property is not known here so try the special classes.
        Result = FFeedbackLayer->SetProperty(Property, Value);
        break;
      };
  };

  if (Result)
    Invalidate();

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the viewport of the canvas. The viewport is the area in the viewer where the canvas may draw its content.
 * Hence the viewport is given in pixels in the viewer (window) space.
 *
 * @param NewViewport The new viewport to use.
 */
void CGenericCanvas::SetViewportV(TGCViewport* NewViewport)
{
  FViewport = *NewViewport;
  ApplyViewport();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the viewport of the canvas. The viewport is the area in the viewer where the canvas may draw its content.
 * Hence the viewport is given in pixels in the viewer (window) space.
 *
 * @param Left The left pixel position for the output area.
 * @param Top The upper pixel position for the output area.
 * @param Width The width for the output area.
 * @param Height The height for the output area.
 */
void CGenericCanvas::SetViewport(int Left, int Top, int Width, int Height)
{
  FViewport.Left = Left;
  FViewport.Top = Top;
  FViewport.Width = Width;
  FViewport.Height = Height;
  ApplyViewport();
}

//----------------------------------------------------------------------------------------------------------------------

void CGenericCanvas::SetZoom(double X, double Y)
{
  FZoomX = X;
  FZoomY = Y;
  Invalidate();
}

//----------------------------------------------------------------------------------------------------------------------

void CGenericCanvas::ShowSelection(bool Visible)
{
  FFeedbackLayer->SetVisible(Visible);
}

//----------------------------------------------------------------------------------------------------------------------

