// diskfile.h 

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


// DiskFile is the class wrapper for on-disk files or files to be written to
// disk.  It is a modification of the old File class from the Gnu libg++
// distribution.

#ifndef DISKFILE_H
#ifdef __GNUG__
#pragma interface
#endif
#define DISKFILE_H

#include <builtin.h>
#include <stdio.h>
#include <stddef.h>
#include <sys/ioctl.h>
#include <InterViews/resource.h>

// The following should allow the use of the State_value vars instead of
// the integer versions found on the SGI machine

#if defined(sgi) && !defined(__GNUG__)
#define _eof INT_eof
#define _fail INT_fail
#define _bad INT_bad
#include <unistd.h>
#include <fcntl.h>
#endif	/* sgi && !__GNUG__ */

enum State_value                // File states
{ 
  _good         = 0,            // all is well
  _eof          = 1,            // at eof
  _fail         = 2,            // logical or physical IO error
  _bad          = 4             // unopened/corrupted
};

enum io_mode                    // known unix file IO modes
{
  io_readonly   = 0,            
  io_writeonly  = 1,
  io_readwrite  = 2, 
  io_appendonly = 3,
  io_append     = 4            // append, plus allow reads
};

enum access_mode                // ways to open a file
{
  a_createonly  = 0,            // create, fail if file exists
  a_create      = 1,            // create if doesn't exist, else truncate
  a_useonly     = 2,            // use (no truncate)  fail if doesn't exist
  a_use         = 3            // use (no truncate), create if doesn't exist
};

class DiskFile : virtual public Resource {
protected:
	FILE*		_fp;              // _iobuf file pointer
	char*		_nm;		// file name (dynamically allocated)
	char		_rw;		//  1 = read; 2 = write; 3 = readwrite
	State_value	_state;		// _good/_eof/_fail/_bad
	long		_stat;		// last read/write/... return value

	void	initialize();
	void	reinitialize(const char*);
	static char* fopen_cmd_arg(io_mode);
public:
	DiskFile();
	virtual		~DiskFile();

// binding, rebinding, unbinding to physical files

	virtual DiskFile&	open(const char* filename, io_mode m, access_mode a);
	virtual DiskFile&	open(const char* filename, const char* m);
	DiskFile& 			reOpen(const char* m);
	void *memmap(int, int);
	void unmap(void *, int);

	virtual DiskFile&	close();
	virtual DiskFile&	remove();
	virtual DiskFile&	rename(const char* newname);

	operator FILE *() { return _fp; }
	int	fdesc() const;
	virtual const char*	name();
	void		setname(const char* newname);
	int		iocount();
	int		rdstate();
	int		eof();
	int		fail();
	int		bad();
	int		good();
	
	// other status queries
	
	virtual int	readable() const;
	virtual int	writable();
	virtual int	is_open();
	
	static int exists(const char* filename);
	
	virtual		operator void*();
	
	// error handling
	
	void		error();
	void		clear(State_value f = _good); // poorly named
	void		set(State_value f); // set corresponding but
	void		unset(State_value f); // clear corresponding bit
	DiskFile&		failif(int cond);
	virtual void	check_state();
	
	// binary IO
	
	virtual DiskFile&	read(void* b, int bytes);
	virtual DiskFile&	write(const void* b, int bytes);

	// buffer control

	DiskFile&    setbuf(int buffer_kind); // legal vals: _IONBF, _IOFBF, _IOLBF
	DiskFile&    setbuf(int size, char* buf);
	DiskFile&    raw();
	
	// position control
	
	virtual DiskFile&	seek(long pos, int seek_mode=0);
	virtual long	tell();
};

// error handlers

extern void  verbose_File_error_handler(const char*);
extern void  quiet_File_error_handler(const char*);
extern void  fatal_File_error_handler(const char*);
extern one_arg_error_handler_t File_error_handler;
extern one_arg_error_handler_t set_File_error_handler(one_arg_error_handler_t);

inline int DiskFile::fdesc() const {
	return (_fp != 0) ? fileno(_fp) : 0;
}

inline const char* DiskFile::name() { 
	return _nm; 
}

inline int DiskFile::iocount() { 
	return _stat; 
}

inline void DiskFile::clear(State_value flag) { 
	_state = flag;
}

inline void DiskFile::set(State_value flag) { 
	_state = State_value(int(_state) | int(flag));
}

inline void DiskFile::unset(State_value flag) { 
	_state = State_value(int(_state) & ~int(flag));
}

inline DiskFile& DiskFile::failif(int cond) { 
	if (cond) set(_fail);  return *this; 
}

inline int DiskFile::is_open() { 
	return (_fp != 0);
}

inline int DiskFile::rdstate() { 
	check_state();  return _state; // check_state is necessary in rare but
}				      // possible circumstances

inline DiskFile::operator void*() { 
	check_state(); return (int(_state) & (int(_bad)|int(_fail)))? 0 : this ; 
}

inline int DiskFile::eof() { 
	check_state(); return _state & _eof; 
}

inline int DiskFile::fail() { 
	check_state(); return _state & _fail; 
}

inline int DiskFile::bad() { 
 	check_state(); return _state & _bad; 
}

inline int DiskFile::good() { 
	check_state(); return rdstate() == _good; 
}

inline int DiskFile::readable() const
{ 
  if (_fp != 0) {
      if (feof(_fp))
	  ((DiskFile &)(*this)).set(_eof); // ugh
      if (ferror(_fp))
	  ((DiskFile &)(*this)).set(_bad); // ugh
  }
  return (_state == _good && (_rw & 01));
}

inline int DiskFile::writable()
{ 
  if (_fp != 0 && ferror(_fp)) set(_bad);
  return ((int(_state) & (int(_fail)|int(_bad))) == 0 && (_rw & 02));
}

inline DiskFile& DiskFile::seek(long pos, int seek_mode) { 
	return failif(!is_open() || fseek(_fp, pos, seek_mode) < 0); 
}

inline long DiskFile::tell() { 
	failif(!is_open() || ((_stat = ftell(_fp)) < 0));
	return _stat;
}

inline DiskFile& DiskFile::raw()
{ 
  return this->DiskFile::setbuf(_IONBF); 
}

inline DiskFile& DiskFile::read(void* buf, int bytes) {
	return failif (!readable() || (_stat = fread((char *)buf, bytes, 1, _fp)) != 1);
}

inline DiskFile& DiskFile::write(const void* buf, int bytes) { 
	return failif (!writable() || (_stat = fwrite((const char *)buf, bytes, 1, _fp)) != 1);
}

#endif /* DISKFILE_H */

