//=========================================================
//  MusE
//  Linux Music Editor
//  $Id: wave.cpp,v 1.2 2002/02/13 11:42:56 muse Exp $
//
//  (C) Copyright 2000 Werner Schweer (ws@seh.de)
//=========================================================

#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <sys/mman.h>
#include <errno.h>

#include <qmessagebox.h>

#include "xml.h"
#include "song.h"
#include "wave.h"
#include "app.h"
#include "device.h"
#include "filedialog.h"
#include "arranger/arranger.h"
#include "globals.h"
#include "event.h"
#include "sf/sndfile.h"
#include "midithread.h"

const char* audioFilePattern[] = {
      "Wave/Binary (*.wav *.bin)",
      "Wave (*.wav)",
      "Binary (*.bin)",
      "All Files (*)",
      0
      };

//---------------------------------------------------------
//   getSnd
//---------------------------------------------------------

SndFile* getWave(const QString& name)
      {
      SndFile* f = sndFiles->search(name);
      if (f == 0) {
            f = new SndFile(name);
            if (f->openRead()) {
                  fprintf(stderr, "open wave file(%s) failed: %s\n",
                     name.latin1(), strerror(errno));
                  delete f;
                  f = 0;
                  }
            else {
                  sndFiles->push_back(f);
                  }
            }
      return f;
      }

//---------------------------------------------------------
//   importAudio
//---------------------------------------------------------

void MusE::importWave()
      {
      Track* track = arranger->curTrack();
      if (track == 0 || track->type() != Track::WAVE) {
            QMessageBox::critical(this, "MusE",
              "to import a audio file you have first to select"
              "a audio track");
            return;
            }
      QString fn = getOpenFileName("waves", audioFilePattern, this,
         "Import Wave File");
      if (!fn.isEmpty()) {
            importWave(fn);
            }
      }

//---------------------------------------------------------
//   importWave
//---------------------------------------------------------

bool MusE::importWave(const QString& name)
      {
      WaveTrack* track = (WaveTrack*)(arranger->curTrack());
      SndFile* f = getWave(name);

      if (f == 0) {
            printf("import audio file failed\n");
            return true;
            }
      int samples = f->samples();
      track->setChannels(f->channels());
      Clip* clip = new Clip(f, 0, samples);

      WavePart* part = new WavePart(track);
      part->setPosSample(0);
      part->setLenSample(samples);

      WaveEvent* event = new WaveEvent(
         track->outPort(), track->Track::outChannel(),
         0, clip);

      part->addEvent(event);

      part->setName(QFileInfo(name).baseName());
      midiThread->msgAddPart(part);
      int endTick = part->posTick() + part->lenTick();
      if (song->len() < endTick)
            song->setLen(endTick);
      return false;
      }

//---------------------------------------------------------
//   Clip
//---------------------------------------------------------

Clip::Clip()
      {
      waveClips->add(this);
      refs = 0;
      }

Clip::Clip(SndFile* file, int start, int l)
      {
      assert(f);
      f = file;
      int irefs = f->incRef();
      _name.sprintf("%s.%d", file->basename().latin1(), irefs);
      _spos = start;
      len  = l;
      refs = 0;
      waveClips->add(this);
      }

//---------------------------------------------------------
//   read
//---------------------------------------------------------

void Clip::read(unsigned offset, float** buffer, int channel,
   unsigned n, int* meter, const double mag)
      {
      f->seek(offset + _spos, 0);
      if (channel != int(f->channels())) {
            fprintf(stderr, "Clip::read(): channel mismatch\n");
            abort();
            }
      f->read(buffer, n);

      float* f = &buffer[0][0];
      for (int k = 0; k < channel; ++k) {
            double mv = 0;
            for (unsigned i = 0; i < n; ++i) {
                  if (*f > mv)
                        mv = *f;
                  *f *= mag;
                  ++f;
                  }
            meter[k] = int(mv * 32767.0);
            }
      }

Clip::~Clip()
      {
      waveClips->remove(this);
      if (f->decRef() <= 0)
            delete f;
      }

//---------------------------------------------------------
//   incRef
//---------------------------------------------------------

void Clip::incRef()
      {
      ++refs;
//      printf("CLip: inc ref %d\n", refs);
      }

//---------------------------------------------------------
//   decRef
//---------------------------------------------------------

int Clip::decRef()
      {
      --refs;
//      printf("CLip: dec ref %d\n", refs);
      return refs;
      }

//---------------------------------------------------------
//   ClipList::write(level, xml)
//---------------------------------------------------------

void ClipList::write(int level, Xml& xml) const
      {
      for (ciClip i = begin(); i != end(); ++i)
            (*i)->write(level, xml);
      }

//---------------------------------------------------------
//   Clip::write(level, xml)
//---------------------------------------------------------

void Clip::write(int level, Xml& xml) const
      {
      xml.tag(level++, "clip");
      xml.strTag(level, "file", f->path());
      xml.strTag(level, "name", _name);
      xml.intTag(level, "tick", _spos);
      xml.intTag(level, "len", len);
      xml.etag(level, "clip");
      }

//---------------------------------------------------------
//   Clip::read
//---------------------------------------------------------

void Clip::read(Xml& xml)
      {
      for (;;) {
            Xml::Token token = xml.parse();
            const QString& tag = xml.s1();
            switch (token) {
                  case Xml::Error:
                  case Xml::End:
                        return;
                  case Xml::TagStart:
                        if (tag == "file") {
                              f = getWave(xml.parse1());
                              if (f)
                                    f->incRef();
                              }
                        else if (tag == "name")
                              _name = xml.parse1();
                        else if (tag == "tick")
                              _spos = xml.parseInt();
                        else if (tag == "len")
                              len = xml.parseInt();
                        else
                              xml.unknown("Clip");
                        break;
                  case Xml::TagEnd:
                        if (tag == "clip")
                              return;
                  default:
                        break;
                  }
            }
      }

//---------------------------------------------------------
//   search
//---------------------------------------------------------

Clip* ClipList::search(const QString& name) const
      {
      for (ciClip i = begin(); i != end(); ++i)
            if ((*i)->name() == name)
                  return *i;
      printf("ClipList: clip <%s> not found\n", name.latin1());
      return 0;
      }

//---------------------------------------------------------
//   remove
//---------------------------------------------------------

void ClipList::remove(Clip* clip)
      {
      for (iClip i = begin(); i != end(); ++i) {
            if (*i == clip) {
                  erase(i);
                  return;
                  }
            }
      printf("ClipList:remove: clip not found\n");
      }

//---------------------------------------------------------
//   idx
//---------------------------------------------------------

int ClipList::idx(Clip* clip) const
      {
      int n = 0;
      for (ciClip i = begin(); i != end(); ++i, ++n) {
            if (*i == clip)
                  return n;
            }
      printf("ClipList: idx -1\n");
      return -1;
      }

