//=========================================================
//  MusE
//  Linux Music Editor
//  $Id: app.cpp,v 1.16 2002/02/28 09:02:25 muse Exp $
//
//  (C) Copyright 1999/2000 Werner Schweer (ws@seh.de)
//=========================================================

#include "config.h"

#include <assert.h>
#include <getopt.h>
#include <errno.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>
#include <signal.h>
#include <stdarg.h>

#include <qbuttongroup.h>
#include <qpopupmenu.h>
#include <qmessagebox.h>
#include <qclipboard.h>
#include <qsocketnotifier.h>
#include <qtextcodec.h>

#include "app.h"
#include "transport.h"
#include "bigtime.h"
#include "arranger.h"
#include "pianoroll.h"
#include "score.h"
#include "xml.h"
#include "midi.h"
#include "conf.h"
#include "listedit.h"
#include "master/masteredit.h"
#include "master/lmaster.h"
#include "drumedit.h"
#include "ttoolbar.h"
#include "mixer.h"
#include "mixer/amixer.h"
#include "cliplist/cliplist.h"
#include "midiport.h"
#include "audioport.h"
#include "audiodev.h"
#include "driver/mididev.h"
#include "waveedit.h"
#include "sf/sndfile.h"
#include "icons.h"
#include "minstrument.h"
#include "mixdowndialog.h"
// #include "midiitransform.h"
#include "midictrl.h"
#include "midithread.h"

#include "filedialog.h"
#include "plugins/plugin.h"
#include "marker/markerview.h"
#include "transpose.h"
#include "appearance.h"
#include "themes/themes.h"
#include "seq.h"
#include "gatetime.h"
#include "metronome.h"
#include "debug.h"

extern void initIcons();

const char* fileOpenText =
      "Click this button to open a <em>new song</em>.<br>"
      "You can also select the <b>Open command</b> from the File menu.";
const char* fileSaveText = "Click this button to save the song you are "
      "editing.  You will be prompted for a file name.\n"
      "You can also select the Save command from the File menu.";
const char* fileNewText     = "Create New Song";
const char* fileArchiveText = "select new song from midi file archive";

#define PROJECT_LIST_LEN  6
QString* projectList[PROJECT_LIST_LEN];

bool configLoaded = false;

#ifdef AUDIO

#ifdef OSS
extern bool initOssAudio();
#endif

#ifdef ALSA
extern void initMidiSynth();
extern bool  initAlsaAudio();
#ifdef ALSACVS
#include <alsa/asoundlib.h>
#else
#include <sys/asoundlib.h>
#endif

extern void initMidiSynth();

#if (SND_LIB_MAJOR==0) && (SND_LIB_MINOR==9)
static void error_handler(const char *file, int line, const char *function, int err, const char *fmt, ...)
      {
      va_list arg;

      if (err == ENOENT) {      // Ignore those misleading "warnings"
            return;
            }
      va_start(arg, fmt);
      fprintf(stderr, "ALSA lib %s:%i:(%s) ", file, line, function);
      vfprintf(stderr, fmt, arg);
      if (err)
            fprintf(stderr, ": %s", snd_strerror(err));
      putc('\n', stderr);
      va_end(arg);
      }
#endif      // SND_LIB_MAJOR==0 && SND_LIB_MINOR==9
#endif      // ALSA

#ifdef JACK
extern bool initJackAudio();
#endif

#endif      // AUDIO

//---------------------------------------------------------
//   addProject
//---------------------------------------------------------

static void addProject(const QString& name)
      {
      for (int i = 0; i < PROJECT_LIST_LEN; ++i) {
            if (projectList[i] == 0)
                  break;
            if (name == *projectList[i]) {
                  int dst = i;
                  int src = i+1;
                  int n = PROJECT_LIST_LEN - i - 1;
                  delete projectList[i];
                  for (int k = 0; k < n; ++k)
                        projectList[dst++] = projectList[src++];
                  projectList[dst] = 0;
                  break;
                  }
            }
      QString** s = &projectList[PROJECT_LIST_LEN - 2];
      QString** d = &projectList[PROJECT_LIST_LEN - 1];
      if (*d)
            delete *d;
      for (int i = 0; i < PROJECT_LIST_LEN-1; ++i)
            *d-- = *s--;
      projectList[0] = new QString(name);
      }

//---------------------------------------------------------
//   MusE
//---------------------------------------------------------

MusE::MusE(int argc, char** argv) : MainWindow(0, "mainwindow")
      {
      midiMixer             = 0;
      clipListEdit          = 0;
      midiSyncConfig        = 0;
      midiRemoteConfig      = 0;
      midiPortConfig        = 0;
      printerConfig         = 0;
      metronomeConfig       = 0;
      audioConfig           = 0;
      midiFileConfig        = 0;
      midiFilterConfig      = 0;
      midiInputTransform    = 0;
      midiRhythmGenerator   = 0;
      globalSettingsConfig  = 0;
      markerView            = 0;
      helpViewer            = 0;
      softSynthesizerConfig = 0;
      midiTransformerDialog = 0;
      printerType           = 1;        // print to file
      printerFile           = "mops.ps";
      printerCommand        = "lpr -";
      previewCommand        = "gv %s&";
      appName               = "MusE";

      configName            = getenv("HOME");
      configName           += "/.MusE";
      configTheme	          = "default";
      _currentTheme	    = configTheme;

      song = new Song("song");

      //---------------------------------------------------
      //    undo/redo
      //---------------------------------------------------

      undoRedo = new QActionGroup(this, "UndoRedo", false);
      undoAction = new QAction("undo", QIconSet(*undoIconS, *undoIcon), "Und&o",
        CTRL+Key_Z, undoRedo, "undo");
      redoAction = new QAction("redo", QIconSet(*redoIconS, *redoIcon), "Re&do",
        CTRL+Key_Y, undoRedo, "redo");
      undoAction->setWhatsThis("undo last change to song");
      redoAction->setWhatsThis("redo last undo");
      undoAction->setEnabled(false);
      redoAction->setEnabled(false);
      connect(redoAction, SIGNAL(activated()), song, SLOT(redo()));
      connect(undoAction, SIGNAL(activated()), song, SLOT(undo()));

      initMidiInstruments();
      initMidiPorts();
      initMidiDevices();
      configGeometryMain.setHeight(300);
      //_fontSize = qApp->font().pixelSize();


      //----Actions

//      QMimeSourceFactory::defaultFactory()->setPixmap("fileopen", *openIcon);

      QAction* fileNewAction = new QAction("new",
        QIconSet(*filenewIconS, *filenewIcon), "&New", CTRL+Key_N, this, "new");
      fileNewAction->setToolTip(fileNewText);

      QAction* fileOpenAction = new QAction("open",
        QIconSet(*openIconS, *openIcon), "&Open", CTRL+Key_O, this, "open");
      fileOpenAction->setToolTip(fileOpenText);

      QAction* fileSaveAction = new QAction("save",
        QIconSet(*saveIconS, *saveIcon), "&Save", CTRL+Key_S, this, "save");
      fileSaveAction->setToolTip(fileSaveText);

      QAction* pianoAction = new QAction("pianoroll",
        *pianoIconSet, "Pianoroll", CTRL+Key_E, this, "pianoroll");
      connect(pianoAction, SIGNAL(activated()), SLOT(startPianoroll()));

      QAction* scoreAction = new QAction("score",
        *scoreIconSet, "Score", CTRL+Key_R, this, "score");
      connect(scoreAction, SIGNAL(activated()), SLOT(startScoreEditor()));

      QAction* markerAction = new QAction("marker", QIconSet(*flagIcon), "Marker",
        0, this, "marker");
      connect(markerAction, SIGNAL(activated()), SLOT(startMarkerView()));

      connect(fileNewAction,     SIGNAL(activated()), SLOT(newSong()));
      connect(fileOpenAction,    SIGNAL(activated()), SLOT(loadProject()));
      connect(fileSaveAction,    SIGNAL(activated()), SLOT(save()));

      //--------------------------------------------------
      //    Toolbar
      //--------------------------------------------------

      tools = new ToolBar("File Buttons", this);
      fileNewAction->addTo(tools);
      fileOpenAction->addTo(tools);
      fileSaveAction->addTo(tools);

      //
      //    Whats This
      //
      QWhatsThis::whatsThisButton(tools);

      tools->addSeparator();
      undoRedo->addTo(tools);

      ToolBar* tools1 = new EditToolBar(this,
         PointerTool | PencilTool | RubberTool | CutTool | GlueTool);

      new TransportToolbar(this, "pr-tr-tools");

      seq = new Sequencer();

      //---------------------------------------------------
      //    Popups
      //---------------------------------------------------
      //-------------------------------------------------------------
      //    popup File
      //-------------------------------------------------------------

      menu_file = new QPopupMenu(this);
      menuBar()->insertItem(tr("&File"), menu_file);
      fileNewAction->addTo(menu_file);
      fileOpenAction->addTo(menu_file);
      openRecent = new QPopupMenu(this);
      connect(openRecent, SIGNAL(aboutToShow()), this, SLOT(openRecentMenu()));
      connect(openRecent, SIGNAL(activated(int)), this, SLOT(selectProject(int)));
      menu_file->insertItem(tr("Open &Recent"), openRecent, CTRL+Key_S);
      menu_file->insertSeparator();
      fileSaveAction->addTo(menu_file);
      menu_file->insertItem(tr("Save &As"), this, SLOT(saveAs()), CTRL+Key_A, -2);
      menu_file->insertSeparator();
      menu_file->insertItem(*printIconS, tr("Config &Printer"), this, SLOT(configPrinter()), CTRL+Key_P, -2);
      menu_file->insertSeparator();
      menu_file->insertItem(*openIconS, tr("Import Midifile"), this, SLOT(importMidi()), 0, -2);
      menu_file->insertItem(*saveIconS, tr("Export Midifile"), this, SLOT(exportMidi()), 0, -2);
      menu_file->insertSeparator();
      menu_file->insertItem(*openIconS, tr("Import Wave File"), this, SLOT(importWave()), 0, -2);
      menu_file->insertSeparator();
      menu_file->insertItem(*exitIconS, tr("&Quit"), this, SLOT(quitDoc()), CTRL+Key_Q, -2);
      menu_file->insertSeparator();

      //-------------------------------------------------------------
      //    popup Edit
      //-------------------------------------------------------------

      menuEdit = new QPopupMenu(this);
      undoRedo->addTo(menuEdit);
      menuEdit->insertSeparator();
      menuBar()->insertItem(tr("&Edit"), menuEdit);

      menuEdit->insertItem(*editcutIconSet, tr("C&ut"),   CMD_CUT);
      menuEdit->setAccel(CTRL+Key_X, CMD_CUT);
      menuEdit->insertItem(*editcopyIconSet, tr("&Copy"),  CMD_COPY);
      menuEdit->setAccel(CTRL+Key_C, CMD_COPY);
      menuEdit->insertItem(*editpasteIconSet, tr("&Paste"), CMD_PASTE);
      menuEdit->setAccel(CTRL+Key_V, CMD_PASTE);
      menuEdit->insertSeparator();
      menuEdit->insertItem(tr("Delete Track"), CMD_DELETE_TRACK);
      menuEdit->setAccel(Key_Delete, CMD_DELETE_TRACK);
      menuEdit->insertItem(tr("Add Track"), CMD_ADD_TRACK);
      menuEdit->setAccel(CTRL+Key_J, CMD_ADD_TRACK);
      QPopupMenu* select = new QPopupMenu(this);
      select->insertItem(tr("Select &All"),  CMD_SELECT_ALL);
      select->setAccel(CTRL+Key_Z, CMD_SELECT_ALL);
      select->insertItem(tr("&Deselect All"), CMD_SELECT_NONE);
      select->setAccel(CTRL+Key_B, CMD_SELECT_NONE);
      menuEdit->insertSeparator();
      select->insertItem(tr("Invert &Selection"), CMD_SELECT_INVERT);
      select->insertItem(tr("&Inside Loop"), CMD_SELECT_ILOOP);
      select->insertItem(tr("&Outside Loop"), CMD_SELECT_OLOOP);
      select->insertItem(tr("All &Parts on Track"), CMD_SELECT_PARTS);
      menuEdit->insertItem(tr("Select"), select);
      menuEdit->insertSeparator();
      pianoAction->addTo(menuEdit);
      scoreAction->addTo(menuEdit);
      menuEdit->insertItem(tr("Drums"), this, SLOT(startDrumEditor()), CTRL+Key_D);
      menuEdit->insertItem(tr("List"), this, SLOT(startListEditor()), CTRL+Key_L);

      QPopupMenu* master = new QPopupMenu(this);
      master->setCheckable(false);
      master->insertItem(tr("Graphic"), this, SLOT(startMasterEditor()), CTRL+Key_M);
      master->insertItem(tr("List"), this, SLOT(startLMasterEditor()), SHIFT+CTRL+Key_M);
      menuEdit->insertItem(tr("Mastertrack"), master, Key_F);

      markerAction->addTo(menuEdit);
      connect(menuEdit, SIGNAL(activated(int)), SLOT(cmd(int)));
      connect(select, SIGNAL(activated(int)), SLOT(cmd(int)));

      menuEdit->insertItem(tr("Midi &Transform"), this, SLOT(startMidiTransformer()), CTRL+Key_T);
      QPopupMenu* midiEdit = new QPopupMenu(this);
      midiEdit->setCheckable(false);
#if 0  // TODO
      midiEdit->insertItem(tr("Modify Gate Time"), this, SLOT(modifyGateTime()));
      midiEdit->insertItem(tr("Modify Velocity"),  this, SLOT(modifyVelocity()));
      midiEdit->insertItem(tr("Crescendo"),        this, SLOT(crescendo()));
      midiEdit->insertItem(tr("Transpose"),        this, SLOT(transpose()));
      midiEdit->insertItem(tr("Thin Out"),         this, SLOT(thinOut()));
      midiEdit->insertItem(tr("Erase Event"),      this, SLOT(eraseEvent()));
      midiEdit->insertItem(tr("Note Shift"),       this, SLOT(noteShift()));
      midiEdit->insertItem(tr("Move Clock"),       this, SLOT(moveClock()));
      midiEdit->insertItem(tr("Copy Measure"),     this, SLOT(copyMeasure()));
      midiEdit->insertItem(tr("Erase Measure"),    this, SLOT(eraseMeasure()));
      midiEdit->insertItem(tr("Delete Measure"),   this, SLOT(deleteMeasure()));
      midiEdit->insertItem(tr("Create Measure"),   this, SLOT(createMeasure()));
      midiEdit->insertItem(tr("Mix Track"),        this, SLOT(mixTrack()));
#endif
      midiEdit->insertItem(tr("Transpose"),        this, SLOT(transpose()));
      menuEdit->insertItem(tr("Midi"), midiEdit);

      //-------------------------------------------------------------
      //    popup Display
      //-------------------------------------------------------------

      menu_display = new QPopupMenu(this);
      menu_display->setCheckable(false);
      menuBar()->insertItem(tr("&Display"), menu_display);

      tr_id = menu_display->insertItem(tr("Transport Panel"), this, SLOT(toggleTransport()), Key_F11);
      bt_id = menu_display->insertItem(tr("Bigtime window"), this, SLOT(toggleBigTime()), Key_F12);
      menu_display->setCheckable(true);

      //-------------------------------------------------------------
      //    popup Config
      //-------------------------------------------------------------

      follow = new QPopupMenu(this);
      follow->setCheckable(false);
      fid0 = follow->insertItem(tr("dont follow Song"), CMD_FOLLOW_NO);
      fid1 = follow->insertItem(tr("follow page"), CMD_FOLLOW_JUMP);
      fid2 = follow->insertItem(tr("follow continuous"), CMD_FOLLOW_CONTINUOUS);
      follow->setItemChecked(fid1, true);
      connect(follow, SIGNAL(activated(int)), SLOT(cmd(int)));

      menu_config = new QPopupMenu(this);
      menu_config->setCheckable(false);
      menuBar()->insertItem(tr("&Config"), menu_config);
      menu_config->insertItem(tr("Global Settings"), this, SLOT(configGlobalSettings()));
      menu_config->insertItem(tr("follow song"), follow, Key_F);
      menu_config->insertItem(tr("Metronome"), this, SLOT(configMetronome()));
      menu_config->insertSeparator();
      menu_config->insertItem(tr("Midi Sync"), this, SLOT(configMidiSync()));
      menu_config->insertItem(tr("Midi File Config"), this, SLOT(configMidiFile()));
      menu_config->insertSeparator();
      menu_config->insertItem(tr("Appearance settings"), this, SLOT(configAppearance()));
      menu_config->insertSeparator();
      menu_config->insertItem(tr("Soft Synthesizer"), this, SLOT(configSoftSynthesizer()));
      menu_config->insertItem(tr("Midi Ports"), this, SLOT(configMidiPorts()));
      menu_config->insertItem(tr("Audio System"), this, SLOT(configAudioPorts()));
      menu_config->insertSeparator();
      menu_config->insertItem(tr("Save Configuration"), this, SLOT(writeConfiguration()));

      //-------------------------------------------------------------
      //    popup Midi
      //-------------------------------------------------------------

      midiInputPlugins = new QPopupMenu(this);
      midiInputPlugins->setCheckable(false);
      mpid0 = midiInputPlugins->insertItem(tr("Transpose"), 0);
      mpid1 = midiInputPlugins->insertItem(tr("Midi Input Transform"), 1);
      mpid2 = midiInputPlugins->insertItem(tr("Midi Input Filter"), 2);
      mpid3 = midiInputPlugins->insertItem(tr("Midi Remote Control"), 3);
      mpid4 = midiInputPlugins->insertItem(tr("Random Rhythm Generator"), 4);
      connect(midiInputPlugins, SIGNAL(activated(int)), SLOT(startMidiInputPlugin(int)));

      QPopupMenu* menu_functions = new QPopupMenu(this);
      menu_functions->setCheckable(false);
      menuBar()->insertItem(tr("&Midi"), menu_functions);
//      menu_functions->insertTearOffHandle();
      menu_functions->setCaption(tr("Midi"));
      menu_functions->insertItem(tr("Mixer"), this, SLOT(startMidiMixer()));
      menu_functions->insertItem(tr("Define Controller"), this, SLOT(startDefineController()));
      menu_functions->insertItem(tr("Input Plugins"), midiInputPlugins, Key_P);
      menu_functions->insertSeparator();
      menu_functions->insertItem(tr("Reset Instr."), midiThread, SLOT(resetDevices()));
      menu_functions->insertItem(tr("Init Instr."), midiThread, SLOT(msgInitDevices()));
      menu_functions->insertItem(tr("local off"), midiThread, SLOT(localOff()));

      //-------------------------------------------------------------
      //    popup Audio
      //-------------------------------------------------------------

      QPopupMenu* menu_audio = new QPopupMenu(this);
      menuBar()->insertItem(tr("&Audio"), menu_audio);
      menu_audio->setCaption(tr("Audio"));
      int aid1 = menu_audio->insertItem(tr("Mixer"), this, SLOT(startAudioMixer()));
      int aid2 = menu_audio->insertItem(tr("Cliplist"), this, SLOT(startClipList()));
      int aid3 = menu_audio->insertItem(*openIconS, tr("MixdownFile"), this, SLOT(setMixdownFile()));

      //---------------------------------------------------
      //    popup Help
      //---------------------------------------------------

      menuBar()->insertSeparator();
      menu_help = new QPopupMenu(this);
      menu_help->setCheckable(false);
      menuBar()->insertItem(tr("&Help"), menu_help);

      menu_help->insertItem(tr("Browser"), this, SLOT(helpBrowser()), Key_F1);
      menu_help->insertItem(tr("&About"), this, SLOT(about()));
      menu_help->insertItem(tr("About&Qt"), this, SLOT(aboutQt()));
      menu_help->insertSeparator();
      menu_help->insertItem(tr("What's &This"), this, SLOT(whatsThis()), SHIFT+Key_F1);

      //---------------------------------------------------
      //    Central Widget
      //---------------------------------------------------

      arranger = new Arranger(this, "arranger");
      setCentralWidget(arranger);

      connect(tools1, SIGNAL(toolChanged(int)), arranger, SLOT(setTool(int)));
      connect(arranger, SIGNAL(editPart(Track*)), SLOT(startEditor(Track*)));
      connect(arranger, SIGNAL(dropFile(const QString&)), SLOT(loadProjectFile(const QString&)));
      connect(arranger, SIGNAL(startEditor(PartList*,int)),  SLOT(startEditor(PartList*,int)));

      genTransport();
      genBigTime();

      //---------------------------------------------------
      //  read list of "Recent Projects"
      //---------------------------------------------------

      QString prjPath(getenv("HOME"));
      prjPath += "/.musePrj";
      FILE* f = fopen(prjPath.latin1(), "r");
      if (f == 0) {
            perror("open projectfile");
            for (int i = 0; i < PROJECT_LIST_LEN; ++i)
                  projectList[i] = 0;
            }
      else {
            for (int i = 0; i < PROJECT_LIST_LEN; ++i) {
                  char buffer[256];
                  if (fgets(buffer, 256, f)) {
                        int n = strlen(buffer);
                        if (n && buffer[n-1] == '\n')
                              buffer[n-1] = 0;
                        projectList[i] = *buffer ? new QString(buffer) : 0;
                        }
                  else
                        break;
                  }
            fclose(f);
            }

      bool noAlsa = true;
      bool noOss = true;
      bool noJack = true;
      if (!noAudio) {
#ifdef AUDIO
#ifdef JACK
            noJack = initJackAudio();
#endif
#ifdef ALSA
#if (SND_LIB_MAJOR==0) && (SND_LIB_MINOR==9)
            snd_lib_error_set_handler(error_handler);
#endif
            noAlsa = initAlsaAudio();
#endif      // ALSA

#ifdef OSS
            noOss = initOssAudio();
#endif
#endif  // AUDIO
            }

      if (noAudio || (noAlsa && noOss && noJack)) {
            noAudio = true;
            menu_audio->setItemEnabled(aid1, false);
            menu_audio->setItemEnabled(aid2, false);
            menu_audio->setItemEnabled(aid3, false);
            }
      else {
#ifdef ALSA
            initMidiSynth();
#endif
            }

      //---------------------------------------------------
      //  load project
      //---------------------------------------------------

      newSongFlag = argc >= 2;

      QString name;
      if (argc < 2) {
            name = projectList[0] ? *projectList[0] : "";
            }
      else
            name = argv[0];

      // set defaults:
      configTransportHandleColor   = transport->getHandleColor();
      configActivityColor          = arranger->getActivityColor();
      configActivityMode           = arranger->getActivityMode();
      configSelectedTrackColor     = arranger->getSelectedTrackColor();
      configBigTimeVisible         = false;
      configBigTimeForegroundColor = bigtime->getFgColor();
      configBigTimeBackgroundColor = bigtime->getBgColor();

      loadProjectFile(name);

      if (!configLoaded) {
            // open default devices:
            if (midiPorts[0].device() == 0) {
                  if (!audioDevices.empty()) {
                        MidiDevice* dev = midiDevices.front();
                        midiThread->setMidiDevice(&midiPorts[0], dev);
                        }
                  }
            if (audioPort.device() == 0 && !noAudio) {
                  if (!audioDevices.empty()) {
                        AudioDevice* dev = audioDevices.front();
                        audioThread->msgSetAudioDevice(&audioPort, dev);
                        }
                  }
            }

      if (_currentTheme != configTheme)
            loadTheme(configTheme);
      if (configFontSize) {
            QFont font = QApplication::font();
            font.setPointSize(configFontSize);
            QApplication::setFont(font, true);
            }

      menu_display->setItemChecked(tr_id, configTransportVisible);
      menu_display->setItemChecked(bt_id, configBigTimeVisible);
      if (configTransportVisible)
            transport->show();
      if (configBigTimeVisible)
            bigtime->show();

      setGeometry(configGeometryMain);
      transport->setGeometry(configGeometryTransport);
      transport->setMasterFlag(song->masterFlag());
      transport->setHandleColor(configTransportHandleColor);

      if (configGeometryBigTime.width())
            bigtime->setGeometry(configGeometryBigTime);
      bigtime->setFgColor(configBigTimeForegroundColor);
      bigtime->setBgColor(configBigTimeBackgroundColor);

      arranger->setActivityColor(configActivityColor);
      arranger->setActivityMode(configActivityMode);
      arranger->setSelectedTrackColor(configSelectedTrackColor);

      song->updatePos();
      connect(song, SIGNAL(recordChanged(bool)), SLOT(checkRecord(bool)));
      QClipboard* cb = QApplication::clipboard();
      connect(cb, SIGNAL(dataChanged()), SLOT(clipboardChanged()));
      connect(arranger, SIGNAL(selectionChanged()), SLOT(selectionChanged()));
      clipboardChanged(); // enable/disable "Paste"
      selectionChanged(); // enable/disable "Copy" & "Paste"
      seq->startThread();
      //
      // after starting the Sequencer set effective user id to uid;
      // this resets root rights, if startet as suid root
      //
      setuid(getuid());
      song->update();
      }

//---------------------------------------------------------
//   loadProjectFile
//    load *.med, *.mid, *.kar
//---------------------------------------------------------

void MusE::loadProjectFile(const QString& name)
      {
      if (name.isEmpty()) {
            if (clearSong())
                  return;
            readConfiguration();
            song->update();

	      // only open default.med if file exists
            int default_med = open("default.med", O_RDONLY);
            if (default_med > 0) {
                  close(default_med);
	            project.setFile("default.med");
	            }

            setCaption(QString("Song: untitled")); // + project.baseName()
            newSongFlag = true;
            return;
            }
      project.setFile(name);
      QString ex = project.extension(true).lower();
      if (ex.length() == 3)
            ex += ".";
      ex = ex.left(4);
      if (ex.isEmpty() || ex == "med.") {
            load();
            return;
            }
      if (!configLoaded)
            readConfiguration();
      if (ex == "mid." || ex == "kar.") {
            importMidi(name);
            return;
            }
      QMessageBox::critical(this, "MusE", "Unknown File Format");
      }

//---------------------------------------------------------
//   load
//    load "project"
//---------------------------------------------------------

void MusE::load()
      {
      // reset midi ports to "unknown state"
      for (int i = 0; i < MIDI_PORTS; ++i) {
            for (int k = 0; k < MIDI_CHANNELS; ++k) {
                  midiPorts[i].resetIstate(k, false);
                  }
            }
      song->removeSoftSynth();
      bool popenFlag;
      FILE* f = fileOpen(this, project.filePath(), ".med", "r", popenFlag, newSongFlag);
      if (f == 0) {
            song->removeSoftSynth();
            readConfiguration();
            if (newSongFlag) {
                  newSongFlag = false;
                  song->dirty = true;
                  addProject(project.absFilePath());
                  song->update();
                  setCaption(QString("Song: ") + project.baseName());
                  }
            return;
            }
      if (clearSong())
            return;
      song->removeSoftSynth();
      newSongFlag = false;
      Xml xml(f);
      read(xml);
      if (ferror(f)) {
            popenFlag ? pclose(f) : fclose(f);
            song->clear();
            song->setName("none");
            readConfiguration();
            newSongFlag = true;
            return;
            }

      popenFlag ? pclose(f) : fclose(f);
      addProject(project.absFilePath());
      song->update();
      arranger->soloChanged(song->solo());

//      seq->setRecTrack(dynamic_cast<MidiTrack*>(arranger->curTrack()));
      setCaption(QString("Song: ") + project.baseName());
      setFollow();
      }

//---------------------------------------------------------
//   setFollow
//---------------------------------------------------------

void MusE::setFollow()
      {
      Song::FollowMode fm = song->follow();
      follow->setItemChecked(fid0, fm == Song::NO);
      follow->setItemChecked(fid1, fm == Song::JUMP);
      follow->setItemChecked(fid2, fm == Song::CONTINUOUS);
      }

//---------------------------------------------------------
//   genTransport
//    generates the transport widget
//---------------------------------------------------------

void MusE::genTransport()
      {
      transport = new Transport(this, "transport");
      }

//---------------------------------------------------------
//   genBigTime
//    generates the bigtime widget
//---------------------------------------------------------

void MusE::genBigTime()
      {
      bigtime = new BigTime(0, this);
      connect(song, SIGNAL(posChanged(int, int, bool)), bigtime, SLOT(setPos(int, int, bool)));
      }

//---------------------------------------------------------
//   setTransportHandleColor
//---------------------------------------------------------

void MusE::setTransportHandleColor(QColor c)
      {
      if (transport)
	      transport->setHandleColor(c);
      }


//---------------------------------------------------------
//   setBigTimeForegroundColor
//---------------------------------------------------------

void MusE::setBigTimeForegroundColor(QColor c)
      {
      if (bigtime)
	      bigtime->setFgColor(c);
      }

//---------------------------------------------------------
//   setBigTimeBackgroundColor
//---------------------------------------------------------

void MusE::setBigTimeBackgroundColor(QColor c)
      {
      if (bigtime)
	      bigtime->setBgColor(c);
      }

//---------------------------------------------------------
//   getTransportHandleColor
//---------------------------------------------------------

QColor MusE::getTransportHandleColor()
      {
      if (transport)
	      return(transport->getHandleColor());

      return QColor(0, 0, 0xff);
      }

//---------------------------------------------------------
//   getBigTimeForegroundColor
//---------------------------------------------------------

QColor MusE::getBigTimeForegroundColor()
      {
      if (bigtime)
	      return(bigtime->getFgColor());

      return QColor(0xff, 0, 0);
      }

//---------------------------------------------------------
//   getBigTimeBackgroundColor
//---------------------------------------------------------

QColor MusE::getBigTimeBackgroundColor()
      {
      if (bigtime)
	      return(bigtime->getBgColor());

      return QColor(0, 0, 0);
      }

//---------------------------------------------------------
//   MusE::~MusE
//---------------------------------------------------------

MusE::~MusE()
      {
      }

//---------------------------------------------------------
//   newSong
//---------------------------------------------------------

void MusE::newSong()
      {
      loadProjectFile(QString());
      }

//---------------------------------------------------------
//   MusE::loadProject
//---------------------------------------------------------

void MusE::loadProject()
      {
      QString fn = getOpenFileName("", med_file_pattern, this);
      if (!fn.isEmpty())
            loadProjectFile(fn);
      if(QFileInfo(fn).isFile())
            museProject = QFileInfo(fn).dirPath(true);
      }

//---------------------------------------------------------
//   save
//---------------------------------------------------------

bool MusE::save()
      {
      if (newSongFlag)
            return saveAs();
      else
            return save(project.filePath(), false);
      }

//---------------------------------------------------------
//   save
//---------------------------------------------------------

bool MusE::save(const QString& name, bool overwriteWarn)
      {
      bool popenFlag;
      FILE* f = fileOpen(this, name, ".med", "w", popenFlag, false, overwriteWarn);
      if (f == 0)
            return false;
      Xml xml(f);
      write(xml);
      if (ferror(f)) {
            QString s = "Write File\n" + name + "\nfailed: "
               + strerror(errno);
            QMessageBox::critical(this,
               "MusE: Write File failed", s);
            popenFlag? pclose(f) : fclose(f);
            unlink(name.latin1());
            return false;
            }
      else {
            popenFlag? pclose(f) : fclose(f);
            song->dirty = false;
            return true;
            }
      }

//---------------------------------------------------------
//   quitDoc
//---------------------------------------------------------

void MusE::quitDoc()
      {
      close(true);
      }

//---------------------------------------------------------
//   closeEvent
//---------------------------------------------------------

extern void terminateSynthGuis();

void MusE::closeEvent(QCloseEvent*)
      {
      song->setPlay(false);
      //
      // wait for sequencer
      //
      while (song->play()) {
            printf("waiting for sequencer to stop\n");
            sleep(1);
            qApp->processEvents();
            }
      if (song->dirty) {
            int n = 0;
            n = QMessageBox::warning(this, appName,
               tr("The current Project contains unsaved data\n"
               "Save Current Project?"),
               tr("&Save"), tr("&Nosave"), tr("&Abort"), 0, 2);
            if (n == 0) {
                  if (!save())      // dont quit if save failed
                        return;
                  }
            else if (n == 2)
                  return;
            }
      seq->stopThread();
      terminateSynthGuis();

      // save "Open Recent" list
      QString prjPath(getenv("HOME"));
      prjPath += "/.musePrj";
      FILE* f = fopen(prjPath.latin1(), "w");
      if (f) {
            for (int i = 0; i < PROJECT_LIST_LEN; ++i) {
                  fprintf(f, "%s\n", projectList[i] ? projectList[i]->latin1() : "");
                  }
            fclose(f);
            }
      if (seq)
            delete seq;
      qApp->quit();
      }

//---------------------------------------------------------
//   about
//---------------------------------------------------------

void MusE::about()
      {
      QMessageBox* mb = new QMessageBox();
      mb->setCaption("MusE");
      mb->setText("Linux Music Editor\n"
       "Version " VERSION "\n"
       "(C) Copyright 1999-2001 Werner Schweer\n"
       "see http://muse.seh.de for new versions\n"
       "and more information\n"
       "Published under the GNU Public Licence");
      mb->setIconPixmap(QPixmap("muse1.png"));
      mb->exec();
      }

//---------------------------------------------------------
//   aboutQt
//---------------------------------------------------------

void MusE::aboutQt()
      {
      QMessageBox::aboutQt(this, "MusE");
      }

//---------------------------------------------------------
//   toggleTransport
//---------------------------------------------------------

void MusE::toggleTransport()
      {
      bool flag = true;
      if (transport->isVisible()) {
            transport->hide();
            flag = false;
            }
      else {
            transport->show();
            }
      menu_display->setItemChecked(tr_id, flag);
      }

//---------------------------------------------------------
//   toggleBigTime
//---------------------------------------------------------

void MusE::toggleBigTime()
      {
      bool flag = true;
      if (bigtime->isVisible()) {
            bigtime->hide();
            flag = false;
            }
      else {
            bigtime->show();
//          bigtime->raise();
            }
      menu_display->setItemChecked(bt_id, flag);
      }

//---------------------------------------------------------
//   saveAs
//---------------------------------------------------------

bool MusE::saveAs()
      {
      QString name = getSaveFileName(museProject, med_file_pattern, this);
      bool ok = false;
      if (!name.isEmpty()) {
            ok = save(name, true);
            if (newSongFlag) {
                  project.setFile(name);
                  setCaption(QString("Song: ") + project.baseName());
                  addProject(name.latin1());
                  newSongFlag = false;
                  }
            }
      if (QFileInfo(name).isFile())
            museProject = QFileInfo(name).dirPath(true);
      return ok;
      }

//---------------------------------------------------------
//   importMidi
//---------------------------------------------------------

void MusE::importMidi()
      {
      QString fn = getOpenFileName("midis", midi_file_pattern, this, "Import Midi");
      if (!fn.isEmpty()) {
            importMidi(fn);
            }
      }

//---------------------------------------------------------
//   exportMidi
//---------------------------------------------------------

void MusE::exportMidi()
      {
      MFile file("midis", ".mid");

      FILE* fp = file.open("w", midi_file_pattern, this, false, true);
      if (fp == 0)
            return;
      MidiFile mf(fp);
      mf.save();
      }

//---------------------------------------------------------
//   printVersion
//---------------------------------------------------------

static void printVersion(const char* prog)
      {
      fprintf(stderr, "%s: Linux Music Editor; Version %s\n", prog, VERSION);
      }

//---------------------------------------------------------
//   importMidi
//---------------------------------------------------------

bool MusE::importMidi(const QString name)
      {
      if (clearSong())
            return true;

      bool popenFlag;
      FILE* fp = fileOpen(this, name, ".mid", "r", popenFlag);
      if (fp == 0)
            return true;
      MidiFile mf(fp);
      bool rv = mf.read();
      popenFlag ? pclose(fp) : fclose(fp);
      if (rv)
            QMessageBox::critical(this, "MusE", tr("reading midifile failed"));
      else
            initNewSong(name.latin1());
      return rv;
      }

//---------------------------------------------------------
//   startEditor
//---------------------------------------------------------

void MusE::startEditor(PartList* pl, int type)
      {
      switch (type) {
            case 0: startPianoroll(pl); break;
            case 1: startListEditor(pl); break;
            case 2: startScoreEditor(pl); break;
            case 3: startDrumEditor(pl); break;
            case 4: startWaveEditor(pl); break;
            }
      }

//---------------------------------------------------------
//   startEditor
//---------------------------------------------------------

void MusE::startEditor(Track* t)
      {
      switch (t->type()) {
            case Track::MIDI: startPianoroll(); break;
            case Track::DRUM: startDrumEditor(); break;
            case Track::WAVE: startWaveEditor(); break;
            }
      }

//---------------------------------------------------------
//   getMidiPartsToEdit
//---------------------------------------------------------

PartList* MusE::getMidiPartsToEdit()
      {
      PartList* pl = song->getSelectedMidiParts();
      if (pl->empty()) {
            QMessageBox::critical(this, "MusE", tr("Nothing to edit"));
            return 0;
            }
      return pl;
      }

//---------------------------------------------------------
//   startPianoroll
//---------------------------------------------------------

void MusE::startPianoroll()
      {
      PartList* pl = getMidiPartsToEdit();
      if (pl == 0)
            return;
      startPianoroll(pl);
      }

void MusE::startPianoroll(PartList* pl)
      {
      PianoRoll* pianoroll = new PianoRoll(pl, this);
      pianoroll->show();
      toplevels.push_back(Toplevel(Toplevel::PIANO_ROLL, int(pianoroll), pianoroll));
      connect(pianoroll, SIGNAL(deleted(int)), SLOT(toplevelDeleted(int)));
      }

//---------------------------------------------------------
//   startScoreEditor
//---------------------------------------------------------

void MusE::startScoreEditor()
      {
      PartList* pl = getMidiPartsToEdit();
      if (pl == 0)
            return;
      startScoreEditor(pl);
      }

void MusE::startScoreEditor(PartList* pl)
      {
      Score* scoreEditor = new Score(pl);
      scoreEditor->show();
      toplevels.push_back(Toplevel(Toplevel::NOTEN, int(scoreEditor), scoreEditor));
      connect(scoreEditor, SIGNAL(deleted(int)), SLOT(toplevelDeleted(int)));
      }

//---------------------------------------------------------
//   startListenEditor
//---------------------------------------------------------

void MusE::startListEditor()
      {
      PartList* pl = getMidiPartsToEdit();
      if (pl == 0)
            return;
      startListEditor(pl);
      }

void MusE::startListEditor(PartList* pl)
      {
      ListEdit* listEditor = new ListEdit(pl);
      listEditor->show();
      toplevels.push_back(Toplevel(Toplevel::LISTE, int(listEditor), listEditor));
      connect(listEditor, SIGNAL(deleted(int)), SLOT(toplevelDeleted(int)));
      }

//---------------------------------------------------------
//   startMasterEditor
//---------------------------------------------------------

void MusE::startMasterEditor()
      {
      MasterEdit* masterEditor = new MasterEdit();
      masterEditor->show();
      toplevels.push_back(Toplevel(Toplevel::MASTER, int(masterEditor), masterEditor));
      connect(masterEditor, SIGNAL(deleted(int)), SLOT(toplevelDeleted(int)));
      }

//---------------------------------------------------------
//   startLMasterEditor
//---------------------------------------------------------

void MusE::startLMasterEditor()
      {
      LMaster* lmaster = new LMaster();
      lmaster->show();
      toplevels.push_back(Toplevel(Toplevel::LMASTER, int(lmaster), lmaster));
      connect(lmaster, SIGNAL(deleted(int)), SLOT(toplevelDeleted(int)));
      }

//---------------------------------------------------------
//   startDrumEditor
//---------------------------------------------------------

void MusE::startDrumEditor()
      {
      PartList* pl = getMidiPartsToEdit();
      if (pl == 0)
            return;
      startDrumEditor(pl);
      }

void MusE::startDrumEditor(PartList* pl)
      {
      DrumEdit* drumEditor = new DrumEdit(pl, this);
      drumEditor->show();
      toplevels.push_back(Toplevel(Toplevel::DRUM, int(drumEditor), drumEditor));
      connect(drumEditor, SIGNAL(deleted(int)), SLOT(toplevelDeleted(int)));
      }

//---------------------------------------------------------
//   startWaveEditor
//---------------------------------------------------------

void MusE::startWaveEditor()
      {
      PartList* pl = song->getSelectedWaveParts();
      if (pl->empty()) {
            QMessageBox::critical(this, "MusE", "Nothing to edit");
            return;
            }
      startWaveEditor(pl);
      }

void MusE::startWaveEditor(PartList* pl)
      {
      WaveEdit* waveEditor = new WaveEdit(pl);
      waveEditor->show();
      toplevels.push_back(Toplevel(Toplevel::WAVE, int(waveEditor), waveEditor));
      connect(waveEditor, SIGNAL(deleted(int)), SLOT(toplevelDeleted(int)));
      }

//---------------------------------------------------------
//   midiMixer
//---------------------------------------------------------

void MusE::startMidiMixer()
      {
      if (midiMixer == 0) {
            midiMixer = new Mixer();
            toplevels.push_back(Toplevel(Toplevel::M_MIXER, int(midiMixer), midiMixer));
            connect(midiMixer, SIGNAL(ctrlChanged(int,int,int,int)), song, SLOT(ctrlChanged(int,int,int,int)));
            connect(midiMixer, SIGNAL(masterVolChanged(int)), song, SLOT(masterVolChanged(int)));
            }
      midiMixer->show();
      }

//---------------------------------------------------------
//   startDefineController
//---------------------------------------------------------

void MusE::startDefineController()
      {
      configMidiController();
      }

//---------------------------------------------------------
//   startMarkerView
//---------------------------------------------------------

void MusE::startMarkerView()
      {
      if (markerView == 0) {
            markerView = new MarkerView(this);
            toplevels.push_back(Toplevel(Toplevel::MARKER, int(markerView), markerView));
//            connect(markerView, SIGNAL(deleted(int)), SLOT(toplevelDeleted(int)));
            }
      markerView->show();
      }

//---------------------------------------------------------
//   startAudioMixer
//---------------------------------------------------------

TopWin* MusE::startAudioMixer()
      {
      if (audioMixer == 0) {
            audioMixer = new AudioMixerApp();
            toplevels.push_back(Toplevel(Toplevel::A_MIXER, int(audioMixer), audioMixer));
//            connect(audioMixer, SIGNAL(deleted(int)), SLOT(toplevelDeleted(int)));
            }
      audioMixer->show();
      return audioMixer;
      }

//---------------------------------------------------------
//   startClipList
//---------------------------------------------------------

void MusE::startClipList()
      {
      if (clipListEdit == 0) {
            clipListEdit = new ClipListEdit();
            toplevels.push_back(Toplevel(Toplevel::CLIPLIST, int(clipListEdit), clipListEdit));
//            connect(clipListEdit, SIGNAL(deleted(int)), SLOT(toplevelDeleted(int)));
            }
      clipListEdit->show();
      }


//---------------------------------------------------------
//   clearSong
//    return true if aborted
//---------------------------------------------------------

bool MusE::clearSong()
      {
      if (song->dirty) {
            int n = 0;
            n = QMessageBox::warning(this, appName,
               tr("The current Project contains unsaved data\n"
               "Load overwrites current Project:\n"
               "Save Current Project?"),
               tr("&Save"), tr("&Overwrite"), tr("&Abort"), 0, 2);
            switch (n) {
                  case 0:
                        if (!save())      // abort if save failed
                              return true;
                        break;
                  case 1:
                        break;
                  case 2:
                        return true;
                  default:
                        printf("InternalError: gibt %d\n", n);
                  }
            }
      //
      //  remove soft synthis
      //
      for (iSynthI ii = synthiInstances.begin(); ii != synthiInstances.end();) {
            iSynthI iii = ii;
            ++iii;
            int port;
            for (port = 0; port < MIDI_PORTS; ++port) {
                  if (midiPorts[port].instrument() == *ii)
                        break;
                  }
            if (port != MIDI_PORTS) {
                  // synthi is attached
                  midiThread->setMidiDevice(&midiPorts[port], 0);
                  midiPorts[port].setInstrument(genericMidiInstrument);
                  }
            audioThread->msgRemoveSynthI(*ii);
            ii = iii;
            }

again:
      for (iToplevel i = toplevels.begin(); i != toplevels.end(); ++i) {
            Toplevel tl = *i;
            int obj = tl.object();
            switch (tl.type()) {
                  case Toplevel::A_MIXER:
                  case Toplevel::M_MIXER:
                  case Toplevel::CLIPLIST:
                  case Toplevel::MARKER:
                        break;
                  case Toplevel::PIANO_ROLL:
                  case Toplevel::NOTEN:
                  case Toplevel::LISTE:
                  case Toplevel::DRUM:
                  case Toplevel::MASTER:
                  case Toplevel::WAVE:
                  case Toplevel::LMASTER:
                  case Toplevel::MIXDOWN:
                        ((QWidget*)(obj))->close(true);
                        goto again;
                  }
            }
      song->clear(true);
      return false;
      }

//---------------------------------------------------------
//   fileMenu
//---------------------------------------------------------

void MusE::openRecentMenu()
      {
      openRecent->clear();
      for (int i = 0; i < PROJECT_LIST_LEN; ++i) {
            if (projectList[i] == 0)
                  break;
            const char* path = projectList[i]->latin1();
            const char* p = strrchr(path, '/');
            if (p == 0)
                  p = path;
            else
                  ++p;
            openRecent->insertItem(p, i);
            }
      }

//---------------------------------------------------------
//   selectProject
//---------------------------------------------------------

void MusE::selectProject(int id)
      {
      if (id < 0)
            return;
      assert(id < PROJECT_LIST_LEN);
      QString* name = projectList[id];
      if (name == 0)
            return;
      loadProjectFile(*name);
      }

//---------------------------------------------------------
//   toplevelDeleted
//---------------------------------------------------------

void MusE::toplevelDeleted(int tl)
      {
//      printf("toplevel deleted\n");

      for (iToplevel i = toplevels.begin(); i != toplevels.end(); ++i) {
            if (i->object() == tl) {
                  switch(i->type()) {
                        case Toplevel::M_MIXER:
                              // midiMixer = 0;
                              break;
                        case Toplevel::MARKER:
//                              markerView = 0;
                              break;
                        case Toplevel::A_MIXER:
                              // audioMixer = 0;
                              return;
//                              break;
                        case Toplevel::CLIPLIST:
//                              clipListEdit = 0;
                              break;
                        // the followin editors can exist in more than
                        // one instantiation:
                        case Toplevel::MIXDOWN:
                        case Toplevel::PIANO_ROLL:
                        case Toplevel::NOTEN:
                        case Toplevel::LISTE:
                        case Toplevel::DRUM:
                        case Toplevel::MASTER:
                        case Toplevel::WAVE:
                        case Toplevel::LMASTER:
                              break;
                        }
                  toplevels.erase(i);
                  return;
                  }
            }
      assert(false);
      }

//---------------------------------------------------------
//   initNewSong
//    wird nach dem Import eines Midi-Files aufgerufen
//---------------------------------------------------------

void MusE::initNewSong(const char* filename)
      {
      int z, n;

      song->initLen();
      sigmap.timesig(0, z, n);

      int tempo = tempomap.tempo(0);
      transport->setTimesig(z, n);
      transport->setTempo(tempo);

      bool masterF = !tempomap.empty();
      song->setMasterFlag(masterF);
      transport->setMasterFlag(masterF);

      song->updatePos();
      song->setLoop(false);

      project.setFile(filename);
      addProject(filename);

      if (song->name().isEmpty())
            song->setName(project.baseName());
      setCaption(QString("Song: ") + project.baseName());
      arranger->reset();
      arranger->setMode(int(song->mtype()));
      newSongFlag = true;
      song->dirty = false;

      //---------------------------------------------------
      //    calculate song len
      //---------------------------------------------------

      TrackList* tl = song->tracks();
      int lastTick = 0;
      for (iTrack it = tl->begin(); it != tl->end(); ++it) {
            Track* mt = *it;
            PartList* pl = mt->parts();
            for (iPart ip = pl->begin(); ip != pl->end(); ++ip) {
                  Part* mp = ip->second;
                  int lt =  mp->posTick() + mp->lenTick();
                  if (lt > lastTick)
                        lastTick = lt;
                  }
            }
      lastTick = song->roundUpBar(lastTick);
      song->setLen(lastTick);
      }

//---------------------------------------------------------
//   ctrlChanged
//    midi ctrl value changed
//---------------------------------------------------------

void MusE::ctrlChanged()
      {
      if (midiMixer)
            midiMixer->updateValues(true);      // set current values
      }


//---------------------------------------------------------
//   kbAccel
//---------------------------------------------------------

void MusE::kbAccel(int key)
      {
      switch (key) {
            case Key_Space:
                  if (song->play())
                        song->setPlay(false);
                  else if (song->cpos() != song->lpos())
                        song->setPos(0, song->lpos());
                  else
                        song->setPos(0, 0);
                  break;
            case Key_Insert:
                  song->setPlay(false);
                  break;
            case Key_Enter:
                  song->setPlay(true);
                  break;
            case Key_End:
                  if (!song->record())
                        song->setPos(0, song->lpos());
                  break;
            case Key_Down:
                  if (!song->record())
                        song->setPos(0, song->rpos());
                  break;
            case Key_Slash:
                  song->setLoop(!song->loop());
                  break;
            case Key_Asterisk:
                  if (!song->play())
                        song->setRecord(true);
                  break;
            case Key_F11:
                  toggleTransport();
                  break;
            case Key_F12:
                  toggleBigTime();
                  break;
            default:
                  if (debugMsg)
                        printf("unknown kbAccel 0x%x\n", key);
                  break;
            }
      }

#if defined(HAVE_KDE)
#define Application KApplication
#else
#define Application QApplication
#endif

//---------------------------------------------------------
//   MuseApplication
//---------------------------------------------------------

class MuseApplication : public Application {
      MusE* muse;

   public:
      MuseApplication(int& argc, char** argv)
#if defined(HAVE_KDE)
         : KApplication(argc, argv, "MusE")
#else
         : QApplication(argc, argv)
#endif
            {
            muse = 0;
            }
      void setMuse(MusE* m) { muse = m; }

      bool notify(QObject* receiver, QEvent* event) {
            bool flag = QApplication::notify(receiver, event);
            if (event->type() == QEvent::KeyPress) {
                  QKeyEvent* ke = (QKeyEvent*)event;
                  globalKeyState = ke->stateAfter();
                  bool accepted = ke->isAccepted();
                  if (!accepted && seq) {
                        muse->kbAccel(ke->key());
                        return true;
                        }
                  }
            if (event->type() == QEvent::KeyRelease) {
                  QKeyEvent* ke = (QKeyEvent*)event;
                  globalKeyState = ke->stateAfter();
                  }

            return flag;
            }
      };

//---------------------------------------------------------
//   usage
//---------------------------------------------------------

static void usage(const char* prog, const char* txt)
      {
      fprintf(stderr, "%s: %s\nusage: %s flags midifile\n   Flags:",
         prog, txt, prog);
      fprintf(stderr, "   -v  print version\n");
      fprintf(stderr, "   -d  debug mode: no threads\n");
      fprintf(stderr, "   -D  debug mode: enable some debug messages\n");
      fprintf(stderr, "   -m  debug mode: trace midi Input\n");
      fprintf(stderr, "   -M  debug mode: trace midi Output\n");
      fprintf(stderr, "   -s  debug mode: trace sync\n");
      fprintf(stderr, "   -R  enable real time scheduling\n");
      fprintf(stderr, "   -P  real time priority (default: 50)\n");
      fprintf(stderr, "   -L  lock memory (implies -R)\n");
      fprintf(stderr, "   -a  no audio\n");
      }

//---------------------------------------------------------
//   getCapabilities
//---------------------------------------------------------

void getCapabilities()
      {
#ifdef RTCAP
#ifdef __linux__
      const char* napp = getenv("GIVERTCAP");
      system(napp ? napp : "givertcap");
#endif // __linux__
#endif
      }

//---------------------------------------------------------
//   catchSignal
//---------------------------------------------------------

static void catchSignal(int sig)
      {
//      fprintf(stderr, "MusE: signal %d catched\n", sig);
      if (sig == SIGSEGV) {
            fprintf(stderr, "MusE: segmentation fault\n");
            abort();
            }
//      if (sig == SIGINT)
//            exit(0);
      if (sig == SIGCHLD) {
            //
            // try to catch software synthesizer gui exit
            //
            M_DEBUG("caught SIGCHLD - child died\n");
            int status;
            int n = waitpid (-1, &status, WNOHANG);
            if (n > 0) {
                  for (iSynthI ii = synthiInstances.begin(); ii != synthiInstances.end(); ++ii) {
                        SynthI* si = *ii;
                        if (!si) {
                              M_DEBUG("null synth - exiting loop\n");
                              break;
                              }
                        else
                              if (si->guiPid() == n) {
                                     si->guiExited();
                                     break;
                                     }
                        }
                  }
            }
      }

//---------------------------------------------------------
//   main
//---------------------------------------------------------

int main(int argc, char* argv[])
      {
      ruid = getuid();
      euid = geteuid();
      undoSetuid();

      srand(time(0));   // initialize random number generator

      for (int i = 0; i < _NSIG; ++i) {
            if (i != SIGABRT)
                  signal(i, catchSignal);
            }

      QApplication::setColorSpec(QApplication::ManyColor);

      MuseApplication app(argc, argv);
      qApp = &app;

      int c;
      while ((c = getopt(argc, argv, "vdDmMRLsP:a")) != EOF) {
            switch (c) {
                  case 'v': printVersion(argv[0]); return 0;
                  case 'd': debugMode = true; break;
                  case 'D': debugMsg = true; break;
                  case 'm': midiInputTrace = true; break;
                  case 'M': midiOutputTrace = true; break;
                  case 'R': realTimeScheduling = true; break;
                  case 'L': lockMemory = true; break;
                  case 's': debugSync = true; break;
                  case 'P': realTimePriority = atoi(optarg); break;
                  case 'a': noAudio = true; break;
                  default:  usage(argv[0], "bad argument"); return -1;
                  }
            }
      argc -= optind;
      ++argc;

      museUser = getenv("MUSEHOME");
      if (museUser == 0)
            museUser = getenv("HOME");
      QString museGlobal;
      const char* p = getenv("MUSE");
      if (p)
            museGlobal = p;

      if (museGlobal.isEmpty()) {
            QString museGlobal(INSTPREFIX);
            museGlobalLib   =  museGlobal + "/lib/muse";
            museGlobalShare =  museGlobal + "/share/muse";
            }
      else {
            museGlobalShare = museGlobal + "/share";
            museGlobalLib = museGlobal + "/lib";
            }
      museProject = getcwd(0, 0);
      if (debugMsg) {
            printf("global lib:   <%s>\n", museGlobalLib.latin1());
            printf("global share: <%s>\n", museGlobalShare.latin1());
            printf("muse home:    <%s>\n", museUser.latin1());
            printf("project dir:  <%s>\n", museProject.latin1());
            }

      QString wp(museGlobalShare);
      wp += "/rawwaves/";
      setenv("RAWWAVE_PATH", wp.latin1(), 0);  // hack for stk lib

      static QTranslator translator(0);
      QString locale(QTextCodec::locale());
      if (locale != "C") {
            QString loc("muse_");
            loc += QTextCodec::locale();
            if (translator.load(loc, ".") == false) {
                  QString lp(museGlobalShare);
                  lp += "/locale";
                  if (translator.load(loc, lp) == false) {
                        printf("no locale <%s>/<%s>\n", loc.latin1(), lp.latin1());
                        }
                  }
            app.installTranslator(&translator);
            }

      if (locale == "de") {
            printf("locale de\n");
            hIsB = false;
            }

      waveClips = new ClipList();
      sndFiles = new SndFileList();

//      QApplication::setStyle(new QCDEStyle(true));

      if (!noAudio)
            initPlugins();
      initIcons();
      muse = new MusE(argc, &argv[optind]);
      app.setMuse(muse);
      muse->show();
      return app.exec();
      }

//---------------------------------------------------------
//   checkRecord
//---------------------------------------------------------

void MusE::checkRecord(bool flag)
      {
      if (!flag)
            return;
      for (int i = 0; i < MIDI_PORTS; ++i) {
            MidiDevice* dev = midiPorts[i].device();
            if (dev && (dev->rwFlags() & 0x2))
                  return;
            }
      QMessageBox::critical(this, "MusE: Record",
         "There are no midi devices configured for recording");
      song->setRecord(false);
      }
#if 0
//---------------------------------------------------------
//   configPart
//---------------------------------------------------------

void MusE::configPart(int id)
      {
      if (id < 3) {
            partConfig->setItemChecked(0, id == 0);
            partConfig->setItemChecked(1, id == 1);
            partConfig->setItemChecked(2, id == 2);
            arranger->setShowPartType(id);
            for (int i = 3; i < 10; ++i) {
                  partConfig->setItemEnabled(i, id == 2);
                  }
            }
      else {
            bool flag = !partConfig->isItemChecked(id);
            partConfig->setItemChecked(id, flag);
            int val = arranger->showPartEvent();
            if (flag) {
                  val |= 1 << (id-3);
                  }
            else {
                  val &= ~(1 << (id-3));
                  }
            arranger->setShowPartEvent(val);
            }
      }
#endif

//---------------------------------------------------------
//   cmd
//    some cmd's from pulldown menu
//---------------------------------------------------------

void MusE::cmd(int cmd)
      {
      TrackList* tracks = song->tracks();
      int l = song->lpos();
      int r = song->rpos();

      switch(cmd) {
            case CMD_CUT:
                  arranger->cmd(Arranger::CMD_CUT_PART);
                  break;
            case CMD_COPY:
                  arranger->cmd(Arranger::CMD_COPY_PART);
                  break;
            case CMD_PASTE:
                  arranger->cmd(Arranger::CMD_PASTE_PART);
                  break;
            case CMD_DELETE_TRACK:
                  midiThread->msgRemoveTracks();
                  break;
            case CMD_ADD_TRACK:
                  song->addTrack();
                  break;
            case CMD_SELECT_ALL:
            case CMD_SELECT_NONE:
            case CMD_SELECT_INVERT:
            case CMD_SELECT_ILOOP:
            case CMD_SELECT_OLOOP:
                  for (iTrack i = tracks->begin(); i != tracks->end(); ++i) {
                        PartList* parts = (*i)->parts();
                        for (iPart p = parts->begin(); p != parts->end(); ++p) {
                              bool f = false;
                              int t1 = p->second->posTick();
                              int t2 = t1 + p->second->lenTick();
                              bool inside =
                                 ((t1 >= l) && (t1 < r))
                                 ||  ((t2 >= l) && (t2 < r))
                                 ||  ((t1 <= l) && (t2 > r));
                              switch(cmd) {
                                    case CMD_SELECT_INVERT:
                                          f = !p->second->selected();
                                          break;
                                    case CMD_SELECT_NONE:
                                          f = false;
                                          break;
                                    case CMD_SELECT_ALL:
                                          f = true;
                                          break;
                                    case CMD_SELECT_ILOOP:
                                          f = inside;
                                          break;
                                    case CMD_SELECT_OLOOP:
                                          f = !inside;
                                          break;
                                    }
                              p->second->setSelected(f);
                              }
                        }
                  song->update();
                  break;

            case CMD_SELECT_PARTS:
                  for (iTrack i = tracks->begin(); i != tracks->end(); ++i) {
                        if (!(*i)->selected())
                              continue;
                        PartList* parts = (*i)->parts();
                        for (iPart p = parts->begin(); p != parts->end(); ++p)
                              p->second->setSelected(true);
                        }
                  song->update();
                  break;
            case CMD_FOLLOW_NO:
                  song->setFollow(Song::NO);
                  setFollow();
                  break;
            case CMD_FOLLOW_JUMP:
                  song->setFollow(Song::JUMP);
                  setFollow();
                  break;
            case CMD_FOLLOW_CONTINUOUS:
                  song->setFollow(Song::CONTINUOUS);
                  setFollow();
                  break;
            }
      }

//---------------------------------------------------------
//   clipboardChanged
//---------------------------------------------------------

void MusE::clipboardChanged()
      {
      QCString subtype("partlist");
      QMimeSource* ms = QApplication::clipboard()->data();
      if (ms && ms->format(0)) {
            bool flag = strcmp(ms->format(0), "text/partlist;charset=UTF-8") == 0;
            menuEdit->setItemEnabled(CMD_PASTE, flag);
            }
      }

//---------------------------------------------------------
//   selectionChanged
//---------------------------------------------------------

void MusE::selectionChanged()
      {
      bool flag = arranger->isSingleSelection();
      menuEdit->setItemEnabled(CMD_CUT, flag);
      menuEdit->setItemEnabled(CMD_COPY, flag);
//      seq->setRecTrack(dynamic_cast<MidiTrack*>(arranger->curTrack()));
      }

//---------------------------------------------------------
//   setMixdownFile
//---------------------------------------------------------

void MusE::setMixdownFile()
      {
      SndFile* sf = audioThread->mixdownFile();
      SndFile* nsf = getSndFile(sf, this, "recFileDialog");
      if (nsf)
            audioThread->msgSetMixdownFile(nsf);
      }

//---------------------------------------------------------
//   transpose
//---------------------------------------------------------

void MusE::transpose()
      {
      Transpose *w = new Transpose();
      w->show();
      }

//---------------------------------------------------------
//   modifyGateTime
//---------------------------------------------------------

void MusE::modifyGateTime()
      {
      GateTime* w = new GateTime(this);
      w->show();
      }

//---------------------------------------------------------
//   modifyVelocity
//---------------------------------------------------------

void MusE::modifyVelocity()
      {
      printf("not implemented\n");
      }

//---------------------------------------------------------
//   crescendo
//---------------------------------------------------------

void MusE::crescendo()
      {
      printf("not implemented\n");
      }

//---------------------------------------------------------
//   thinOut
//---------------------------------------------------------

void MusE::thinOut()
      {
      printf("not implemented\n");
      }

//---------------------------------------------------------
//   eraseEvent
//---------------------------------------------------------

void MusE::eraseEvent()
      {
      printf("not implemented\n");
      }

//---------------------------------------------------------
//   noteShift
//---------------------------------------------------------

void MusE::noteShift()
      {
      printf("not implemented\n");
      }

//---------------------------------------------------------
//   moveClock
//---------------------------------------------------------

void MusE::moveClock()
      {
      printf("not implemented\n");
      }

//---------------------------------------------------------
//   copyMeasure
//---------------------------------------------------------

void MusE::copyMeasure()
      {
      printf("not implemented\n");
      }

//---------------------------------------------------------
//   eraseMeasure
//---------------------------------------------------------

void MusE::eraseMeasure()
      {
      printf("not implemented\n");
      }

//---------------------------------------------------------
//   deleteMeasure
//---------------------------------------------------------

void MusE::deleteMeasure()
      {
      printf("not implemented\n");
      }

//---------------------------------------------------------
//   createMeasure
//---------------------------------------------------------

void MusE::createMeasure()
      {
      printf("not implemented\n");
      }

//---------------------------------------------------------
//   mixTrack
//---------------------------------------------------------

void MusE::mixTrack()
      {
      printf("not implemented\n");
      }

//---------------------------------------------------------
//   configAppearance
//---------------------------------------------------------

void MusE::configAppearance()
      {
      Appearance *a = new Appearance(this, arranger);
      a->show();
      }

//---------------------------------------------------------
//   setBigTimeChecked
//---------------------------------------------------------

void MusE::setBigTimeChecked(bool c)
      {
      menu_display->setItemChecked(bt_id, c);
      }

//---------------------------------------------------------
//   loadTheme
//---------------------------------------------------------

void MusE::loadTheme(QString s)
      {
      if (s == _currentTheme)
            return;

      QFont appFont = QApplication::font();

      if (s == "default") {
            // shit. how do we get rid of a loaded theme?
            //qApp->setStyle(0); // segfault
            }
      else if (s == "Norwegian Wood") {
            qApp->setStyle(new NorwegianWoodStyle);
            qApp->setFont(appFont, true);
            }
      else if (s == "Metal") {
            qApp->setStyle(new MetalStyle);
            qApp->setFont(appFont, true);
            }
      else if (s == "MusE") {
            qApp->setStyle(new MuseStyle);
            qApp->setFont(appFont, true);
            }
      else if (s == "Platinum") {
            qApp->setStyle(new QPlatinumStyle);
            QPalette p( QColor(239, 239, 239));
            qApp->setPalette(p, true);
            qApp->setFont( appFont, true);
            }

       else if (s == "Windows") {
            qApp->setStyle(new QWindowsStyle);
            qApp->setFont(appFont, true);
            }

       else if(s == "CDE") {
		qApp->setStyle(new QCDEStyle(true));

		QPalette p(QColor( 75, 123, 130 ) );
		p.setColor(QPalette::Active, QColorGroup::Base, QColor( 55, 77, 78 ) );
		p.setColor(QPalette::Inactive, QColorGroup::Base, QColor( 55, 77, 78 ) );
		p.setColor(QPalette::Disabled, QColorGroup::Base, QColor( 55, 77, 78 ) );
		p.setColor(QPalette::Active, QColorGroup::Highlight, Qt::white );
		p.setColor(QPalette::Active, QColorGroup::HighlightedText, QColor( 55, 77, 78 ) );
		p.setColor(QPalette::Inactive, QColorGroup::Highlight, Qt::white );
		p.setColor(QPalette::Inactive, QColorGroup::HighlightedText, QColor( 55, 77, 78 ) );
		p.setColor(QPalette::Disabled, QColorGroup::Highlight, Qt::white );
		p.setColor(QPalette::Disabled, QColorGroup::HighlightedText, QColor( 55, 77, 78 ) );
		p.setColor(QPalette::Active, QColorGroup::Foreground, Qt::white );
		p.setColor(QPalette::Active, QColorGroup::Text, Qt::white );
		p.setColor(QPalette::Active, QColorGroup::ButtonText, Qt::white );
		p.setColor(QPalette::Inactive, QColorGroup::Foreground, Qt::white );
		p.setColor(QPalette::Inactive, QColorGroup::Text, Qt::white );
		p.setColor(QPalette::Inactive, QColorGroup::ButtonText, Qt::white );
		p.setColor(QPalette::Disabled, QColorGroup::Foreground, Qt::lightGray );
		p.setColor(QPalette::Disabled, QColorGroup::Text, Qt::lightGray );
		p.setColor(QPalette::Disabled, QColorGroup::ButtonText, Qt::lightGray );
		qApp->setPalette( p, true );
		qApp->setFont( QFont( "times", appFont.pointSize() ), true );
            }

       else if (s == "Motif") {
               qApp->setStyle(new QMotifStyle);
               QPalette p( QColor(192, 192, 192));
               qApp->setPalette(p, true);
               qApp->setFont(appFont, true);
               }
       else if (s == "Motif Plus") {
               qApp->setStyle( new QMotifPlusStyle);
               QPalette p( QColor( 192, 192, 192));
               qApp->setPalette( p, true);
               qApp->setFont( appFont, true);
               }
       else {
            printf("MusE::loadTheme(): oops! unknown theme: %s\n", s.data());
            return;
            }

      _currentTheme = s;
      }

//---------------------------------------------------------
//   configChanged
//---------------------------------------------------------

void MusE::configChanged()
      {
      for (iToplevel i = toplevels.begin(); i != toplevels.end(); ++i) {
            Toplevel tl = *i;
            if (tl.type() == Toplevel::WAVE) {
                  int obj = tl.object();
                  ((WaveEdit*)(obj))->setBg(WaveEdit::configBg());
                  }
            }
      }

//---------------------------------------------------------
//   configMetronome
//---------------------------------------------------------

void MusE::configMetronome()
      {
      if (!metronomeConfig)
            metronomeConfig = new MetronomeConfig(this, "metronome");
      metronomeConfig->show();
      }


