/*
 * Copyright (C) 2001  CodeFactory AB
 * Copyright (C) 2001  Richard Hult
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult <rhult@codefactory.se>
 */

#include <gtk/gtk.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <time.h>
#include <math.h>
#include "util/type-utils.h"
#include "util/time-utils.h"
#include "gantt-scale.h"

#define DEFAULT_VIEWPORT 5000

enum {
	UNITS_CHANGED,
	SCALE_CHANGED,
	VIEWPORT_CHANGED,
	LAST_SIGNAL
};

static void gantt_scale_init		   (GanttScale		 *gantt_scale);
static void gantt_scale_class_init	   (GanttScaleClass	 *klass);

GNOME_CLASS_BOILERPLATE (GanttScale, gantt_scale, GtkObject, gtk_object);

static guint signals[LAST_SIGNAL] = { 0 };

static void
gantt_scale_class_init (GanttScaleClass *klass)
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass*) klass;

	signals[UNITS_CHANGED] = 
		gtk_signal_new ("units_changed",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GanttScaleClass, units_changed),
				gtk_marshal_NONE__NONE,
				GTK_TYPE_NONE, 0);

	signals[SCALE_CHANGED] = 
		gtk_signal_new ("scale_changed",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GanttScaleClass, scale_changed),
				gtk_marshal_NONE__NONE,
				GTK_TYPE_NONE, 0);
	
	signals[VIEWPORT_CHANGED] = 
		gtk_signal_new ("viewport_changed",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GanttScaleClass, viewport_changed),
				gtk_marshal_NONE__NONE,
				GTK_TYPE_NONE, 0);

	gtk_object_class_add_signals (object_class, signals, LAST_SIGNAL);

	klass->units_changed = NULL;
}

static void
gantt_scale_init (GanttScale *scale)
{
}

static void
update_viewport_coords (GanttScale *scale)
{
	gdouble x1, x2, dx;

	x1 = scale->t1 * scale->scale;
	x2 = scale->t2 * scale->scale;
	
	if (scale->min_width != -1 && (x2 - x1) < scale->min_width) {
		/* Too narrow, we must expand. */
		dx = (scale->min_width - (x2 - x1)) / 2;
		x1 -= dx;
		x2 += dx;
	}

	scale->x1 = x1;
	scale->x2 = x2;
}

GanttScale *
gantt_scale_new (gdouble scale_factor, time_t t1, time_t t2)
{
	GanttScale *scale;

	scale = gtk_type_new (TYPE_GANTT_SCALE);

	scale->min_width = -1;
	scale->scale     = scale_factor;
	scale->t1        = t1;
	scale->t2        = t2;

	update_viewport_coords (scale);

	return scale;
}

/* Time to world coordinates. */
gdouble
gantt_scale_t2w (GanttScale *scale, time_t t)
{
	g_return_val_if_fail (scale != NULL, 0);
	g_return_val_if_fail (IS_GANTT_SCALE (scale), 0);
	g_return_val_if_fail (t >= 0, 0);

	return t * scale->scale;
}

/* World coordinates to time. */
time_t
gantt_scale_w2t (GanttScale *scale, gdouble x)
{
	g_return_val_if_fail (scale != NULL, 0);
	g_return_val_if_fail (IS_GANTT_SCALE (scale), 0);
	g_return_val_if_fail (x >= 0 && x <= G_MAXINT, 0);

	return floor (x / scale->scale + 0.5);
}

/* Set the major and minor units for the scale. */
void
gantt_scale_set_units (GanttScale *scale, GanttUnit major, GanttUnit minor)
{
	g_return_if_fail (scale != NULL);
	g_return_if_fail (IS_GANTT_SCALE (scale));

	if (major != GANTT_UNIT_NONE) {
		scale->major_unit = major;
	}
	if (minor != GANTT_UNIT_NONE) {
		scale->minor_unit = minor;
	}

	if (scale->major_unit < scale->minor_unit) {
		g_warning ("Can't set major to less than minor.");
		scale->minor_unit = MAX (scale->major_unit - 1, GANTT_UNIT_MINUTE);
	}

	gtk_signal_emit (GTK_OBJECT (scale), signals[UNITS_CHANGED], NULL);
}

void
gantt_scale_set_viewport (GanttScale *scale, time_t t1, time_t t2)
{
	g_return_if_fail (scale != NULL);
	g_return_if_fail (IS_GANTT_SCALE (scale));

	if (t1 > -1) {
		/* Snap the first shown date to the minor scale, so that
		 * we can add e.g. a quarter and end up on an even quarter.
		 */
		scale->t1 = gantt_scale_snap_time (scale, GANTT_UNIT_MINOR, t1);
	}
	if (t2 > -1) {
		scale->t2 = t2; /* FIXME: snap this too? Not neccessary... */
	}

	update_viewport_coords (scale);
	gtk_signal_emit (GTK_OBJECT (scale), signals[VIEWPORT_CHANGED], NULL);
}

void
gantt_scale_move_viewport (GanttScale *scale, time_t dt)
{
	gdouble t1, t2;
	
	g_return_if_fail (scale != NULL);
	g_return_if_fail (IS_GANTT_SCALE (scale));

	t1 = scale->t1 + dt;
	t2 = scale->t2 + dt;

	t1 = CLAMP (t1, 0, G_MAXINT);
	t2 = CLAMP (t2, 0, G_MAXINT);

	scale->t1 = t1;
	scale->t2 = t2;

	update_viewport_coords (scale);
	
	gtk_signal_emit (GTK_OBJECT (scale), signals[VIEWPORT_CHANGED], NULL);
}

/* Set the minimal viewport width, in world coordinates. */
void
gantt_scale_set_min_viewport_width (GanttScale *scale, gdouble width)
{
	gdouble old_width;
	
	g_return_if_fail (scale != NULL);
	g_return_if_fail (IS_GANTT_SCALE (scale));
	g_return_if_fail (width > 0 && width < G_MAXINT);

	scale->min_width = width;

	old_width = scale->x2 - scale->x1;
	update_viewport_coords (scale);
	if (old_width != (scale->x2 - scale->x1)) {
		gtk_signal_emit (GTK_OBJECT (scale), signals[VIEWPORT_CHANGED], NULL);
	}
}

/* Set the horizontal scale factor. */
void
gantt_scale_set_scale_factor (GanttScale *scale, gdouble scale_factor)
{
	g_return_if_fail (scale != NULL);
	g_return_if_fail (IS_GANTT_SCALE (scale));
	g_return_if_fail (scale_factor > 0);

	scale->scale = scale_factor;
	update_viewport_coords (scale);
	
	gtk_signal_emit (GTK_OBJECT (scale), signals[SCALE_CHANGED], NULL);
}

void
gantt_scale_set_scale_factor_ex (GanttScale *scale, gdouble scale_factor, time_t center)
{
	gdouble t1, t2;
	gdouble dt, s;

	g_return_if_fail (scale != NULL);
	g_return_if_fail (IS_GANTT_SCALE (scale));
	g_return_if_fail (scale_factor > 0);

	s = scale->scale;

	scale->scale = scale_factor;

	t1 = scale->x1 / scale_factor;
	t2 = scale->x2 / scale_factor;

	/* new time span */
	dt = (t2 - t1) / 2;
	
	scale->t1 = floor (center - dt + 0.5);
	scale->t2 = floor (center + dt + 0.5);

	update_viewport_coords (scale);

	gtk_signal_emit (GTK_OBJECT (scale), signals[SCALE_CHANGED], NULL);
}

void
gantt_scale_zoom (GanttScale *scale,
		  time_t      t1,
		  time_t      t2,
		  gdouble     width)
{
	g_return_if_fail (scale != NULL);
	g_return_if_fail (IS_GANTT_SCALE (scale));
	g_return_if_fail (t1 < t2);
	g_return_if_fail (width > 0);

	scale->scale = width / (t2 - t1);
	update_viewport_coords (scale);

	gtk_signal_emit (GTK_OBJECT (scale), signals[SCALE_CHANGED], NULL);
}

time_t
gantt_scale_snap_time (GanttScale *scale, GanttUnitType type, time_t t)
{
	GanttUnit unit;
	
	g_return_val_if_fail (scale != NULL, 0);
	g_return_val_if_fail (IS_GANTT_SCALE (scale), 0);
	g_return_val_if_fail (t >= 0, 0);

	if (type == GANTT_UNIT_MINOR) {
		unit = scale->minor_unit;
	} else {
		unit = scale->major_unit;
	}
	
	switch (unit) {
	case GANTT_UNIT_YEAR:
		t = time_year_begin (t);
		break;
		
	case GANTT_UNIT_QUARTER:
		t = time_quarter_begin (t);
		break;
		
	case GANTT_UNIT_MONTH:
		t = time_month_begin (t);
		break;

	case GANTT_UNIT_WEEK:
		t = time_week_begin (t, scale->week_starts_on_monday);
		break;
		
	case GANTT_UNIT_DAY:
		t = time_day_begin (t);
		break;

	case GANTT_UNIT_HOUR:
		t = time_hour_begin (t);
		break;

	case GANTT_UNIT_MINUTE:
		t = time_minute_begin (t);
		break;

	default:
		g_warning ("Unit not set.");
		break;
	}
		
	return t;
}
	
gboolean
gantt_scale_is_on_tick (GanttScale *scale, GanttUnitType type, time_t t)
{
	g_return_val_if_fail (scale != NULL, FALSE);
	g_return_val_if_fail (IS_GANTT_SCALE (scale), FALSE);
	g_return_val_if_fail (t >= 0, FALSE);

	return (t == gantt_scale_snap_time (scale, type, t));
}

/* FIXME: Should we make this step to the next even year/quarter/month etc? */
static time_t
increase_one_tick (GanttUnit unit, time_t t)
{
	gint days_in_month, year, month, i;
	
	switch (unit) {
	case GANTT_UNIT_YEAR:
		/* Add twelve months from t (t must be snapped to a year). */
		for (i = 0; i < 12; i++) {
			time_split (t, &year, &month, NULL);
			days_in_month = time_days_in_month (year, month);
			t += days_in_month * 60 * 60 * 24;
		}
		break;
		
	case GANTT_UNIT_QUARTER:
		/* Add three months from t (t must be snapped to a quarter). */
		for (i = 0; i < 3; i++) {
			time_split (t, &year, &month, NULL);
			days_in_month = time_days_in_month (year, month);
			t += days_in_month * 60 * 60 * 24;
		}
		break;
		
	case GANTT_UNIT_MONTH:
		time_split (t, &year, &month, NULL);
		days_in_month = time_days_in_month (year, month);
		
		t += days_in_month * 60 * 60 * 24;
		break;

	case GANTT_UNIT_WEEK:
		t += 60 * 60 * 24 * 7;
		break;
		
	case GANTT_UNIT_DAY:
		t += 60 * 60 * 24;
		break;

	case GANTT_UNIT_HOUR:
		t += 60 * 60;
		break;

	case GANTT_UNIT_MINUTE:
		t += 60;
		break;

	default:
		g_warning ("Invalid unit.");
	}

	return t;
}

time_t
gantt_scale_increase_one_tick (GanttScale *scale, GanttUnitType type, time_t t)
{
	GanttUnit unit;
	
	g_return_val_if_fail (scale != NULL, t);
	g_return_val_if_fail (IS_GANTT_SCALE (scale), t);
	g_return_val_if_fail (t >= 0, t);

	if (type == GANTT_UNIT_MINOR) {
		unit = scale->minor_unit;
	} else {
		unit = scale->major_unit;
	}
	
	return increase_one_tick (unit, t);
}

gchar *
gantt_scale_format_time (GanttScale *scale, GanttUnitType type, time_t t)
{
	GanttUnit     unit;
	struct tm    *tm;
	static gchar  buf[64], *tmp;
	gint          month;

	g_return_val_if_fail (scale != NULL, NULL);
	g_return_val_if_fail (IS_GANTT_SCALE (scale), NULL);
	g_return_val_if_fail (t >= 0, NULL);

	if (type == GANTT_UNIT_MINOR) {
		unit = scale->minor_unit;
	} else {
		unit = scale->major_unit;
	}

	buf[0] = '\0';
	tm = localtime (&t);
	
	switch (unit) {
	case GANTT_UNIT_YEAR:
		strftime (buf, sizeof (buf) - 1, "%Y", tm);
		break;

	case GANTT_UNIT_QUARTER:
		strftime (buf, sizeof (buf) - 1, "%Y", tm);

		time_split (t, NULL, &month, NULL);
		if (month >= 0 && month <= 2) {
			tmp = _("Q1");
		}
		else if (month >= 3 && month <= 5) {
			tmp = _("Q2");
		}
		else if (month >= 6 && month <= 8) {
			tmp = _("Q3");
		}
		else if (month >= 9 && month <= 11) {
			tmp = _("Q4");
		}
		else {
			tmp = _("Q?");
		}

		return g_strdup_printf ("%s %s", tmp, buf);
		break;
		
	case GANTT_UNIT_MONTH:
		strftime (buf, sizeof (buf) - 1, "%Y %b", tm);
		break;

	case GANTT_UNIT_WEEK:
		strftime (buf, sizeof (buf) - 1, "%b%e", tm);
		break;
		
	case GANTT_UNIT_DAY:
		/*strftime (buf, sizeof (buf) - 1, "%a %b%e", tm);*/
		strftime (buf, sizeof (buf) - 1, "%a", tm);
		break;

	case GANTT_UNIT_HOUR:
		strftime (buf, sizeof (buf) - 1, "%R", tm);
		break;

	case GANTT_UNIT_MINUTE:
		strftime (buf, sizeof (buf) - 1, "%R", tm);
		break;

	default:
		g_warning ("Unit not set.");
	}

	return g_strdup (buf);
}

void
gantt_scale_set_week_starts_on_monday (GanttScale	*scale,
				       gboolean          monday)

{
	g_return_if_fail (scale != NULL);
	g_return_if_fail (IS_GANTT_SCALE (scale));
	
	scale->week_starts_on_monday = monday;
	gtk_signal_emit (GTK_OBJECT (scale), signals[UNITS_CHANGED], NULL);
}

/* experimental stuff */

void
gantt_scale_show_time (GanttScale *scale,
		       time_t      t)
{
	gdouble  t1, t2, tpad;
	gboolean changed;
	
	g_return_if_fail (scale != NULL);
	g_return_if_fail (IS_GANTT_SCALE (scale));

	/* Some padding. */
	tpad = gantt_scale_get_minor_tick (scale) * 2;
	t1 = t - tpad;
	t2 = t + tpad;
	
	/* We might need to expand the viewport. */
	changed = FALSE;
	if (t1 < scale->t1) {
		scale->t1 = MIN (scale->t1, t1);
		changed = TRUE;
	}
	if (t2 > scale->t2) {
		scale->t2 = MAX (scale->t2, t2);
		changed = TRUE;
	}

	if (changed) {
		update_viewport_coords (scale);
		gtk_signal_emit (GTK_OBJECT (scale),
				 signals[VIEWPORT_CHANGED],
				 NULL);
	}
}

time_t
gantt_scale_get_minor_tick (GanttScale *scale)
{
	time_t t;
	
	g_return_val_if_fail (scale != NULL, 0);
	g_return_val_if_fail (IS_GANTT_SCALE (scale), 0);

	t = gantt_scale_increase_one_tick (scale, GANTT_UNIT_MINOR, 0);

	return t;
}

time_t
gantt_scale_get_major_tick (GanttScale *scale)
{
	time_t t;
	
	g_return_val_if_fail (scale != NULL, 0);
	g_return_val_if_fail (IS_GANTT_SCALE (scale), 0);

	t = gantt_scale_increase_one_tick (scale, GANTT_UNIT_MAJOR, 0);

	return t;
}

time_t
gantt_scale_unit_get_period (GanttUnit unit)
{
	return increase_one_tick (unit, 0);
}
