/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 Anders Carlsson <andersca@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <bonobo.h>
#include <bonobo/bonobo-listener.h>
#include <bonobo/bonobo-shlib-factory.h>
#include <glib.h>
#include <liboaf/liboaf.h>
#include <glade/glade.h>

#include "util/corba-utils.h"
#include "libmrproject/GNOME_MrProject.h"
#include "client/manager-clients/task-manager-client.h"
#include "client/manager-clients/allocation-manager-client.h"
#include "client/manager-clients/resource-manager-client.h"
#include "util/id-map.h"
#include "util/time-utils.h"
#include "client/widgets/goto-popup.h"

#include "calendar-component.h"
#include "month-view.h"
#include "month-view-task-item.h"
#include "resource-filter-dialog-gui.h"
#include "resource-filter.h"

#define DEBUG 1
#include "util/debug.h"

#define OAFIID "OAFIID:GNOME_MrProject_CalendarComponent"

static void calendar_component_init       (CalendarComponent       *component);
static void calendar_component_class_init (CalendarComponentClass  *klass);
static void cc_task_inserted_cb           (TaskManagerClient       *tmc, 
					   GM_Task                 *task, 
					   GM_Id                    sibling_id,
					   GM_TaskOrderType         type,
					   CalendarComponent       *component);
static void cc_tasks_removed_cb           (TaskManagerClient        *tmc, 
					   GSList                  *tasks, 
					   CalendarComponent       *component);
static void cc_task_updated_cb            (TaskManagerClient       *tmc, 
					   GM_Task                 *task, 
					   CalendarComponent       *component);
static void cc_tasks_repositioned_cb      (TaskManagerClient       *tmc, 
					   GM_Id                    task_id, 
					   GM_Id                    sibling_id, 
					   GM_TaskOrderType         type, 
					   CalendarComponent       *component);
static void cc_task_reparented_cb         (TaskManagerClient       *tmc, 
					   GM_Id                    task_id, 
					   GM_Id                    new_parent_id, 
					   CalendarComponent       *component);
static void cc_tasks_linked_cb            (TaskManagerClient       *tmc, 
					   GM_Dependency           *dependency, 
					   CalendarComponent       *component);
static void cc_tasks_unlinked_cb          (TaskManagerClient       *tmc, 
					   GM_Dependency           *dependency, 
					   CalendarComponent       *component);
static void cc_items_clear_cb             (TaskManagerClient       *mc, 
					   CalendarComponent       *component);
static void cc_shell_event                (ShellComponent          *component,
					   const gchar             *event_name,
					   CORBA_any               *any);
static void cc_allocation_added_cb        (AllocationManagerClient *amc,
					   GM_Allocation           *alloc,
					   CalendarComponent       *component);
static void cc_allocations_removed_cb     (AllocationManagerClient *amc,
					   GSList                  *allocations,
					   CalendarComponent       *component);

static void calendar_control_activate     (BonoboControl           *control,
					   CalendarComponent       *calendar_component);
static void control_activate_cb           (BonoboControl           *control,
					   gboolean                 activate,
					   gpointer                 user_data);


static void cc_resource_inserted_cb       (ResourceManagerClient   *rmc,
					   GM_Resource             *resource,
					   CalendarComponent       *component);
static void cc_resource_updated_cb        (ResourceManagerClient   *rmc,
					   GM_Resource             *resource,
					   CalendarComponent       *component);
static void cc_resources_removed_cb       (ResourceManagerClient   *rmc,
					   GSList                  *resources,
					   CalendarComponent       *component);

static BonoboControl *cc_control_new      (CalendarComponent       *component);


#define PARENT_TYPE SHELL_COMPONENT_TYPE
static GtkObjectClass *parent_class;

/* Private members. */
struct _CalendarComponentPriv {
	BonoboControl           *control;
	GtkWidget               *month_view;
	GtkWidget               *date_label;

	ResourceFilterDialogGui *gui;

	gboolean                 goto_dialog_shown;
	gboolean                 resource_filter_dialog_shown;
	gboolean                 filter_tasks;

	TaskManagerClient       *tm_client;
	AllocationManagerClient *am_client;
	ResourceManagerClient   *rm_client;

	ResourceFilter          *filter;

	IdMap                   *tasks; /* containing all tasks */
	IdMap                   *resources; /* containing all resources */
};

static gboolean
calendar_component_add_to_shell (ShellComponent *component,
				 GM_Shell        shell)
{
	CORBA_Environment      ev;
	CalendarComponentPriv *priv;

	g_return_val_if_fail (component != NULL, FALSE);
	g_return_val_if_fail (IS_CALENDAR_COMPONENT (component), FALSE);

	d(puts(__FUNCTION__));

	priv = CALENDAR_COMPONENT (component)->priv;

	CORBA_exception_init (&ev);

	priv->control = cc_control_new (CALENDAR_COMPONENT (component));

	GNOME_MrProject_Shell_addControl (component->shell, 
					  BONOBO_OBJREF (component),
					  BONOBO_OBJREF (priv->control),
					  OAFIID,
					  _("Calendar"),
					  MRPROJECT_IMAGEDIR "calendar.png",
					  NULL);

	priv->tm_client = TASK_MANAGER_CLIENT (task_mc_new (shell, FALSE, &ev));

	gtk_signal_connect (GTK_OBJECT (priv->tm_client), 
			    "task_inserted", 
			    GTK_SIGNAL_FUNC (cc_task_inserted_cb), 
			    GTK_OBJECT (component));
	gtk_signal_connect (GTK_OBJECT (priv->tm_client), 
			    "tasks_removed", 
			    GTK_SIGNAL_FUNC (cc_tasks_removed_cb), 
			    GTK_OBJECT (component));
	gtk_signal_connect (GTK_OBJECT (priv->tm_client), 
			    "task_updated", 
			    GTK_SIGNAL_FUNC (cc_task_updated_cb), 
			    GTK_OBJECT (component));
	gtk_signal_connect (GTK_OBJECT (priv->tm_client), 
			    "task_repositioned", 
			    GTK_SIGNAL_FUNC (cc_tasks_repositioned_cb), 
			    GTK_OBJECT (component));
	gtk_signal_connect (GTK_OBJECT (priv->tm_client), 
			    "task_reparented", 
			    GTK_SIGNAL_FUNC (cc_task_reparented_cb), 
			    GTK_OBJECT (component));
	gtk_signal_connect (GTK_OBJECT (priv->tm_client), 
			    "tasks_linked", 
			    GTK_SIGNAL_FUNC (cc_tasks_linked_cb), 
			    GTK_OBJECT (component));
	gtk_signal_connect (GTK_OBJECT (priv->tm_client), 
			    "tasks_unlinked", 
			    GTK_SIGNAL_FUNC (cc_tasks_unlinked_cb), 
			    GTK_OBJECT (component));
	gtk_signal_connect (GTK_OBJECT (priv->tm_client),
			    "items_clear",
			    GTK_SIGNAL_FUNC (cc_items_clear_cb),
			    GTK_OBJECT (component));

	CORBA_exception_free (&ev);

	CORBA_exception_init (&ev);
	priv->am_client = ALLOCATION_MANAGER_CLIENT (allocation_mc_new (shell, 
									FALSE, 
									&ev));

	gtk_signal_connect (GTK_OBJECT (priv->am_client),
			    "allocation_added",
			    GTK_SIGNAL_FUNC (cc_allocation_added_cb),
			    GTK_OBJECT (component));
	gtk_signal_connect (GTK_OBJECT (priv->am_client),
			    "allocations_removed",
			    GTK_SIGNAL_FUNC (cc_allocations_removed_cb),
			    GTK_OBJECT (component));

	CORBA_exception_free (&ev);

	CORBA_exception_init (&ev);
	priv->rm_client = RESOURCE_MANAGER_CLIENT (resource_mc_new (shell, 
								    FALSE, 
								    &ev));
	gtk_signal_connect (GTK_OBJECT (priv->rm_client),
			    "resource_inserted", 
			    GTK_SIGNAL_FUNC (cc_resource_inserted_cb),
			    GTK_OBJECT (component));
	gtk_signal_connect (GTK_OBJECT (priv->rm_client),
			    "resource_updated",
			    GTK_SIGNAL_FUNC (cc_resource_updated_cb),
			    GTK_OBJECT (component));
	gtk_signal_connect (GTK_OBJECT (priv->rm_client),
			    "resources_removed",
			    GTK_SIGNAL_FUNC (cc_resources_removed_cb),
			    GTK_OBJECT (component));

	CORBA_exception_free (&ev);

	gtk_signal_connect (GTK_OBJECT (priv->control), "activate",
			    GTK_SIGNAL_FUNC (control_activate_cb),
			    component);

	return TRUE;
}

static void
calendar_component_remove_from_shell (ShellComponent *component)
{
	CalendarComponentPriv *priv;

	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_CALENDAR_COMPONENT (component));

	d(puts(__FUNCTION__));

	priv = CALENDAR_COMPONENT (component)->priv;

	if (priv->tm_client) {
		gtk_object_destroy (GTK_OBJECT (priv->tm_client));
		priv->tm_client = NULL;
	}

	if (priv->am_client) {
		gtk_object_destroy (GTK_OBJECT (priv->am_client));
		priv->am_client = NULL;
	}

	if (priv->rm_client) {
		gtk_object_destroy (GTK_OBJECT (priv->rm_client));
		priv->rm_client = NULL;
	}
}

static void
calendar_component_set_debug_flags (ShellComponent *component,
				    long flags)
{
	d(puts(__FUNCTION__));
}

static void
year_back_clicked (GtkWidget *button, CalendarComponent *component)
{
	CalendarComponentPriv *priv;
	GDate                  date;
	
	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_CALENDAR_COMPONENT (component));

	priv = component->priv;
	
	month_view_get_first_day_shown (MONTH_VIEW (priv->month_view), 
					&date);

	g_date_subtract_years (&date, 1);

	month_view_set_first_day_shown (MONTH_VIEW (priv->month_view), 
					&date);
}

static void
year_forward_clicked (GtkWidget *button, CalendarComponent *component)
{
	CalendarComponentPriv *priv;
	GDate                  date;
	
	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_CALENDAR_COMPONENT (component));

	priv = component->priv;

	month_view_get_first_day_shown (MONTH_VIEW (priv->month_view), 
					&date);

	g_date_add_years (&date, 1);

	month_view_set_first_day_shown (MONTH_VIEW (priv->month_view), 
					&date);
}

static void
month_back_clicked (GtkWidget *button, CalendarComponent *component)
{
	CalendarComponentPriv *priv;
	GDate                  date;
	
	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_CALENDAR_COMPONENT (component));

	priv = component->priv;
	
	month_view_get_first_day_shown (MONTH_VIEW (priv->month_view), 
					&date);

	g_date_subtract_months (&date, 1);

	month_view_set_first_day_shown (MONTH_VIEW (priv->month_view), 
					&date);
}

static void
month_forward_clicked (GtkWidget *button, CalendarComponent *component)
{
	CalendarComponentPriv *priv;
	GDate                  date;
	
	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_CALENDAR_COMPONENT (component));

	priv = component->priv;

	month_view_get_first_day_shown (MONTH_VIEW (priv->month_view), 
					&date);

	g_date_add_months (&date, 1);

	month_view_set_first_day_shown (MONTH_VIEW (priv->month_view), 
					&date);
}

static void
goto_today_clicked (GtkWidget *button, CalendarComponent *component)
{
	CalendarComponentPriv *priv;
	GDate                  date;
	
	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_CALENDAR_COMPONENT (component));

	priv = component->priv;

	g_date_set_time (&date, time (NULL));

	month_view_set_first_day_shown (MONTH_VIEW (priv->month_view), 
					&date);
}

static void
first_day_changed (MonthView *month_view, CalendarComponent *component)
{
	gchar buf[64];

	buf[0] = '\0';
	g_date_strftime (buf,
			 sizeof (buf),
			 _("%Y %B %e"),
			 &month_view->first_day_shown);

	gtk_label_set_text (GTK_LABEL (component->priv->date_label), buf);
}

static BonoboControl *
cc_control_new (CalendarComponent *component)
{
	CalendarComponentPriv *priv;
	GDate                 *date;
	GtkWidget             *vbox, *hbox, *button;
	GtkWidget             *label, *arrow;

	priv = component->priv;

	vbox = gtk_vbox_new (FALSE, 0);

	hbox = gtk_hbox_new (FALSE, 0);

	/* Set up buttons for navigating the calendar.*/
	
	button = gtk_button_new ();
	GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
	gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
	gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
	arrow = gtk_arrow_new (GTK_ARROW_LEFT, GTK_SHADOW_OUT);
	gtk_container_add (GTK_CONTAINER (button), arrow);

	gtk_signal_connect (GTK_OBJECT (button),
			    "clicked",
			    month_back_clicked,
			    component);

	label = gtk_label_new (_("Month"));
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 8);
	
	button = gtk_button_new ();
	GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
	gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
	gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
	arrow = gtk_arrow_new (GTK_ARROW_RIGHT, GTK_SHADOW_OUT);
	gtk_container_add (GTK_CONTAINER (button), arrow);

	gtk_signal_connect (GTK_OBJECT (button),
			    "clicked",
			    month_forward_clicked,
			    component);

	/* Spacing. */
	label = gtk_label_new (" ");
	gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);

	priv->date_label = gtk_label_new ("");
	gtk_box_pack_start (GTK_BOX (hbox), priv->date_label, FALSE, FALSE, 0);
	
	/* Spacing. */
	label = gtk_label_new (" ");
	gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);

	button = gtk_button_new ();
	GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
	gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
	gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
	arrow = gtk_arrow_new (GTK_ARROW_LEFT, GTK_SHADOW_OUT);
	gtk_container_add (GTK_CONTAINER (button), arrow);

	gtk_signal_connect (GTK_OBJECT (button),
			    "clicked",
			    year_back_clicked,
			    component);
	
	label = gtk_label_new (_("Year"));
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 8);
	
	button = gtk_button_new ();
	GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
	gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
	gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
	arrow = gtk_arrow_new (GTK_ARROW_RIGHT, GTK_SHADOW_OUT);
	gtk_container_add (GTK_CONTAINER (button), arrow);

	gtk_signal_connect (GTK_OBJECT (button),
			    "clicked",
			    year_forward_clicked,
			    component);

	gtk_widget_show_all (hbox);

	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
	
	priv->month_view = month_view_new ();
	gtk_widget_show (priv->month_view);

	gtk_signal_connect (GTK_OBJECT (priv->month_view),
			    "first-day-changed",
			    first_day_changed,
			    component);
	
	gtk_box_pack_start_defaults (GTK_BOX (vbox), priv->month_view);

	priv->control = bonobo_control_new (vbox);

	date = g_date_new ();
	g_date_set_time (date, time_month_begin (time (NULL)));

	/* just for making it work */
	month_view_set_first_day_shown (MONTH_VIEW (priv->month_view), 
					date);

	g_date_free (date); /* Fixing a leak that andersca so gracefully pointed out */

	gtk_widget_show (bonobo_control_get_widget (priv->control));

	return priv->control;
}

static void
goto_date_clicked (GtkWidget         *widget,
		   time_t             t,
		   CalendarComponent *component)
{
	GDate date;
	
	g_date_set_time (&date, t);
	
	month_view_set_first_day_shown (MONTH_VIEW (component->priv->month_view), 
					&date);
}

static void
goto_start_clicked (GtkWidget         *widget,
		    CalendarComponent *component)
{
	GDate date;

	g_date_set_time (&date, time (NULL));
	
	month_view_set_first_day_shown (MONTH_VIEW (component->priv->month_view), 
					&date);
}

static gboolean
goto_dialog_shown (GtkWidget         *widget,
		   CalendarComponent *component)
{
	if (component->priv->goto_dialog_shown) {
		return FALSE;
	}

	component->priv->goto_dialog_shown = TRUE;
	return TRUE;
}

static void
goto_dialog_hidden (GtkWidget         *widget,
		    CalendarComponent *component)
{
	component->priv->goto_dialog_shown = FALSE;
}

static void
cc_apply_resource_filter (CalendarComponent *component)
{
	GSList *list, *remove_list = NULL, *node;
	GM_Task *task;
	CalendarComponentPriv *priv;

	g_assert (component != NULL);
	g_assert (IS_CALENDAR_COMPONENT (component));

	d(puts(__FUNCTION__));

	priv = component->priv;

	list = id_map_get_objects (priv->tasks);
	for (node = list; node; node = node->next) {
		g_assert (node != NULL);

		task = (GM_Task *)node->data;

		if (!priv->filter_tasks || 
		    resource_filter_task_is_visible (priv->filter, task->taskId)) {
			month_view_insert_task (MONTH_VIEW (priv->month_view), task);
		} else {
			remove_list = g_slist_prepend (remove_list, 
						       GINT_TO_POINTER (task->taskId));
		}
	}

	if (remove_list) {
		month_view_remove_task (MONTH_VIEW (priv->month_view), 
					remove_list);
		g_slist_free (remove_list);
	}
	g_slist_free (list);
}

static void
cc_resource_filter_apply (GtkWidget         *widget, 
			  GSList            *filter, 
			  CalendarComponent *component)
{
	GM_Id id;
	CalendarComponentPriv *priv;
	GSList *node;

	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_CALENDAR_COMPONENT (component));

	d(puts(__FUNCTION__));

	priv = component->priv;

	resource_filter_all_resources_set_state (priv->filter, FALSE);

	for (node = filter; node; node = node->next) {
		g_assert (node->data != NULL);

		id = GPOINTER_TO_INT (node->data);

		resource_filter_resource_set_state (priv->filter, 
						    id, TRUE);
	}

	priv->filter_tasks = TRUE; /* we have a filter */

	cc_apply_resource_filter (component);
}

static void
cc_resource_filter_clear (GtkWidget *widget, 
			  CalendarComponent *component)
{
	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_CALENDAR_COMPONENT (component));

	d(puts(__FUNCTION__));

	component->priv->filter_tasks = FALSE;
	cc_apply_resource_filter (component);
}

static void
cc_resource_filter_dialog_destroy (GtkObject *object, 
				   CalendarComponent *component)
{
	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_CALENDAR_COMPONENT (component));

	d(puts(__FUNCTION__));

	component->priv->resource_filter_dialog_shown = FALSE;
	component->priv->gui = NULL;
}

static void
cc_cmd_resource_filter (BonoboUIComponent *component, 
			gpointer           data, 
			const char        *cname)
{
	CalendarComponent *calendar;
	CalendarComponentPriv *priv;
	ResourceFilterDialogGui *gui;
	GSList *node, *list;
	GM_Resource *resource;
	gboolean selected;

	d(puts(__FUNCTION__));
	g_return_if_fail (IS_CALENDAR_COMPONENT (data));

	calendar = CALENDAR_COMPONENT (data);
	priv = calendar->priv;

	if (priv->resource_filter_dialog_shown) {
		return;
	}

	priv->resource_filter_dialog_shown = TRUE;

	gui = RESOURCE_FILTER_DIALOG_GUI (resource_filter_dialog_gui_new ());

	priv->gui = gui;

	list = id_map_get_objects (priv->resources);

	for (node = list; node; node = node->next) {
		g_assert (node->data != NULL);

		resource = (GM_Resource *)node->data;

		selected = resource_filter_resource_get_state (priv->filter, 
							       resource->resourceId);

		resource_filter_dialog_gui_add_resource (gui, 
							 resource, 
							 selected);
	}
	g_slist_free (list);

	gtk_signal_connect (GTK_OBJECT (gui), 
			    "apply_filter", 
			    GTK_SIGNAL_FUNC (cc_resource_filter_apply),
			    GTK_OBJECT (calendar));
	gtk_signal_connect (GTK_OBJECT (gui),
			    "clear_filter",
			    GTK_SIGNAL_FUNC (cc_resource_filter_clear),
			    GTK_OBJECT (calendar));
	gtk_signal_connect (GTK_OBJECT (gui),
			    "destroy",
			    GTK_SIGNAL_FUNC (cc_resource_filter_dialog_destroy),
			    GTK_OBJECT (calendar));

	gtk_widget_show_all (GTK_WIDGET (gui));
}

static BonoboUIVerb verbs [] = {
	BONOBO_UI_VERB ("ResourceFilter", cc_cmd_resource_filter),
	BONOBO_UI_VERB_END
};

static void
cc_set_pixmap (BonoboUIComponent *uic, const char *xml_path, const char *icon)
{
	char      *path;
	GdkPixbuf *pixbuf;

	d(puts (__FUNCTION__));

	path   = g_concat_dir_and_file (MRPROJECT_IMAGEDIR, icon);
	pixbuf = gdk_pixbuf_new_from_file (path);

	if (pixbuf == NULL) {
		g_warning ("Cannot load image -- %s", path);
		g_free (path);
		return;
	}

	bonobo_ui_util_set_pixbuf (uic, xml_path, pixbuf);

	gdk_pixbuf_unref (pixbuf);

	g_free (path);
}

static void
calendar_control_activate (BonoboControl     *control,
			   CalendarComponent *component)
{
	Bonobo_UIContainer  remote_ui_container;
	BonoboUIComponent  *ui_component;
	BonoboControl      *goto_control;
	GtkWidget          *goto_widget;

	ui_component = bonobo_control_get_ui_component (control);
	g_assert (ui_component != NULL);

	remote_ui_container = bonobo_control_get_remote_ui_container (control);
	bonobo_ui_component_set_container (ui_component, remote_ui_container);
	bonobo_object_release_unref (remote_ui_container, NULL);

	bonobo_ui_component_freeze (ui_component, NULL);

	/* Add when we get verbs. */
	bonobo_ui_component_add_verb_list_with_data (ui_component,
						     verbs,
						     component);


	bonobo_ui_util_set_ui (ui_component,
			       GNOME_DATADIR,
			       "GNOME_MrProject_CalendarComponent.ui",
			       "mrproject-calendar-component");

	cc_set_pixmap (ui_component, "/Toolbar/ToolResourceFilter",
		       "24_assign.png");

	goto_widget = goto_popup_new (_("Go to - Calendar"),
				      GOTO_TODAY |
				      GOTO_START);
	goto_control = bonobo_control_new (goto_widget);
	
	gtk_widget_show (goto_widget);

	gtk_signal_connect (GTK_OBJECT (goto_widget),
			    "date-clicked",
			    goto_date_clicked,
			    component);

	gtk_signal_connect (GTK_OBJECT (goto_widget),
			    "start-clicked",
			    goto_start_clicked,
			    component);

	gtk_signal_connect (GTK_OBJECT (goto_widget),
			    "today-clicked",
			    goto_today_clicked,
			    component);

	gtk_signal_connect (GTK_OBJECT (goto_widget),
			    "dialog_shown",
			    GTK_SIGNAL_FUNC (goto_dialog_shown),
			    component);
	
	gtk_signal_connect (GTK_OBJECT (goto_widget),
			    "dialog_hidden",
			    goto_dialog_hidden,
			    component);
	
	bonobo_ui_component_object_set (ui_component,
                                        "/Toolbar/CalendarGoto",
                                        BONOBO_OBJREF (goto_control),
                                        NULL);

	bonobo_ui_component_thaw (ui_component, NULL);
}

static void
calendar_control_deactivate (BonoboControl *control)
{
	BonoboUIComponent *ui_component;

	ui_component = bonobo_control_get_ui_component (control);
	g_assert (ui_component != NULL);

	bonobo_ui_component_rm (ui_component, "/", NULL);
 	bonobo_ui_component_unset_container (ui_component);
}

static void
control_activate_cb (BonoboControl *control,
		     gboolean       activate,
		     gpointer       user_data)
{
	if (activate) {
		calendar_control_activate (control, user_data);
	} else {
		calendar_control_deactivate (control);
	}
}

static void
calendar_component_destroy (GtkObject *object)
{
	CalendarComponent     *component;
	CalendarComponentPriv *priv;
	GSList *list, *node;
	GM_Resource *resource;
	GM_Task *task;

	g_return_if_fail (IS_CALENDAR_COMPONENT (object));

	d(puts(__FUNCTION__));

	component = CALENDAR_COMPONENT (object);
	priv      = component->priv;

	if (priv) {
		if (priv->tm_client) {
			gtk_object_destroy (GTK_OBJECT (priv->tm_client));
		}
		if (priv->am_client) {
			gtk_object_destroy (GTK_OBJECT (priv->am_client));
		}
		if (priv->rm_client) {
			gtk_object_destroy (GTK_OBJECT (priv->rm_client));
		}
		if (priv->tasks) {
			list = id_map_get_objects (priv->tasks);
			for (node = list; node; node = node->next) {
				g_assert (node->data != NULL);

				task = (GM_Task *)node->data;
				id_map_remove (priv->tasks, task->taskId);
				CORBA_free (task);
			}
			g_slist_free (list);
			gtk_object_destroy (GTK_OBJECT (priv->tasks));
		}
		if (priv->resources) {
			list = id_map_get_objects (priv->resources);
			for (node = list; node; node = node->next) {
				g_assert (node->data != NULL);

				resource = (GM_Resource *)node->data;
				id_map_remove (priv->resources, 
					       resource->resourceId);
				CORBA_free (resource);
			}
			g_slist_free (list);
			gtk_object_destroy (GTK_OBJECT (priv->resources));
		}
		if (priv->filter) {
			gtk_object_destroy (GTK_OBJECT (priv->filter));
		}
		g_free (priv);
		component->priv = NULL;
	}

	if (priv->control) {
		/* Destroy the control as well. */
		bonobo_object_unref (BONOBO_OBJECT (priv->control));
		priv->control = NULL;
	}

	(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
calendar_component_class_init (CalendarComponentClass *klass)
{
	GtkObjectClass        *object_class;
	ShellComponentClass   *shell_component_class;

	object_class                             = (GtkObjectClass *) klass;
	shell_component_class                    = SHELL_COMPONENT_CLASS (klass);
	parent_class                             = gtk_type_class (PARENT_TYPE);

	object_class->destroy                    = calendar_component_destroy;

	shell_component_class->add_to_shell      = calendar_component_add_to_shell;
	shell_component_class->remove_from_shell = calendar_component_remove_from_shell;
	shell_component_class->set_debug_flags   = calendar_component_set_debug_flags;
	shell_component_class->shell_event       = cc_shell_event;
}

static void
calendar_component_init (CalendarComponent *component)
{
	CalendarComponentPriv   *priv;

	d(puts(__FUNCTION__));
	
	priv = g_new0 (CalendarComponentPriv, 1);

	component->priv = priv;

	priv->resource_filter_dialog_shown = FALSE;
	priv->filter_tasks = FALSE;
	priv->tasks = id_map_new (0);
	priv->resources = id_map_new (0);
	priv->filter = resource_filter_new ();
}

static BonoboObject *
calendar_component_factory (BonoboGenericFactory   *this,
			    const char             *object_id,
			    void                   *data)
{
	ShellComponent *component;
	
	g_return_val_if_fail (object_id != NULL, NULL);

	if (!strcmp (object_id, OAFIID)) {
		component = SHELL_COMPONENT (gtk_type_new (CALENDAR_COMPONENT_TYPE));
		shell_component_construct (component,
					   "=Bonobo/Property:change:Start");

		return BONOBO_OBJECT (component);
	} else {
		g_warning ("Failing to manufacture a '%s'", object_id);
	}
	
	return NULL;
}

static void
cc_resource_inserted_cb (ResourceManagerClient *rmc,
			 GM_Resource           *resource,
			 CalendarComponent     *component)
{
	CalendarComponentPriv *priv;
	GM_Resource *tmp_res;
	gboolean selected;

	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_CALENDAR_COMPONENT (component));
	g_return_if_fail (resource != NULL);

	d(puts(__FUNCTION__));

	priv = component->priv;

	/* just to make sure it's not there! */
	if (id_map_lookup (priv->resources, resource->resourceId) != NULL) {
		g_warning ("Resource: %d already existing!", resource->resourceId);
		return;
	}

	tmp_res = corba_util_resource_duplicate (resource);

	resource_filter_add_resource (priv->filter, tmp_res); /* add it to the filter */
	selected = resource_filter_resource_get_state (priv->filter,
						       tmp_res->resourceId);

	/* we keep a copy locally, just in case */
	id_map_insert_id (priv->resources, resource->resourceId, tmp_res);
	
	/* should notify the filter-table-dialog-gui here too */
	if (priv->resource_filter_dialog_shown) {
		resource_filter_dialog_gui_add_resource (priv->gui,
							 tmp_res, 
							 selected);
	}
}

static void
cc_resource_updated_cb (ResourceManagerClient *rmc,
			GM_Resource *resource,
			CalendarComponent *component)
{
	GM_Resource *old_resource;
	CalendarComponentPriv *priv;

	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_CALENDAR_COMPONENT (component));
	g_return_if_fail (resource != NULL);

	d(puts(__FUNCTION__));

	priv = component->priv;

	old_resource = (GM_Resource *)id_map_lookup (priv->resources,
						 resource->resourceId);
	g_assert (old_resource != NULL);

	corba_util_resource_update (old_resource, resource);

	/* notify everything about the update */
	if (priv->resource_filter_dialog_shown) {
		resource_filter_dialog_gui_update_resource (priv->gui,
							    resource);
	}
}

static void
cc_resources_removed_cb (ResourceManagerClient *rmc,
			 GSList *resources,
			 CalendarComponent *component)
{
	GM_Resource *resource;
	CalendarComponentPriv *priv;
	GSList *node;
	GM_Id id;

	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_CALENDAR_COMPONENT (component));
	g_return_if_fail (resources != NULL);

	d(puts(__FUNCTION__));

	priv = component->priv;

	for (node = resources; node; node = node->next) {
		g_assert (node->data != NULL);

		id = GPOINTER_TO_INT (node->data);
		resource = (GM_Resource *)id_map_lookup (priv->resources, id);

		if (resource != NULL) {
			id_map_remove (priv->resources, id);
			CORBA_free (resource);
		}
		/* notify everything about the deletion */
		resource_filter_remove_resource (priv->filter, resource->resourceId);

		if (priv->resource_filter_dialog_shown) {
			resource_filter_dialog_gui_remove_resource (priv->gui, id);
		}
	}
}

static void
cc_allocation_added_cb (AllocationManagerClient *amc,
			GM_Allocation           *alloc,
			CalendarComponent       *component)
{
	CalendarComponentPriv *priv;

	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_CALENDAR_COMPONENT (component));
	g_return_if_fail (alloc != NULL);

	d(puts(__FUNCTION__));

	priv = component->priv;

	resource_filter_add_allocation (priv->filter, alloc);

	cc_apply_resource_filter (component); /* should have a nicer function here */
}

static void 
cc_allocations_removed_cb (AllocationManagerClient *amc,
			   GSList                  *allocations,
			   CalendarComponent       *component)
{
	GSList *alloc;
	GM_Allocation *allocation;
	CalendarComponentPriv *priv;

	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_CALENDAR_COMPONENT (component));
	
	d(puts(__FUNCTION__));

	priv = component->priv;

	/* traverse all allocations to be removed */
	for (alloc = allocations; alloc; alloc = alloc->next) {
		g_assert (alloc->data != NULL);

		allocation = (GM_Allocation *)alloc->data;

		resource_filter_remove_allocation (priv->filter, allocation);
	}

	cc_apply_resource_filter (component); /* should have a nicer function here */
}


static void 
cc_task_inserted_cb (TaskManagerClient       *tmc, 
		     GM_Task                 *task,
		     GM_Id                    sibling_id,
		     GM_TaskOrderType         type,
		     CalendarComponent       *component) 
{
	GM_Task *tmp_task;

	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_CALENDAR_COMPONENT (component));
	g_return_if_fail (task != NULL);

	d(puts(__FUNCTION__));

	d(g_print ("Inserting Task_ID: %d\n", task->taskId));
	g_assert (component->priv != NULL);
	g_assert (component->priv->month_view != NULL);


	tmp_task = corba_util_task_duplicate (task);
	g_assert (tmp_task != NULL);

	/* should be done in a better way - we get two copies of task! */
	id_map_insert_id (component->priv->tasks, 
			  task->taskId, tmp_task);

	resource_filter_add_task (component->priv->filter, tmp_task);

	if (component->priv->filter_tasks == FALSE) { 
		/* we only show the task if there is no filter present, 
		 * else it won't have a resource attached to it.
		 */
		month_view_insert_task (MONTH_VIEW (component->priv->month_view), tmp_task);
	}
}

static void 
cc_tasks_removed_cb (TaskManagerClient       *tmc, 
		     GSList                  *tasks, 
		     CalendarComponent       *component)
{
	GSList *node;
	GM_Task *task;
	GM_Id id;

	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_CALENDAR_COMPONENT (component));
	g_return_if_fail (tasks != NULL);

	g_assert (IS_MONTH_VIEW (component->priv->month_view));

	d(puts(__FUNCTION__));

	/* remove all tasks from internal idmap */
	for (node = tasks; node; node = node->next) {
		g_assert (node->data != NULL);

		id = GPOINTER_TO_INT (node->data);
		
		task = (GM_Task *)id_map_lookup (component->priv->tasks, id);
		if (task) {
			id_map_remove (component->priv->tasks, id);

			CORBA_free (task);
		}
		resource_filter_remove_task (component->priv->filter, id);
	}

	month_view_remove_task (MONTH_VIEW (component->priv->month_view), tasks);
}

static void 
cc_task_updated_cb (TaskManagerClient       *tmc, 
		    GM_Task                 *task, 
		    CalendarComponent       *component)
{
	GM_Task *tmp_task;

	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_CALENDAR_COMPONENT (component));
	g_return_if_fail (task != NULL);

	g_assert (IS_MONTH_VIEW (component->priv->month_view));

	d(puts(__FUNCTION__));

	tmp_task = (GM_Task *)id_map_lookup (component->priv->tasks, task->taskId);
	corba_util_task_update (tmp_task, task, TASK_CHANGE_ALL);

	month_view_update_task (MONTH_VIEW (component->priv->month_view), task);
}

static void 
cc_tasks_repositioned_cb (TaskManagerClient       *tmc, 
			  GM_Id                    task_id, 
			  GM_Id                    sibling_id, 
			  GM_TaskOrderType         type, 
			  CalendarComponent       *component)
{
	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_CALENDAR_COMPONENT (component));

	d(puts(__FUNCTION__));
}

static void 
cc_task_reparented_cb (TaskManagerClient       *tmc, 
		       GM_Id                    task_id, 
		       GM_Id                    new_parent_id, 
		       CalendarComponent       *component)
{
	GM_Task *task;

	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_CALENDAR_COMPONENT (component));
	g_assert (IS_MONTH_VIEW (component->priv->month_view));

	d(g_print ("%s: Id: %d, newparent: %d\n", __FUNCTION__, task_id, new_parent_id));

	d(puts(__FUNCTION__));

	task = (GM_Task *)id_map_lookup (component->priv->tasks, task_id);
	g_assert (task != NULL);
	task->parentId = new_parent_id; 

	month_view_reparent_task (MONTH_VIEW (component->priv->month_view),
				  task_id, new_parent_id);
}

static void 
cc_tasks_linked_cb (TaskManagerClient       *tmc, 
		    GM_Dependency           *dependency, 
		    CalendarComponent       *component)
{
	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_CALENDAR_COMPONENT (component));
	g_return_if_fail (dependency != NULL);

	d(puts(__FUNCTION__));
}

static void 
cc_tasks_unlinked_cb (TaskManagerClient       *tmc, 
		      GM_Dependency           *dependency, 
		      CalendarComponent       *component)
{
	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_CALENDAR_COMPONENT (component));
	g_return_if_fail (dependency != NULL);

	d(puts(__FUNCTION__));
}

static void
cc_items_foreach_clear (gpointer key, gpointer value, gpointer user_data)
{
	GM_Task *task;

	g_assert (value != NULL);

	task = (GM_Task *)value;

	CORBA_free (task);
}

static void
cc_items_clear_cb (TaskManagerClient *mc, 
		   CalendarComponent *component)
{
	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_CALENDAR_COMPONENT (component));

	d(puts(__FUNCTION__));

	month_view_clear (MONTH_VIEW (component->priv->month_view));

	id_map_foreach (component->priv->tasks, cc_items_foreach_clear, NULL);
	gtk_object_destroy (GTK_OBJECT (component->priv->tasks));
	component->priv->tasks = id_map_new (0);
}

#define IS_EVENT(event_name, str) (!strncmp (event_name, str, sizeof (str)-1))

static void
cc_shell_event (ShellComponent *component,
		const gchar    *event_name,
		CORBA_any      *any)
{
	CalendarComponentPriv *priv;
	gchar                 *subtype, *kind;

	subtype = bonobo_event_subtype (event_name);
	kind = bonobo_event_kind (event_name);
		
	d(g_print ("CalendarComp: got event '%s', subtype: '%s', kind: '%s'\n",
		 event_name,
		 subtype,
		 kind));

	priv = CALENDAR_COMPONENT (component)->priv;

	if (IS_EVENT (event_name, "Bonobo/Property:")) {
		/* We have a BonoboProperty event. */

		if (!strcmp (subtype, "Start")) {
			GDate  date;
			time_t t = *(time_t *) any->_value;

			if (t > 0) {
				g_date_set_time (&date, t);
				month_view_set_first_day_shown (MONTH_VIEW (priv->month_view), &date);
			}
		}
	} else {
		d(g_print ("CalendarComponent: got unhandled event: %s\n", event_name));
	}

	g_free (kind);
	g_free (subtype);
}

BONOBO_OAF_SHLIB_FACTORY_MULTI ("OAFIID:GNOME_MrProject_CalendarComponentFactory",
				"Mr Project calendar component factory",
				calendar_component_factory,
				NULL);

BONOBO_X_TYPE_FUNC (CalendarComponent, 
		    PARENT_TYPE,
		    calendar_component);








