/* some handling of tracks for encoding

   Written by Matthias Hensler
   Copyright WSPse 1999+2000
   eMail: wsp@gmx.de

Created: 1999/08/22
Updated: 2000/07/27
*/

/* Copying:
   This program is free software; you can redistribute it and/or modify it under
   the terms of the GNU Gerneral Public License as published by the Free Soft-
   ware Foundation; either version 2 of License, or (at your option) any later
   version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILTY or
   FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 675 Mass
   Ave, Cambridge, MA 02139, USA.
   */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <ncurses.h>
#include "mp3creat.h"

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

/* Externals */
extern void wuuush(int);
extern void free_song_typ(song_typ **anchor);
extern void calc_tot_frm();
extern void fill_in_songs(WINDOW *win);
extern char *return_track_tmpname(song_typ *track);
extern int enc_non_fly(song_typ *track, BOOL del_tmp_file);
extern int set_mp3_inf(song_typ *track);
extern int add_to_m3u(song_typ *song);
extern void add_field_entry(field_select_typ **anchor, char *des, int nr, char *value);
extern field_select_typ *select_field_box(field_select_typ *fs_anchor, int max_length,
					  char *stat_text,
					  BOOL use_old, BOOL allow_space);
extern void popup_error_win(char *tx);
extern void free_field_select(field_select_typ **anchor);
extern int read_from_sock2(char **line, int sock_fd, BOOL wait);
extern int create_sub_dirs(char *filename, BOOL mode);
extern int lock_lock_file(char *lockfile, BOOL extension);
extern int lock_unlock_file(char *lockfile, BOOL extension);
extern song_typ *lay_global_anchor[2];
extern song_typ *lay_top_anchor[2];
extern song_typ *lay_curr_anchor[2];
extern int lay_select_line[2];
extern int lay_act_side;
extern unsigned long int lay_tot_frm[2];
extern int cache_remain;

/* Globals */
int num_ripped_tracks;       /* number of tracks in "->to encode" list */

/* copy char-pointer */
char *copy_char_str(char *old)
{
  char *new;

  if(! old) return NULL;
  new = (char *) malloc(sizeof(char) * (strlen(old)+1));
  if(! new) {
    wuuush(1);
  }
  strcpy(new, old);

  return new;
}

/* duplicate song_typ */
song_typ *copy_song_typ(song_typ *old)
{
  song_typ *new;

  new = (song_typ *) malloc(sizeof(song_typ));
  if(! new) {
    wuuush(1);
  }

  new->convert      = old->convert;
  new->artist       = copy_char_str(old->artist);
  new->title        = copy_char_str(old->title);
  new->album        = copy_char_str(old->album);
  new->comment      = copy_char_str(old->comment);
  new->year         = old->year;
  new->genre        = old->genre;
  new->filename     = copy_char_str(old->filename);
  new->dirname      = copy_char_str(old->dirname);
  new->on_fly       = old->on_fly;
  new->toc          = old->toc;
  new->fn_auto      = old->fn_auto;
  new->cddb_id      = old->cddb_id;
  new->tmp_wav_file = copy_char_str(old->tmp_wav_file);
  new->frame_len    = old->frame_len;
  new->sampler      = old->sampler;
  new->next         = old->next;
  new->prev         = old->prev;

  return new;
}

void rhand_add_ripped(song_typ *track, WINDOW *win)
{
  song_typ *new, *curr;
 
  /* do not add non-existing tracks to list */
  if(! track->tmp_wav_file) {
    if(access(return_track_tmpname(track), R_OK) != 0) {
      return;
    }
  } else {
    if(access(track->tmp_wav_file, R_OK) != 0) {
      return;
    }
  }
    
  new = copy_song_typ(track);
  new->on_fly = TRUE;        /* delete after encoding */
  new->convert = TRUE;       /* should be encoded     */
  track->convert = FALSE;    /* file is ripped        */
 
  if(! new->tmp_wav_file) new->tmp_wav_file = copy_char_str(return_track_tmpname(track));
  if(new->tmp_wav_file) {
    curr = lay_global_anchor[1];
    while(curr) {
      if((curr->tmp_wav_file) && (strcmp(new->tmp_wav_file, curr->tmp_wav_file) == 0)) {
	new->next = NULL;
	new->prev = NULL;
	free_song_typ(&new);
	new = NULL;
	return;
      }
      curr = curr->next;
    }
  }
  
  new->next = NULL;
  if(! lay_global_anchor[1]) {
    lay_global_anchor[1] = new;
    new->prev            = NULL;
    lay_top_anchor[1]    = new;
    lay_curr_anchor[1]   = new;
  } else {
    curr = lay_global_anchor[1];
    while(curr->next) curr = curr->next;
    curr->next = new;
    new->prev  = curr;
  }

  num_ripped_tracks++;
  calc_tot_frm();
  fill_in_songs(win);
}

void rhand_delete_from_list(song_typ *track, WINDOW *win)
{
  int i;
  song_typ *curr;
  
  if(! track) return;
  
  if(track->on_fly && track->tmp_wav_file)
    unlink(track->tmp_wav_file);

  lay_tot_frm[1] -= track->frame_len;

  if(track->prev) track->prev->next = track->next;
  else lay_global_anchor[1] = track->next;
  if(track->next) track->next->prev = track->prev;

  if(lay_top_anchor[1] == track) {
    if(lay_top_anchor[1]->next) lay_top_anchor[1] = lay_top_anchor[1]->next;
    else if(lay_top_anchor[1]->prev) lay_top_anchor[1] = lay_top_anchor[1]->prev;
    else lay_top_anchor[1] = NULL;
  }

  if(track == lay_curr_anchor[1] && track->next == NULL) {
    if(lay_select_line[1] > 1) lay_select_line[1]--;
  } else if(lay_top_anchor[1] && lay_curr_anchor[1]) {
    curr = lay_top_anchor[1];
    i=1;
    while(curr) {
      if(curr == lay_curr_anchor[1]) {
	lay_select_line[1] = i;
	break;
      }
      curr = curr->next;
      i++;
    }
  }
    
  track->next = NULL;
  track->prev = NULL;
  free_song_typ(&track);
  track = NULL;

  num_ripped_tracks--;
  if(! num_ripped_tracks) lay_act_side = 0;
  if(win) fill_in_songs(win);
}
  
void rhand_do_enc(song_typ *track, WINDOW *win)
{
  if((! enc_non_fly(track, track->on_fly)) && (! set_mp3_inf(track)) &&
     (! add_to_m3u(track))) {
    rhand_delete_from_list(track, win);
  }
}

void rhand_del_actual(WINDOW *win)
{
  if(lay_curr_anchor[1]) {
    rhand_delete_from_list(lay_curr_anchor[1], win);
  }
}

void rhand_del_marked(WINDOW *win)
{
  song_typ *curr, *next;

  curr = lay_global_anchor[1];
  while(curr) {
    next = curr->next;
    if(curr->convert) {
      rhand_delete_from_list(curr, NULL);
    }
    curr = next;
  }

  fill_in_songs(win);
}

void rhand_export_track(FILE *fd, song_typ *track, BOOL clear_del) {
  fputs("#MP3c#\n", fd);
  
  /* save track datas */
  if(track->artist) fputs(track->artist, fd);
  fputc('\n', fd);
  if(track->title) fputs(track->title, fd);
  fputc('\n', fd);
  if(track->album) fputs(track->album, fd);
  fputc('\n', fd);
  if(track->comment) fputs(track->comment, fd);
  fprintf(fd, "\n%d\n%d\n", track->year, track->genre);
  if(track->filename) fputs(track->filename, fd);
  fputc('\n', fd);
  if(track->dirname) fputs(track->dirname, fd);
  fprintf(fd, "\n%d\n%d %d\n%ld\n", track->toc, track->fn_auto, track->sampler, track->cddb_id);
  if(track->tmp_wav_file) fputs(track->tmp_wav_file, fd);
  fprintf(fd, "\n%ld\n", track->frame_len);

  if(clear_del) track->on_fly = FALSE;
}

/* export ripped list
   file_name - file to where to store list
   clear_del - if delete flag should be set to FALSE after export
   sel_code  - 0-save actual, 1-save marked, 2-save all
 */
int rhand_save_ripped(char *file_name, BOOL clear_del, int sel_code)
{
  FILE *save_fd;
  char *error_str;
  field_select_typ *anchor, *new;
  int code;
  song_typ *curr;

  if(sel_code < 0 || sel_code > 2) return 1;

  if(! lay_global_anchor[1]) return 0;
  
  create_sub_dirs(file_name, TRUE);

  /* lock list */
  if(lock_lock_file(file_name, TRUE) == 3) {
    popup_error_win(_("listfile is locked, write aborted"));
    return 1;
  }

  if(access(file_name, F_OK) == 0) {
    if(access(file_name, W_OK) != 0) {
      error_str = (char *) malloc(sizeof(char) * (strlen(file_name) + 100));
      if(! error_str) {
	wuuush(1);
      }
      sprintf(error_str, _("file \"%s\" exists, and isn't writeable"), file_name);
      popup_error_win(error_str);
      free(error_str);
      error_str = NULL;
      lock_unlock_file(file_name, TRUE);
      return 1;
    } else {
      anchor = NULL;
      add_field_entry(&anchor, _("Overwrite file"), 1, NULL);
      add_field_entry(&anchor, _("Append to file"), 2, NULL);
      add_field_entry(&anchor, _("Abort"), 3, NULL);
      error_str = (char *) malloc(sizeof(char) * (strlen(file_name) + 100));
      if(! error_str) {
	wuuush(1);
      }
      sprintf(error_str, _("file \"%s\" exists, what to do?"), file_name);
      new = select_field_box(anchor, 60, error_str, FALSE, FALSE);
      free(error_str);
      error_str = NULL;
      if(! new) code = 0;
      else      code = (int) ((unsigned char) *(new->information));
      free_field_select(&anchor);
      anchor = NULL;
      if(code == 1) {
	unlink(file_name);
      } else if(code == 0) {
	lock_unlock_file(file_name, TRUE);
	return 1;
      } else if(code == 3) {
	lock_unlock_file(file_name, TRUE);
	return 0;
      }
    }
  }

  save_fd = fopen(file_name, "a");
  if(! save_fd) {
    error_str = (char *) malloc(sizeof(char) * (strlen(file_name) + 100));
    sprintf(error_str, _("could not open file \"%s\" for writing"), file_name);
    popup_error_win(error_str);
    free(error_str);
    error_str = NULL;
    lock_unlock_file(file_name, TRUE);
    return 1;
  }
  
  if(sel_code == 0) {
    rhand_export_track(save_fd, lay_curr_anchor[1], clear_del);
    fclose(save_fd);

    popup_error_win(_("Track exported"));
    lock_unlock_file(file_name, TRUE);
    return 0;
  }
  curr = lay_global_anchor[1];
  while(curr) {
    if(curr->convert || sel_code == 2) {
      rhand_export_track(save_fd, curr, clear_del);
    }
    curr = curr->next;
  }

  fclose(save_fd);

  popup_error_win(_("Tracks exported"));
  lock_unlock_file(file_name, TRUE);
  return 0;
}

int rhand_load_ripped(char *file_name, WINDOW *win)
{
  int load_fd;
  char *error_str;
  song_typ *curr;
  char *inp_line;
  BOOL ignore_mode;
  field_select_typ *anchor, *new;
  int i;
  char *pat;

  if(lock_lock_file(file_name, TRUE) == 3) {
    popup_error_win(_("listfile is locked, read aborted"));
    return 1;
  }

  load_fd = open(file_name, O_RDONLY);
  if(load_fd < 1) {
    error_str = (char *) malloc(sizeof(char) * (strlen(file_name) + 100));
    sprintf(error_str, _("could not open \"%s\""), file_name);
    popup_error_win(error_str);
    free(error_str);
    error_str = NULL;
    lock_unlock_file(file_name, TRUE);
    return 1;
  }
 
  cache_remain = 0;
  ignore_mode  = FALSE;

  while(read_from_sock2(&inp_line, load_fd, FALSE) == 0) {
    curr = NULL;
    if(strcmp(inp_line, "#MP3c#") != 0) {
      popup_error_win(_("importfile is in illegal format!"));
      if(inp_line) {
	free(inp_line);
	inp_line = NULL;
      }
      close(load_fd);
      lock_unlock_file(file_name, TRUE);
      return 1;
    }
    if(inp_line) {
      free(inp_line);
      inp_line = NULL;
    }
    curr = (song_typ *) malloc(sizeof(song_typ));
    if(! curr) {
      wuuush(1);
    }
    curr->convert      = TRUE;
    curr->artist       = NULL;
    curr->title        = NULL;
    curr->album        = NULL;
    curr->comment      = NULL;
    curr->year         = 1999;
    curr->genre        = 0;
    curr->filename     = NULL;
    curr->dirname      = NULL;
    curr->on_fly       = FALSE;
    curr->toc          = 1;
    curr->fn_auto      = TRUE;
    curr->cddb_id      = 0L;
    curr->tmp_wav_file = NULL;
    curr->frame_len    = 1;
    curr->sampler      = FALSE;
    curr->next         = NULL;
    curr->prev         = NULL;
    
    /* load track datas */
    if(read_from_sock2(&inp_line, load_fd, FALSE) != 0) break;
    curr->artist = inp_line;
    if(read_from_sock2(&inp_line, load_fd, FALSE) != 0) break;
    curr->title = inp_line;
    if(read_from_sock2(&inp_line, load_fd, FALSE) != 0) break;
    curr->album = inp_line;
    if(read_from_sock2(&inp_line, load_fd, FALSE) != 0) break;
    curr->comment = inp_line;
    if(read_from_sock2(&inp_line, load_fd, FALSE) != 0) break;
    if(strlen(inp_line) > 0 && strlen(inp_line) < 5)
      curr->year = atoi(inp_line);
    free(inp_line);
    inp_line = NULL;
    if(read_from_sock2(&inp_line, load_fd, FALSE) != 0) break;
    if(strlen(inp_line) > 0 && strlen(inp_line) < 4) {
      curr->genre = atoi(inp_line);
      if(curr->genre < 0 || curr->genre > TOT_GENRES) curr->genre = 0;
    }
    free(inp_line);
    inp_line = NULL;
    if(read_from_sock2(&inp_line, load_fd, FALSE) != 0) break;
    curr->filename = inp_line;
    if(read_from_sock2(&inp_line, load_fd, FALSE) != 0) break;
    curr->dirname = inp_line;
    if(read_from_sock2(&inp_line, load_fd, FALSE) != 0) break;
    if(strlen(inp_line) > 0 && strlen(inp_line) < 3) {
      curr->toc = atoi(inp_line);
    }
      if(curr->toc < 0 || curr->toc > 99) curr->toc = 0;
    free(inp_line);
    inp_line = NULL;
    if(read_from_sock2(&inp_line, load_fd, FALSE) != 0) break;
    if(strlen(inp_line) > 0 && strlen(inp_line) < 5) {
      pat = strchr(inp_line, ' ');
      if(pat) *pat = 0;
      curr->fn_auto = atoi(inp_line);
      if(pat) curr->sampler = atoi(pat+1);
    }
    if(curr->fn_auto != TRUE && curr->fn_auto != FALSE) curr->fn_auto = TRUE;
    if(curr->sampler != TRUE && curr->sampler != FALSE) curr->sampler = FALSE;
    free(inp_line);
    inp_line = NULL;
    if(read_from_sock2(&inp_line, load_fd, FALSE) != 0) break;
    if(strlen(inp_line) > 0 && strlen(inp_line) < 12) 
      curr->cddb_id = atol(inp_line);
    free(inp_line);
    inp_line = NULL;
    if(read_from_sock2(&inp_line, load_fd, FALSE) != 0) break;
    curr->tmp_wav_file = inp_line;
    if(read_from_sock2(&inp_line, load_fd, FALSE) != 0) break;
    if(strlen(inp_line) > 0 && strlen(inp_line) < 12) {
      curr->frame_len = atol(inp_line);
      if(curr->frame_len < 0) curr->frame_len = 1;
    }
    free(inp_line);
    inp_line = NULL;

    if(curr->tmp_wav_file) {
      if(access(curr->tmp_wav_file, R_OK) != 0) {
	if(! ignore_mode) {
	  anchor = NULL;
	  add_field_entry(&anchor, _("Ignore"), 1, NULL);
	  add_field_entry(&anchor, _("Ignore all"), 2, NULL);
	  add_field_entry(&anchor, _("Abort"), 3, NULL);
	  error_str = (char *) malloc(sizeof(char) * (strlen(curr->tmp_wav_file) + 100));
	  if(! error_str) {
	    wuuush(1);
	  }
	  sprintf(error_str, _("trackfile \"%s\" is not available"), curr->tmp_wav_file);
	  new = select_field_box(anchor, 72, error_str, FALSE, FALSE);
	  free(error_str);
	  error_str = NULL;
	  if(! new) {
	    i = 0;
	  } else {
	    i = (int) ((unsigned char) *(new->information));
	  }
	  free_field_select(&anchor);
	  if(i == 2) {
	    ignore_mode = TRUE;
	  } else if(i == 3) {
	    free_song_typ(&curr);
	    curr = NULL;
	    if(inp_line) {
	      free(inp_line);
	      inp_line = NULL;
	    }
	    close(load_fd);
	    popup_error_win(_("Aborted"));
	    lock_unlock_file(file_name, TRUE);
	    return 0;
	  }
	}
      } else {
	rhand_add_ripped(curr, win);
      }
    }
    free_song_typ(&curr);
    curr = NULL;
    inp_line = NULL;
  }  
  
  if(inp_line) {
    free(inp_line);
    inp_line = NULL;
  }
  if(curr) {
    free_song_typ(&curr);
    curr = NULL;
  }
  close(load_fd);

  popup_error_win(_("Tracks imported"));
  lock_unlock_file(file_name, TRUE);
  return 0;
}

void rhand_remove_dead_files(WINDOW *win)
{
  song_typ *curr, *next;

  curr = lay_global_anchor[1];
  while(curr) {
    next = curr->next;
    if(access(curr->tmp_wav_file, R_OK) != 0) {
      rhand_delete_from_list(curr, win);
    }
    curr = next;
  }
}

