/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.kneschke.de/projekte/modlogan
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: generate.c,v 1.50 2002/01/03 12:16:18 ostborn Exp $
*/

#include <libintl.h>
#include <locale.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>


#include "mconfig.h"
#include "mstate.h"
#include "mlocale.h"
#include "mhash.h"
#include "mlist.h"
#include "mdatatypes.h"
#include "datatypes/count/datatype.h"
#include "datatypes/visited/datatype.h"
#include "mplugins.h"

#include "pictures.h"
#include "plugin_config.h"

#define HIGHLIGHT	1
#define GROUPING	2
#define VISITS		4
#define INDEX		8
#define BROKEN_LINK	16
#define PERCENT		32
#define RESOLVE_TLD     64
#define VISITS_TRAFFIC  128
#define SORT_BY_KEY     256
#define VIEW_DURATION   512
#define TIME            1024

#define BOTTOM_THRESHOLD	16

static int html_indent_depth = 0;

/* tla -> Three Letter Abbreviation */
char *get_month_string(int m, int tla) {
	static char monthname[255];
	
	struct tm *tm;
	time_t t = time(NULL);
	
	tm = localtime(&t);
	
	tm->tm_mon = m > 0 ? m - 1 : 11;
	
	strftime(monthname, sizeof(monthname)-1, tla ? "%b" : "%B", tm);
	return monthname;
}

char *get_duration_string(time_t t) {
	static char str[255];
	time_t days    = (t/60/60/24);
	time_t hours   = (t/60/60)%24;
	time_t mins    = (t/60)%60;
	time_t secs    = (t)%60;

	if (days) {
		sprintf(str, "%ld %s %.2ld:%.2ld:%.2ld", days, days==1?_("day"):_("days"), hours, mins, secs);
	} else {
		sprintf(str, "%.2ld:%.2ld:%.2ld", hours, mins, secs);
	}

	return str;
}


char *htmlencode(const char *s) {
	char *p;
	char *q = NULL;
	int q_len = 0;
	if (!s) return NULL;
	
	q_len = strlen(s) * 2 + 1;
	q = malloc(q_len);

	p = q;
	*p = '\0';
	
	while (*s) {
		switch(*s) {
			case '<':
				*p = '&';
				*(++p) = 'l';
				*(++p) = 't';
				*(++p) = ';';
				break;
			case '>':
				*p = '&';
				*(++p) = 'g';
				*(++p) = 't';
				*(++p) = ';';
				break;
			default:
				*p = *s;
				break;
		}
		*(++p) = '\0';
		s++;
		
		if (strlen(q) > (q_len - 4)) {
			q_len += 128;
			q = realloc(q, q_len);
			
			p = q + strlen(q);
		}
	}
	
	return q;
}

int show_mhash (mconfig *ext_conf, FILE *f, mhash *h, int count, int opt) {
	int i, html_indent;
	config_output *conf = ext_conf->plugin_conf;
	long sum = 0;
	mdata **md;

	if (!h) return 0;
	
	sum = mhash_sumup(h);
	
	if (opt & SORT_BY_KEY) {
		md = mhash_sorted_to_marray(h, M_SORTBY_KEY, M_SORTDIR_ASC);
	} else {
		md = mhash_sorted_to_marray(h, M_SORTBY_COUNT, M_SORTDIR_DESC);
	}

	for (i = 0; md[i] && (i < count); i++) {
		mdata *data = md[i];

		if (data) {
			char *enc_url = NULL;
			int cut_url = 0;
			unsigned int c = 0;

			if (!conf->dont_escape_entities) {
				enc_url = htmlencode(data->key);
			} else {	
				enc_url = malloc(strlen(data->key)+1);
				strcpy(enc_url, data->key);
			}

			cut_url = (strlen(enc_url) > 40) && !conf->dont_cut_urls;
			
			if (cut_url) {
				enc_url[40] = '\0';
			}
			
			for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
			html_indent_depth++;
			fprintf(f,"<tr>\n");
			
			if (opt & INDEX) {
				for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
				fprintf(f,"<td align=\"right\">%d</td>\n", i+1);
			}

			c = mdata_get_count(data);

			for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
			if (opt & TIME) {
				fprintf(f,"<td align=\"right\">%s</td>\n", get_duration_string(c));
			} else {
				fprintf(f,"<td align=\"right\">%d</td>\n", c);
			}
			
			if (opt & PERCENT && sum) {
				for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
				fprintf(f,"<td align=\"right\">%.2f</td>\n", c * 100.0 / sum);
			}
			
			if (opt & VISITS && data->type == M_DATA_TYPE_VISITED ) {
				/* if the vcount is used as 'traffic' */
				for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
				if (opt & VISITS_TRAFFIC) {
					char c = ' ';
					double tr = data->data.visited.vcount;
					if (tr > 1024) {tr /= 1024; c = 'k';}
					if (tr > 1024) {tr /= 1024; c = 'M';}
					if (tr > 1024) {tr /= 1024; c = 'G';}
					fprintf(f,"<td align=\"right\">%.2f&nbsp;%cB</td>\n", tr, c);
				} else {
					fprintf(f,"<td align=\"right\">%d</td>\n", data->data.visited.vcount);
				}
			}

			if (opt & VIEW_DURATION && data->type == M_DATA_TYPE_VISITED ) {
				/* use count as duration and vcount as hits per page view */
				for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
				fprintf(f,"<td align=\"right\">%d</td>\n", data->data.visited.vcount);
				fprintf(f,"<td align=\"right\">%s</td>\n", data->data.visited.vcount ? 
						get_duration_string(data->data.visited.count / data->data.visited.vcount) :
						"--");
			}

			if ((opt & GROUPING) && mdata_is_grouped(data)) {
				for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
				fprintf(f,"<td class=\"grouping\">%s%s</td>\n", enc_url, cut_url ? "..." : "");
			} else {
				if (opt & HIGHLIGHT) {
					for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
					if (conf->assumedprotocol == NULL || strstr(data->key, "://")) {
						fprintf(f,"<td><a href=\"%s\">%s</a>%s</td>\n", data->key, enc_url, cut_url ? "..." : "");
					} else {
						fprintf(f,"<td><a href=\"%s://%s%s%s\">%s</a>%s</td>\n", conf->assumedprotocol, conf->hostname, *data->key == '/' ? "" : "/", data->key, enc_url, cut_url ? "..." : "");
					}
				} else {
					/* added option to resolve tlds on the fly */
					for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
					if (opt & RESOLVE_TLD) {
						char *c = htmlencode(misoname(data->key));
						fprintf(f,"<td>%s</td>\n", c);
						free(c);
					} else {
						fprintf(f,"<td>%s%s</td>\n", enc_url, cut_url ? "..." : "");
					}
				}
			}

			if (opt & BROKEN_LINK && data->type == M_DATA_TYPE_BROKENLINK) {
				struct tm *_tm;
				char timebuf[32] = "";
				
				for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");

				if (data->data.brokenlink.referrer) {
					if (strcmp(data->data.brokenlink.referrer, "-")) {
						free(enc_url);
						enc_url = htmlencode(data->data.brokenlink.referrer);
						cut_url = strlen(enc_url) > 40;
			
						if (cut_url) {
							enc_url[40] = '\0';
						}
						fprintf(f,"<td><a href=\"%s\">%s</a>%s</td>\n", data->data.brokenlink.referrer, enc_url, cut_url ? "..." : "");
					} else {
						fprintf(f,"<td>%s</td>\n", data->data.brokenlink.referrer);
					}
				} else {
					fprintf(f,"<td>%s</td>\n", "-");
				}

				_tm = localtime(&(data->data.brokenlink.timestamp));
				if (strftime(timebuf, sizeof(timebuf)-1, "%x", _tm) == 0) {
					fprintf(stderr, "output::modlogan.show_mhash: strftime failed\n");
				}
				for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
				fprintf(f,"<td>%s</td>\n", timebuf);
			}
			html_indent_depth--;
			for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
			fprintf(f,"</tr>\n");

			free(enc_url);
		}
	}
	
	free(md);

	return 0;
}

mlist * get_next_element(mhash *h) {
	mlist *ret = NULL;
	int max = 0, i;
	
	for ( i = 0; i < h->size; i++) {
		mlist *l = h->data[i]->list;
		while (l) {
			if (l->data) {
				mdata *data = l->data;
				
				if ( mdata_get_count(data) > max) {
					max = mdata_get_count(data);
					ret = l;
				}
			}
			l = l->next;
		}
	}
	/* invert the value */
	if (ret) {
		mdata_set_count(ret->data, -mdata_get_count(ret->data));
	}
	
	return ret;
}

int cleanup_elements(mhash *h) {
	int i;
	for ( i = 0; i < h->size; i++) {
		mlist *l = h->data[i]->list;
		while (l) {
			if (l->data) {
				mdata *data = l->data;
				
				if (mdata_get_count(data) > 0) {
/*					fprintf(stderr, "%s.%d: count (%d) > 0 in cleanup\n", 
						__FILE__, __LINE__,
						mdata_get_count(data));*/
				} else {
					mdata_set_count(data, -mdata_get_count(data));
				}
				
			}
			l = l->next;
		}
	}
	
	return 0;
}

int show_visit_path (mconfig *ext_conf, FILE *f, mhash *h, int count, int opt) {
	int i = 0, html_indent;
	config_output *conf = ext_conf->plugin_conf;
	mlist *l;
	long sum;

	if (!h) return 0;
	
	sum = mhash_sumup(h);
	
	while ((l = get_next_element(h)) && i < count) {
			
		for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
		html_indent_depth++;
		fprintf(f,"<tr>\n");
		
		if (l->data) {
			mdata *data = l->data;
			mlist *sl = data->data.sublist.sublist;
			int c;
			
			c = -data->data.sublist.count;
			
			i++;
			for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
			fprintf(f,"<td align=\"right\">%d</td>\n", i);
			
			fprintf(f,"<td align=\"right\">%d</td>\n", c);
			fprintf(f,"<td align=\"right\">%.2f</td>\n", c * 100.0 / sum);
			
			fprintf(f,"<td align=\"left\">");
			
			while (sl) {
				if (sl->data) {
					mdata *sld = sl->data;
					
					fprintf(f, "%s<br />", sld->key);
				}
				
				sl = sl->next;
			}
			fprintf(f,"</td>");
			
		}
		
		html_indent_depth--;
		for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
		fprintf(f,"</tr>\n");
	}
	
	cleanup_elements(h);
	
	return 0;
}

mhash *get_entry_pages(mhash *h) {
	mhash *ret;
	int i;

	if (!h) return 0;

	ret = mhash_init( 32 );
	
	for ( i = 0; i < h->size; i++) {
		mlist *l = h->data[i]->list;
		while (l) {
			if (l->data) {
				mdata *data = l->data;
				mlist *sl = data->data.sublist.sublist;
				
				if (data->type != M_DATA_TYPE_SUBLIST) {
					M_DEBUG2(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
						 "datatype not a sublist: %d - %s\n", data->type, data->key);
					return NULL;
				}
			
				if (sl && sl->data) {
					mdata *sld = sl->data;
					mdata *insd;
					
					insd = mdata_Count_create(sld->key, 1, M_DATA_STATE_PLAIN);
					mhash_insert_sorted(ret, insd);
				}
			}
			l = l->next;
		}
	}
	
	return ret;
}

mhash *get_exit_pages(mhash *h) {
	mhash *ret;
	int i;

	if (!h) return 0;
	
	ret = mhash_init( 32 );
	
	for ( i = 0; i < h->size; i++) {
		mlist *l = h->data[i]->list;
		while (l) {
			if (l->data) {
				mdata *data = l->data;
				mlist *sl = data->data.sublist.sublist;
				
				if (sl) {
					/* go to the last element */
					while (sl->next) sl = sl->next;
					
					if (sl->data) {
						mdata *sld = sl->data;
						mdata *insd;
						
						insd = mdata_Count_create(sld->key, 1, M_DATA_STATE_PLAIN);
						mhash_insert_sorted(ret, insd);
					}
				}
			}
			l = l->next;
		}
	}
	
	return ret;
}

mhash *get_visit_path_length(mhash *h) {
	mhash *ret;
	int i;

	if (!h) return 0;
	
	ret = mhash_init( 32 );
	
	for ( i = 0; i < h->size; i++) {
		mlist *l = h->data[i]->list;
		while (l) {
			if (l->data) {
				mdata *data = l->data;
				mlist *sl = data->data.sublist.sublist;
				
				if (sl) {
					/* go to the last element */
					long c = 0;
					char str[255];
					mdata *insd;
					while (sl) {
						c++;
						sl = sl->next;
					}
					
					sprintf(str, "%5ld", c);
					
					insd = mdata_Count_create(str, 1, M_DATA_STATE_PLAIN);
					mhash_insert_sorted(ret, insd);
				}
			}
			l = l->next;
		}
	}
	
	return ret;
}

mhash *get_visit_duration(mhash *h) {
	mhash *ret;
	int i;

	if (!h) return 0;
	
	ret = mhash_init( 32 );
	
	for ( i = 0; i < h->size; i++) {
		mlist *l = h->data[i]->list;
		while (l) {
			if (l->data) {
				mdata *data = l->data;
				mlist *sl = data->data.sublist.sublist;
			
				if (sl) {
				/* go to the last element */
					char str[255];
					mdata *insd;
					mdata *sld = sl->data;
					
					if (sld) {
						time_t t1 = sld->data.brokenlink.timestamp, t2;
						
						while (sl->next) 
							sl = sl->next;
						t2 = sl->data->data.brokenlink.timestamp;
						
						if (t2 - t1) { 
							sprintf(str, "%5ld %s", (t2 - t1) / 60, _("min"));
						} else {
							sprintf(str, " < 1 %s", _("min"));
						}
				
						insd = mdata_Count_create(str, 1, M_DATA_STATE_PLAIN);
						mhash_insert_sorted(ret, insd);
					}
				}
			}
			l = l->next;
		}
	}
	
	return ret;
}

double get_visit_full_path_length(mhash *h) {
	double d = 0;
	int i;

	if (!h) return 0;
	
	for ( i = 0; i < h->size; i++) {
		mlist *l = h->data[i]->list;
		while (l) {
			if (l->data) {
				mdata *data = l->data;
				
				d += mlist_count(data->data.sublist.sublist);
			}
			l = l->next;
		}
	}
	
	return d;
}

double get_visit_full_duration(mhash *h) {
	double d = 0;
	int i;
	if (!h) return 0;
	
	
	for ( i = 0; i < h->size; i++) {
		mlist *l = h->data[i]->list;
		while (l) {
			if (l->data) {
				mdata *data = l->data;
				mlist *sl = data->data.sublist.sublist;
			
				if (sl) {
					/* go to the last element */
					mdata *sld = sl->data;
					
					if (sld) {
						time_t t1 = sld->data.brokenlink.timestamp, t2;
						
						while (sl->next) 
							sl = sl->next;
						t2 = sl->data->data.brokenlink.timestamp;
				
						d += (t2 - t1);
					}
				}
			}
			l = l->next;
		}
	}
	
	return d;
}

mhash *get_path_length(mhash *h) {
	mhash *ret;
	mlist *l;

	if (!h) return 0;
	
	ret = mhash_init( 32 );
	
	while ((l = get_next_element(h))) {
		if (l->data) {
			mdata *data = l->data;
			mlist *sl = data->data.sublist.sublist;
			
			if (sl) {
				/* go to the last element */
				long c = 0;
				char str[255];
				mdata *insd;
				while (sl) {
					c++;
					sl = sl->next;
				}
				
				sprintf(str, "%5ld", c);
				
				insd = mdata_Count_create(str, 1, M_DATA_STATE_PLAIN);
				mhash_insert_sorted(ret, insd);
			}
		}
	}
	
	cleanup_elements(h);
	
	return ret;
}

int show_status_mhash (FILE *f, mhash *h, int count) {
	int i, html_indent;
	mdata **md;
	
	if (!h) return 0;
	
	md = mhash_sorted_to_marray(h, M_SORTBY_KEY, M_SORTDIR_ASC);

	for (i = 0; md[i] && (i < count); i++) {
		mdata *data = md[i];
	
		if (data) {
			for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
			fprintf(f,"<tr><td align=\"right\">%i</td><td>%s - %s</td></tr>\n", 
				data->data.count.count, data->key, 
				mhttpcodes(strtol(data->key, NULL, 10)));
		}
	}
	
	free(md);

	return 0;
}

void table_start(FILE *f, char *str, int colspan) {
	int html_indent;
	if (f == NULL) return;
	
	for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
	fprintf(f, "<p />\n");
	
	for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
	html_indent_depth++;
	fprintf(f, "<center>\n");
	
	for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
	html_indent_depth++;
	fprintf(f, "<table border=\"1\" %s cellspacing=\"1\" cellpadding=\"3\">\n", colspan < 0 ? "width=\"100%\"" : "");
	
	for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
	fprintf(f, "<tr><th colspan=\"%d\">%s</th></tr>\n", colspan < 0 ? -colspan : colspan, str);
}

void table_end(FILE *f) {
	int html_indent;
	
	html_indent_depth--;
	for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
	fprintf(f, "</table>\n");
	
	html_indent_depth--;
	for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
	fprintf(f, "</center>\n");
	
	for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
	fprintf(f, "<p />\n");
}

void file_start(FILE *f, mconfig *ext_conf, time_t timestamp) {
	char buf[255];
	struct tm *_tm;
	time_t t;
	config_output *conf = ext_conf->plugin_conf;
	int show_default = 1;
	int html_indent;
	
	if (conf->html_header) {
		FILE *inf = fopen (conf->html_header, "r");
		
		if (inf == NULL) {
			fprintf(stderr, "%s.%d: %s: %s", __FILE__, __LINE__, _("Can't open page header"),strerror(errno));
		} else {
			char buf[255];
			
			while (fgets(buf, sizeof(buf)-1, inf)) {
				if (fputs(buf, f) == EOF) {
					fprintf(stderr, "%s.%d: %s: %s", __FILE__, __LINE__, _("Can't write header"),strerror(errno));
					break;
				}
			}
			
			show_default = 0;
			fclose(inf);
		}
	} 
	
	if (show_default) {
		fprintf(f, "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
			"<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\" \"DTD/xhtml1-transitional.dtd\">\n"
			"<html xmlns=\"http://www.w3.org/1999/xhtml\" xml:lang=\"en\" lang=\"en\">\n\n"
			"<head>\n"
			" <title>%s</title>\n"
			" <link rel=\"stylesheet\" href=\"modlogan.css\" type=\"text/css\" />\n"
			" <meta http-equiv=\"Content-Type\" content=\"text/html; charset=%s\" />\n"
			" <meta http-equiv=\"Content-Language\" content=\"%s\" />\n"
			"</head>\n"
			"<body>\n",
			_("Statistics"), conf->cont_charset, conf->cont_language);
	
		fprintf(f, "<h1>%s %s</h1>\n", _("Statistics for"), conf->hostname);
		if (timestamp != 0) {
			t = timestamp;
			_tm = localtime(&t);
			strftime(buf, sizeof(buf), "%X %x", _tm);
			fprintf(f, "<b>%s: </b>%s<br />\n", _("Last record"), buf);
		}
		t = time(NULL);
		_tm = localtime(&t);
		strftime(buf, sizeof(buf), "%X %x", _tm);
		fprintf(f, "<b>%s: </b>%s<br />\n", _("Generated at"), buf);
		fprintf(f, "<hr /><br />\n");
	}
	
	for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
	html_indent_depth++;
	fprintf(f, "<table width=\"100%%\" cellpadding=\"4\" cellspacing=\"0\">\n");
	
	for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
	html_indent_depth++;
	fprintf(f, "<tr valign=\"top\">\n");
	
	for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
	html_indent_depth++;
	fprintf(f, "<td class=\"skeleton\">\n");
}

void file_end(FILE *f, mconfig *ext_conf) {
	config_output *conf = ext_conf->plugin_conf;
	int show_default = 1, html_indent;
	
	html_indent_depth--;
	for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
	fprintf(f, "</td>\n");
	
	html_indent_depth--;
	for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
	fprintf(f, "</tr>\n");
	
	html_indent_depth--;
	for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
	fprintf(f, "</table>\n");
	
	if (conf->html_footer) {
		FILE *inf = fopen (conf->html_footer, "r");
		
		if (inf == NULL) {
			fprintf(stderr, "%s.%d: %s: %s", __FILE__, __LINE__, _("Can't open page footer"),strerror(errno));
		} else {
			char buf[255];
			
			while (fgets(buf, sizeof(buf)-1, inf)) {
				if (fputs(buf, f) == EOF) {
					fprintf(stderr, "%s.%d: %s: %s", __FILE__, __LINE__, _("Can't write footer"),strerror(errno));
					break;
				}
			}
			
			show_default = 0;
			fclose(inf);
		}
	}
	
	if (show_default) {

		fprintf(f, "<hr />");
		
		if (conf->show_validation_links) {
/* Tell the users that this is valid XHTML 1.0  :) */
			fprintf(f, "<a href=\"http://validator.w3.org/check/referer\">"
				" <img border=\"0\" src=\"http://www.w3.org/Icons/valid-xhtml10\" "
				"alt=\"Valid XHTML 1.0!\" height=\"31\" width=\"88\" align=\"right\" />\n"
				"</a>\n"
				"<a href=\"http://jigsaw.w3.org/css-validator/check/referer/\">\n"
				" <img border=\"0\" width=\"88\" height=\"31\""
				"src=\"http://jigsaw.w3.org/css-validator/images/vcss.gif\" "
				"alt=\"Valid CSS!\" align=\"right\" />\n"
				"</a>");
		}
		
		fprintf(f, "%s <a href=\"%s\">%s %s</a>\n", _("Output generated by"),"http://www.kneschke.de/projekte/modlogan/", PACKAGE, VERSION);
		fprintf(f, "</body></html>\n");
	}
}

void file_start_index(FILE *f, mconfig *ext_conf, time_t timestamp) {
	char buf[255];
	struct tm *_tm;
	time_t t;
	config_output *conf = ext_conf->plugin_conf;
	
	int show_default = 1, html_indent;
	
	if (conf->html_header) {
		FILE *inf = fopen (conf->html_header, "r");
		
		if (inf == NULL) {
			fprintf(stderr, "%s.%d: %s: %s", __FILE__, __LINE__, _("Can't open page header"),strerror(errno));
		} else {
			char buf[255];
			
			while (fgets(buf, sizeof(buf)-1, inf)) {
				if (fputs(buf, f) == EOF) {
					fprintf(stderr, "%s.%d: %s: %s", __FILE__, __LINE__, _("Can't write header"),strerror(errno));
					break;
				}
			}
			
			show_default = 0;
			fclose(inf);
		}
	} 
	
	if (show_default) {
		fprintf(f, "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
			"<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\" \"DTD/xhtml1-transitional.dtd\">\n"
			"<html xmlns=\"http://www.w3.org/1999/xhtml\" xml:lang=\"en\" lang=\"en\">\n\n"
			"<head>\n"
			" <title>%s</title>\n"
			" <link rel=\"stylesheet\" href=\"modlogan.css\" type=\"text/css\" />\n"
			" <meta http-equiv=\"Content-Type\" content=\"text/html; charset=%s\" />\n"
			" <meta http-equiv=\"Content-Language\" content=\"%s\" />\n"
			"</head>\n"
			"<body>\n",
			_("Statistics"), conf->cont_charset, conf->cont_language);
	
		fprintf(f, "<h1>%s %s</h1>\n", _("Statistics for"), conf->hostname);
		if (timestamp != 0) {
			t = timestamp;
			_tm = localtime(&t);
			strftime(buf, sizeof(buf), "%X %x", _tm);
			fprintf(f, "<b>%s: </b>%s<br />\n", _("Last record"), buf);
		}
		t = time(NULL);
		_tm = localtime(&t);
		strftime(buf, sizeof(buf), "%X %x", _tm);
		fprintf(f, "<b>%s: </b>%s<br />\n", _("Generated at"), buf);
		fprintf(f, "<hr /><br />\n");
	}
	
	for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
	html_indent_depth++;
	fprintf(f, "<table width=\"100%%\" cellpadding=\"4\" cellspacing=\"0\">\n");
	
	for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
	html_indent_depth++;
	fprintf(f, "<tr valign=\"top\">\n");
	
	for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
	html_indent_depth++;
	fprintf(f, "<td class=\"skeleton\">\n");
}

void file_end_index(FILE *f, mconfig *ext_conf) {
	config_output *conf = ext_conf->plugin_conf;
	int show_default = 1, html_indent;
	
	html_indent_depth--;
	for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
	fprintf(f, "</td>\n");
	
	html_indent_depth--;
	for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
	fprintf(f, "</tr>\n");
	
	html_indent_depth--;
	for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
	fprintf(f, "</table>\n");
	
	if (conf->html_footer) {
		FILE *inf = fopen (conf->html_footer, "r");
		
		if (inf == NULL) {
			fprintf(stderr, "%s.%d: %s: %s", __FILE__, __LINE__, _("Can't open page footer"),strerror(errno));
		} else {
			char buf[255];
			
			while (fgets(buf, sizeof(buf)-1, inf)) {
				if (fputs(buf, f) == EOF) {
					fprintf(stderr, "%s.%d: %s: %s", __FILE__, __LINE__, _("Can't write footer"),strerror(errno));
					break;
				}
			}
			
			show_default = 0;
			fclose(inf);
		}
	}
	
	if (show_default) {
/* Tell the users that this is valid HTML 4.0  :) */
		fprintf(f, "<hr />");
		if (conf->show_validation_links) {
			fprintf(f, "<a href=\"http://validator.w3.org/check/referer\">"
				" <img border=\"0\" src=\"http://www.w3.org/Icons/valid-xhtml10\" "
				"alt=\"Valid XHTML 1.0!\" height=\"31\" width=\"88\" align=\"right\" />\n"
				"</a>\n"
				"<a href=\"http://jigsaw.w3.org/css-validator/check/referer/\">\n"
				" <img border=\"0\" width=\"88\" height=\"31\""
				"src=\"http://jigsaw.w3.org/css-validator/images/vcss.gif\" "
				"alt=\"Valid CSS!\" align=\"right\" />\n"
				"</a>");
		}
		fprintf(f, "%s <a href=\"%s\">%s %s</a>\n", _("Output generated by"),"http://www.kneschke.de/projekte/modlogan/", PACKAGE, VERSION);
		fprintf(f, "</body></html>\n");
	}
}

char *table_header(int maxcount, int count, const char *str) {
	static char trans_buf[255];
	
	sprintf(trans_buf, "%d %s %d %s", (maxcount > count) || (maxcount < 0) ? count : maxcount, _("of"), count, str);
	
	return trans_buf;
	
}

#define	M_TYPE_REPORT	1
#define M_TYPE_PAGE	2

typedef enum { M_REPORT_UNSET, M_REPORT_REQ_URL, M_REPORT_REF_URL, 
		M_REPORT_OS, M_REPORT_HOSTS, M_REPORT_ENTRY_PAGES,
		M_REPORT_EXIT_PAGES, M_REPORT_USERAGENT, M_REPORT_INDEXED,
		M_REPORT_REQ_PROT, M_REPORT_REQ_METH, M_REPORT_STATUS_CODES,
		M_REPORT_ROBOTS, M_REPORT_BOOKMARKS, M_REPORT_BROKEN_LINKS,
		M_REPORT_INTERNAL_ERROR, M_REPORT_SEARCH_ENGINE, 
		M_REPORT_SEARCH_STRINGS, M_REPORT_COUNTRIES, M_REPORT_SUMMARY,
		M_REPORT_HOURLY, M_REPORT_DAILY, M_REPORT_EXTENSION,
		M_REPORT_VISIT_PATH, M_REPORT_VISIT_DURATION,
		M_REPORT_VISIT_PATH_LENGTH, M_REPORT_VIEW_DURATION,
		M_REPORT_VHOSTS,
		
		M_PAGE_INDEX = 128, M_PAGE_001, M_PAGE_002, M_PAGE_003,
		M_PAGE_004, M_PAGE_000
} mreport_type;

int get_menu_items (mconfig *ext_conf, mstate *state, mlist *l) {
	mdata *data;
	config_output *conf = ext_conf->plugin_conf;
	int i = 0;
	mstate_web *staweb = state->ext;
	
	
	data = mdata_Count_create("/000", M_TYPE_PAGE, M_PAGE_INDEX);
	mlist_insert(l, data);
/* page 0 */
	data = mdata_Count_create("/000/000", M_TYPE_PAGE, M_PAGE_000);
	mlist_insert(l, data);
	
	data = mdata_Count_create("/000/000/000", M_TYPE_REPORT, M_REPORT_SUMMARY);
	mlist_insert(l, data);
	
	data = mdata_Count_create("/000/000/001", M_TYPE_REPORT, M_REPORT_DAILY);
	mlist_insert(l, data);
	
	data = mdata_Count_create("/000/000/002", M_TYPE_REPORT, M_REPORT_HOURLY);
	mlist_insert(l, data);
	
/* page 1 */	
	i = 0;
	if (conf->max_req_urls && mhash_count(staweb->req_url_hash)) {
		data = mdata_Count_create("/000/001/001", M_TYPE_REPORT, M_REPORT_REQ_URL);
		mlist_insert(l, data);
		i = 1;
	}
	
	if (conf->max_ref_urls && mhash_count(staweb->ref_url_hash)) {
		data = mdata_Count_create("/000/001/002", M_TYPE_REPORT, M_REPORT_REF_URL);
		mlist_insert(l, data);
		i = 1;
	}
	
	if (mhash_count(staweb->visits)) {
		if (conf->max_entry_pages) {
			data = mdata_Count_create("/000/001/003", M_TYPE_REPORT, M_REPORT_ENTRY_PAGES);
			mlist_insert(l, data);
			i = 1;
		}
	
		if (conf->max_exit_pages) {
			data = mdata_Count_create("/000/001/004", M_TYPE_REPORT, M_REPORT_EXIT_PAGES);
			mlist_insert(l, data);
			i = 1;
		}
		
		if (conf->max_visit_paths) {
			data = mdata_Count_create("/000/001/005", M_TYPE_REPORT, M_REPORT_VISIT_PATH);
			mlist_insert(l, data);
			i = 1;
		}
		
		if (conf->max_visit_path_lengths) {
			data = mdata_Count_create("/000/001/006", M_TYPE_REPORT, M_REPORT_VISIT_PATH_LENGTH);
			mlist_insert(l, data);
			i = 1;
		}
		
		if (conf->max_visit_durations) {
			data = mdata_Count_create("/000/001/007", M_TYPE_REPORT, M_REPORT_VISIT_DURATION);
			mlist_insert(l, data);
			i = 1;
		}
	}

	if (conf->max_view_durations && mhash_count(staweb->views)) {
		data = mdata_Count_create("/000/001/008", M_TYPE_REPORT, M_REPORT_VIEW_DURATION);
		mlist_insert(l, data);
		i = 1;
	}
	
	if (conf->max_extensions && mhash_count(staweb->extension)) {
		data = mdata_Count_create("/000/001/009", M_TYPE_REPORT, M_REPORT_EXTENSION);
		mlist_insert(l, data);
		i = 1;
	}
	
	if (conf->max_vhosts && mhash_count(staweb->vhost_hash)) {
		data = mdata_Count_create("/000/001/010", M_TYPE_REPORT, M_REPORT_VHOSTS);
		mlist_insert(l, data);
		i = 1;
	}
	
	if (i) {
		data = mdata_Count_create("/000/001", M_TYPE_PAGE, M_PAGE_001);
		mlist_insert(l, data);
	}

/* page 2 */	
	i = 0;
	if (conf->max_os && mhash_count(staweb->os_hash)) {
		data = mdata_Count_create("/000/002/001", M_TYPE_REPORT, M_REPORT_OS);
		mlist_insert(l, data);
		i = 1;
	}
	
	if (conf->max_hosts && mhash_count(staweb->host_hash)) {
		data = mdata_Count_create("/000/002/002", M_TYPE_REPORT, M_REPORT_HOSTS);
		mlist_insert(l, data);
		i = 1;
	}
	
	if (conf->max_ua && mhash_count(staweb->ua_hash)) {
		data = mdata_Count_create("/000/002/003", M_TYPE_REPORT, M_REPORT_USERAGENT);
		mlist_insert(l, data);
		i = 1;
	}
	
	if (conf->max_countries && mhash_count(staweb->country_hash)) {
		data = mdata_Count_create("/000/002/004", M_TYPE_REPORT, M_REPORT_COUNTRIES);
		mlist_insert(l, data);
		i = 1;
	}
	
	if (i) {
		data = mdata_Count_create("/000/002", M_TYPE_PAGE, M_PAGE_002);
		mlist_insert(l, data);
	} 
/* page 3 */	
	
	i = 0;
	if (conf->max_indexed_pages && mhash_count(staweb->indexed_pages)) {
		data = mdata_Count_create("/000/003/001", M_TYPE_REPORT, M_REPORT_INDEXED);
		mlist_insert(l, data);
		i = 1;
	}
	
	if (conf->max_robots && mhash_count(staweb->robots)) {
		data = mdata_Count_create("/000/003/002", M_TYPE_REPORT, M_REPORT_ROBOTS);
		mlist_insert(l, data);
		i = 1;
	}
	
	if (conf->max_bookmarks && mhash_count(staweb->bookmarks)) {
		data = mdata_Count_create("/000/003/003", M_TYPE_REPORT, M_REPORT_BOOKMARKS);
		mlist_insert(l, data);
		i = 1;
	}
	
	if (conf->max_search_engines && mhash_count(staweb->searchsite)) {
		data = mdata_Count_create("/000/003/004", M_TYPE_REPORT, M_REPORT_SEARCH_ENGINE);
		mlist_insert(l, data);
		i = 1;
	}
	
	if (conf->max_search_strings && mhash_count(staweb->searchstring)) {
		data = mdata_Count_create("/000/003/005", M_TYPE_REPORT, M_REPORT_SEARCH_STRINGS);
		mlist_insert(l, data);
		i = 1;
	}
	
	if (i) {
		data = mdata_Count_create("/000/003", M_TYPE_PAGE, M_PAGE_003);
		mlist_insert(l, data);
	}

/* page 4 */	
	i = 0;
	if (conf->max_req_prot && mhash_count(staweb->req_prot_hash)) {
		data = mdata_Count_create("/000/004/001", M_TYPE_REPORT, M_REPORT_REQ_PROT);
		mlist_insert(l, data);
		i = 1;
	}
	
	if (conf->max_req_meth && mhash_count(staweb->req_meth_hash)) {
		data = mdata_Count_create("/000/004/002", M_TYPE_REPORT, M_REPORT_REQ_METH);
		mlist_insert(l, data);
		i = 1;
	}
	
	if (conf->max_status_codes && mhash_count(staweb->status_hash)) {
		data = mdata_Count_create("/000/004/003", M_TYPE_REPORT, M_REPORT_STATUS_CODES);
		mlist_insert(l, data);
		i = 1;
	}
	
	if (conf->max_broken_links && mhash_count(staweb->status_missing_file)) {
		data = mdata_Count_create("/000/004/004", M_TYPE_REPORT, M_REPORT_BROKEN_LINKS);
		mlist_insert(l, data);
		i = 1;
	}
	
	if (conf->max_internal_errors && mhash_count(staweb->status_internal_error)) {
		data = mdata_Count_create("/000/004/005", M_TYPE_REPORT, M_REPORT_INTERNAL_ERROR);
		mlist_insert(l, data);
		i = 1;
	}
	
	if (i) {
		data = mdata_Count_create("/000/004", M_TYPE_PAGE, M_PAGE_004);
		mlist_insert(l, data);
	}
	
	return 0;
}

char * get_menu_item(int type) {
	switch(type) {
	case M_REPORT_REQ_URL:
		return _("Requested URL's");
	case M_REPORT_REF_URL:
		return _("Referrers");
	case M_REPORT_OS:	
		return _("Operating system");
	case M_REPORT_HOSTS:
		return _("Hosts");
	case M_REPORT_ENTRY_PAGES:
		return _("Entry Pages");
	case M_REPORT_EXIT_PAGES:
		return _("Exit Pages");
	case M_REPORT_USERAGENT:
		return _("Browsers");
	case M_REPORT_INDEXED:
		return _("Indexed Pages");
	case M_REPORT_REQ_PROT:
		return _("Request Protocol");
	case M_REPORT_REQ_METH:
		return _("Request Method");
	case M_REPORT_STATUS_CODES:
		return _("Status Code");
	case M_REPORT_ROBOTS:
		return _("Robots");
	case M_REPORT_BOOKMARKS:
		return _("Bookmarked Pages");
	case M_REPORT_BROKEN_LINKS:
		return _("Broken Links");
	case M_REPORT_INTERNAL_ERROR:
		return _("Internal Errors");
	case M_REPORT_SEARCH_ENGINE:
		return _("SearchEngines");
	case M_REPORT_SEARCH_STRINGS:
		return _("SearchStrings");
	case M_REPORT_COUNTRIES:
		return _("Countries");
	case M_REPORT_SUMMARY:
		return _("Summary");
	case M_REPORT_HOURLY:
		return _("Hourly Statistics");
	case M_REPORT_DAILY:
		return _("Daily Statistics");
	case M_REPORT_EXTENSION:
		return _("Extensions");
	case M_REPORT_VISIT_PATH:
		return _("Visit Path");
	case M_REPORT_VISIT_PATH_LENGTH:
		return _("Path Length");
	case M_REPORT_VISIT_DURATION:
		return _("Visit Duration");
	case M_REPORT_VIEW_DURATION:
		return _("View Duration");
	case M_REPORT_VHOSTS:
		return _("Vhosts");
	case M_PAGE_INDEX:
		return _("Index");
	case M_PAGE_000:
		return _("Overview");
	case M_PAGE_001:
		return _("URLs");
	case M_PAGE_002:
		return _("User");
	case M_PAGE_003:
		return _("Searchengines");
	case M_PAGE_004:
		return _("Server Internals");
	default:
		return "(null)";
	}
}

char *get_url(mconfig *ext_conf, int year, int month, char *sub, char *report) {
	static char filename[255];
	config_output *conf = ext_conf->plugin_conf;
	
	/* Added by Georges 'Melkor' Goncalves <melkor@lords.com>
	** courtesy of Jan Kneschke
	*/
	/* suffix for the generated statistics page files */
	if (conf->page_style && !strcasecmp(conf->page_style, "onepage")) {
		sprintf(filename, "m_usage_%04d%02d.html%s%.3s%s", 
			year, month, 
			sub ? "#" : "", 
			sub ? sub : "",
			report ? report : "");
	} else if (conf->page_style && !strcasecmp(conf->page_style, "seppage")) {
		sprintf(filename, "m_usage_%04d%02d_%.3s_%s.%s", 
			year, month, 
			sub ? sub : "",
			report ? report : "", 
			conf->pages_suffix );
	} else {
		sprintf(filename, "m_usage_%04d%02d_%.3s.html%s%s", 
			year, month, sub, report ? "#" : "",
			report ? report : "");	
	}
		
	return filename;
}

int write_menu_page(mconfig *ext_conf, mstate *state,FILE *f, int type, char *sub, char *report) {
	int html_indent;
	for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
	fprintf(f, "<tr><td class=\"menu\">&nbsp;&nbsp;<a href=\"%s\">[%s]</a></td></tr>\n", 
		get_url(ext_conf, state->year, state->month, sub, report), get_menu_item(type));
	
	return 0;
}

int write_menu_report(mconfig *ext_conf, mstate *state,FILE *f, int type, char *sub, char *report, int highlight) {
	int html_indent;
	for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
	fprintf(f, "<tr><td class=\"%s\">&nbsp;&nbsp;&nbsp;&nbsp;<a href=\"%s\">[%s]</a></td></tr>\n", 
		highlight ? "menu_highlight" : "menu", 
		get_url(ext_conf, state->year, state->month, sub, report), 
		get_menu_item(type));
	
	return 0;
}


int write_menu (mconfig *ext_conf, mstate *state,FILE *f, mlist *l, char *sub, int cur_item) {
	config_output *conf = ext_conf->plugin_conf;
	int html_indent;
	
	for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
	html_indent_depth++;
	fprintf(f, "<table width=\"150\">\n");
	while (l) {
		mdata *data = l->data;
		char *sep_main, *sep_sub, *sep_report;
		
	/* seperate menu string */
		sep_main = strchr(data->key, '/');
		sep_sub = strchr(sep_main+1, '/');
		
		sep_main++;
		if (!sep_sub) {
			for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
			/* Added by Georges 'Melkor' Goncalves <melkor@lords.com>
			** courtesy of Jan Kneschke
			*/
			/* suffix for the generated statistics page files */
			fprintf(f, "<tr><td class=\"menu\"><a href=\"index.%s\">[%s]</a></td></tr>\n",
			  conf->pages_suffix,
				get_menu_item(data->data.count.grouped));
		} else {
			sep_sub++;
			sep_report = strchr(sep_sub, '/');
			if (!sep_report) {
				if (conf->page_style && (!strcasecmp(conf->page_style, "seppage") ||
				!strcasecmp(conf->page_style, "onepage"))
				) {
					if (l->next) {
						mdata *_data = l->next->data;
						char *_sep_main, *_sep_sub, *_sep_report;
		
				/* seperate menu string */
						_sep_main = strchr(_data->key, '/');
						_sep_sub = strchr(_sep_main+1, '/');
			
						_sep_main++;
						if (_sep_sub) {
							_sep_sub++;
							_sep_report = strchr(_sep_sub, '/');
							if (_sep_report) {
								_sep_report++;
								write_menu_page(ext_conf, state, f, data->data.count.grouped, sep_sub, _sep_report);
							}
						}
					}
				} else {
					write_menu_page(ext_conf, state, f, data->data.count.grouped, sep_sub, NULL);
				}
			} else {
				sep_report++;
				
				if (conf->page_style && !strcasecmp(conf->page_style, "onepage")) {
					write_menu_report(ext_conf, state, f, data->data.count.grouped, sep_sub, sep_report, cur_item == data->data.count.grouped);
				} else if (!strncmp(sub, sep_sub,3))
					write_menu_report(ext_conf, state, f, data->data.count.grouped, sep_sub, sep_report, cur_item == data->data.count.grouped);
			}
		}
		l = l->next;
	}
	
	html_indent_depth--;
	for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
	fprintf(f, "</table>\n");
	
	return 0;
}

int write_css(mconfig *ext_conf, const char *subpath) {
	char filename[255];
	int f;
	FILE *f2;
	char buf[512+1];
	int len;
	config_output *conf = ext_conf->plugin_conf;
	
	/* build destination filename */
	sprintf(filename, "%s%s%s/modlogan.css", 
		ext_conf->outputdir ? ext_conf->outputdir : ".",
		subpath ? "/" : "",
		subpath ? subpath : "");

	/* open the css file, search includepath as well */
	if ((f2 = mfopen (ext_conf, conf->cssfile, "r")) == NULL) {
		fprintf(stderr, "can't open %s: %s\n", conf->cssfile, strerror(errno));
		return -1;
	}

	if ((f = open (filename, O_CREAT | O_EXCL | O_WRONLY, S_IROTH | S_IRGRP | S_IWUSR | S_IRUSR)) != -1) {
		if (ext_conf->debug_level > 2)
			fprintf(stderr, "writing CSS-definition for %s\n", subpath ? subpath : "(null)");
		while( (len = read(fileno(f2), buf, sizeof(buf)-1 )) != -1 ) {
			if (len<=0) break;
			write(f,buf,len);
		}
		close(f);
	} else if (errno != EEXIST) {
		fprintf(stderr, "writing CSS-definition for %s failed: %s\n", subpath ? subpath : "(null)", strerror(errno));
	} else {
		if (ext_conf->debug_level > 2)
			fprintf(stderr, "writing CSS-definition for %s skipped: %s\n", subpath ? subpath : "(null)", strerror(errno));
	}
	/* Close the css source file */
	fclose(f2);
	
	return 0;
}

int write_report_header(mconfig *ext_conf, FILE *f, char *sub, char *report) {
	config_output *conf = ext_conf->plugin_conf;
	int html_indent;
	
	if (f == NULL) return -1;
	
	for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
	if (conf->page_style && !strcasecmp(conf->page_style, "onepage")) {
		fprintf(f, "<center><a name=\"%.3s%s\"></a><a href=\"#000\">[top]</a></center>", sub, report);
	} else if (conf->page_style && !strcasecmp(conf->page_style, "seppage")) {
		fprintf(f, "<center><a name=\"%s\"></a></center>", report);
	} else {
		fprintf(f, "<center><a name=\"%s\"></a><a href=\"#000\">[top]</a></center>", report);
	}
	
	return 0;
}

/**
 * data container for the coloums of the tables
 * 
 */

typedef struct {
	const char *name;  /*< visible name of the coloumn */
	const char *class; /*< CSS class, may be NULL */
} fields_def;

/**
 * data container for generic reports 
 * 
 */

typedef struct {
	mreport_type type; /*< type of report */
	mhash *data;       /*< data source (staweb->...) */
	int max;           /*< maximum fields to show */
	const char *title; /*< visible title of the table */
	int options;       /*< options for show_mhash */
	int show_graph;    /*< for the graph ? (conf->show...) */
	char * (*draw_graph)(mconfig * ext_conf, mstate * state, const char *subpath);
	                   /*< graph function (may be NULL) */
	fields_def fields[5];
	                   /*< coloumns */
} reports_def;

int mplugins_output_modlogan_generate_monthly_output(mconfig * ext_conf, mstate * state, const char *subpath) {
	unsigned int i, s_200, s_304;
	unsigned int min, sec;
	double d = 0;
	FILE *f = NULL;
	char filename[255];
	data_WebHistory sumdat, maxdat;
	int last_day = 1, html_indent;
	config_output *conf = ext_conf->plugin_conf;
	mlist *menu_items = mlist_init(), *l;
	char last_sub[4] = { 0, 0, 0, 0 };
	char last_report[4] = { 0, 0, 0, 0 };
	mstate_web *staweb = NULL;
	
	/* each 'generic' report is defined in this structure 
	 * 
	 * NOTE: the data field has to be NULL in the declaration as staweb isn't 
	 *       checked yet. The real assignment is done later
	 */

	reports_def reports[] = {
		/* 0 */
		  /* id             data  max. entry to show  title */
		{ M_REPORT_REQ_URL, NULL, conf->max_req_urls, _("Requested URL's"),
			/* options */
			HIGHLIGHT | GROUPING | INDEX | PERCENT | VISITS | VISITS_TRAFFIC,
				0, /* show graph */
				NULL, /* graph function */
			{ /* fields */
				{ _("Hits"), "hits" },
				{ _("Traffic"), "traffic" },
				{ _("URL"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 1 */
		{ M_REPORT_REF_URL, NULL, conf->max_ref_urls, _("Referring URL's"),
			HIGHLIGHT | GROUPING | INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Referrer"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 2 */
		{ M_REPORT_VIEW_DURATION, NULL, conf->max_req_urls, _("View Durations"),
			HIGHLIGHT | GROUPING | INDEX | PERCENT | VIEW_DURATION | TIME,
				0,
				NULL,
			{
				{ _("Duration"), "duration" },
				{ _("Hits"), "hits" },
				{ _("Average"), "duration" },
				{ _("URL"), NULL },
				{ NULL, NULL }
			}
		},
		/* 3 */
		{ M_REPORT_OS, NULL, conf->max_os, _("Used Operating Systems"),
			GROUPING | VISITS | INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Visits"), "visits" },
				{ _("Operating System"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
				
			}
		},
		/* 4 */
		{ M_REPORT_HOSTS, NULL, conf->max_hosts, _("Hosts"),
			GROUPING | INDEX | VISITS | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Visits"), "visits" },
				{ _("Host"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 5 */
		{ M_REPORT_ENTRY_PAGES, NULL, conf->max_entry_pages, _("Entry Pages"),
			HIGHLIGHT | INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Entry Page"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 6 */
		{ M_REPORT_EXIT_PAGES, NULL, conf->max_exit_pages, _("Exit Pages"),
			HIGHLIGHT | INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Exit Page"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 7 */
		{ M_REPORT_INDEXED, NULL, conf->max_indexed_pages, _("Indexed Pages"),
			HIGHLIGHT | INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Indexed Page"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 8 */
		{ M_REPORT_USERAGENT, NULL, conf->max_ua, ("Used Browsers"),
			GROUPING | VISITS | INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Visits"), "visits" },
				{ _("Browser"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 9 */
		{ M_REPORT_REQ_PROT, NULL, conf->max_req_prot, _("Used Request Protocol"),
			INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Protocol"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 10 */
		{ M_REPORT_REQ_METH, NULL, conf->max_req_meth, _("Used Request Method"),
			INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Method"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 11 */
		{ M_REPORT_ROBOTS, NULL, conf->max_robots, _("Robots"),
			INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Robot"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 12 */
		{ M_REPORT_BOOKMARKS, NULL, conf->max_bookmarks, _("Bookmarked Pages"),
			HIGHLIGHT | INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Bookmarked Page"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 13 */
		{ M_REPORT_BROKEN_LINKS, NULL, conf->max_broken_links, _("Missing File / Broken Link"),
			GROUPING | INDEX | BROKEN_LINK | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Broken Link"), NULL },
				{ _("last referrering URL"), NULL },
				{ _("Last Hit"), NULL },
				{ NULL, NULL }
			}
		},
		/* 14 */
		{ M_REPORT_INTERNAL_ERROR, NULL, conf->max_internal_errors, _("Internal Errors"),
			INDEX | BROKEN_LINK,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Broken Link"), NULL },
				{ _("last referrering URL"), NULL },
				{ _("Last Hit"), NULL },
				{ NULL, NULL }
			}
		},
		/* 15 */
		{ M_REPORT_SEARCH_STRINGS, NULL, conf->max_search_strings, _("SearchStrings"),
			INDEX | PERCENT | GROUPING,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Search String"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 16 */
		{ M_REPORT_SEARCH_ENGINE, NULL, conf->max_search_engines, _("SearchEngines"),
			HIGHLIGHT | GROUPING | INDEX | PERCENT,
				0,
				NULL,
				
			{
				{ _("Hits"), "hits" },
				{ _("Search Engine"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 17 */
		{ M_REPORT_EXTENSION, NULL, conf->max_extensions, _("Extensions"),
			HIGHLIGHT | GROUPING |INDEX | PERCENT | VISITS | VISITS_TRAFFIC,
				1,
				create_pic_ext,
			{
				{ _("Hits"), "hits" },
				{ _("Traffic"), "traffic" },
				{ _("Extenstions"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 18 */
		{ M_REPORT_VISIT_DURATION, NULL, conf->max_visit_durations, _("Time Per Visit"),
			INDEX | PERCENT | SORT_BY_KEY,
				1,
				create_pic_vd,
			{
				{ _("Visits"), "visits" },
				{ _("Time per Visit"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 19 */
		{ M_REPORT_VISIT_PATH_LENGTH, NULL, conf->max_visit_path_lengths, _("Visit Path Length"),
			INDEX | PERCENT | SORT_BY_KEY,
				1,
				create_pic_vpl,
			
			{
				{ _("Visits"), "visits" },
				{ _("Pages per Visit"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 20 */
		{ M_REPORT_COUNTRIES, NULL, conf->max_countries, _("Countries"),
			VISITS | INDEX | PERCENT | RESOLVE_TLD,
				conf->show_country_graph,
				create_pic_countries,
			{
				{ _("Hits"), "hits" },
				{ _("Visits"), "visits" },
				{ _("Country"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 21 */
		{ M_REPORT_VHOSTS, NULL, conf->max_vhosts, _("Vhosts"),
			GROUPING | INDEX | VISITS | PERCENT,
				conf->show_vhost_graph,
				create_pic_vhost,
			{
				{ _("Hits"), "hits" },
				{ _("Visits"), "visits" },
				{ _("Vhost"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},

		{0, NULL, 0, NULL, 0, 0, NULL,
			{
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		}
	};

	if (state == NULL)
		return -1;
	
	if (state->ext == NULL)
		return -1;
	
	if (state->ext_type != M_STATE_TYPE_WEB)
		return -1;
	
	staweb = state->ext;
	
	/*
	 * set the data 
	 */
	reports[0].data = staweb->req_url_hash;
	reports[1].data = staweb->ref_url_hash;
	reports[2].data = staweb->views;
	reports[3].data = staweb->os_hash;
	reports[4].data = staweb->host_hash;
	reports[5].data = get_entry_pages(staweb->visits); /* free me */
	reports[6].data = get_exit_pages(staweb->visits); /* free me */
	reports[7].data = staweb->indexed_pages;
	reports[8].data = staweb->ua_hash;
	reports[9].data = staweb->req_prot_hash;
	reports[10].data = staweb->req_meth_hash;
	reports[11].data = staweb->robots;
	reports[12].data = staweb->bookmarks;
	reports[13].data = staweb->status_missing_file;
	reports[14].data = staweb->status_internal_error;
	reports[15].data = staweb->searchstring;
	reports[16].data = staweb->searchsite;
	reports[17].data = staweb->extension; 
	reports[18].data = get_visit_duration(staweb->visits); /* free me */
	reports[19].data = get_visit_path_length(staweb->visits); /* free me */
	reports[20].data = staweb->country_hash;
	reports[21].data = staweb->vhost_hash;
/*	reports[22].data = ;
	reports[23].data = ;
	reports[24].data = ;
	reports[25].data = ;
	reports[26].data = ;*/
	
	if (subpath) {
		sprintf(filename, "%s/%s/",
			ext_conf->outputdir ? ext_conf->outputdir : ".",
			subpath);
		mkdir(filename, 0755);
	}
	
	if (write_css(ext_conf, subpath)) {
		return -1;
	}
	
	get_menu_items(ext_conf, state, menu_items);
	
	l = menu_items;
	while (l) {
/* sort the list */
		mlist *min_l = NULL;
		mlist *hl = l->next;
		char *min = "";
		char *last = l->data->key;
		
		while (hl) {
			if (hl->data) {
				if ( strcmp(hl->data->key, min) > 0 &&
					strcmp(hl->data->key, last) < 0) {
					min = hl->data->key;
					min_l = hl;
				}
			}
			hl = hl->next;
		}
		
		if (min_l) {
			mdata *d;
			
			d = l->data;
			l->data = min_l->data;
			min_l->data = d;
		}
		
		l = l->next;
	}
/* write the menu */	
	l = menu_items;
	while (l) {
		mdata *data = l->data;
		char *sep_main, *sep_sub, *sep_report = NULL;

	/* seperate menu string */
		sep_main = strchr(data->key, '/');
		sep_main++;
		
		sep_sub = strchr(sep_main, '/');
		if (sep_sub) {
			sep_sub++;
			sep_report = strchr(sep_sub, '/');
			if (sep_report) {
				sep_report++;
			}
		}
	/* open file */
		
		if (conf->page_style && !strcasecmp(conf->page_style, "onepage")) {
			if (sep_main && strncmp(last_sub, sep_main, 3)) {
				strncpy(last_sub, sep_main, 3);
			
				/* Added by Georges 'Melkor' Goncalves <melkor@lords.com>
				** courtesy of Jan Kneschke
				*/
				/* suffix for the generated statistics page files */
				sprintf(filename, "%s%s%s/m_usage_%04i%02i.%s", 
					ext_conf->outputdir ? ext_conf->outputdir : ".",
					subpath ? "/" : "",
					subpath ? subpath : "",
					state->year, state->month, conf->pages_suffix );
			
				if (f) {
					file_end(f, ext_conf);
					fclose(f);
				}
			
				if (!(f = fopen(filename, "w"))) {
					return -1;			}
	/* create menu */
				file_start(f,ext_conf,state->timestamp);
			
				write_menu(ext_conf, state,f, menu_items, last_sub, data->data.count.grouped);
				
				html_indent_depth--;
				for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
				fprintf(f, "</td>\n");
				
				for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
				html_indent_depth++;
				fprintf(f, "<td class=\"skeleton\">\n");
			} 
		} else if (conf->page_style && !strcasecmp(conf->page_style, "seppage")) {
			if (sep_sub && sep_report &&
				(strncmp(last_sub, sep_sub, 3) != 0 ||
				strncmp(last_report, sep_report, 3) != 0)) {
				
				strncpy(last_report, sep_report, 3);
				strncpy(last_sub, sep_sub, 3);
				
				/* Added by Georges 'Melkor' Goncalves <melkor@lords.com>
				** courtesy of Jan Kneschke
				*/
				/* suffix for the generated statistics page files */
				sprintf(filename, "%s%s%s/m_usage_%04i%02i_%.3s_%s.%s", 
					ext_conf->outputdir ? ext_conf->outputdir : ".",
					subpath ? "/" : "",
					subpath ? subpath : "",
					state->year, state->month, 
					sep_sub ? sep_sub : "",
					sep_report ? sep_report : "",
					conf->pages_suffix);

				if (f) {
					file_end(f,ext_conf);
					fclose(f);
				}

				if (!(f = fopen(filename, "w"))) {
					fprintf(stderr, "%s.%d: %s %s\n", __FILE__, __LINE__, _("Can't open file"), filename);
					return -1;			
				}
	/* create menu */
				file_start(f,ext_conf, state->timestamp);
			
				write_menu(ext_conf, state,f, menu_items, last_sub, data->data.count.grouped);
			
				html_indent_depth--;
				for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
				fprintf(f, "</td>\n");
				
				for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
				html_indent_depth++;
				fprintf(f, "<td class=\"skeleton\">\n");
			}
		} else {
			if (sep_sub && strncmp(last_sub, sep_sub, 3)) {
				strncpy(last_sub, sep_sub, 3);
			
				/* Added by Georges 'Melkor' Goncalves <melkor@lords.com>
				** courtesy of Jan Kneschke
				*/
				/* suffix for the generated statistics page files */
				sprintf(filename, "%s%s%s/m_usage_%04i%02i_%s.%s", 
					ext_conf->outputdir ? ext_conf->outputdir : ".",
					subpath ? "/" : "",
					subpath ? subpath : "",
					state->year, state->month, last_sub, conf->pages_suffix);
			
				if (f) {
					file_end(f,ext_conf);
					fclose(f);
				}
			
				if (!(f = fopen(filename, "w"))) {
					return -1;			}
	/* create menu */
				file_start(f,ext_conf, state->timestamp);
			
				write_menu(ext_conf, state,f, menu_items, last_sub, data->data.count.grouped);
			
				html_indent_depth--;
				for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
				fprintf(f, "</td>\n");
				
				for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
				html_indent_depth++;
				fprintf(f, "<td class=\"skeleton\">\n");
			}
		}

		if (f == NULL) {
#if 0
			fprintf(stderr, "%s.%d: no filedescriptor opened !!!\n", __FILE__, __LINE__);
#endif
			l = l->next;
			continue;
		}
		
	/* write report */	
		
		/* data.count.grouped is used in a differend fashion here */
		switch (data->data.count.grouped) {
		case M_REPORT_VISIT_PATH:
			{
				write_report_header(ext_conf, f, sep_sub, sep_report);
				
				table_start(f,
					    table_header(conf->max_visit_paths,
							 mhash_count(staweb->visits),
							 _("Visit Path")), 
					    4);
				
				for (html_indent = 0; html_indent < html_indent_depth; html_indent++)
					fprintf(f, " ");
				
				fprintf(f,
					"<tr><th>#</th><th class=\"%s\">%s</th><th>%%</th><th width=\"200\">%s</th></tr>\n",
					"visits", _("Visits"), 
					_("Visit Path")
					);
				
				show_visit_path(ext_conf, f, staweb->visits,
						conf->max_visit_paths,
						HIGHLIGHT | GROUPING | INDEX | PERCENT);
				
				if (conf->max_visit_paths > BOTTOM_THRESHOLD) {
					for (html_indent = 0; html_indent < html_indent_depth;
					     html_indent++)
						fprintf(f, " ");
					fprintf(f,
						"<tr><th>#</th><th class=\"%s\">%s</th><th>%%</th><th width=\"200\">%s</th></tr>\n",
						"visits", _("Visits"), 
						_("Visit Path")
						);
				}
				table_end(f);
				break;
			}
		case M_REPORT_STATUS_CODES:
			{
				char *ref;
				
				ref = create_pic_status(ext_conf, state, subpath);
				
				if (ref && strlen(ref)) {
					fprintf(f, "%s", ref);
				}
				
				write_report_header(ext_conf, f, sep_sub, sep_report);
				table_start(f, table_header(conf->max_status_codes, mhash_count(staweb->status_hash),_("Status Code")), 2);
				for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
				fprintf(f,"<tr><th class=\"%s\">%s</th><th width=\"200\">%s</th></tr>\n",
					"hits",	_("Hits"),
					_("Status Code")
					);
				show_status_mhash(f,staweb->status_hash, conf->max_status_codes);
				if (conf->max_status_codes > BOTTOM_THRESHOLD) {
					for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
					fprintf(f,"<tr><th class=\"%s\">%s</th><th width=\"200\">%s</th></tr>\n",
						"hits",	_("Hits"),
						_("Status Code")
						);
				}
				table_end(f);
				break;
			}
		case M_REPORT_SUMMARY:
			{
				
				write_report_header(ext_conf, f, sep_sub, sep_report);
				sumdat.files	= maxdat.files		= 0;
				sumdat.xfersize	= maxdat.xfersize	= 0;
				sumdat.hits	= maxdat.hits		= 0;
				sumdat.hosts	= maxdat.hosts		= 0;
				sumdat.pages	= maxdat.pages		= 0;
				sumdat.visits	= maxdat.visits		= 0;
	
				/* count the values */
				for ( i = 0; i < 31; i++) {
					if (staweb->days[i].hits) last_day = i+1;
					sumdat.files	+= staweb->days[i].files;
					sumdat.xfersize	+= staweb->days[i].xfersize;
					sumdat.hits	+= staweb->days[i].hits;
					sumdat.hosts	+= staweb->days[i].hosts;
					sumdat.pages	+= staweb->days[i].pages;
					sumdat.visits	+= staweb->days[i].visits;

					if (maxdat.files < staweb->days[i].files)
						maxdat.files	= staweb->days[i].files;
					if (maxdat.hits < staweb->days[i].hits) 
						maxdat.hits	= staweb->days[i].hits;
					if (maxdat.hosts < staweb->days[i].hosts) 
						maxdat.hosts	= staweb->days[i].hosts;
					if (maxdat.pages < staweb->days[i].pages) 
						maxdat.pages	= staweb->days[i].pages;
					if (maxdat.visits < staweb->days[i].visits) 
						maxdat.visits	= staweb->days[i].visits;
					if (maxdat.xfersize < staweb->days[i].xfersize) 
						maxdat.xfersize	= staweb->days[i].xfersize;
				}
				
				
				
				maxdat.hosts = sumdat.hosts = mhash_count(staweb->host_hash);
				
				table_start(f, _("Summary"), 3);
	
				/* Totals */
				fprintf(f,"<tr><td>%s</td><td colspan=\"2\" align=\"right\">%ld</td></tr>\n", _("Total Hits"), sumdat.hits);
				fprintf(f,"<tr><td>%s</td><td colspan=\"2\" align=\"right\">%ld</td></tr>\n", _("Total Files"), sumdat.files);
				fprintf(f,"<tr><td>%s</td><td colspan=\"2\" align=\"right\">%ld</td></tr>\n", _("Total Pages"), sumdat.pages);
				fprintf(f,"<tr><td>%s</td><td colspan=\"2\" align=\"right\">%ld</td></tr>\n", _("Total Hosts"), sumdat.hosts);
				fprintf(f,"<tr><td>%s</td><td colspan=\"2\" align=\"right\">%ld</td></tr>\n", _("Total Visits"), sumdat.visits);
				fprintf(f,"<tr><td>%s</td><td colspan=\"2\" align=\"right\">%.0f</td></tr>\n", _("Transfered KBytes"), sumdat.xfersize / 1024);
				
				fprintf(f,"<tr><th>&nbsp;</th><th>%s</th><th>%s</th></tr>\n", _("avg"), _("max"));
				
				fprintf(f,"<tr><td>%s</td><td align=\"right\">%ld</td><td align=\"right\">%ld</td></tr>\n", _("Hits per Day"), sumdat.hits / last_day, maxdat.hits);
				fprintf(f,"<tr><td>%s</td><td align=\"right\">%ld</td><td align=\"right\">%ld</td></tr>\n", _("Files per Day"), sumdat.files / last_day, maxdat.files);
				fprintf(f,"<tr><td>%s</td><td align=\"right\">%ld</td><td align=\"right\">%ld</td></tr>\n", _("Pages per Day"), sumdat.pages / last_day, maxdat.pages);
				fprintf(f,"<tr><td>%s</td><td align=\"right\">%ld</td><td align=\"right\">%ld</td></tr>\n", _("Hosts per Day"), sumdat.hosts / last_day, maxdat.hosts);
				fprintf(f,"<tr><td>%s</td><td align=\"right\">%ld</td><td align=\"right\">%ld</td></tr>\n", _("Visits per Day"), sumdat.visits / last_day, maxdat.visits);
				fprintf(f,"<tr><td>%s</td><td align=\"right\">%.0f</td><td align=\"right\">%.0f</td></tr>\n", _("Transfered Kbytes per Day"), (sumdat.xfersize / 1024) / last_day, maxdat.xfersize / 1024);

				if (sumdat.visits) {
					double allvisitduration = get_visit_full_duration(staweb->visits);
					
					d = (allvisitduration / sumdat.visits);
					min = d / 60;
					sec = (int)floor(d) % 60;
				} else {
					min = 0;
					sec = 0;
				}
				fprintf(f,"<tr><td>%s</td><td align=\"right\">%d:%02d %s</td><td align=\"right\">%s</td></tr>\n", _("Time per visit"), min, sec, _("min"), "---");
	
				if (sumdat.visits) {
					double allvisitlength = get_visit_full_path_length(staweb->visits);
					d = (double)allvisitlength / sumdat.visits;
				} else {
					d = 0;
				}
				fprintf(f,"<tr><td>%s</td><td align=\"right\">%.2f</td><td align=\"right\">%s</td></tr>\n", _("Pages per visit"), d, "---");

				if (sumdat.pages && staweb->views) {
					double allviewduration = mhash_sumup(staweb->views);
					d = (double)allviewduration / sumdat.pages;
				} else {
					d = 0;
				}
				fprintf(f,"<tr><td>%s</td><td align=\"right\">%s</td><td align=\"right\">%s</td></tr>\n", _("Duration per page"), get_duration_string(d), "---");

				s_200 = mhash_get_value(staweb->status_hash, "200");
				s_304 = mhash_get_value(staweb->status_hash, "304");
	
				d = ((double)s_304/(s_200+s_304)) * 100;
				fprintf(f,"<tr><td>%s</td><td align=\"right\">%.2f%%</td><td align=\"right\">%s</td></tr>\n", _("Cache Hit ratio"), d, "---");
	
				table_end(f);
				break;
			}
		case M_REPORT_DAILY:
			{
				char *ref;

				write_report_header(ext_conf, f, sep_sub, sep_report);
				if (conf->show_daily_graph) {

					ref = create_pic_31_day(ext_conf, state, subpath);

					if (ref && strlen(ref)) {
						fprintf(f, "%s", ref);
					}
				}

				table_start(f, _("Daily Statistics"), 6);
				for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
				fprintf(f,"<tr><th>%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th></tr>\n",
					_("Day"),
					"hits",	 _("Hits"),
					"files", _("Files"),
					"pages", _("Pages"),
					"visits", _("Visits"),
					"traffic", _("KBytes")
					);

				for ( i = 0; i < last_day; i++) {
					fprintf(f,"<tr><td>%d</td><td align=\"right\">%ld</td><td align=\"right\">%ld</td><td align=\"right\">%ld</td><td align=\"right\">%ld</td><td align=\"right\">%.0f</td></tr>\n",
						i+1, 
						staweb->days[i].hits,
						staweb->days[i].files,
						staweb->days[i].pages,
						staweb->days[i].visits,
						staweb->days[i].xfersize / 1024
						);
				}

				if (last_day > BOTTOM_THRESHOLD) {
					for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
					fprintf(f,"<tr><th>%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th></tr>\n",
						_("Day"),
						"hits", _("Hits"),
						"files", _("Files"),
						"pages", _("Pages"),
						"visits", _("Visits"),
						"traffic", _("KBytes")
						);
				}
				table_end(f);

				break;
			}
		case M_REPORT_HOURLY:
			{
				char *ref;

				write_report_header(ext_conf, f, sep_sub, sep_report);
				if (conf->show_hourly_graph) {
					ref = create_pic_24_hour(ext_conf, state, subpath);
	
					if (ref && strlen(ref)) {
						fprintf(f, "%s", ref);
					}
				}
	
				table_start(f, _("Hourly Statistics"), 6);
				fprintf(f,"<tr><th>%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th></tr>\n",
					_("Hour"),
					"hits", _("Hits"),
					"files", _("Files"),
					"pages", _("Pages"),
					"visits", _("Visits"),
					"traffic", _("KBytes")
					);
				for ( i = 0; i < 24; i++) {
					fprintf(f,"<tr><td>%d</td><td align=\"right\">%ld</td><td align=\"right\">%ld</td><td align=\"right\">%ld</td><td align=\"right\">%ld</td><td align=\"right\">%.0f</td></tr>\n",
						i, 
						staweb->hours[i].hits,
						staweb->hours[i].files,
						staweb->hours[i].pages,
						staweb->hours[i].visits,
						staweb->hours[i].xfersize / 1024
						);
				}
				fprintf(f,"<tr><th>%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th><th class=\"%s\">%s</th></tr>\n",
					_("Hour"),
					"hits", _("Hits"),
					"files", _("Files"),
					"pages", _("Pages"),
					"visits", _("Visits"),
					"traffic", _("KBytes")
					);
				table_end(f);
				break;
			}
		default:
			{
				int i = 0;
				
				/* don't try the generate a HTML-page for the master-pages */
				if (data->data.count.grouped > 128)
					break;
				
				for (i = 0; reports[i].data != NULL; i++) {
					if (reports[i].type == data->data.count.grouped) {
						int w = 0, j;
						
						write_report_header(ext_conf, f, sep_sub, sep_report);
						
						if (reports[i].show_graph && reports[i].draw_graph) {
							char *ref = reports[i].draw_graph(ext_conf, state, subpath);
							
							if (ref && strlen(ref)) {
								fprintf(f, "%s", ref);
							}
						}
						
						/* calculate the number coloumns */
						for (j = 0; reports[i].fields[j].name != NULL; j++, w++);
						if (reports[i].options & INDEX) w++;
						if (reports[i].options & PERCENT) w++;
						
						/* write header */
						table_start(f, 
							    table_header(reports[i].max, mhash_count(reports[i].data), reports[i].title), 
							    w);
						
						for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
						
						fprintf(f, "<tr>");
						if (reports[i].options & INDEX) {
							fprintf(f, "<th>#</th>");
						}
						
						for (j = 0; reports[i].fields[j].name != NULL; j++) {
							if (reports[i].fields[j].class) {
								fprintf(f, "<th class=\"%s\">%s</th>",
									reports[i].fields[j].class,
									reports[i].fields[j].name);
							} else {
								fprintf(f, "<th>%s</th>",
									reports[i].fields[j].name);
							}
							
							if (j == 0 && reports[i].options & PERCENT) {
								fprintf(f, "<th>%%</th>");
							}
						}
						
						fprintf(f, "</tr>\n");
#if 0
						fprintf(stderr, "%s.%d: %d (%s - %d)\n", __FILE__, __LINE__, i, reports[i].title, reports[i].max);
#endif
						show_mhash(ext_conf, f, reports[i].data, reports[i].max, reports[i].options);
						
						if (reports[i].max > BOTTOM_THRESHOLD) {
							for (html_indent = 0; html_indent < html_indent_depth; html_indent++) fprintf(f, " ");
							
							fprintf(f, "<tr>");
							if (reports[i].options & INDEX) {
								fprintf(f, "<th>#</th>");
							}
							
							for (j = 0; reports[i].fields[j].name != NULL; j++) {
								if (reports[i].fields[j].class) {
									fprintf(f, "<th class=\"%s\">%s</th>",
										reports[i].fields[j].class,
										reports[i].fields[j].name);
								} else {
									fprintf(f, "<th>%s</th>",
										reports[i].fields[j].name);
								}
								
								if (j == 0 && reports[i].options & PERCENT) {
									fprintf(f, "<th>%%</th>");
								}
							}
							
							fprintf(f, "</tr>\n");
						}
						
						table_end(f);
						
						break;
					}
				}
				
				if (reports[i].data == NULL) {
					fprintf(stderr,
						"%s.%d: don't know how to generate a report for %d\n",
						__FILE__, __LINE__, 
						data->data.count.grouped);
					return -1;
				}
			}
			break;
		}
		
		l = l->next;
	}
	
	mhash_free(reports[5].data);
	mhash_free(reports[6].data);
	mhash_free(reports[18].data);
	mhash_free(reports[19].data);
	
	mlist_free(menu_items);
	
	if (f) {
		file_end(f,ext_conf);
	
		fclose (f);
	}
	
	return 0;
}

int mplugins_output_modlogan_generate_history_output(mconfig *ext_conf, mlist *history, const char *subpath) {
	mlist *l = history;
	FILE *f;
	char filename[255];
	config_output *conf = ext_conf->plugin_conf;
	data_WebHistory hist, yearly;
	
	if (history == NULL) return -1;
	
	/* Added by Georges 'Melkor' Goncalves <melkor@lords.com>
	** courtesy of Jan Kneschke
	*/
	/* suffix for the generated statistics page files */
	sprintf(filename, "%s%s%s/index.%s", 
		ext_conf->outputdir ? ext_conf->outputdir : ".",
		subpath ? "/" : "",
		subpath ? subpath : "",
		conf->pages_suffix );
	
	if (!(f = fopen(filename, "w"))) {
		return -1;
	}
	
	file_start_index(f,ext_conf,0);
	
	if (conf->show_monthly_graph) {
		char * ref = create_pic_12_month(ext_conf, history, subpath);
	
		if (ref && *ref) {
			fprintf(f, "%s", ref);
		}
	}
	
	table_start(f, _("History"), -11);
	
	fprintf(f,"<tr><th>&nbsp;</th><th colspan=\"5\">%s</th><th colspan=\"5\">%s</th></tr>",
		_("Average/day"),
		_("Totals")
		);
	fprintf(f,"<tr><th>%s</th>"
		"<th class=\"%s\">%s</th><th class=\"%s\">%s</th>"
		"<th class=\"%s\">%s</th><th class=\"%s\">%s</th>"
		"<th class=\"%s\">%s</th>"
		"<th class=\"%s\">%s</th><th class=\"%s\">%s</th>"
		"<th class=\"%s\">%s</th><th class=\"%s\">%s</th>"
		"<th class=\"%s\">%s</th>"
		"</tr>\n",
		_("Month"),
		"hits", _("Hits"),
		"files", _("Files"),
		"pages", _("Pages"),
		"visits", _("Visits"),
		"traffic", _("KBytes"),
		"hits", _("Hits"),
		"files", _("Files"),
		"pages", _("Pages"),
		"visits", _("Visits"),
		"traffic", _("KBytes")
		);

	
#define HIST(x) \
	hist.x = 0;
	
	HIST(hits);
	HIST(files);
	HIST(pages);
	HIST(visits);
	HIST(xfersize);
	HIST(days_passed);
#undef HIST
	
#define HIST(x) \
	yearly.x = 0;
	
	HIST(year);
	HIST(hits);
	HIST(files);
	HIST(pages);
	HIST(visits);
	HIST(xfersize);
	HIST(days_passed);
#undef HIST
	
	/* go to the last element */
	while (l->next) l = l->next;
	
	while (l) {
		mdata *data = l->data;
		
		if (!data) break;
		
		if (data->data.webhist.days_passed != 0) {
			if (yearly.year > data->data.webhist.year) {
				fprintf(f,"<tr><td class=\"centerb\">%04d</td>" 
					"<td class=\"tinyr\">%ld</td><td class=\"tinyr\">%ld</td>" 
					"<td class=\"tinyr\">%ld</td><td class=\"tinyr\">%ld</td>" 
					"<td class=\"tinyr\">%.0f</td>"
					"<td class=\"tinyr\">%ld</td><td class=\"tinyr\">%ld</td>"
					"<td class=\"tinyr\">%ld</td><td class=\"tinyr\">%ld</td>"
					"<td class=\"tinyr\">%.0f</td>"
					"</tr>\n",
					yearly.year,
			
					yearly.hits / yearly.days_passed,
					yearly.files / yearly.days_passed,
					yearly.pages / yearly.days_passed,
					yearly.visits / yearly.days_passed,
					yearly.xfersize / 1024 / yearly.days_passed,
					yearly.hits,
					yearly.files,
					yearly.pages,
					yearly.visits,
					yearly.xfersize / 1024
					);
			}
			fprintf(f,"<tr><td class=\"centerb\"><a href=\"%s\">%s&nbsp;%04i</a></td>" 
				"<td class=\"tinyr\">%ld</td><td class=\"tinyr\">%ld</td>" 
				"<td class=\"tinyr\">%ld</td><td class=\"tinyr\">%ld</td>" 
				"<td class=\"tinyr\">%.0f</td>"
				"<td class=\"tinyr\">%ld</td><td class=\"tinyr\">%ld</td>"
				"<td class=\"tinyr\">%ld</td><td class=\"tinyr\">%ld</td>"
				"<td class=\"tinyr\">%.0f</td>"
				"</tr>\n",
				get_url(ext_conf, data->data.webhist.year, data->data.webhist.month, "000", "000"),
				get_month_string(data->data.webhist.month,1),
				data->data.webhist.year,
			
				data->data.webhist.hits / data->data.webhist.days_passed,
				data->data.webhist.files / data->data.webhist.days_passed,
				data->data.webhist.pages / data->data.webhist.days_passed,
				data->data.webhist.visits / data->data.webhist.days_passed,
				data->data.webhist.xfersize / 1024 / data->data.webhist.days_passed,
				data->data.webhist.hits,
				data->data.webhist.files,
				data->data.webhist.pages,
				data->data.webhist.visits,
				data->data.webhist.xfersize / 1024
				);

			if (yearly.year > data->data.webhist.year) {
				
#define HIST(x) \
	yearly.x = data->data.webhist.x;
				HIST(year);
				HIST(hits);
				HIST(files);
				HIST(pages);
				HIST(visits);
				HIST(xfersize);
				HIST(days_passed);
#undef HIST
	
			} else {
#define HIST(x) \
	yearly.x += data->data.webhist.x
				yearly.year = data->data.webhist.year;
				HIST(hits);
				HIST(files);
				HIST(pages);
				HIST(visits);
				HIST(xfersize);
				HIST(days_passed);
#undef HIST
			}
#define HIST(x) \
	hist.x += data->data.webhist.x
			
			HIST(hits);
			HIST(files);
			HIST(pages);
			HIST(visits);
			HIST(xfersize);
			HIST(days_passed);
#undef HIST
		} else {
			M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_OUTPUT, M_DEBUG_LEVEL_WARNINGS,
				 "count == 0, is this ok ?? splitby for '%s' without an entry ??\n",
				 data->key);
		}
		l = l->prev;
	}
	
	if (yearly.year && yearly.days_passed) {
		fprintf(f,"<tr><td class=\"centerb\">%04d</td>" 
			"<td class=\"tinyr\">%ld</td><td class=\"tinyr\">%ld</td>" 
			"<td class=\"tinyr\">%ld</td><td class=\"tinyr\">%ld</td>" 
			"<td class=\"tinyr\">%.0f</td>"
			"<td class=\"tinyr\">%ld</td><td class=\"tinyr\">%ld</td>"
			"<td class=\"tinyr\">%ld</td><td class=\"tinyr\">%ld</td>"
			"<td class=\"tinyr\">%.0f</td>"
			"</tr>\n",
			yearly.year,
			
			yearly.hits / yearly.days_passed,
			yearly.files / yearly.days_passed,
			yearly.pages / yearly.days_passed,
			yearly.visits / yearly.days_passed,
			yearly.xfersize / 1024 / yearly.days_passed,
			yearly.hits,
			yearly.files,
			yearly.pages,
			yearly.visits,
			yearly.xfersize / 1024
			);
	}
	
	if (hist.days_passed) {
		fprintf(f,"<tr><td class=\"centerb\">%s</td>"
			"<td class=\"tinyr\">%ld</td><td class=\"tinyr\">%ld</td>"
			"<td class=\"tinyr\">%ld</td><td class=\"tinyr\">%ld</td>"
			"<td class=\"tinyr\">%.0f</td>"
			"<td class=\"tinyr\">%ld</td><td class=\"tinyr\">%ld</td>"
			"<td class=\"tinyr\">%ld</td><td class=\"tinyr\">%ld</td>"
			"<td class=\"tinyr\">%.0f</td>"
			"</tr>\n",
			
			_("total"),
			hist.hits / hist.days_passed,
			hist.files / hist.days_passed,
			hist.pages / hist.days_passed,
			hist.visits / hist.days_passed,
			hist.xfersize / 1024 / hist.days_passed,
			hist.hits,
			hist.files,
			hist.pages,
			hist.visits,
			hist.xfersize / 1024
			);
	}
		
	table_end(f);
	
	file_end_index(f,ext_conf);
	
	fclose(f);
	
	return 0;
}
