///////////////////////////////////////////////////////////////////////////////
// $Id: maplayer.cpp,v 1.11 2004/10/03 19:30:45 krake Exp $
//
// Package:   MOAGG Edit - Level Editor for MOAGG
// Copyright: Kevin Krammer, 2003
//
///////////////////////////////////////////////////////////////////////////////
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.
//
///////////////////////////////////////////////////////////////////////////////


/*! \file    maplayer.cpp
    \author  Kevin Krammer, kevin.krammer@gmx.at
    \brief   Implementation of the map layer class
*/

// Qt includes
#include <qfile.h>
#include <qrect.h>
#include <qpixmap.h>
#include <qpoint.h>

// local includes
#include "maplayer.h"
#include "tilecache.h"
#include "tilegroup.h"

///////////////////////////////////////////////////////////////////////////////

const int MapLayer::MAX_VERSION = 2;

///////////////////////////////////////////////////////////////////////////////

MapLayer::MapLayer() :
	m_tileGroup(0),
	m_tileCache(0),
	m_groupIndex(0),
	m_currentTile(0),
	m_rows(1),
	m_cols(1),
	m_alpha(255),
	m_brightness(128),
	m_background(false),
	m_modified(false),
	m_visible(true)
{
	init(1,1);
}

///////////////////////////////////////////////////////////////////////////////

MapLayer::~MapLayer()
{
}

///////////////////////////////////////////////////////////////////////////////
	
int MapLayer::byteSize(int version, int rows, int cols)
{
	if (rows < 1 || cols < 1) return 0;
	
	int size = 0;
	switch (version)
	{
		case 1:
			size = rows * cols + 1; // area + tile index
			break;
		
		case 2:
			size = rows * cols + 4; // area + tile index + image parameters
			break;
			
		default:
			break;
	}
	
	return size;
}

///////////////////////////////////////////////////////////////////////////////

void MapLayer::setTileGroup(TileGroup* group)
{
	delete m_tileCache;
	m_tileCache = 0;
	
	m_tileGroup = group;
}

///////////////////////////////////////////////////////////////////////////////

void MapLayer::setImageParameters(int alpha, int brightness)
{
	if (m_tileGroup == 0) return;
	
	if (alpha < 0) alpha = 0;
	else if (alpha > 255) alpha = 255;
	
	if (brightness < 0) brightness = 0;
	else if (brightness > 255) brightness = 255;
	
	bool recreateCache = false;
	if (alpha != m_alpha)
	{
		m_alpha       = alpha;
		m_modified    = true;
		recreateCache = true;
	}
	if (brightness != m_brightness)
	{
		m_brightness  = brightness;
		m_modified    = true;
		recreateCache = true;
	}
	
	if (recreateCache)
	{
		delete m_tileCache;
		m_tileCache = m_tileGroup->createPixmapCache(m_alpha, m_brightness);
		
		for (uint i = 0; i < m_tileField.count(); ++i)
		{
			m_pixField[i] = m_tileCache->pixmap(m_tileField[i]);
		}
	}
}

///////////////////////////////////////////////////////////////////////////////

QPixmap* MapLayer::pixmap(int row, int col)
{
	if (!m_visible) return 0;
	
	int index = row * m_cols + col;

	if (index < 0 || static_cast<uint>(index) >= m_pixField.size()) return 0;

	return m_pixField[index];
}

///////////////////////////////////////////////////////////////////////////////

void MapLayer::drawTile(int row, int col)
{
	if (m_tileGroup == 0) return;
	if (m_tileCache == 0)
	{
		m_tileCache = m_tileGroup->createPixmapCache(m_alpha, m_brightness);
	}

	int index = row * m_cols + col;

	uchar tile = static_cast<uchar>(m_currentTile);

	QPixmap* pix = m_tileCache->pixmap(m_currentTile);

	m_tileField[index] = tile;
	m_pixField[index] = pix;

	m_modified = true;
}

///////////////////////////////////////////////////////////////////////////////

void MapLayer::init(int rows, int cols)
{
	if (rows < 1 || cols < 1) return;

	m_tileField.resize(rows*cols);
	m_pixField.resize(m_tileField.size());

	uchar tile = 0;

	QPixmap* pix = 0;
	if (m_tileCache != 0)
	{
		pix = m_tileCache->pixmap(0);
	}

	int index = 0;
	for (int r = 0; r < rows; ++r)
	{
		for (int c = 0; c < cols; ++c, ++index)
		{
			m_tileField[index] = tile;
			m_pixField[index] = pix;
		}
	}

	m_cols = cols;
	m_rows = rows;
}

///////////////////////////////////////////////////////////////////////////////

void MapLayer::resize(int rows, int cols)
{
	if (rows < 1 || cols < 1) return;

	QMemArray<uchar> tileField;
	QMemArray<QPixmap*> pixField;

	tileField.fill(0, rows*cols);
	pixField.fill(0, tileField.size());

	int index = 0;
	int minRows = QMIN(rows, m_rows);
	int minCols = QMIN(cols, m_cols);

	for (int r = 0; r < minRows; ++r)
	{
		for (int c = 0; c < minCols; ++c, ++index)
		{
			int i = r * m_cols + c; // calculate index for current dimensions
			tileField[index] = m_tileField[i];
			pixField[index] = m_pixField[i];
		}
		if (minCols < cols)
			index += cols - minCols;
	}

	m_tileField = tileField;
	m_pixField = pixField;
	m_rows = rows;
	m_cols = cols;

	m_modified = true;
}

///////////////////////////////////////////////////////////////////////////////

bool MapLayer::readFromDevice(QIODevice* device, int version)
{
	if (device == 0) return false;

	// if not current version, try to import
	if (version != 2) return importFromDevice(device, version);
	
	char header[3];
	if (device->readBlock(header, 3) != 3)
	{
		return false;
	}

	bool background = header[0] != 0;
	int alpha       = (header[1] & 0xFF);
	int brightness  = (header[2] & 0xFF);

	qDebug("Layer-Header: 0x%x, 0x%x, 0x%x",
	       0xFF & header[0], 0xFF & header[1], 0xFF & header[2]);
	qDebug("background=%s, alpha=%d, brightness=%d",
	       (background ? "true" : "false"), alpha, brightness);
	   
	m_alpha      = alpha;
	m_brightness = brightness;
	m_background = background;
	
	char* buffer = new char[m_cols];
	for (int r = 0; r < m_rows; ++r)
	{
		if (device->readBlock(buffer, m_cols) != m_cols)
		{
			delete[] buffer;
			return false;
		}

		for (int c = 0; c < m_cols; ++c)
		{
			setCurrentTile(0xFF & buffer[c]);
			drawTile(r, c);
		}
	}
	delete[] buffer;
	
	return true;
}

///////////////////////////////////////////////////////////////////////////////

bool MapLayer::writeToDevice(QIODevice* device) const
{
	if (device == 0) return false;

	char header[3];
	header[0] = (m_background ? 0x01 : 0x00);
	header[1] = static_cast<char>(m_alpha & 0xFF);
	header[2] = static_cast<char>(m_brightness & 0xFF);
	
	qDebug("background=%s, alpha=%d, brightness=%d",
	       (m_background ? "true" : "false"), m_alpha, m_brightness);
	qDebug("Layer-Header: 0x%x, 0x%x, 0x%x",
	       0xFF & header[0], 0xFF & header[1], 0xFF & header[2]);
	
	if (device->writeBlock(header, 3) != 3) return false;
	
	char* buffer = new char[m_cols];
	for (int r = 0; r < m_rows; ++r)
	{
		for (int c = 0; c < m_cols; ++c)
		{
			int index = r * m_cols + c;
			buffer[c] = m_tileField[index];
		}

		if (device->writeBlock(buffer, m_cols) != m_cols)
		{
			delete[] buffer;
			return false;
		}
	}
	delete[] buffer;

	return true;
}

///////////////////////////////////////////////////////////////////////////////

bool MapLayer::copyData(const QRect& area, const MapLayer& source,
                        const QPoint& destpoint)
{
	if (area.isNull()) return false;
	if (destpoint.x() < 0 || destpoint.y() < 0) return false;

	QRect sourceRect(0, 0, source.numCols(), source.numRows());
	if (!sourceRect.contains(area)) return false;

	QRect rect(0, 0, numCols(), numRows());
	QRect destRect = area;
	destRect.moveTopLeft(destpoint);
	if (!rect.contains(destRect)) return false;
	
	// if there is no tile cache yet, create on
	if (m_tileCache == 0)
	{
		// if the data source has the same tile group instance
		// simply copy the time cache
		if (m_tileGroup == source.m_tileGroup)
		{
			m_tileCache = new TileCache(*(source.m_tileCache));
		}
		else
		{
			m_tileCache = m_tileGroup->createPixmapCache(m_alpha, m_brightness);
		}
	}
	
	int srcX  = area.x();
	int destX = destRect.x();
	for (; srcX <= area.right(); ++srcX, ++destX)
	{
		int srcY  = area.y();
		int destY = destRect.y();
		for (; srcY <= area.bottom(); ++srcY, ++destY)
		{
			uint srcIndex = srcY * source.numCols() + srcX;
			uint destIndex = destY * numCols() + destX;

			if (source.m_tileField[srcIndex] >= m_tileGroup->count())
				srcIndex = 0;

			m_tileField[destIndex] = source.m_tileField[srcIndex];
			m_pixField[destIndex]  = m_tileCache->pixmap(m_tileField[destIndex]);
		}
	}

	return true;
}

bool MapLayer::importFromDevice(QIODevice* device, int version)
{
	// sanity checks
	if (device == 0 || version < 1) return false;
	
	if (version == 1)
	{
		m_alpha      = 255;
		m_brightness = 128;
		m_background = m_groupIndex == 0;
		
		char* buffer = new char[m_cols];
		for (int r = 0; r < m_rows; ++r)
		{
			if (device->readBlock(buffer, m_cols) != m_cols)
			{
				delete[] buffer;
				return false;
			}
	
			for (int c = 0; c < m_cols; ++c)
			{
				setCurrentTile(0xFF & buffer[c]);
				drawTile(r, c);
			}
		}
		delete[] buffer;
		return true;
	}
	
	return false;
}

// End of file

