#ifndef MATHTOOLS_H
#define MATHTOOLS_H

#include <cmath>

//----------------------------------------------------------------------------
/**
 * This namespace contains some common needed functions
 * related to mathematical and geometric operations.
 */
namespace MATH_TOOLS
{
    //------------------------------------------------------------------------
    class Vector
    {
      public:
        //--------------------------------------------------------------------
        Vector();
        Vector(double x, double y);
        ~Vector();

        //--------------------------------------------------------------------
        inline void set(double x, double y) { m_x = x; m_y = y; }
        inline void setX(double x) { m_x = x; }
        inline double getX() const { return m_x; }
        inline void setY(double y) { m_y = y; }
        inline double getY() const { return m_y; }

        //--------------------------------------------------------------------
        inline void flipX() { m_x = -m_x; }
        inline void flipY() { m_y = -m_y; }

        //--------------------------------------------------------------------
        double getAbsValue() const;
        double getAngle() const;

        //--------------------------------------------------------------------
        inline void add(double x, double y) { m_x += x; m_y += y; }
        inline void addX(double x) { m_x += x; }
        inline void addY(double y) { m_y += y; }
        inline void sub(double x, double y) { m_x -= x; m_y -= y; }
        inline void subX(double x) { m_x -= x; }
        inline void subY(double y) { m_y -= y; }

        //--------------------------------------------------------------------
        Vector &setExponential(double absValue, double angle);
        Vector &addExponential(double absValue, double angle);

        //--------------------------------------------------------------------
        Vector &operator+=(const Vector &other);
        Vector &operator-=(const Vector &other);

        Vector &operator*=(const double scale);
        Vector &operator/=(const double scale);

        //--------------------------------------------------------------------
        friend Vector operator+(const Vector &v1, const Vector &v2);
        friend Vector operator-(const Vector &v1, const Vector &v2);

      private:
        //--------------------------------------------------------------------
        double m_x;
        double m_y;
    };


    //------------------------------------------------------------------------
    /**
     * Returns the angle between (0,0)(x,y) and the y axis,
     * counted in clockwise order and inside the value range from [0..360).
     */
    double getAngle(double x, double y);

    /**
     * Returns the angle between (x1,y1)(x2,y2) and the y axis,
     * counted in clockwise order and inside the value range from [0..360).
     */
    inline double getAngle(double x1, double y1, double x2, double y2)
    {
        return getAngle(x2-x1, y2-y1);
    }

    inline int getAngle(int x, int y)
    {
        return (int)rint(getAngle(1.0*x, 1.0*y));
    }

    inline int getAngle(int x1, int y1, int x2, int y2)
    {
        return getAngle(x2-x1, y2-y1);
    }
}

#endif //MATHTOOLS_H
