#-----------------------------------------------------------------------------
#
#  Copyright (c) 2005, Enthought, Inc.
#  All rights reserved.
#
#-----------------------------------------------------------------------------

""" A Traits editor that uses a ResourceTree as its content control, thus
    giving the benefits of extensible resource types to the visualization.
"""

# Standard library imports.
import logging

# Enthought library imports
from enthought.envisage import get_application
from enthought.naming.api import Binding
from enthought.envisage.resource.ui import ResourceTree
from enthought.traits.api import Any, Instance
from enthought.traits.ui.wx.editor import Editor


# Setup a logger for this module.
logger=logging.getLogger(__name__)


class ResourceTreeEditor(Editor):
    """ A Traits editor that uses a ResourceTree as its content control, thus
        giving the benefits of extensible resource types to the visualization.

        The keyword arguments used to initialize our ResourceTree come from
        our factory's 'kwargs' trait.

        The resource manager used for the ResourceTree can be configured by
        specifying the ID string of the service object within Envisage as the
        factory's id, but this is only used if the keyword arguments don't
        explicitly provide a 'resource_manager' argument.

        We setup listeners on the ResourceTree for selection and node
        activation events and pass these on to either the handlers registered
        with the factory or, if there are no factory handlers, to associated
        traits on the object being edited.  The associated traits should have
        these names:
            * <name>_selection
            * <name>_activated
    """

    ##########################################################################
    # Traits
    ##########################################################################

    ### protected 'ResourceTreeEditor' interface #############################

    # The resource manager we're using
    _resource_manager = Any

    # Our ResourceTree control
    _tree = Instance(ResourceTree)


    ##########################################################################
    # 'Editor' interface
    ##########################################################################

    ### public interface #####################################################

    def init(self, parent):
        """ Finishes initializing the editor by creating the underlying toolkit
            widget.
        """
        # Ensure that we only have a single reference to a resource manager
        kwargs = self.factory.kwargs.copy()
        if 'resource_manager' in kwargs:
            self._resource_manager = self.factory.kwargs['resource_manager']
        else:
            self._resource_manager = get_application().get_service(self.factory.id)
            if self._resource_manager is None:
                raise "Unable to find resource manager using id: '%s'" \
                    % self.factory.id
            kwargs['resource_manager'] = self._resource_manager

        # Build a resource tree and listen to standard events
        root = self._generate_binding(self.value)
        self._tree = ResourceTree(parent,
            _name            = 'Traits Resource Tree Editor',
            root             = root,
            **kwargs
            )
        self._tree.on_trait_change(self._on_select, 'selection')
        self._tree.on_trait_change(self._on_double_click, 'node_activated')

        # Our control is actually the wx widget that is our tree
        self.control = self._tree.control


    def update_editor(self):
        """ Updates the editor when the object trait changes external to the
            editor.
        """
        self._tree.root = self._generate_binding(self.value)


    ##########################################################################
    # 'ResourceTreeEditor' interface
    ##########################################################################

    ### protected interface ##################################################

    def _generate_binding(self, object):
        """ Called to generate a binding for the specified object.

            If the object can't be mapped to a resource type, then the object
            is bound to the name 'Root'.  Otherwise, it's name is retrieved
            from the resource type's method to determine the name.
        """
        # If no object value was passed, the binding is None
        if object is None:
            return None

        # Otherwise, create a binding for the object
        name = 'Root'
        resource_type = self._resource_manager.get_type_of(object)
        if resource_type is not None:
            name = resource_type.get_name(object)
        return Binding(name=name, obj=object)


    def _on_select(self, obj, trait_name, old, new):
        """ Called when the user changes the selection in the tree control.
        """
        # Build a list of selected node objects
        objects = [node.obj for node in new]

        # If the user provided a selection handler to the factory, then call
        # that and we're done.
        if self.factory.on_select is not None:
            self.ui.evaluate(self.factory.on_select, objects)
            return

        # Otherwise, if the user provided a selection trait associated with
        # the trait we're editing, then update that trait to our selection.
        trait_name = self.name + "_selection"
        if hasattr(self.object, trait_name):
            logger.debug('Passing selection [%s] to [%s]', new, self.object)
            setattr(self.object, trait_name, objects)

        # Otherwise, warn that no one is receiving selection notices.
        else:
            logger.debug("Unable to pass selection to [%s]", self.object)


    def _on_double_click(self, obj, trait_name, old, new):
        """ Called when the user double-clicks a node in the tree control.
        """
        # If the user provided a double-click handler to the factory, then
        # call that and we're done.
        if self.factory.on_dclick is not None:
            self.ui.evaluate(self.factory.on_dclick, new.obj)
            return

        # Otherwise, if the user provided a node_activated trait associated
        # with the trait we're editing, then update that trait to our
        # activated object.
        trait_name = self.name + "_activated"
        if hasattr(self.object, trait_name):
            setattr(self.object, trait_name, new.obj)

        # Otherwise, warn that no one is receiving activation notices.
        else:
            logger.debug("Unable to pass node_activated to [%s]", self.object)


#### EOF #####################################################################
