""" The resource manager looks after all resource types etc. """


# Enthought library imports.
from enthought.pyface.action.api import Group, MenuManager
from enthought.traits.api import Dict, HasTraits, Instance, List, Property
from enthought.type_manager import TypeManager


# Class name used in trait definitions.
RESOURCE_TYPE = 'enthought.envisage.resource.ResourceType'


class ResourceManager(HasTraits):
    """ The resource manager looks after all resource types etc. """
    
    #### 'ResourceManager' interface ##########################################

    # The context menu 'template'.  This menu manager should contain ONLY
    # actions that are available on ALL resources regardless of type (note
    # however that just because an action is available doesn't mean it has
    # to be enabled).
    context_menu = Instance(MenuManager, ())

    # The resource type that is associated with a resource when no matching
    # resource type can be found.
    default_resource_type = Instance(RESOURCE_TYPE)

    # The parent resource manager.
    #
    # By default this is None, but you can use it to set up a hierarchy of
    # resource managers. If a resource manager fails to find the type of an
    # object then it will give its parent a chance to do so.
    parent = Instance('ResourceManager')

    # All of the resource types known to the manager.
    #
    # fixme: All hell breaks loose (i.e., a system error - null passed to
    # internal routine) if we try to attach the type information to the
    # property here 8^() I thought that this was meant to work now! See Dave!
    resource_types = Property # (List(Instance(RESOURCE_TYPE)))

    # The type manager in effect for the resource manager.
    type_manager = Instance(TypeManager, ())
    
    #### Private interface ####################################################

    # Shadow trait for the 'resource_types' property.
    _resource_types = List(Instance(RESOURCE_TYPE))

    # A resource is considered 'dirty' when it has been modified but not
    # saved. When a resource becomes dirty it is the responsibility of
    # whoever detected the change to register a a dirty resource handler
    # with the resource manager.  If and when the resource is considered clean
    # the handler should be removed.
    _dirty_resources = Dict # { id(obj) : dirty_resource_handler }
    
    ###########################################################################
    # 'ResourceManager' interface.
    ###########################################################################

    #### Defaults #############################################################

    # context_menu
    def _context_menu_default(self):
        """ Returns the context menu 'template'. """

        menu = MenuManager(
            Group(
                id = 'system_top'
            ),

            Group(
                id = 'edit_group'
            ),
            
            Group(
                id = 'system_bottom'
            )
        )

        return menu

    # default_resource_type
    def _default_resource_type_default(self):
        """ Initializes the default resource type. """

        # fixme: Circular import.
        from object_resource_type import ObjectResourceType
        
        return ObjectResourceType(resource_manager=self)
    
    #### Properties ###########################################################

    # resource_types
    def _get_resource_types(self):
        """ Returns the manager's resource types. """

        return self._resource_types[:]

    def _set_resource_types(self, resource_types):
        """ Sets the manager's resource types. """

        for resource_type in self._resource_types:
            self.remove_resource_type(resource_type)
        
        for resource_type in resource_types:
            self.add_resource_type(resource_type)

        return
   
    #### Methods ##############################################################

    def add_resource_type(self, resource_type):
        """ Adds a new resource type to the manager. """

        self._resource_types.append(resource_type)

        # A resource type can be managed by exactly ONE resource manager!
        resource_type.resource_manager = self

        # If the resource type has a context adapter factory then register it
        # with the type manager.
        factory = resource_type.context_adapter_factory
        if factory is not None:
            self.type_manager.register_type_adapters(
                factory, factory.adaptee_class
            )
                
        return

    def remove_resource_type(self, resource_type):
        """ Removes a resource type from the manager. """

        resource_type.resource_manager = None
        self._resource_types.remove(resource_type)

        # If the resource type has a context adapter factory then unregister it
        # from the type manager.
        factory = resource_type.context_adapter_factory
        if factory is not None:
            self.type_manager.unregister_type_adapters(factory)
        
        return

    def lookup(self, id):
        """ Returns the resource type with the specified Id.

        Returns None if no such resource type is found.

        """

        for resource_type in self.resource_types:
            if resource_type.id == id:
                break

        else:
            resource_type = None

        return resource_type

    def get_type_of(self, obj):
        """ Returns the resource type of an object.

        Returns None if none of the manager's resource types 'recognize' the
        object.

        """

        # Attempt to find a resource type that 'recognizes' the object.
        #
        # fixme: We currently take the first resource type that recognizes the
        # resource.  This obviously means that ordering of types is important,
        # but we don't have an interface for controlling the order.  Maybe sort
        # on some 'precedence' trait on the type?
        for resource_type in self.resource_types:
            if resource_type.is_type_for(obj):
                break

        else:
            # If this resource manager couldn't do the job, then give its
            # parent a go!
            if self.parent is not None:
                resource_type = self.parent.get_type_of(obj)

            else:
                resource_type = None
                
            # Note that 'default_resource_type' *can* be None (if the creator
            # of the resource manager explicitly chose to set it to None).
            if resource_type is None:
                resource_type = self.default_resource_type

        return resource_type

    def add_dirty_resource_handler(self, binding, dirty_resource_handler):
        """ Adds a handler for a dirty resource. """

        key = binding.namespace_name
        
        self._dirty_resources[key] = (binding, dirty_resource_handler)

        return

    def get_dirty_resource_handler(self, binding):
        """ Returns the dirty handler for the specified object.

        Returns None, iff the object is not considered dirty.

        """

        key = binding.namespace_name

        binding, handler = self._dirty_resources.get(key, (None, None))

        return handler
    
    def remove_dirty_resource_handler(self, binding):
        """ Removes a dirty handler. """

        key = binding.namespace_name
        try:
            del self._dirty_resources[key]

        except KeyError:
            pass

        return

    def dirty_resource_count(self):
        """ Returns the number of dirty resources known to the manager. """

        return len(self._dirty_resources)

    def get_dirty_resources(self):
        """ Return a list of all dirty resources. """
        
        return self._dirty_resources.values()

    def identity(self, x, y):
        """ Returns True if two objects represent the same resource. """

        resource_type_x = self.get_type_of(x)
        resource_type_y = self.get_type_of(y)
        
        if resource_type_x is not None and resource_type_y is not None \
           and resource_type_x is resource_type_y:
            id_x = resource_type_x.get_id(x)
            id_y = resource_type_y.get_id(y)
            
            result = id_x == id_y

        else:
            result = False

        return result

    ###########################################################################
    # Debugging interface.
    ###########################################################################

    def dump(self):
        """ Dumps the contents of the manager to stdout. """

        print 'Resource Manager', id(self)

        for type in self._resource_types:
            print 'Type:', type.id

        print '--------------------------'
        
        return
    
#### EOF ######################################################################
