""" The resource type for Python instances. """


# Enthought library imports.
from enthought.naming.adapter import InstanceContextAdapterFactory
from enthought.traits.api import List, Str

# Local imports.
from instance_node_type import InstanceNodeType
from object_resource_type import ObjectResourceType


class InstanceResourceType(ObjectResourceType):
    """ The resource type for Python instances. """

    # All simple i.e., non-container types.
    SIMPLE_TYPES = [bool, complex, float, int, long, str, unicode]

    #### 'InstanceResourceType' interface #####################################
    
    # By default every public attribute of an instance is exposed. Use the
    # following traits to either include or exclude attributes as appropriate.
    #
    # Regular expressions that describe the names of attributes to exclude.  By
    # default we exclude 'protected' and 'private' attributes and any
    # attributes that are artifacts of the traits mechanism.
    exclude = List(Str, ['_', 'trait_'])

    # Regular expressions that describe the names of attributes to include.
    include = List(Str)
    
    ###########################################################################
    # 'ResourceType' interface.
    ###########################################################################

    #### Initializers #########################################################

    def _node_type_default(self):
        """ Initializes the node type. """

        return InstanceNodeType(resource_type=self)

    def _context_adapter_factory_default(self):
        """ Initializes the context adapter factory. """

        factory = InstanceContextAdapterFactory(
            include = self.include,
            exclude = self.exclude
        )

        return factory
    
    #### Methods ##############################################################

    def is_type_for(self, obj): 
        """ Returns True if the resource type 'recognizes' an object. """

        return obj is not None and not type(obj) in self.SIMPLE_TYPES

    def get_name(self, obj):
        """ Returns the name of a resource of this type. """

        if hasattr(obj, 'name'):
            name = obj.name

        elif hasattr(obj, 'id'):
            name = str(obj.id)

        else:
            name = str(obj)
            
        return name
    
    def set_name(self, obj, name):
        """ Sets the name of a resource of this type. """

        if hasattr(obj, 'name'):
            obj.name = name

        elif hasattr(obj, 'id'):
            obj.id = name

        return

##### EOF #####################################################################
