#------------------------------------------------------------------------------
# Copyright (c) 2005, Enthought, Inc.
# All rights reserved.
# 
# This software is provided without warranty under the terms of the BSD
# license included in enthought/LICENSE.txt and may be redistributed only
# under the conditions described in the aforementioned license.  The license
# is also available online at http://www.enthought.com/licenses/BSD.txt
# Thanks for using Enthought open source!
# 
# Author: Enthought, Inc.
# Description: <Enthought naming package component>
#------------------------------------------------------------------------------
""" An example of using the resource tree. """


# Enthought library imports.
from enthought.envisage.resource import ResourceManager
from enthought.envisage.resource import FolderResourceType, FileResourceType
from enthought.envisage.resource.ui import ResourceTree
from enthought.naming.api import Binding, PyFSContext
from enthought.pyface.api import GUI, PythonShell, SplitApplicationWindow
from enthought.pyface.action.api import Action, MenuManager
from enthought.traits.api import Instance


class MainWindow(SplitApplicationWindow):
    """ The main application window. """

    #### 'Window' interface ###################################################

    title = 'Resource Tree Example'
    
    #### 'SplitApplicationWindow' interface ###################################

    # The direction in which the panel is split.
    direction = 'vertical'

    # The ratio of the size of the left/top pane to the right/bottom pane.
    ratio = 0.3

    #### 'MainWindow' interface ###############################################
    
    # The root binding (usually a binding to a context!).
    root = Instance(Binding)
    
    ###########################################################################
    # Protected 'SplitApplicationWindow' interface.
    ###########################################################################

    def _create_lhs(self, parent):
        """ Creates the left hand side or top depending on the style. """

        return self._create_tree(parent)

    def _create_rhs(self, parent):
        """ Creates the panel containing the selected preference page. """

        return self._create_python_shell(parent)

    ###########################################################################
    # Private interface.
    ###########################################################################
    
    def _create_tree(self, parent):
        """ Creates the tree. """

        from enthought.envisage.resource.ui.action import NewFolder

        file_resource_type = FileResourceType()

        folder_resource_type = FolderResourceType()
        folder_resource_type.node_type.context_menu = MenuManager(
            MenuManager(
                NewFolder(),
                name = 'New'
            )
        )
        
        # Create a resource manager.
        resource_manager = ResourceManager()
        resource_manager.add_resource_type(file_resource_type)
        resource_manager.add_resource_type(folder_resource_type)
        
        # A resource tree is just a naming tree, hence the root is a binding.
        import tempfile, os
        root = Binding(name='Root',
                       obj=PyFSContext(path=os.path.dirname(tempfile.mktemp())))

        self._tree = tree = ResourceTree(
            parent,
            resource_manager = resource_manager,
            root             = root,
        )

        return tree.control
    
    def _create_python_shell(self, parent):
        """ Creates the Python shell. """

        self._python_shell = python_shell = PythonShell(parent)

        # Bind useful names.
        python_shell.bind('widget', self._tree)
        python_shell.bind('w', self._tree)
        python_shell.bind('window', self)
        
        # Execute useful commands to bind useful names ;^)
        python_shell.execute_command('from enthought.naming.api import *')
        
        return python_shell.control


# Application entry point.
if __name__ == '__main__':
    # Create the GUI (this does NOT start the GUI event loop).
    gui = GUI()

    # Create and open the main window.
    window = MainWindow()
    window.open()

    # Start the GUI event loop.
    gui.event_loop()
    
##### EOF #####################################################################
