""" A view containing the navigator tree. """


# Enthought library imports.
from enthought.envisage.resource.ui import ResourceTree
from enthought.envisage.resource.ui.resource_sorter import FoldersBeforeFiles
from enthought.envisage.ui import View
from enthought.naming.api import Binding

# Plugin imports.
from enthought.envisage.project.action.delete_action import DeleteAction
from enthought.envisage.project.action.open_action import OpenAction
from enthought.envisage.resource.resource_plugin import ResourcePlugin


class NavigatorView(View):
    """ A view containing the navigator tree.

    The navigator tree is a tree view of the workspace.

    """

    ###########################################################################
    # 'View' interface.
    ###########################################################################

    def _create_contents(self, parent):
        """ Creates the toolkit-specific control that represents the view.

        'parent' is the toolkit-specific control that is the view's parent.

        """

        return self._create_navigator_tree(parent)
    
    ###########################################################################
    # Private interface.
    ###########################################################################

    def _create_navigator_tree(self, parent):
        """ Creates the navigator tree. """

        # The root of the workspace.
        from enthought.envisage import get_application
        workspace = get_application().service_registry.get_service(
                  'enthought.envisage.project.IWorkspace')
        resource_manager = get_application().service_registry.get_service(
                  ResourcePlugin.IRESOURCE_MANAGER)


        # A resource tree is just a naming tree, hence the root is a binding.
        root = Binding(name='Workspace', obj=workspace)

        self._resource_tree = tree = ResourceTree(
            parent,

            # Debugging.
            _name = 'Navigator tree',
            
            # Content.
            resource_manager = resource_manager,
            root             = root,

            # Visualization.
            show_root = True,
            
            # Filtering/sorting.
            sorter    = FoldersBeforeFiles()
        )

        # The tree provides the view's selection.
        tree.on_trait_change(self._on_selection_changed, 'selection')
        tree.on_trait_change(self._on_node_activated_changed, 'node_activated')
        tree.on_trait_change(self._on_key_pressed_changed, 'key_pressed')

        return tree.control

    #### Trait event handlers #################################################
    
    def _on_selection_changed(self, obj, trait_name, old, new):
        """ Called when the selection is changed in the tree. """

        self.selection = new

        return

    def _on_node_activated_changed(self, node):
        """ Called when a node has been activated (i.e., double-clicked). """

        # fixme: Actions should really just invoke methods on a UI model.
        action = OpenAction(window=self.window)
        if action.enabled:
            action.perform()

        return

    def _on_key_pressed_changed(self, event):
        """ Called when a key is pressed while the tree has focus. """

        if event.key_code == 127:
            # fixme: Actions should really just invoke methods on a UI model.
            action = DeleteAction(window=self.window)
            if action.enabled:
                action.perform(None)
                
        return
        
    def get_resource_tree(self):
        return self._resource_tree

#### EOF ######################################################################
