""" The base class for all preferences. """


# Standard library imports.
import logging, os, pickle

# Enthought library imports.
from enthought.io.api import File
from enthought.traits.api import Any, Dict, Event, HasTraits, Str


# Setup a logger for this module.
logger=logging.getLogger(__name__)


class PreferenceChangedEvent(HasTraits):
    """ The event fired when a preference is changed. """

    # The key of the preference that changed.
    key = Str

    # The old value.
    old = Any

    # The new value.
    new = Any


class Preferences(HasTraits):
    """ The base class for all preferences. """

    #### 'Preferences' interface ##############################################

    # The default values of the preferences.
    defaults = Dict(Str, Any)

    # Any user overrides to the values of the preferences.
    user = Dict(Str, Any)

    # Fired when user preference has been changed.
    preference_changed = Event(PreferenceChangedEvent)

    ###########################################################################
    # 'Preferences' interface.
    ###########################################################################

    def contains(self, key):
        """ Returns '''True''' if the specified preference exists. """

        return key in self.user or key in self.defaults

    def load(self, filename):
        """ Loads the user preferences from the specified file. """

        # If a preferences file exists then load it.
        if os.path.isfile(filename):
            f = file(filename, 'rb')
            try:
                self.user = pickle.load(f)
            except:
                logger.error("error loading preference file '%s', using defaults" % filename)
                self.user = {}
            f.close()

        # Otherwise, there is no preferences file which means that the user has
        # not overridden any of the default preferences.
        else:
            self.user = {}

        return

    def save(self, filename):
        """ Saves the user preferences to the specified file. """

        defaults = self.defaults
        user = self.user

        # We only save any preferences that the user has changed from the
        # defaults.
        delta = user.copy()
        for key, value in user.items():
            if defaults.has_key(key) and defaults[key] == value:
                del delta[key]

        # If there is something to save then save 'em!
        if len(delta) > 0:
            f = file(filename, 'wb')
            pickle.dump(delta, f)
            f.close()

        # Otherwise, make sure that any old, saved preferences are cleaned up.
        else:
            f = File(filename)
            if f.is_file:
                f.delete()

        return

    def get(self, key, default=None):
        """ Returns the value of the preference with the specified key. """

        # Has the user overridden the default value for this preference?
        try:
            value = self.user[key]

        # Obviously not!
        except KeyError:
            # Did the plugin initialize a default value for it?
            try:
                value = self.defaults[key]

            # Obviously not again!
            except:
                value = default

        return value

    def set(self, key, value):
        """ Sets the value of the preference with the specified key. """

        old = self.get(key, None)

        self.user[key] = value

        if old != value:
            self.preference_changed = PreferenceChangedEvent(
                key=key, old=old, new=value
            )

        return

    def get_default(self, key):
        """ Returns the default value of a preference. """

        return self.defaults[key]

#### EOF ######################################################################
