""" Manages contributions of action sets. """


# Standard library imports
import logging

# Enthought library imports.
from enthought.envisage import Application
from enthought.traits.api import Any, HasTraits, Instance

# Plugin definition imports.
from action_plugin_definition import ActionSet


# Setup a logger for this module
logger = logging.getLogger(__name__)


class ActionSetManager(HasTraits):
    """ Manages contributions of action sets. """

    #### 'ActionSetManager' interface #########################################

    # The application that the manager is part of.
    application = Instance(Application)

    # The action sets that this manager manages.
    action_sets = Any

    ###########################################################################
    # 'ActionSetManager' interface.
    ###########################################################################

    # fixme: In all of these methods we should organize and cache the
    # contributions!
    def get_action(self, id, root):
        """
        Returns the action, if any, having the specified ID within the
        specified root location.

        """

        actions = self.get_actions(root)
        for action in actions:
            if action.id == id:
                result = action
                break
        else:
            result = None

        return result

    def get_actions(self, root):
        """ Returns all action contributions for a root. """

        logger.debug('ActionSetManager [%s] retrieving actions matching '
            'root [%s]', self, root)

        actions = {}
        keys = []
        for action_set in self._get_action_sets():
            aliases = action_set.aliases
            for action in action_set.actions:
                for location in action.locations:
                    if self._get_location_root(location, aliases) == root:
                        logger.debug('\tFound action with id [%s] in '
                            'action set with id [%s]', action.id,
                            action_set.id)
                        action._action_set_ = action_set
                        key = action.id
                        if hasattr(action, 'location'):
                            key += action.location.path
                        actions[key] = action
                        if key not in keys:
                            keys.append(key)
                        break

        return [actions[x] for x in keys]
    
    def get_groups(self, root):
        """ Returns all group contributions for a root. """

        groups = {}
        keys = []
        for action_set in self._get_action_sets():
            aliases = action_set.aliases
            for group in action_set.groups:
                if self._get_location_root(group.location, aliases) == root:
                    group._action_set_ = action_set
                    key = group.location.path + group.id
                    if key not in keys:
                        keys.append(key)
                    groups[key] = group

        return [groups[x] for x in keys]

    def get_menus(self, root):
        """ Returns all menu contributions for a root. """

        menus = {}
        keys = []
        for action_set in self._get_action_sets():
            aliases = action_set.aliases
            for menu in action_set.menus:
                if self._get_location_root(menu.location, aliases) == root:
                    menu._action_set_ = action_set
                    key = menu.location.path + menu.id
                    if key not in keys:
                        keys.append(key)
                    menus[key] = menu

        return [menus[x] for x in keys]

    ###########################################################################
    # 'Private' interface.
    ###########################################################################

    def _get_action_sets(self):
        """ Returns all action set contributions. """

        return self.action_sets
        ##return self.application.load_extensions(ActionSet)

    def _get_location_root(self, location, aliases):
        """ Returns the effective root for a location. """

        components = location.path.split('/')
        if components[0] in aliases:
            location_root = aliases[components[0]]

        else:
            location_root = components[0]

        return location_root

#### EOF ######################################################################
