/*
 * SysInfo.xs
 *
 * $Header$
 * $DateTime$
 * $Change$
 */

static char szWhatHeader[] = 
	"@(#) $Header$";

#ifndef __cplusplus
#ifndef HAS_BOOL
typedef char bool;
#endif
#endif

#ifndef WIN32
#define UNIX UNIX
#endif

#ifdef UNIX
#include <sys/utsname.h>
#endif

#ifdef WIN32
#include <malloc.h>
#ifndef alloca
#define alloca(size) _alloca(size)
#endif
#define snprintf _snprintf
#endif

#ifdef __hpux
#define DONT_DECLARE_STD
#endif

#include "EXTERN.h"
#include "perl.h"
#include "XSUB.h"

#ifdef PERL_OBJECT
#define FILE struct _PerlIO
#endif

#if defined WIN32
#define NEED_TYPEDEF_UCHAR_T NEED_TYPEDEF_UCHAR_T
#define NEED_TYPEDEF_UNIT16_T NEED_TYPEDEF_UNIT16_T
#define NEED_TYPEDEF_UNIT32_T NEED_TYPEDEF_UNIT32_T
#define NEED_TYPEDEF_UNIT64_T NEED_TYPEDEF_UNIT64_T
#elif defined linux
#define NEED_TYPEDEF_UCHAR_T NEED_TYPEDEF_UCHAR_T
#elif defined __hpux && defined __hppa
#define NEED_TYPEDEF_UCHAR_T NEED_TYPEDEF_UCHAR_T
#elif defined __hpux && defined __ia64 && ! defined HAS_STRTOLL
#define NEED_TYPEDEF_UCHAR_T NEED_TYPEDEF_UCHAR_T
#elif defined SNI
#define NEED_TYPEDEF_UNIT16_T NEED_TYPEDEF_UNIT16_T
#define NEED_TYPEDEF_UNIT32_T NEED_TYPEDEF_UNIT32_T
#define NEED_TYPEDEF_UNIT64_T NEED_TYPEDEF_UNIT64_T
#elif defined __osf__
#define NEED_TYPEDEF_UNIT16_T NEED_TYPEDEF_UNIT16_T
#define NEED_TYPEDEF_UNIT32_T NEED_TYPEDEF_UNIT32_T
#define NEED_TYPEDEF_UNIT64_T NEED_TYPEDEF_UNIT64_T
#define NEED_SNPRINTF NEED_SNPRINTF
#define HAS_SPRINTF_INTRINSIC HAS_SPRINTF_INTRINSIC
#endif

#ifdef NEED_TYPEDEF_UCHAR_T
typedef unsigned char uchar_t;
#endif
#ifdef NEED_TYPEDEF_UNIT16_T
typedef unsigned short uint16_t;
#endif
#ifdef NEED_TYPEDEF_UNIT32_T
typedef unsigned int   uint32_t;
#endif
#ifdef NEED_TYPEDEF_UNIT64_T
typedef unsigned long  uint64_t;
#endif

#ifdef _AIX
#define NEED_PRAGMA_ALLOCA NEED_PRAGMA_ALLOCA
#endif
#if defined UNIX && !defined _AIX && !defined SNI
#define NEED_ALLOCA_H NEED_ALLOCA_H
#endif

#ifdef NEED_ALLOCA_H
#include <alloca.h>
#endif

#ifdef NEED_PRAGMA_ALLOCA
#pragma alloca
#endif

#if defined __hpux || defined sun || defined _AIX
#define NEED_SYS_ERRLIST_PROTOTYPE NEED_SYS_ERRLIST_PROTOTYPE
#endif

/*
 * expect compiler warning 4113 when using Microsoft C
 * in c code generated by xs preprocessor
 */
#if _MSC_VER >= 1200
#pragma warning(disable : 4113)
#endif

/*
 * prototype and dummy function to find out libc version
 */
#if defined linux
char *gnu_get_libc_version ();
#else
static char *
gnu_get_libc_version () {
	return 0;
}
#endif

#include "SysInfo.h"

/*
 * common used type definitions
 */
typedef struct {
	int value;
	char name[64];
} names_by_value_t;

typedef struct {
	char their_name[64];
	char my_name[64];
} names_by_name_t;

typedef struct {
	char *system;
	char *version;
	char *subversion;
	char *architecture;
	char *c_runtime;
	char *cpp_runtime;
	char *cpp_runtime64;
} sysinfo_t;

/*
 * function prototypes
 */
static char *get_archname ();
static sysinfo_t *get_sysinfo ();

/*
 * common used helper funtions
 */
static char *
get_name_by_value (int value, names_by_value_t *ptr) {
	while (ptr->value >= 0) {
		if (ptr->value == value) {
			return (ptr->name);
		}
		ptr++;
	}
	return 0;
}

static char *
get_name_by_name (char *their_name, names_by_name_t *ptr) {
	if (ptr == 0) {
		return 0;
	}

	while (strlen (ptr->their_name) > 0) {
		if (strcmp (ptr->their_name, their_name) == 0) {
			return (ptr->my_name);
		}
		ptr++;
	}
	return 0;
}

#ifdef UNIX
static names_by_name_t os_names[] = {
	{"AIX",           "AIX"},
	{"HP-UX",         "HPUX"},
	{"Linux",         "Linux"},
	{"OSF1",          "Tru64"},
	{"SunOS",         "Solaris"},
	{"ReliantUNIX-N", "Reliant"},
	{"ReliantUNIX-Y", "Reliant"},
	{"SINIX-N",       "Reliant"},
	{"SINIX-Y",       "Reliant"},
	{"",              "unknown"}
};

static names_by_name_t arch_names[] = {
	{"alpha",  "ALPHA"},
	{"i386",   "I386"},
	{"i486",   "I386"},
	{"i586",   "I386"},
	{"i686",   "I386"},
	{"i786",   "I386"},
	{"i886",   "I386"},
	{"i986",   "I386"},
	{"ia64",   "IA64"},
	{"IA64",   "IA64"},
	{"x86_64", "X86-64"},
	{"x86-64", "X86-64"},
	{"X86_64", "X86-64"},
	{"X86-64", "X86-64"},
	{"ppc",    "PPC"},
	{"PPC",    "PPC"},
	{"ppc64",  "PPC"},
	{"PPC64",  "PPC"},
	{"s390",   "S390"},
	{"S390",   "S390"},
	{"s390x",  "S390"},
	{"S390X",  "S390"},
	{"",       "unknown"}
};

static names_by_name_t solaris_release_names[] = {
	{"5.4",   "2.4"},
	{"5.5.1", "2.5.1"},
	{"5.6",   "2.6"},
	{"",      "unknown"}
};

/*
 * get_sysinfo ()
 */
static sysinfo_t *
get_sysinfo () {
	struct utsname un[1];
	int rc;

	sysinfo_t *sysinfo;
	
	char *os_name;
	char *arch_name;
	char *version;
	char *subversion;
	char *c_runtime;
	char *cpp_runtime;
	char *cpp_runtime64;

	rc = uname (un);
	if (rc < 0) {
		return 0;
	}

	os_name = get_name_by_name (un->sysname, os_names);	
	if (os_name == 0) {
		return 0;
	}

	sysinfo = (sysinfo_t *) malloc (sizeof (sysinfo_t));
	if (sysinfo == 0) {
		return 0;
	}

	memset (sysinfo, 0, sizeof (sysinfo_t));

	subversion = 0;
	c_runtime = 0;
	cpp_runtime = 0;
	cpp_runtime64 = 0;

	sysinfo->system = malloc (strlen (os_name) + 1);
	if (sysinfo->system == 0) {
		free (sysinfo);
		return 0;
	}
	strcpy (sysinfo->system, os_name);

	if (strcmp (os_name, "AIX") == 0) {
		arch_name = "PowerPC";
	} else if (strcmp (os_name, "HPUX") == 0) {
		if (memcmp (un->machine, "9000", 4) == 0) {
			arch_name = "PA-RISC";
		} else {
			arch_name = get_name_by_name (un->machine, arch_names);
		}
	} else if (strcmp (os_name, "Reliant") == 0) {
		arch_name = "MIPS";
	} else if (memcmp (un->machine, "sun4", 4) == 0) {
		arch_name = "SPARC";
	} else {
		arch_name = get_name_by_name (un->machine, arch_names);
	}

	if (arch_name == 0) {
		sysinfo->architecture = malloc (1);
		if (sysinfo->architecture == 0) {
			free (sysinfo->system);
			free (sysinfo);
			return 0;
		}
		strcpy (sysinfo->architecture, "");
	} else {
		sysinfo->architecture = malloc (strlen (arch_name) + 1);
		if (sysinfo->architecture == 0) {
			free (sysinfo->system);
			free (sysinfo);
			return 0;
		}
		strcpy (sysinfo->architecture, arch_name);
	}

	if (strcmp (os_name, "AIX") == 0) {
		/*
		 * AIX
		 * try to get aix_release.level
		 */
		version = 0;

		{
		int rc;
		char *ptr;

		dSP;
		ENTER;
		SAVETMPS;
		PUSHMARK (sp);
		
		rc = perl_call_pv (
		"SAPDB::Install::SysInfo::get_aix_version", G_SCALAR);

		SPAGAIN;
		if (rc == 1) {
			ptr = POPp;
			if (ptr != 0) {
				version = alloca (strlen (ptr) + 1);
				strcpy (version, ptr);
			}
		}

		PUTBACK;
		FREETMPS;
		LEAVE;
		}

		/*
		 * try uname when we missed first chance
		 * we have to concat version and release
		 */
		if (version == 0) {
			version = alloca (strlen (un->version) + 1 +
			                  strlen (un->release) + 1);	

			strcpy (version, un->version);
			strcat (version, ".");
			strcat (version, un->release);
		}
	} else if (strcmp (os_name, "HPUX") == 0) {
		/*
		 * HP-UX
		 * release name looks like 'B.<dd>.<dd>'
		 * so we only have to get rid of 'B.'
		 */
		if (memcmp (un->release, "B.", 2) != 0) {
			version = alloca (1);
			strcpy (version, "");
		} else {
			version =
			alloca (strlen ((char *)(un->release) + 2) + 1);
			strcpy (version, (char *)(un->release) + 2); 
		}
	} else if (strcmp (os_name, "Linux") == 0) {
		/*
		 * Linux
		 * release name looks like '<d>.<d>.<d>'
		 * or '<d>.<d>.<d>-<some _text>
		 * so we have to get rid of the trailing text
		 */
		char *ptr;
		char *glibc_version;

		ptr = strchr (un->release, '-');
		if (ptr != 0) {
			int len;

			len = ptr - (char *)(un->release);
			version = alloca (len + 1);
			memcpy (version, (char *)(un->release), len);
			version[len] = '\0';
			
			subversion = alloca (strlen (ptr + 1) + 1);
			strcpy (subversion, ptr + 1);

		} else {
			version = alloca (strlen ((char *)(un->release)) + 1);
			strcpy (version, (char *)(un->release));
		}

		glibc_version = gnu_get_libc_version ();
		if (glibc_version != 0) {
			char *glibc_name = "GLIBC";

			c_runtime = alloca (strlen (glibc_name) + 1 +
			                    strlen (glibc_version) + 1);
			strcpy (c_runtime, glibc_name);
			strcat (c_runtime, " ");
			strcat (c_runtime, glibc_version);
		}
	} else if (strcmp (os_name, "Reliant") == 0) {
		/*
		 * Reliant
		 * really amazing, we an use the output of uname
		 */
		version = alloca (strlen ((char *)(un->release)) + 1);
		strcpy (version, (char *)(un->release));
	} else if (strcmp (os_name, "Solaris") == 0) {
		/*
		 * Solaris
		 * use mapping table for known releases or
		 * get rid of the heading '5.'
		 */
		version = get_name_by_name (un->release, solaris_release_names);
		if (version == 0 || strlen (version) == 0) {
			if (memcmp (un->release, "5.", 2) == 0) {
				version =
				alloca (strlen ((char *)(un->release) + 2) + 1);
				strcpy (version, (char *)(un->release) + 2); 
			} else {
				version = alloca (1);
				strcpy (version, "");
			}
		}

		/*
		 * get applied patch for C++-runtime
		 * using 'showrev -p'
		 */
		{
			int rc;
			char *ptr;

			dSP;
			ENTER;
			SAVETMPS;
			PUSHMARK (sp);
		
			XPUSHs (sv_2mortal (newSVpv (version, strlen (version))));
			PUTBACK;
		
			rc = perl_call_pv (
			"SAPDB::Install::SysInfo::get_solaris_cppruntime", G_ARRAY);

			SPAGAIN;
			if (rc == 1) {
				ptr = POPp;
				if (ptr != 0) {
					cpp_runtime = alloca (strlen (ptr) + 1);
					strcpy (cpp_runtime, ptr);
				}
			}

			if (rc == 2) {
				ptr = POPp;
				if (ptr != 0) {
					cpp_runtime64 = alloca (strlen (ptr) + 1);
					strcpy (cpp_runtime64, ptr);
				}

				ptr = POPp;
				if (ptr != 0) {
					cpp_runtime = alloca (strlen (ptr) + 1);
					strcpy (cpp_runtime, ptr);
				}
			}

			PUTBACK;
			FREETMPS;
			LEAVE;
		}
	} else if (strcmp (os_name, "Tru64") == 0) {
		/*
		 * get version of running kernel using 'sizer -v'
		 */
		{
		int rc;
		char *ptr;

		dSP;
		ENTER;
		SAVETMPS;
		PUSHMARK (sp);
		
		rc = perl_call_pv (
		"SAPDB::Install::SysInfo::get_tru64_version", G_SCALAR);

		SPAGAIN;
		if (rc == 1) {
			ptr = POPp;
			if (ptr != 0) {
				version = alloca (strlen (ptr) + 1);
				strcpy (version, ptr);
			} else {
				version = alloca (1);
				strcpy (version, "");
			}
		} else {
			version = alloca (1);
			strcpy (version, "");
		}

		PUTBACK;
		FREETMPS;
		LEAVE;
		}

		/*
		 * get latest applied patch kit from
		 * /var/log/patch/log/event.log
		 */
		{
		int rc;
		char *ptr;

		dSP;
		ENTER;
		SAVETMPS;
		PUSHMARK (sp);
		
		rc = perl_call_pv (
		"SAPDB::Install::SysInfo::get_tru64_subversion", G_SCALAR);

		SPAGAIN;
		if (rc == 1) {
			ptr = POPp;
			if (ptr != 0) {
				subversion = alloca (strlen (ptr) + 1);
				strcpy (subversion, ptr);
			}
		}

		PUTBACK;
		FREETMPS;
		LEAVE;
		}
	}

	if (version == 0) {
		sysinfo->version = malloc (1);
		if (sysinfo->version == 0) {
			free (sysinfo->system);
			free (sysinfo->architecture);
			free (sysinfo);
			return 0;
		}
		strcpy (sysinfo->version, "");
	} else {
		sysinfo->version = malloc (strlen (version) + 1);
		if (sysinfo->version == 0) {
			free (sysinfo->system);
			free (sysinfo->architecture);
			free (sysinfo);
			return 0;
		}
		strcpy (sysinfo->version, version);
	}

	if (subversion != 0) {
		sysinfo->subversion = malloc (strlen (subversion) + 1);
		if (sysinfo->subversion == 0) {
			free (sysinfo->version);
			free (sysinfo->system);
			free (sysinfo->architecture);
			free (sysinfo);
			return 0;
		}
		strcpy (sysinfo->subversion, subversion);
	}

	if (c_runtime != 0) {
		sysinfo->c_runtime = malloc (strlen (c_runtime) + 1);
		if (sysinfo->c_runtime == 0) {
			free (sysinfo->version);
			free (sysinfo->system);
			free (sysinfo->architecture);
			if (sysinfo->subversion) {
				free (sysinfo->subversion);
			}
			free (sysinfo);
			return 0;
		}
		strcpy (sysinfo->c_runtime, c_runtime);
	}

	if (cpp_runtime != 0) {
		sysinfo->cpp_runtime = malloc (strlen (cpp_runtime) + 1);
		if (sysinfo->cpp_runtime == 0) {
			free (sysinfo->version);
			free (sysinfo->system);
			free (sysinfo->architecture);
			if (sysinfo->subversion) {
				free (sysinfo->subversion);
			}
			if (sysinfo->c_runtime) {
				free (sysinfo->c_runtime);
			}
			free (sysinfo);
			return 0;
		}
		strcpy (sysinfo->cpp_runtime, cpp_runtime);
	}

	if (cpp_runtime64 != 0) {
		sysinfo->cpp_runtime64 = malloc (strlen (cpp_runtime64) + 1);
		if (sysinfo->cpp_runtime64 == 0) {
			free (sysinfo->version);
			free (sysinfo->system);
			free (sysinfo->architecture);
			if (sysinfo->subversion) {
				free (sysinfo->subversion);
			}
			if (sysinfo->c_runtime) {
				free (sysinfo->c_runtime);
			}
			if (sysinfo->cpp_runtime) {
				free (sysinfo->cpp_runtime);
			}
			free (sysinfo);
			return 0;
		}
		strcpy (sysinfo->cpp_runtime64, cpp_runtime64);
	}

	return (sysinfo);
}
#endif

#ifdef WIN32                                                 
/*
 * defines for processor architecture 
 */
#ifndef PROCESSOR_ARCHITECTURE_INTEL
#define PROCESSOR_ARCHITECTURE_INTEL    0
#endif

#ifndef PROCESSOR_ARCHITECTURE_MIPS
#define PROCESSOR_ARCHITECTURE_MIPS     1
#endif

#ifndef PROCESSOR_ARCHITECTURE_ALPHA
#define PROCESSOR_ARCHITECTURE_ALPHA    2
#endif

#ifndef PROCESSOR_ARCHITECTURE_PPC
#define PROCESSOR_ARCHITECTURE_PPC      3
#endif

#ifndef PROCESSOR_ARCHITECTURE_IA64
#define PROCESSOR_ARCHITECTURE_IA64     6
#endif

static names_by_value_t arch_names[] = {
	{PROCESSOR_ARCHITECTURE_INTEL, "I386"},
	{PROCESSOR_ARCHITECTURE_MIPS,  "MIPS"},
	{PROCESSOR_ARCHITECTURE_ALPHA, "ALPHA"},
	{PROCESSOR_ARCHITECTURE_PPC,   "PPC"},
	{PROCESSOR_ARCHITECTURE_IA64,  "IA64"},
	{-1,                           "unknown"}
};

static names_by_value_t os_names[] = {
	{VER_PLATFORM_WIN32_WINDOWS, "WIN32"},
	{VER_PLATFORM_WIN32_NT,      "NT"},
	{-1,                         "unknown"}
};

static char *
get_version () {
	OSVERSIONINFO os_info[1];
	char vers[4096];
	char *name;
	char *system;
	
	os_info->dwOSVersionInfoSize = sizeof (OSVERSIONINFO);
	GetVersionEx (os_info);

	name = get_name_by_value (os_info->dwPlatformId, os_names);
	snprintf (vers, sizeof (vers),
	          "%d.%d", os_info->dwMajorVersion, os_info->dwMinorVersion);

	system = (char *) malloc (strlen (name) + 1 + strlen (vers) + 1);
	if (system == 0)
		return 0;
	
	strcpy (system , name);
	strcat (system, " ");
	strcat (system, vers);

	return (system);
}

static char *
get_subversion () {
	OSVERSIONINFO os_info[1];
	char *subversion;
	
	os_info->dwOSVersionInfoSize = sizeof (OSVERSIONINFO);
	GetVersionEx (os_info);

	if (os_info->szCSDVersion == 0 || 
    strlen (os_info->szCSDVersion) == 0)
		return 0;

	subversion = (char *) malloc (strlen (os_info->szCSDVersion) + 1);
	if (subversion == 0)
		return 0;
	
	strcpy (subversion, os_info->szCSDVersion);
	return (subversion);
}

/*
 * get_archname ()
 */
static char *
get_archname () {
	char *archname;
	
	SYSTEM_INFO sys_info[1];
	HANDLE hKERNEL32;
	BOOL IsWow64;
	BOOL (WINAPI *fIsWow64Process)(HANDLE, PBOOL);

	GetSystemInfo (sys_info);

	archname =
	get_name_by_value (sys_info->wProcessorArchitecture, arch_names);

	/*
	 * return on anything else then I386 or unknown
	 */
	if ((strcmp (archname, "I386") != 0) &&
	    (strcmp (archname, "unknown") != 0))
		return (archname);

	/*
	 * find out if we are running in an i386 emulator called wow64
	 * we have to use dynamic loading,
	 * beause the loader cannot find the neccesary symbols on windows nt 4.0
	 */
	hKERNEL32 = LoadLibrary ("kernel32.dll");
	if (hKERNEL32 == 0)
		return (archname);
	
	fIsWow64Process = (BOOL (WINAPI *)(HANDLE, PBOOL))
	GetProcAddress (hKERNEL32, "IsWow64Process");

	if (fIsWow64Process == 0) {
		FreeLibrary (hKERNEL32);
		return (archname);
	}

	if ((*fIsWow64Process) (GetCurrentProcess, &IsWow64) == 0) {
		FreeLibrary (hKERNEL32);
		return (archname);
	}

	FreeLibrary (hKERNEL32);
	if (IsWow64 != 0)
		return (archname);

	return ("WOW64");
}

/*
 * get_sysinfo ()
 */
static sysinfo_t *
get_sysinfo () {
	sysinfo_t *sysinfo;
	char *system = "Windows";
	char *version = 0; 
	char *architecture;
	
	sysinfo = (sysinfo_t *) malloc (sizeof (sysinfo_t));
	if (sysinfo == 0) {
		return 0;
	}

	memset (sysinfo, 0, sizeof (sysinfo_t));

	sysinfo->system = (char *) malloc (strlen (system) + 1);
	if (sysinfo->system == 0) {
		free (sysinfo);
		return 0;
	}
	strcpy (sysinfo->system, system);

	architecture = get_archname ();
	if (architecture == 0) {
		sysinfo->architecture = (char *) malloc (1);
		if (sysinfo->architecture == 0) {
			free (sysinfo->system);
			free (sysinfo);
			return 0;
		} else {
			strcpy (sysinfo->architecture, "");
		}		
	} else {
		sysinfo->architecture = (char *) malloc (strlen (architecture) + 1);
		if (sysinfo->architecture == 0) {
			free (sysinfo->system);
			free (sysinfo);
			return 0;
		} else {
			strcpy (sysinfo->architecture, architecture);
		}
	}

	sysinfo->version = get_version ();
	if (sysinfo->version == 0) {
		sysinfo->version = (char *) malloc (1);
		if (sysinfo->version == 0) {
			free (sysinfo->system);
			free (sysinfo->architecture);
			free (sysinfo);
			return 0;
		} else {
			strcpy (sysinfo->version, "");
		}
	}
 
	sysinfo->subversion = get_subversion ();
	if (sysinfo->subversion == 0) {
		sysinfo->subversion = (char *) malloc (1);
		if (sysinfo->subversion == 0) {
			free (sysinfo->system);
			free (sysinfo->version);
			free (sysinfo->architecture);
			free (sysinfo);
			return 0;
		} else {
			strcpy (sysinfo->subversion, "");
		}
	}
 
	return (sysinfo);
}
#endif

MODULE = SAPDB::Install::SysInfo	PACKAGE = SAPDB::Install::SysInfo

PROTOTYPES: DISABLE

BOOT:
	perl_eval_pv ((char *) sz_text, 1);

SV *
GetSystemInfo (...)
PREINIT:
	sysinfo_t *sysinfo;
	HV *hv;
	SV *rv;
PPCODE:
	sysinfo = get_sysinfo ();
	
	if (sysinfo == 0) {
		XSRETURN_UNDEF;
	}

	hv = newHV ();
	
	if (sysinfo->system != 0) {
		char *key = "system";

		hv_store (hv, key, strlen (key), 
		newSVpv (sysinfo->system, strlen (sysinfo->system)), 0);

		free (sysinfo->system);
	}

	if (sysinfo->version != 0) {
		char *key = "version";

		hv_store (hv, key, strlen (key), 
		newSVpv (sysinfo->version, strlen (sysinfo->version)), 0);

		free (sysinfo->version);
	}

	if (sysinfo->architecture != 0) {
		char *key = "architecture";

		hv_store (hv, key, strlen (key), 
		newSVpv (sysinfo->architecture,
		strlen (sysinfo->architecture)), 0);

		free (sysinfo->architecture);
	}

	if (sysinfo->subversion != 0) {
		char *key = "subversion";

		hv_store (hv, key, strlen (key), 
		newSVpv (sysinfo->subversion,
		strlen (sysinfo->subversion)), 0);

		free (sysinfo->subversion);
	}

	if (sysinfo->c_runtime != 0) {
		char *key = "c_runtime";

		hv_store (hv, key, strlen (key), 
		newSVpv (sysinfo->c_runtime,
		strlen (sysinfo->c_runtime)), 0);

		free (sysinfo->c_runtime);
	}

	if (sysinfo->cpp_runtime != 0) {
		char *key = "cpp_runtime";

		hv_store (hv, key, strlen (key), 
		newSVpv (sysinfo->cpp_runtime,
		strlen (sysinfo->cpp_runtime)), 0);

		free (sysinfo->cpp_runtime);
	}

	if (sysinfo->cpp_runtime64 != 0) {
		char *key = "cpp_runtime64";

		hv_store (hv, key, strlen (key), 
		newSVpv (sysinfo->cpp_runtime64,
		strlen (sysinfo->cpp_runtime64)), 0);

		free (sysinfo->cpp_runtime64);
	}

	free (sysinfo);
	rv = sv_2mortal (newRV ((SV *) hv));
	SvREFCNT_dec (hv);

	XPUSHs (rv);
	XSRETURN (1);
