/*
                 Manual Page Editor: Editor Structure
 */

#ifndef EDITOR_H
#define EDITOR_H

#include <sys/types.h>
#include <gtk/gtk.h>

#include "editorfip.h"

#ifndef gbool
# define gbool	guchar
#endif


/*
 *	Editor ctree item types:
 */
#define EditorItemTypeFile	0	/* Main trunk. */
#define EditorItemTypeHeader	1
#define EditorItemTypeSection	2


/*
 *	Editor ctree item structure:
 *
 *	Used as row data on the layout_ctree.
 *
 *	When adding new members be sure to update functions
 *	EditorItemNew(), EditorItemDup(), and EditorItemDelete()
 *	so that they properly allocate, duplicate, and deallocate the
 *	new members.
 *
 *	Note that not all members who are pointers should be
 *	deallocated.
 */
typedef struct {

	int type;		/* One of EditorItemType*. */
	void *editor_ptr;	/* Pointer back to editor. */
	gbool has_changes;

	GtkCTree *ctree;		/* Pointer back to parent ctree. */
	GtkCTreeNode *parent;		/* Parent branch. */
	GtkCTreeNode *this_branch;	/* Pointer back to this item. */
	gbool is_leaf;

	/* Members for type EditorItemTypeFile. */
	char *file_name;	/* Just the file's name without a path. */
	char *file_name_full;	/* Full path to file, if this is NULL
				 * then it implies it's never been
				 * saved before.
				 */

	/* Members for type EditorItemTypeHeader. */
	char *header_name;
	char *header_section_number;
	char *header_version;
	char *header_author;
	char *header_catagory;

	/* Members for type EditorItemTypeSection. */
	char *section_name;

	/* Common members. */
	char **line;
	int total_lines;

} editor_item_struct;


/*
 *	Editor structure:
 */
typedef struct {

	gbool initialized;
	gbool map_state;
	gbool processing;
	gbool syntax_highlighting;

	void *core_ptr;
	int viewer_num;		/* Number of manual page viewer on core,
				 * can be -1 for none.
				 */

	GtkWidget *toplevel;
	GtkWidget *main_vbox;

	GtkWidget *menu_bar_dock, *menu_bar;
	GtkWidget *tool_bar_dock, *tool_bar;

	GtkWidget *layout_ctree;

	/* List of trunk branch nodes on the layout ctree, these may
	 * not be in any particular order and not in sync on the
	 * ctree. These should all have user item data type set to
	 * EditorItemTypeFile, which implies the number of files loaded.
	 */
	GtkCTreeNode **layout_trunk;
	int total_layout_trunks;

	GtkCTreeNode *selected_branch;	/* NULL for none. */

	/* Right pane editing widgets parent, note the type
	 * corresponds to EditorItemType* and maximum is
	 * MEDIT_EDIT_PANEL_MAX which must be one greater than
	 * the highest EditorItemType* code.
	 */
#define MEDIT_EDIT_PANEL_MAX	3
	GtkWidget *edit_panel_vbox[MEDIT_EDIT_PANEL_MAX];

	GtkWidget	*header_name_entry,
			*header_section_number_entry,
			*header_version_entry,
			*header_author_entry,
			*header_catagory_entry,
			*header_text;	/* For comments. */

	GtkWidget	*section_name_entry,
			*section_text;

	/* Syntax highlight timeout callback values. */
	guint synhl_timeout_cb_id;	/* None if (guint)(-1). */
	int synhl_timeout_cb_cursor_pos;
	int synhl_timeout_cb_start_pos;
	int synhl_timeout_cb_end_pos;

	/* Important menu items. */
	GtkWidget	*file_mh,
			*new_mi,
			*new_from_template_mi,
			*open_mi,
			*save_mi,
			*save_as_mi,
			*revert_mi,
			*close_manual_mi,
			*close_mi,
			*exit_mi,

			*edit_mh,
			*undo_mi,
			*undo_milabel,
			*redo_mi,
			*redo_milabel,
			*cut_mi,
			*copy_mi,
			*paste_mi,
			*select_all_mi,
			*unselect_all_mi,
			*find_in_pages_mi,

                        *view_mh,
                        *preview_mi,
                        *syntax_highlight_cmi,

			*layout_mh,
			*add_header_mi,
			*add_section_mi,
			*remove_mi,
			*properties_mi,

                        *fmt_mh,
                        *fmt_pl_mi,
                        *fmt_pr_mi,
                        *fmt_pi_mi,
                        *fmt_li_mi,
                        *fmt_b_mi,
                        *fmt_u_mi,
                        *fmt_br_mi,
                        *fmt_amp_mi,
                        *fmt_lt_mi,
                        *fmt_gt_mi,
			*fmt_strip_tags_mi,

                        *windows_mh,
                        *windows_new_editor,
                        *windows_new_viewer;

	/* Tool bar buttons. */
	GtkWidget	*new_btn,
			*new_btn_new_menu,
			*new_btn_new_mi,
			*new_btn_new_from_template_mi,
			*open_btn,
			*save_btn,
			*save_as_btn,
			*close_btn,

			*undo_btn,
			*redo_btn,

			*cut_btn,
			*copy_btn,
			*paste_btn,

			*add_header_btn,
			*add_section_btn,
			*remove_btn,
			*properties_btn,

			*preview_btn;

	/* Formatting bar buttons. */
	GtkWidget	*fmt_bar_dock,
			*fmt_bar,
			*fmt_pl_btn,
			*fmt_pr_btn,
			*fmt_pi_btn,
			*fmt_li_btn,
			*fmt_b_btn,
			*fmt_u_btn,
			*fmt_br_btn,
			*fmt_amp_btn,
			*fmt_lt_btn,
			*fmt_gt_btn,
			*fmt_strip_tags_btn;

	/* Search bar. */
	GtkWidget	*find_bar_dock,
			*find_bar,
			*find_combo,
			*find_btn,
			*replace_combo,
			*replace_btn,
			*replace_all_btn;

	/* Find bar button find menu. */
	GtkWidget	*find_bar_find_menu,
			*find_bar_find_in_pages_mi;

	/* Find bar button replace menu. */
	GtkWidget       *find_bar_replace_menu,
                        *find_bar_replace_entire_page_mi;

	/* Right-click menus. */
	GtkWidget *layout_menu, *edit_panel_menu;

	/* Important menu items on right-click menus. */
	GtkWidget	*layout_menu_expand_mi,
			*layout_menu_preview_mi,
			*layout_menu_add_header_mi,
                        *layout_menu_add_section_mi,
                        *layout_menu_remove_mi,
			*layout_menu_properties_mi,
			*layout_menu_new_manual_mi,
			*layout_menu_new_from_template_mi,
                        *layout_menu_open_mi,
                        *layout_menu_save_mi,
                        *layout_menu_save_as_mi,
			*layout_menu_revert_mi,
			*layout_menu_close_manual_mi;

	GtkWidget	*edit_panel_undo_mi,
			*edit_panel_undo_milabel,
			*edit_panel_redo_mi,
			*edit_panel_redo_milabel,
			*edit_panel_cut_mi,
			*edit_panel_copy_mi,
			*edit_panel_paste_mi,
			*edit_panel_select_all_mi,
			*edit_panel_unselect_all_mi;
	/* Format submenu on edit panel menu. */
	GtkWidget	*edit_panel_fmt_sub_menu,
                        *edit_panel_fmt_pl_mi,
                        *edit_panel_fmt_pr_mi,
                        *edit_panel_fmt_pi_mi,
                        *edit_panel_fmt_li_mi,
                        *edit_panel_fmt_b_mi,
                        *edit_panel_fmt_u_mi,
                        *edit_panel_fmt_br_mi,
                        *edit_panel_fmt_amp_mi,
                        *edit_panel_fmt_lt_mi,
                        *edit_panel_fmt_gt_mi,
			*edit_panel_fmt_strip_tags_mi;

	/* Status bar. */
	GtkWidget	*status_bar_dock,
			*status_bar_toplevel,	/* An hbox. */
			*status_bar_progress,
			*status_bar_label,
			*status_bar_cursor_label;

	/* Last progress position value from 0.0 to 1.0. */
	gfloat status_bar_progress_pos_last;

	/* Undo/redo structures. */
	void **undo;
	int total_undos;

	void **redo;
	int total_redos;

	int max_undos;	/* Maximum undos and maximum redos. */


	/* Last open file path. */
	char *last_open_path;

	/* Last save as file path. */
	char *last_save_as_path;


	/* Editor Find In Pages dialog. */
	editor_fip_struct *fip_dialog;

} editor_struct;



extern editor_item_struct *EditorItemNew(
        int type,               /* One of EditorItemType*. */
        editor_struct *editor,
        GtkCTree *ctree,        /* Pointer back to parent ctree. */
        GtkCTreeNode *parent,   /* Parent branch. */
        GtkCTreeNode *this_branch,      /* Pointer back to this item. */
        gbool is_leaf,
        char **line,            /* Eatened, not copied. */
        int total_lines
);
extern editor_item_struct *EditorItemDup(editor_item_struct *item);
extern void EditorItemDelete(editor_item_struct *item);

extern void EditorLayoutTrunkDeleteAll(editor_struct *editor);

extern void EditorBranchSelect(
        editor_struct *editor, GtkCTreeNode *branch
);

extern void EditorBranchSetData(
	GtkCTree *ctree, GtkCTreeNode *branch,
	editor_item_struct *item_ptr, GtkDestroyNotify destroy
);
extern editor_item_struct *EditorBranchGetData(
	GtkCTree *ctree, GtkCTreeNode *branch
);
extern void EditorItemSetToplevelHasChanges(
        editor_struct *editor, GtkCTreeNode *branch, gbool has_changes
);

extern GtkCTreeNode *EditorItemGetFirstToplevel(editor_struct *editor);
extern GtkCTreeNode *EditorItemGetParent(
        editor_struct *editor, GtkCTreeNode *branch
);
extern GtkCTreeNode *EditorItemGetToplevel(
        editor_struct *editor, GtkCTreeNode *branch
);

extern void EditorSetBusy(editor_struct *editor);
extern void EditorSetReady(editor_struct *editor);

extern editor_struct *EditorNew(void *core_ptr);
extern void EditorUpdateMenus(editor_struct *editor);
extern void EditorSetStatusPosition(editor_struct *editor, int row, int column);
extern void EditorSetStatusMessage(editor_struct *editor, const char *mesg);
extern void EditorSetStatusProgress(editor_struct *editor, gfloat percent);
extern void EditorRecordPositions(editor_struct *editor);
extern void EditorResetUndos(editor_struct *editor);
extern void EditorReset(editor_struct *editor, gbool need_unmap);
extern void EditorMap(editor_struct *editor);
extern void EditorUnmap(editor_struct *editor);
extern void EditorDelete(editor_struct *editor);



#endif	/* EDITOR_H */
