/* This file is part of Malaga, a system for Natural Language Analysis.
 * Copyright (C) 1995-1999 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* description ==============================================================*/

/* Options for malaga and functions to start and terminate malaga. */

/* includes =================================================================*/

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <stdlib.h>
#include <setjmp.h>
#include <time.h>
#include "basic.h"
#include "pools.h"
#include "values.h"
#include "input.h"
#include "commands.h"
#include "options.h"
#include "malaga_lib.h"
#include "rule_type.h"
#include "rules.h"
#include "files.h"
#include "analysis.h"
#include "cache.h"
#include "symbols.h"
#include "lexicon.h"
#include "transmit.h"
#include "display.h"

#ifdef HANGUL
#include "hangul.h"
#endif

#undef GLOBAL
#define GLOBAL
#include "malaga_lib.h"

/* functions ================================================================*/

LOCAL void do_pruning_option (string_t arguments)
/* Enable/disable pruning_rule. */
{
  if (*arguments == EOS) 
    printf ("pruning: %s\n", 
	    get_analysis_option (PRUNING_OPTION) ? "yes" : "no");
  else
    set_analysis_option (PRUNING_OPTION, parse_yes_no (&arguments));

  parse_end (arguments);
}

LOCAL command_t pruning_option =
{
  "pruning p", do_pruning_option,
  "Usage:\n"
  "  set pruning yes -- turn pruning_rule on\n"
  "  set pruning no -- turn pruning_rule off\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_robust_option (string_t arguments)
/* Enable/disable robust parsing. */
{
  if (*arguments == EOS)
    printf ("robust: %s\n", 
	    get_analysis_option (ROBUST_OPTION) ? "yes" : "no");
  else
    set_analysis_option (ROBUST_OPTION, parse_yes_no (&arguments));
  
  parse_end (arguments);
}

LOCAL command_t robust_option =
{
  "robust", do_robust_option,
  "Usage:\n"
  "  set robust yes -- turn robust_rule on\n"
  "  set robust no -- turn robust_rule off\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_mor_out_filter_option (string_t arguments)
/* Enable/disable morphology output filter. */
{
  if (*arguments == EOS)
    printf ("mor-out-filter: %s\n", 
	    get_analysis_option (MOR_OUT_FILTER_OPTION) ? "yes" : "no");
  else
    set_analysis_option (MOR_OUT_FILTER_OPTION, parse_yes_no (&arguments));
  
  parse_end (arguments);
}

LOCAL command_t mor_out_filter_option =
{
  "mor-out-filter mofil", do_mor_out_filter_option,
  "Usage:\n"
  "  set mor-out-filter yes -- turn morphology output filter on\n"
  "  set mor-out-filter no -- turn morphology output filter off\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_syn_out_filter_option (string_t arguments)
/* Enable/disable syntax output filter. */
{
  if (*arguments == EOS)
    printf ("syn-out-filter: %s\n", 
	    get_analysis_option (SYN_OUT_FILTER_OPTION) ? "yes" : "no");
  else
    set_analysis_option (SYN_OUT_FILTER_OPTION, parse_yes_no (&arguments));
  
  parse_end (arguments);
}
LOCAL command_t syn_out_filter_option =
{
  "syn-out-filter sofil", do_syn_out_filter_option,
  "Usage:\n"
  "  set syn-out-filter yes -- turn syntax output filter on\n"
  "  set syn-out-filter no -- turn syntax output filter off\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_syn_in_filter_option (string_t arguments)
/* Enable/disable syntax input filter. */
{
  if (*arguments == EOS)
    printf ("syn-in-filter: %s\n", 
	    get_analysis_option (SYN_IN_FILTER_OPTION) ? "yes" : "no");
  else
    set_analysis_option (SYN_IN_FILTER_OPTION, parse_yes_no (&arguments));
  
  parse_end (arguments);
}
LOCAL command_t syn_in_filter_option =
{
  "syn-in-filter sifil", do_syn_in_filter_option,
  "Usage:\n"
  "  set syn-in-filter yes -- turn syntax input filter on\n"
  "  set syn-in-filter no -- turn syntax input filter off\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_cache_size_option (string_t arguments)
/* Change the cache size. */
{
  if (*arguments == EOS)
    printf ("cache-size: %ld (%ld used)\n", 
	    get_cache_size (TRUE), get_cache_size (FALSE));
  else
  {
    int_t size = parse_int (&arguments);

    if (size < 0)
      error ("don't use a negative cache size");

    set_cache_size (size);
    set_analysis_option (CACHE_OPTION, (size > 0));
  }
  parse_end (arguments);
}

LOCAL command_t cache_size_option =
{
  "cache-size", do_cache_size_option,
  "Usage: set cache-size <size> -- Set cache size to <size>\n"
  "If <size> == 0, cache is switched off.\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_tree_option (string_t arguments)
/* Determine if tree is shown after analysis. */
{
  if (*arguments == EOS)
    printf ("tree: %s\n", show_tree ? "yes" : "no");
  else
    show_tree = parse_yes_no (&arguments);
  parse_end (arguments);
}

LOCAL command_t tree_option =
{
  "tree", do_tree_option,
  "Usage:\n"
  "  set tree yes -- show tree after analysis\n"
  "  set tree no -- don't show tree after analysis\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_output_format_option (string_t arguments)
/* Change analysis success format to "arguments". */
{
  if (*arguments == EOS)
  {
    string_t format = new_string_readable (output_format, NULL);

    printf ("output-format: %s\n", format);
    free_mem (&format);
  }
  else
  {
    string_t new_format = parse_word (&arguments);

    free_mem (&output_format);
    output_format = new_format;
  }

  parse_end (arguments);
}

LOCAL command_t output_format_option =
{
  "output-format", do_output_format_option,
  "Describe the format in which analysis results will be printed.\n"
  "Usage: set output-format <output_format_string>\n"
  "The output format string may contain the following special sequences:\n"
  "  %s -- surface\n"
  "  %l -- input line number\n"
  "  %c -- result category\n"
  "  %r -- the ambiguity index\n"
};

/*---------------------------------------------------------------------------*/

/* Change unknown analysis format to "arguments". */

LOCAL void do_unknown_format_option (string_t arguments)
{
  if (*arguments == EOS)
  {
    string_t format = new_string_readable (unknown_format, NULL);

    printf ("unknown-format: %s\n", format);
    free_mem (&format);
  }
  else
  {
    string_t new_format = parse_word (&arguments);

    free_mem (&unknown_format);
    unknown_format = new_format;
  }
  
  parse_end (arguments);
}

LOCAL command_t unknown_format_option =
{
  "unknown-format", do_unknown_format_option,
  "Describe the format in which unknown forms will be printed.\n"
  "Usage: set unknown-format <unknown_format_string>\n"
  "The unknown format string may contain the following special sequences:\n"
  "  %s -- surface\n"
  "  %l -- input line number\n"
  "  %n -- number of analysis states\n"
};

/*===========================================================================*/

LOCAL string_t morphology_file, syntax_file, lexicon_file;
LOCAL string_t symbol_file, extended_symbol_file;

/*---------------------------------------------------------------------------*/

/* Read the project file. */

LOCAL void read_project_file (string_t project_file)
{
  FILE *project_stream;
  
  project_stream = open_stream (project_file, "r");
  while (! feof (project_stream)) 
  {
    string_t project_line, project_line_ptr, argument, extension, path;
    string_t *file_name;
    
    project_line = read_line (project_stream);
    project_line_ptr = project_line;
    
    if (*project_line_ptr != EOS)
    {
      argument = parse_word (&project_line_ptr);
      
      extension = NULL;
      file_name = NULL;
      if (strcmp_no_case (argument, "sym:") == 0) 
      {
	file_name = &symbol_file;
	extension = "sym_c";
      } 
      if (strcmp_no_case (argument, "esym:") == 0) 
      {
	file_name = &extended_symbol_file;
	extension = "esym_c";
      } 
      else if (strcmp_no_case (argument, "lex:") == 0) 
      {
	file_name = &lexicon_file;
	extension = "lex_c";
      } 
      else if (strcmp_no_case (argument, "mor:") == 0) 
      {
	file_name = &morphology_file;
	extension = "mor_c";
      }
      else if (strcmp_no_case (argument, "syn:") == 0)
      {
	file_name = &syntax_file;
	extension = "syn_c";
      }
      else if (strcmp_no_case (argument, "include:") == 0)
      {
	string_t include_file = parse_word (&project_line_ptr);
	string_t include_path = absolute_path (include_file, project_file);
	
	parse_end (project_line_ptr);
	read_project_file (include_path);
	free_mem (&include_path);
	free_mem (&include_file);
      }
      else if (strcmp_no_case (argument, "morinfo:") == 0
	       || strcmp_no_case (argument, "syninfo:") == 0)
      {
	grammar_t grammar = ((strcmp_no_case (argument, "morinfo:") == 0) ?
			     MORPHOLOGY : SYNTAX);
	
	if (info[grammar] == NULL)
	  info[grammar] = concat_strings (project_line_ptr, "\n", NULL);
	else
	{
	  string_t old_info = info[grammar];
	  
	  info[grammar] = concat_strings (info[grammar], project_line_ptr, 
					  "\n", NULL);
	  free_mem (&old_info);
	}
      }
      
      free_mem (&argument);
      
      if (file_name != NULL && *file_name == NULL && *project_line_ptr != EOS) 
      {
	argument = parse_word (&project_line_ptr);
	path = absolute_path (argument, project_file);
	free_mem (&argument);
	set_file_name (file_name, path, extension);
	free_mem (&path);
      }
      
    }
    free_mem (&project_line);
  }
  close_stream (&project_stream, project_file);
}

/*---------------------------------------------------------------------------*/

LOCAL command_t *malaga_options[] = 
/* the commands that can be called on startup, in alphabetical order */
{
  &alias_option, &cache_size_option, &display_option,
#ifdef HANGUL
  &hangul_option,
#endif
  &hidden_option, &mor_out_filter_option, &output_option,
  &output_format_option, &pruning_option, &result_option, &robust_option, 
  &sort_records_option, &switch_option, &syn_in_filter_option, 
  &syn_out_filter_option, &transmit_option, &tree_option, 
  &unknown_format_option, &variables_option,
  NULL
};

/*---------------------------------------------------------------------------*/

GLOBAL void init_malaga (string_t project_file)
/* Initialise this module. */
{
  string_t malagarc_name;

#ifdef HANGUL
  init_hangul ();
#endif

  if (! has_extension (project_file, "pro"))
    error ("project file \"%s\" must have extension \"pro\"", project_file);

  read_project_file (project_file);

  if (morphology_file == NULL)
    error ("missing morphology rules");

  if (lexicon_file == NULL)
    error ("missing lexicon");

  if (symbol_file == NULL)
    error ("missing symbol file");

  init_values ();

  if (extended_symbol_file != NULL)
    init_symbols (extended_symbol_file);
  else
    init_symbols (symbol_file);

  init_lexicon (lexicon_file);

  init_transmit ();
  init_analysis (morphology_file, syntax_file);

  output_format = new_string ("%l: %s: %c", NULL);
  unknown_format = new_string ("%l: %s: unknown", NULL);
  show_output = TRUE;
  show_tree = show_result = FALSE;

  options = malaga_options;
  execute_set_commands (project_file, "malaga:");
  malagarc_name = absolute_path ("~/.malagarc", NULL);
  if (file_exists (malagarc_name))
    execute_set_commands (malagarc_name, "malaga:");
  free_mem (&malagarc_name);
}

/*---------------------------------------------------------------------------*/

GLOBAL void terminate_malaga (void)
/* Terminate this module. */
{
  free_mem (&unknown_format);
  free_mem (&output_format);

  terminate_analysis ();

  free_mem (&syntax_file);
  free_mem (&morphology_file);
  terminate_transmit ();
  terminate_lexicon ();
  free_mem (&lexicon_file);
  terminate_symbols ();
  free_mem (&extended_symbol_file);
  free_mem (&symbol_file);
  terminate_values ();
  free_mem (&info[MORPHOLOGY]);
  free_mem (&info[SYNTAX]);
#ifdef HANGUL
  terminate_hangul ();
#endif
}

/* end of file ==============================================================*/
