/***************************************************************************
 *
 * This file is covered by a dual licence. You can choose whether you
 * want to use it according to the terms of the GNU GPL version 2, or
 * under the terms of Zorp Professional Firewall System EULA located
 * on the Zorp installation CD.
 *
 * $Id: registry.c,v 1.11 2004/08/18 11:46:46 bazsi Exp $
 *
 * Author  : Bazsi
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/registry.h>
#include <zorp/log.h>

#include <glib.h>
#include <string.h>

static GHashTable *registry[MAX_REGISTRY_TYPE];

#define MAX_REGISTRY_NAME 32

/*
 * An entry in the registry. The registry is a hash table indexed by a
 * string and storing an opaque pointer and an integer type
 * value. 
 *
 * ZRegistryEntry is an entry in the registry hash.
 */
typedef struct _ZRegistryEntry
{
  gint type;
  gchar name[MAX_REGISTRY_NAME];
  gpointer value;
} ZRegistryEntry;

/**
 * z_registry_init:
 *
 * Initialize the registry.
 **/
void
z_registry_init(void)
{
  int i;
  for (i = 0; i < MAX_REGISTRY_TYPE; i++)
    registry[i] = g_hash_table_new(g_str_hash, g_str_equal);
}


/**
 * z_registry_destroy:
 *
 * Deinitialize the registry subsystem.
 **/
void
z_registry_destroy(void)
{
  int i;
  for (i = 0; i<MAX_REGISTRY_TYPE; i++)
    g_hash_table_destroy(registry[i]);
}

/**
 * z_registry_add:
 * @name: the key of this entry
 * @type: the type of this entry
 * @value: the pointer associated with this entry
 * 
 * Add an entry to the registry with the given name and type.
 **/
void
z_registry_add(const gchar *name, gint type, gpointer value)
{
  ZRegistryEntry *ze = g_new0(ZRegistryEntry, 1);
  
  if (type < 0 || type > MAX_REGISTRY_TYPE)
    {
      /*LOG
        This message indicates that an internal error occurred,
        a buggy/incompatible loadable module wanted to register
        an unsupported module type. Please report this event to the 
	 Balabit QA Team (devel@balabit.com).
       */
      z_log(NULL, CORE_ERROR, 0, "Internal error, bad registry type; name='%s', type='%d'", name, type);
      return;
    }
  g_strlcpy(ze->name, name, sizeof(ze->name));
  ze->value = value;
  ze->type = type;
  g_hash_table_insert(registry[type], ze->name, ze);
}

/**
 * z_registry_get_one:
 * @name: name of the entry to fetch
 * @type: pointer to an int, where the type of the entry is stored.
 *
 * Fetch an item from the registry with the given name returning its
 * type and pointer.
 *
 * Returns: NULL if the entry was not found, the associated pointer otherwise
 **/
ZRegistryEntry *
z_registry_get_one(const gchar *name, gint type)
{
  ZRegistryEntry *ze = NULL;
  
  z_enter();
  ze = g_hash_table_lookup(registry[type], name);
  z_leave();
  return ze;
}

/**
 * z_registry_get:
 * @name: name of the entry to fetch
 * @type: contains the preferred entry type on input, contains the real type on output
 *
 * This function returns an entry from the registry autoprobing for
 * different types. If @type is NULL or the value pointed by @type is 0,
 * then each possible entry type is checked, otherwise only the value
 * specified will be used.
 *
 * Returns: the value stored in the registry
 **/
gpointer
z_registry_get(const gchar *name, gint *type)
{
  int i;
  ZRegistryEntry *ze = NULL;
  
  z_enter();
  if (type && (*type > MAX_REGISTRY_TYPE || *type < 0))
    {
      z_leave();
      return NULL;
    }
  if (type == NULL || *type == 0)
    {
      for (i = 0; i < MAX_REGISTRY_TYPE && ze == NULL; i++)
        {
          ze = z_registry_get_one(name, i);
        }
    }
  else
    {
      ze = z_registry_get_one(name, *type);
    }
    
  if (ze)
    {
      if (type) 
        *type = ze->type;
      z_leave();
      return ze->value;
    }
  z_leave();
  return NULL;    
}

/**
 * z_registry_has_key:
 * @name: name to search for
 *
 * This function checks whether the given name is found in the registry at
 * all, and returns the accompanying type value if found.
 **/
guint
z_registry_has_key(const gchar *name)
{
  int i;
  ZRegistryEntry *ze = NULL;
  
  for (i = 0; (i < MAX_REGISTRY_TYPE) && (ze == NULL); i++)
    {
      ze = z_registry_get_one(name, i);
    }
  if (ze)
    {
      return i;
    }
  else
    {
      return 0;
    }
}

/**
 * z_registry_foreach:
 * @type: type of entries to iterate over
 * @func: function to call for elements
 * @user_data: pointer to be passed to @func
 * 
 * This function iterates over the set of registry entries having the
 * type @type.
 **/
void
z_registry_foreach(gint type, GHFunc func, gpointer user_data)
{
  g_hash_table_foreach(registry[type], func, user_data); 
}
