/***************************************************************************
 *
 * This file is covered by a dual licence. You can choose whether you
 * want to use it according to the terms of the GNU GPL version 2, or
 * under the terms of Zorp Professional Firewall System EULA located
 * on the Zorp installation CD.
 *
 * $Id: parser.c,v 1.37 2004/05/22 14:04:16 bazsi Exp $
 *
 *
 * Author  : Chaoron
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/parser.h>
#include <zorp/log.h>
#include <string.h>
#include <errno.h>

#include <glib.h>

#include <stdlib.h>
#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif

#ifdef G_OS_WIN32

  //TODO MASHOVA
#  include <io.h>
#define open _open
#define close _close
#define read _read

#endif

#include <fcntl.h>
 
#define Z_PARSER_TAG_FOUND 0x1000


typedef struct _ZParserSection
{
  GHashTable *hash;
  gchar *type;
  gchar *name;
} ZParserSection;

struct _ZParser
{
  GStaticMutex lock;
  gint ref_cnt;
  GHashTable *hash;
  GMarkupParseContext *context;
  gchar *configname;
  GSList *tagstack;
  ZParserSection *actualsect;
  gchar *actualconf;
  GHashTable *taglist;
};

typedef struct _ZParserCalldata
{
  ZParser *parser;
  gchar *section;
  gchar *tag;
  ZParserCallback callback;
  gpointer user_data;
} ZParserCalldata;


static void z_parser_hash_destroy_func(gpointer data);
static void z_parser_value_destroy_func(gpointer data);

static void z_parser_start_tag(GMarkupParseContext *context,
                   const gchar *element_name, const gchar **attribute_names, const gchar **attribute_values,
                   gpointer user_data, GError **error);
static void z_parser_end_tag(GMarkupParseContext *context,
                   const gchar *element_name,
                   gpointer user_data, GError **error);
static void z_parser_text(GMarkupParseContext *context,
                   const gchar *text, gsize text_len,
                   gpointer user_data, GError **error);

static gboolean z_parser_tag_set_value(GValue *value, const gchar *text, gsize text_len);

static void z_parser_lookup_type_foreach(gpointer key, gpointer value, gpointer user_data);
static void z_parser_lookup_tag_foreach(gpointer key, gpointer value, gpointer user_data);


static GMarkupParser parser_funcs =
{
  z_parser_start_tag,
  z_parser_end_tag,
  z_parser_text,
  NULL,
  NULL
};


/**
 * z_parser_new:
 * @configname: configuration section to look for
 * @taglist: list of tags to validate against
 *
 * ZParser is a simple configuration file parser. A configuration file may
 * contain the configuration for several programs each in its own
 * configuration tag. This function constructs a ZParser and returns the
 * constructed instance.
 *
 * Returns ZParser instance
 */
ZParser *
z_parser_new(const gchar *configname, ZParserTag *taglist)
{
  ZParser *self = g_new0(ZParser, 1);
  static gboolean inited = FALSE;
  gint i;
  
  z_enter();
  
  if (!inited)
    {
      g_type_init();
      inited = TRUE;
    }
  
  self->ref_cnt = 1;
  if (taglist != NULL)
    {
      self->taglist = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL);
      i = 0;
      while (taglist[i].name != NULL)
        {
          if (taglist[i].section == NULL)
            g_hash_table_insert(self->taglist, g_strdup(taglist[i].name), &taglist[i]);
          else
            g_hash_table_insert(self->taglist, g_strconcat(taglist[i].section, "___", taglist[i].name, NULL), &taglist[i]);
          i++;
        }
    }
  
  self->configname = g_strdup(configname);
  self->context = g_markup_parse_context_new(&parser_funcs, 0, self, NULL);
  self->hash = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, z_parser_hash_destroy_func);
  
  z_leave();
  return self;
}

/**
 * z_parser_free:
 * @s: ZParser instance
 *
 * Destroy the specified ZParser instance.
 *
 **/
static void
z_parser_free(ZParser *s)
{
  ZParser *self = (ZParser *) s;
  
  z_enter();
  
  g_return_if_fail(s != NULL);
  
  g_hash_table_destroy((GHashTable *) self->hash);
  
  if (self->tagstack != NULL)
    g_slist_free(self->tagstack);
  
  if (self->taglist != NULL)
    g_hash_table_destroy(self->taglist);
  
  if (self->configname != NULL)
    g_free(self->configname);

  g_free(self);
  
  z_leave();
}

/**
 * z_parser_ref:
 * @self: ZParser instance
 *
 * Increment the reference count for this ZParser
 **/
ZParser *
z_parser_ref(ZParser *self)
{
  g_static_mutex_lock(&self->lock);
  g_assert(self->ref_cnt > 0);
  self->ref_cnt++;
  g_static_mutex_unlock(&self->lock);
  return self;
}

/**
 * z_parser_ref:
 * @self: ZParser instance
 *
 * Decrement the reference count for this ZParser, and free it if it reaches
 * zero.
 **/
void
z_parser_unref(ZParser *self)
{
  g_static_mutex_lock(&self->lock);
  g_assert(self->ref_cnt > 0);
  if (--self->ref_cnt == 0)
    {
      g_static_mutex_unlock(&self->lock);
      z_parser_free(self);
      return;
    }
  g_static_mutex_unlock(&self->lock);
}

/**
 * z_parser_check:
 * @parser: ZParser instance
 * @taglist: taglist to validate against
 *
 * Validate parser for a taglist
 *
 * Returns: ???
 */
gchar *
z_parser_check(ZParser *parser, ZParserTag *taglist)
{
  gint i = 0;
  GValue *value;
  gchar *result = NULL;

  z_enter();
  g_return_val_if_fail(parser != NULL, FALSE);
  
  while (taglist[i].name != NULL && result == NULL)
    {
      if (taglist[i].section == NULL)
        {
          i++;
          continue;
        }
      
      value = z_parser_lookup(parser, taglist[i].section, taglist[i].name);
      if (value == NULL && taglist[i].required)
        result = taglist[i].name;
      else if (value && !G_VALUE_HOLDS(value, taglist[i].type))
        result = taglist[i].name;
      
      i++;
    }
  
  z_leave();
  return result;
}


/**
 * z_parser_read_file:
 * @s: ZParser instance
 * @filename: file to read
 * @error: return error condition here
 *
 * FIXME: File parser
 *
 * Returns: whether parsing was successful
 **/
gboolean
z_parser_read_file(ZParser *s, const gchar *filename, GError **error)
{
  ZParser *self = (ZParser *) s;
  int fd;
  gchar buff[256];
  gint readlen;
  gboolean result = TRUE;
  
  z_enter();
  
  g_return_val_if_fail(error == NULL || *error == NULL, FALSE);
  g_return_val_if_fail(s != NULL, FALSE);
  
  fd = open(filename, O_RDONLY);
  if (fd == -1)
    {
      /*LOG
        This message indicates that the given configuration file can not be opened
        for the given reason. It is likely that the given file does not exists, or
        the program is not allowed to read it.
       */
      z_log(NULL, CORE_ERROR, 0, "Could not open configuration file; file='%s', error='%s'", filename, g_strerror(errno));
      g_markup_parse_context_end_parse(self->context, error);
      
      z_leave();
      return FALSE;
    }
  
  while ((readlen = read(fd, &buff, 256)) > 0 && result)
    {
      result = g_markup_parse_context_parse(self->context, buff, readlen, error);
    }
  if (readlen == -1)
    {
      /*LOG
        This message indicates that an error occurred during read of the given
        configuration file.
       */
      z_log(NULL, CORE_ERROR, 2, "Error reading configuration file; file='%s', error='%s'", filename, g_strerror(errno));
      close(fd);
      g_markup_parse_context_end_parse(self->context, error);
      
      z_leave();
      return FALSE;
    }
  close(fd);
  g_markup_parse_context_end_parse(self->context, error);
  
  z_leave();
  return result;
}


/**
 * z_parser_lookup:
 * @s: ZParser instance
 * @sectionname: section to look into
 * @tag: name of the tag to look up
 *
 * Value lookup
 *
 * Returns: ?
 **/
GValue *
z_parser_lookup(ZParser *s, gchar *sectionname, gchar *tag)
{
  ZParser *self = (ZParser *) s;
  ZParserSection *section;
  GValue *data = NULL;
  
  z_enter();
  
  g_return_val_if_fail(s != NULL, NULL);
  
  section = g_hash_table_lookup(self->hash, sectionname);
  if (section != NULL)
    {
      data = g_hash_table_lookup(section->hash, tag);
    }
  
  z_leave();
  return data;
}


/*
 * Type lookup
 */
void
z_parser_foreach_type(ZParser *s, gchar *sectiontype, ZParserCallback callback, gpointer user_data)
{
  ZParser *self = (ZParser *) s;
  ZParserCalldata calldata = { s, sectiontype, NULL, callback, user_data };
  
  g_return_if_fail(s != NULL);
  
  z_enter();
  
  g_hash_table_foreach(self->hash, z_parser_lookup_type_foreach, &calldata);
  
  z_leave();
}


/*
 * Session lookup by tag and value
 */
gchar *
z_parser_find_tag_by_value(ZParser *s, gchar *tag, GValue *value)
{
  ZParser *self = (ZParser *) s;
  ZParserCalldata calldata = {s, NULL, tag, NULL, value};
  
  g_return_val_if_fail(s != NULL, NULL);
  
  g_hash_table_foreach(self->hash, z_parser_lookup_tag_foreach, &calldata);
  
  z_leave();
  return calldata.section;
}


/*
 * Hash destroy functions
 */
static void 
z_parser_hash_destroy_func(gpointer data)
{
  ZParserSection *section = (ZParserSection *) data;
  
  z_enter();
  
  g_hash_table_destroy(section->hash);
  if (section->type != NULL)
    g_free(section->type);
  
  z_leave();
}

static void 
z_parser_value_destroy_func(gpointer data)
{
  GValue *value = (GValue *) data;
  
  z_enter();
  
  g_value_unset(value);
  g_free(value);
  
  z_leave();
}


/*
 * Parser callbacks
 */
static void 
z_parser_start_tag(GMarkupParseContext *context G_GNUC_UNUSED,
                   const gchar *element_name,
                   const gchar **attribute_names, const gchar **attribute_values,
                   gpointer user_data,
                   GError **error)
{
  ZParser *self = (ZParser *) user_data;
  ZParserSection *section;
  gboolean has_attr;
  gchar *name = NULL;
  gchar *type = NULL;
  gint i;  
  
  z_enter();
  
  if (!strcmp(element_name, Z_PARSER_CONFIG_TAG))
    {
      i = 0;
      has_attr = FALSE;
      while (attribute_names[i] != NULL)
        {
          if (!strcmp(attribute_names[i], "name"))
            {
              if (self->actualconf != NULL)
                {
                  g_free(self->actualconf);
                  self->actualconf = NULL;
                }
              if (!strcmp(attribute_values[i], Z_PARSER_DEFAULT_CONFIG) ||
                  (self->configname != NULL && !strcmp(attribute_values[i], self->configname)))
                {
                   self->actualconf = g_strdup(attribute_values[i]);
		   /*LOG
		     This message reports that the parser is reading the given
		     configuration block.
		    */
                   z_log(NULL, CORE_DEBUG, 7, "Parsing configuration block; name='%s'", attribute_values[i]);
                }
              has_attr = TRUE;
            }
          i++;
        }
      if (!has_attr)
        {
          g_set_error(error, G_MARKUP_ERROR, G_MARKUP_ERROR_INVALID_CONTENT, "Configuration tag has no name attribute.");
        }
    }
  else if (!strcmp(element_name, Z_PARSER_SECTION_TAG) && self->actualconf != NULL)
    {
      i = 0;
      has_attr = FALSE;
      while (attribute_names[i] != NULL)
        {
          if (!strcmp(attribute_names[i], "name"))
            {
              has_attr = TRUE;
              name = g_strdup(attribute_values[i]);
            }
          else if (!strcmp(attribute_names[i], "type"))
            {
              type = g_strdup(attribute_values[i]);
            }
          i++;
        }
      if (has_attr)
        {
          section = g_hash_table_lookup(self->hash, name);
          if (section == NULL)
            {
              section = g_new0(ZParserSection, 1);
              section->name = name;
              section->hash = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, z_parser_value_destroy_func);
              g_hash_table_insert(self->hash, name, section);
              /*LOG
                This message reports that the parser is reading the given
                configuration section.
               */
              z_log(NULL, CORE_DEBUG, 7, "Parsing configuration section; name='%s'", name);
            }
          else
            {
              g_free(name);
            }
          section->type = type;
          self->actualsect = section;
        }
      else
        {
          g_set_error(error, G_MARKUP_ERROR, G_MARKUP_ERROR_INVALID_CONTENT, "Section tag has no name attribute.");
        }
    }
  else if (self->actualconf != NULL)
    {
      self->tagstack = g_slist_prepend(self->tagstack, g_strdup(element_name));
    }

  z_leave();
}

static void
z_parser_end_tag(GMarkupParseContext *context G_GNUC_UNUSED,
                   const gchar *element_name,
                   gpointer user_data, GError **error G_GNUC_UNUSED)
{
  ZParser *self = (ZParser *) user_data;
  GSList *item = self->tagstack;
  
  z_enter();
  
  if (!strcmp(element_name, Z_PARSER_CONFIG_TAG))
    {
      if (self->actualconf != NULL)
        {
          g_free(self->actualconf);
        }
      self->actualconf = NULL;
    }
  else if (!strcmp(element_name, Z_PARSER_SECTION_TAG))
    {
      self->actualsect = NULL;
    }
  else if (self->actualconf != NULL)
    {
      self->tagstack = g_slist_next(self->tagstack);
      item->next = NULL;
      g_slist_free(item);
    }
  
  z_leave();
}

static void
z_parser_text(GMarkupParseContext *context G_GNUC_UNUSED,
                   const gchar *text, gsize text_len,
                   gpointer user_data, GError **error)
{
  ZParser *self = (ZParser *) user_data;
  gchar *name, *key;
  GValue *data;
  GValue *value;
  ZParserTag *tag;
  GType tagtype = G_TYPE_STRING;
  
  z_enter();
  
  if (self->actualconf != NULL && self->actualsect != NULL && self->tagstack != NULL)
    {
      name = (gchar *) self->tagstack->data;
      data = g_hash_table_lookup(self->actualsect->hash, name);
      
      key = g_strconcat(self->actualsect->name, "___", name, NULL);
      if (self->taglist != NULL)
        {
          tag = g_hash_table_lookup(self->taglist, key);
          if (tag == NULL)
            tag = g_hash_table_lookup(self->taglist, name);
          
          if (tag)
            tagtype= tag->type;
        }
      g_free(key);
      
      if (data != NULL && strcmp(self->actualconf, Z_PARSER_DEFAULT_CONFIG))
        {
          value = g_new0(GValue, 1);
          value = g_value_init(value, tagtype);
          if (z_parser_tag_set_value(value, text, text_len))
            {
              g_hash_table_remove(self->actualsect->hash, name);
              g_hash_table_insert(self->actualsect->hash, name, value);
            }
          else
            {
              g_set_error(error, G_MARKUP_ERROR, G_MARKUP_ERROR_INVALID_CONTENT, "Tag has invalid type; tag='%s'", name);
              g_value_unset(value);
              g_free(value);
            }
        }
      else if (data == NULL)
        {
          value = g_new0(GValue, 1);
          value = g_value_init(value, tagtype);
          if (z_parser_tag_set_value(value, text, text_len))
            {
	      /*LOG
	        This message reports that the parser is reading the given
		section tag.
	       */
              z_log(NULL, CORE_DEBUG, 7, "Inserting section tag; name='%s', value='%.*s'", name, (gint) text_len, text);
              g_hash_table_insert(self->actualsect->hash, name, value);
            }
          else
            {
              g_set_error(error, G_MARKUP_ERROR, G_MARKUP_ERROR_INVALID_CONTENT, "Tag has invalid type; tag='%s'", name);
              g_value_unset(value);
              g_free(value);
            }
        }
    }
  
  z_leave();
}

static gboolean
z_parser_tag_set_value(GValue *value, const gchar *text, gsize text_len)
{
  gboolean result;
  gchar *data = g_strndup(text, text_len);
  
  z_enter();
  
  switch (G_VALUE_TYPE(value))
    {
      case G_TYPE_INT:
        g_value_set_int(value, atoi(data));
        result = TRUE;
        break;
      
      case G_TYPE_LONG:
        g_value_set_int(value, atol(data));
        result = TRUE;
        break;
      
      case G_TYPE_FLOAT:
        g_value_set_float(value, (gfloat)atof(data));
        result = TRUE;
        break;
      
      case G_TYPE_STRING:
        g_value_set_string(value, data);
        result = TRUE;
        break;
      
      case G_TYPE_BOOLEAN:
        g_value_set_boolean(value, !strcmp(data, "yes") || !strcmp(data, "true") || !strcmp(data, "1"));
        result = TRUE;
        break;
        
      default:
        /*LOG
          This message indicates that the parser is called with wrong parameters.
          Please report this error to the Zorp QA Team.
         */
        z_log(NULL, CORE_ERROR, 4, "Internal error; error='Wrong GValue Type'");
        result = FALSE;
    }
  g_free(data);
  
  z_leave();
  return result;
}


/*
 * Callback on each section for type lookup
 */
static void
z_parser_lookup_type_foreach(gpointer key, gpointer value, gpointer user_data)
{
  ZParserCalldata *data = (ZParserCalldata *) user_data;
  ZParserSection *section;
  
  z_enter();
  
  section = (ZParserSection *) value;
  if (section->type != NULL && !strcmp(section->type, data->section))
    {
      data->callback(data->parser, (gchar *) key, data->user_data);
    }
  
  z_leave();
}

/*
 * Callback on each section for tag lookup
 */
static void
z_parser_lookup_tag_foreach(gpointer key, gpointer value, gpointer user_data)
{
  ZParserCalldata *calldata = (ZParserCalldata *) user_data;
  ZParserSection *section;
  GValue *data;
  gboolean found = FALSE;
  
  z_enter();
  
  section = (ZParserSection *) value;
  data = g_hash_table_lookup(section->hash, calldata->tag);
  if (data != NULL && G_VALUE_TYPE(data) == G_VALUE_TYPE((GValue *) calldata->user_data))
    {
      switch (G_VALUE_TYPE(data))
        {
          case G_TYPE_INT:
            found = (g_value_get_int(data) == g_value_get_int((GValue *) calldata->user_data));
            break;
            
          case G_TYPE_LONG:
            found = (g_value_get_long(data) == g_value_get_long((GValue *) calldata->user_data));
           break;
            
          case G_TYPE_FLOAT:
            found = (g_value_get_float(data) == g_value_get_float((GValue *) calldata->user_data));
            break;
            
          case G_TYPE_STRING:
            found = !strcmp(g_value_get_string(data), g_value_get_string((GValue *) calldata->user_data));
            break;
            
          case G_TYPE_BOOLEAN:
            found = (!g_value_get_boolean(data) == !g_value_get_boolean((GValue *) calldata->user_data));
            break;
          
          default:
            /*LOG
              This message indicates that the parser is experiencing an internal error.
              Please report this error to the Zorp QA Team.
             */
            z_log(NULL, CORE_ERROR, 4, "Internal error; error='Inconsystency in parser'");
        }
    }
  if (found)
    {
      calldata->section = (gchar *) key;
    }
  
  z_leave();
}

/* helper functions for lookup */
gboolean 
z_parser_get_value(ZParser *s, gchar *section, gchar *tag, guint value_type, GValue **value)
{
  GValue *v_value;
  
  v_value = z_parser_lookup(s, section, tag);
  
  if (v_value && G_VALUE_TYPE(v_value) == value_type)
    {
      *value = v_value;
      return TRUE;
    }
  else
    *value = NULL;
  return FALSE;
}

gboolean
z_parser_get_boolean(ZParser *s, gchar *section, gchar *tag, gboolean *value)
{
  GValue *v_value;

  if (z_parser_get_value(s, section, tag, G_TYPE_BOOLEAN, &v_value))
    {
      *value = g_value_get_boolean(v_value);
      return TRUE;
    }
  return FALSE;
}

gboolean
z_parser_get_int(ZParser *s, gchar *section, gchar *tag, gint *value)
{
  GValue *v_value;

  if (z_parser_get_value(s, section, tag, G_TYPE_INT, &v_value))
    {
      *value = g_value_get_int(v_value);
      return TRUE;
    }
  return FALSE;
}

gboolean
z_parser_get_string(ZParser *s, gchar *section, gchar *tag, const gchar **value)
{
  GValue *v_value;

  if (z_parser_get_value(s, section, tag, G_TYPE_STRING, &v_value))
    {
      *value = g_value_get_string(v_value);
      return TRUE;
    }
  return FALSE;
}
