/*
 * libSpiff - XSPF playlist handling library
 *
 * Copyright (C) 2007, Sebastian Pipping / Xiph.Org Foundation
 * All rights reserved.
 *
 * Redistribution  and use in source and binary forms, with or without
 * modification,  are permitted provided that the following conditions
 * are met:
 *
 *     * Redistributions   of  source  code  must  retain  the   above
 *       copyright  notice, this list of conditions and the  following
 *       disclaimer.
 *
 *     * Redistributions  in  binary  form must  reproduce  the  above
 *       copyright  notice, this list of conditions and the  following
 *       disclaimer   in  the  documentation  and/or  other  materials
 *       provided with the distribution.
 *
 *     * Neither  the name of the Xiph.Org Foundation nor the names of
 *       its  contributors may be used to endorse or promote  products
 *       derived  from  this software without specific  prior  written
 *       permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS  IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT  NOT
 * LIMITED  TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS
 * FOR  A  PARTICULAR  PURPOSE ARE DISCLAIMED. IN NO EVENT  SHALL  THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL,    SPECIAL,   EXEMPLARY,   OR   CONSEQUENTIAL   DAMAGES
 * (INCLUDING,  BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES;  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT  LIABILITY,  OR  TORT (INCLUDING  NEGLIGENCE  OR  OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Sebastian Pipping, sping@xiph.org
 */

/**
 * @file SpiffExtensionReaderFactory.h
 * Interface of SpiffExtensionReaderFactory.
 */

#ifndef SPIFF_EXTENSION_READER_FACTROY_H
#define SPIFF_EXTENSION_READER_FACTROY_H



#include "SpiffDefines.h"
#include "SpiffToolbox.h"
#include <map>

namespace Spiff {



class SpiffExtensionReader;
class SpiffReader;
class SpiffExtensionReaderFactoryPrivate;



class SpiffExtensionReaderFactory {

private:
	/// @cond DOXYGEN_NON_API
	SpiffExtensionReaderFactoryPrivate * const d; ///< D pointer
	/// @endcond

protected:
	std::map<const XML_Char *, const SpiffExtensionReader *, Toolbox::SpiffStringCompare> playlistExtensionReaders; ///< Map of extension readers for playlist extensions
	std::map<const XML_Char *, const SpiffExtensionReader *, Toolbox::SpiffStringCompare> trackExtensionReaders; ///< Map of extension readers for track extensions
	const SpiffExtensionReader * playlistCatchAllReader; ///< Catch-all extension reader for playlist extensions
	const SpiffExtensionReader * trackCatchAllReader; ///< Catch-all extension reader for track extensions

public:
	/**
	 * Creates a new SpiffExtensionReaderFactory object.
	 */
	SpiffExtensionReaderFactory();

	/**
	 * Copy constructor.
	 *
	 * @param source  Source to copy from
	 */
	SpiffExtensionReaderFactory(const SpiffExtensionReaderFactory
			& source);

	/**
	 * Assignment operator.
	 *
	 * @param source  Source to copy from
	 */
	SpiffExtensionReaderFactory & operator=(
			const SpiffExtensionReaderFactory & source);

	/**
	 * Destroys this SpiffExtensionReaderFactory object and deletes all
	 * memory associated with it.
	 */
	~SpiffExtensionReaderFactory();

	/**
	 * Overwrites the registered reader for the given application URI.
	 * Pass NULL for the URI to make this the catch-all reader.
	 * The reader will be cloned internally so can safely delete
	 * the instance passed for registration.
	 *
	 * @param example		Representative for the extension reader cleass
	 * @param triggerUri	Application URI associate, must not be NULL
	 */
	void registerPlaylistExtensionReader(const SpiffExtensionReader * example,
			const XML_Char * triggerUri);

	/**
	 * Overwrites the registered reader for the given application URI.
	 * Pass NULL for the URI to make this the catch-all reader.
	 * The reader will be cloned internally so can safely delete
	 * the instance passed for registration.
	 *
	 * @param example		Representative for the extension reader cleass
	 * @param triggerUri	Application URI associate, must not be NULL
	 */
	void registerTrackExtensionReader(const SpiffExtensionReader * example,
			const XML_Char * triggerUri);

	/**
	 * Unregisteres the given application URI.
	 * NOTE: This URI will still be handled if a catch-all
	 * handler has been set.
	 *
	 * @param triggerUri	Application URI to unregister
	 */
	void unregisterPlaylistExtensionReader(
			const XML_Char * triggerUri);

	/**
	 * Unregisteres the given application URI.
	 * NOTE: This URI will still be handled if a catch-all
	 * handler has been set.
	 *
	 * @param triggerUri	Application URI to unregister
	 */
	void unregisterTrackExtensionReader(
			const XML_Char * triggerUri);

	/**
	 * Creates a new SpiffExtensionReader whose type was
	 * registered for this application URI.
	 *
	 * @param applicationUri	Application URI
	 * @param reader			SpiffReader for the extension reader
	 * @return					New playlist extension reader
	 */
	SpiffExtensionReader * newPlaylistExtensionReader(
			const XML_Char * applicationUri, SpiffReader * reader);

	/**
	 * Creates a new SpiffExtensionReader whose type was
	 * registered for this application URI.
	 *
	 * @param applicationUri	Application URI
	 * @param reader			SpiffReader for the extension reader
	 * @return					New track extension reader
	 */
	SpiffExtensionReader * newTrackExtensionReader(
			const XML_Char * applicationUri, SpiffReader * reader);

private:
	/**
	 * Overwrites the registered reader for the given application URI.
	 * Pass NULL for the URI to make this the catch-all reader.
	 * The reader will be cloned internally so can safely delete
	 * the instance passed for registration.
	 *
	 * @param container		Container to unregister from
	 * @param catchAll		Catch-all slot to modifiy
	 * @param example		Reader class representative
	 * @param triggerUri	Application URI to unregister
	 */
	void registerReader(std::map<const XML_Char *,
			const SpiffExtensionReader *,
			Toolbox::SpiffStringCompare> & container,
			const SpiffExtensionReader * & catchAll,
			const SpiffExtensionReader * example,
			const XML_Char * triggerUri);

	/**
	 * Unregisteres the given application URI.
	 *
	 * @param container		Container to unregister from
	 * @param catchAll		Catch-all slot to modifiy
	 * @param triggerUri	Application URI to unregister
	 */
	void unregisterReader(std::map<const XML_Char *,
			const SpiffExtensionReader *,
			Toolbox::SpiffStringCompare> & container,
			const SpiffExtensionReader * & catchAll,
			const XML_Char * triggerUri);

	/**
	 * Creates a new SpiffExtensionReader whose type was
	 * registered for this application URI.
	 *
	 * @param container			Container to use
	 * @param catchAll			Catch-all slot to use
	 * @param applicationUri	Application URI
	 * @param reader			SpiffReader for the extension reader
	 * @return					New extension reader
	 */
	SpiffExtensionReader * newReader(std::map<const XML_Char *,
			const SpiffExtensionReader *,
			Toolbox::SpiffStringCompare> & container,
			const SpiffExtensionReader * catchAll,
			const XML_Char * applicationUri,
			SpiffReader * reader);

	/**
	 * Deletes the extension reader and URI clones kept in a container.
	 * NOTE: The container itself is not deleted.
	 *
	 * @param container		Container to delete in
	 */
	static void freeMap(std::map<const XML_Char *, const SpiffExtensionReader *,
			Toolbox::SpiffStringCompare> & container);

	/**
	 * Copies the extension reader and URI clones kept in a container
	 * over to another container.
	 * NOTE: The destination container is <strong>not</strong>
	 * cleared before.
	 *
	 * @param dest    Container to copy into
	 * @param source  Container to copy from
	 */
	static void copyMap(std::map<const XML_Char *, const SpiffExtensionReader *,
		Toolbox::SpiffStringCompare> & dest,
		const std::map<const XML_Char *, const SpiffExtensionReader *,
		Toolbox::SpiffStringCompare> & source);

};



}

#endif // SPIFF_EXTENSION_READER_FACTROY_H
